require.config({"config": {
        "jsbuild":{"Magento_Ui/js/lib/logger/levels-pool.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore'\n], function (_) {\n    'use strict';\n\n    var LEVELS,\n        CODE_MAP;\n\n    LEVELS = {\n        NONE: 0,\n        ERROR: 1,\n        WARN: 2,\n        INFO: 3,\n        DEBUG: 4,\n        ALL: 5\n    };\n\n    CODE_MAP = _.invert(LEVELS);\n\n    return {\n        /**\n         * Returns the list of available log levels.\n         *\n         * @returns {Object}\n         */\n        getLevels: function () {\n            return LEVELS;\n        },\n\n        /**\n         * Returns name of the log level that matches to the provided code.\n         *\n         * @returns {String}\n         */\n        getNameByCode: function (code) {\n            return CODE_MAP[code];\n        }\n    };\n});\n","Magento_Ui/js/lib/logger/logger-utils.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    /**\n     * Utils methods for logger\n     * @param {Logger} logger\n     */\n    function LogUtils(logger) {\n        this.logger = logger;\n\n    }\n\n    /**\n     * Method for logging asynchronous operations\n     * @param {Promise} promise\n     * @param {Object} config\n     */\n    LogUtils.prototype.asyncLog = function (promise, config) {\n        var levels,\n            messages,\n            wait;\n\n        config = config || {};\n        levels = config.levels || this.createLevels();\n        messages = config.messages || this.createMessages();\n        wait = config.wait || 5000;\n\n        this.logger[levels.requested](messages.requested, config.data);\n        setTimeout(function () {\n            promise.state() === 'pending' ?\n                this.logger[levels.failed](messages.failed, config.data) :\n                this.logger[levels.loaded](messages.loaded, config.data);\n        }.bind(this), wait);\n    };\n\n    /**\n     * Method that creates object of messages\n     * @param {String} requested - log message that showing that request for class is started\n     * @param {String} loaded - log message that show when requested class is loaded\n     * @param {String} failed - log message that show when requested class is failed\n     * @returns {Object}\n     */\n    LogUtils.prototype.createMessages = function (requested, loaded, failed) {\n        return {\n            requested: requested || '',\n            loaded: loaded || '',\n            failed: failed || ''\n        };\n    };\n\n    /**\n     * Method that creates object of log levels\n     * @param {String} requested - log message that showing that request for class is started\n     * @param {String} loaded - log message that show when requested class is loaded\n     * @param {String} failed - log message that show when requested class is failed\n     * @returns {Object}\n     */\n    LogUtils.prototype.createLevels = function (requested, loaded, failed) {\n        return {\n            requested: requested || 'info',\n            loaded: loaded || 'info',\n            failed: failed || 'warn'\n        };\n    };\n\n    return LogUtils;\n});\n","Magento_Ui/js/lib/logger/entry-factory.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    './entry'\n], function (LogEntry) {\n    'use strict';\n\n    return {\n        /**\n         * @param {String} message\n         * @param {Number} level\n         * @param {Object} [messageData]\n         * @returns {LogEntry}\n         */\n        createEntry: function (message, level, messageData) {\n            return new LogEntry(message, level, messageData);\n        }\n    };\n});\n","Magento_Ui/js/lib/logger/entry.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    './levels-pool'\n], function (logLevels) {\n    'use strict';\n\n    /**\n     * @param {String} message\n     * @param {Number} level\n     * @param {Object} [data]\n     */\n    function LogEntry(message, level, data) {\n        /**\n         * @readonly\n         * @type {Number}\n         */\n        this.timestamp = Date.now();\n\n        /**\n         * @readonly\n         * @type {Number}\n         */\n        this.level = level;\n\n        /**\n         * @readonly\n         * @type {String}\n         */\n        this.levelName = logLevels.getNameByCode(level);\n\n        /**\n         * @readonly\n         * @type {Object}\n         */\n        this.data = data;\n\n        /**\n         * @readonly\n         * @type {String}\n         */\n        this.message = message;\n    }\n\n    return LogEntry;\n});\n","Magento_Ui/js/lib/logger/logger.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    './levels-pool'\n], function (logLevels) {\n    'use strict';\n\n    var levels = logLevels.getLevels();\n\n    /**\n     * @param {LogOutputHandler} outputHandler\n     * @param {LogEntryFactory} entryFactory\n     */\n    function Logger(outputHandler, entryFactory) {\n        /**\n         * An array of log entries.\n         *\n         * @protected\n         * @type {Array<LogEntry>}\n         */\n        this.entries_ = [];\n\n        /**\n         * Current display level.\n         *\n         * @protected\n         * @type {Number}\n         */\n        this.displayLevel_ = levels.ERROR;\n\n        /**\n         * An array of display criteria.\n         *\n         * @protected\n         * @type {Array<LogCriteria>}\n         */\n        this.displayCriteria_ = [];\n\n        /**\n         * @protected\n         * @type {LogEntryFactory}\n         */\n        this.entryFactory_ = entryFactory;\n\n        /**\n         * @protected\n         * @type {Array<LogOutputHandler>}\n         */\n        this.outputHandlers_ = [outputHandler];\n\n        this.addDisplayCriteria(this.matchesLevel_);\n    }\n\n    /**\n     * Swaps current display level with the provided one.\n     *\n     * @param {Number} level - Level's code.\n     */\n    Logger.prototype.setDisplayLevel = function (level) {\n        var levelName = logLevels.getNameByCode(level);\n\n        if (!levelName) {\n            throw new TypeError('The provided level is not defined in the levels list.');\n        }\n\n        this.displayLevel_ = level;\n    };\n\n    /**\n     * Sets up the criteria by which log entries will be filtered out from the output.\n     *\n     * @param {LogCriteria} criteria\n     */\n    Logger.prototype.addDisplayCriteria = function (criteria) {\n        this.displayCriteria_.push(criteria);\n    };\n\n    /**\n     * Removes previously defined criteria.\n     *\n     * @param {LogCriteria} criteria\n     */\n    Logger.prototype.removeDisplayCriteria = function (criteria) {\n        var index = this.displayCriteria_.indexOf(criteria);\n\n        if (~index) {\n            this.displayCriteria_.splice(index, 1);\n        }\n    };\n\n    /**\n     * @param {String} message\n     * @param {Object} [messageData]\n     * @returns {LogEntry}\n     */\n    Logger.prototype.error = function (message, messageData) {\n        return this.log_(message, levels.ERROR, messageData);\n    };\n\n    /**\n     * @param {String} message\n     * @param {Object} [messageData]\n     * @returns {LogEntry}\n     */\n    Logger.prototype.warn = function (message, messageData) {\n        return this.log_(message, levels.WARN, messageData);\n    };\n\n    /**\n     * @param {String} message\n     * @param {Object} [messageData]\n     * @returns {LogEntry}\n     */\n    Logger.prototype.info = function (message, messageData) {\n        return this.log_(message, levels.INFO, messageData);\n    };\n\n    /**\n     * @param {String} message\n     * @param {Object} [messageData]\n     * @returns {LogEntry}\n     */\n    Logger.prototype.debug = function (message, messageData) {\n        return this.log_(message, levels.DEBUG, messageData);\n    };\n\n    /**\n     * @protected\n     * @param {String} message\n     * @param {Number} level\n     * @param {Object} [messageData]\n     * @returns {LogEntry}\n     */\n    Logger.prototype.log_ = function (message, level, messageData) {\n        var entry = this.createEntry_(message, level, messageData);\n\n        this.entries_.push(entry);\n\n        if (this.matchesCriteria_(entry)) {\n            this.processOutput_(entry);\n        }\n\n        return entry;\n    };\n\n    /**\n     * @protected\n     * @param {String} message\n     * @param {Number} level\n     * @param {Object} [messageData]\n     * @returns {LogEntry}\n     */\n    Logger.prototype.createEntry_ = function (message, level, messageData) {\n        return this.entryFactory_.createEntry(message, level, messageData);\n    };\n\n    /**\n     * Returns an array of log entries that have been added to the logger.\n     *\n     * @param {LogCriteria} [criteria] - Optional filter criteria.\n     * @returns {Array<LogEntry>}\n     */\n    Logger.prototype.getEntries = function (criteria) {\n        if (criteria) {\n            return this.entries_.filter(criteria);\n        }\n\n        return this.entries_;\n    };\n\n    /**\n     * @param {LogCriteria} [criteria]\n     */\n    Logger.prototype.dump = function (criteria) {\n        var entries;\n\n        if (!criteria) {\n            criteria = this.matchesCriteria_;\n        }\n\n        entries = this.entries_.filter(criteria, this);\n\n        this.outputHandlers_.forEach(function (handler) {\n            handler.dump(entries);\n        });\n    };\n\n    /**\n     * @protected\n     * @param {LogEntry} entry\n     */\n    Logger.prototype.processOutput_ = function (entry) {\n        this.outputHandlers_.forEach(function (handler) {\n            handler.show(entry);\n        });\n    };\n\n    /**\n     * @protected\n     * @param {LogEntry} entry\n     * @returns {Boolean}\n     */\n    Logger.prototype.matchesCriteria_ = function (entry) {\n        return this.displayCriteria_.every(function (criteria) {\n            return criteria.call(this, entry);\n        }, this);\n    };\n\n    /**\n     * Checks that the level of provided entry passes the \"displayLevel_\" threshold.\n     *\n     * @protected\n     * @param {LogEntry} entry - Entry to be checked.\n     * @returns {Boolean}\n     */\n    Logger.prototype.matchesLevel_ = function (entry) {\n        return entry.level <= this.displayLevel_;\n    };\n\n    return Logger;\n});\n","Magento_Ui/js/lib/validation/rules.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'underscore',\n    './utils',\n    'moment',\n    'tinycolor',\n    'jquery/validate',\n    'mage/translate'\n], function ($, _, utils, moment, tinycolor) {\n    'use strict';\n\n    /**\n     * validate credit card number using mod10\n     * @param {String} s\n     * @return {Boolean}\n     */\n    function validateCreditCard(s) {\n        // remove non-numerics\n        var v = '0123456789',\n            w = '',\n            i, j, k, m, c, a, x;\n\n        for (i = 0; i < s.length; i++) {\n            x = s.charAt(i);\n\n            if (v.indexOf(x, 0) !== -1) {\n                w += x;\n            }\n        }\n        // validate number\n        j = w.length / 2;\n        k = Math.floor(j);\n        m = Math.ceil(j) - k;\n        c = 0;\n\n        for (i = 0; i < k; i++) {\n            a = w.charAt(i * 2 + m) * 2;\n            c += a > 9 ? Math.floor(a / 10 + a % 10) : a;\n        }\n\n        for (i = 0; i < k + m; i++) {\n            c += w.charAt(i * 2 + 1 - m) * 1;\n        }\n\n        return c % 10 === 0;\n    }\n\n    /**\n     * Collection of validation rules including rules from additional-methods.js\n     * @type {Object}\n     */\n    return _.mapObject({\n        'min_text_length': [\n            function (value, params) {\n                return _.isUndefined(value) || value.length === 0 || value.length >= +params;\n            },\n            $.mage.__('Please enter more or equal than {0} symbols.')\n        ],\n        'max_text_length': [\n            function (value, params) {\n                return !_.isUndefined(value) && value.length <= +params;\n            },\n            $.mage.__('Please enter less or equal than {0} symbols.')\n        ],\n        'max-words': [\n            function (value, params) {\n                return utils.isEmpty(value) || utils.stripHtml(value).match(/\\b\\w+\\b/g).length < params;\n            },\n            $.mage.__('Please enter {0} words or less.')\n        ],\n        'min-words': [\n            function (value, params) {\n                return utils.isEmpty(value) || utils.stripHtml(value).match(/\\b\\w+\\b/g).length >= params;\n            },\n            $.mage.__('Please enter at least {0} words.')\n        ],\n        'range-words': [\n            function (value, params) {\n                var match = utils.stripHtml(value).match(/\\b\\w+\\b/g) || [];\n\n                return utils.isEmpty(value) || match.length >= params[0] &&\n                    match.length <= params[1];\n            },\n            $.mage.__('Please enter between {0} and {1} words.')\n        ],\n        'letters-with-basic-punc': [\n            function (value) {\n                return utils.isEmpty(value) || /^[a-z\\-.,()\\u0027\\u0022\\s]+$/i.test(value);\n            },\n            $.mage.__('Letters or punctuation only please')\n        ],\n        'alphanumeric': [\n            function (value) {\n                return utils.isEmpty(value) || /^\\w+$/i.test(value);\n            },\n            $.mage.__('Letters, numbers, spaces or underscores only please')\n        ],\n        'letters-only': [\n            function (value) {\n                return utils.isEmpty(value) || /^[a-z]+$/i.test(value);\n            },\n            $.mage.__('Letters only please')\n        ],\n        'no-whitespace': [\n            function (value) {\n                return utils.isEmpty(value) || /^\\S+$/i.test(value);\n            },\n            $.mage.__('No white space please')\n        ],\n        'no-marginal-whitespace': [\n            function (value) {\n                return !/^\\s+|\\s+$/i.test(value);\n            },\n            $.mage.__('No marginal white space please')\n        ],\n        'zip-range': [\n            function (value) {\n                return utils.isEmpty(value) || /^90[2-5]-\\d{2}-\\d{4}$/.test(value);\n            },\n            $.mage.__('Your ZIP-code must be in the range 902xx-xxxx to 905-xx-xxxx')\n        ],\n        'integer': [\n            function (value) {\n                return utils.isEmpty(value) || /^-?\\d+$/.test(value);\n            },\n            $.mage.__('A positive or negative non-decimal number please')\n        ],\n        'vinUS': [\n            function (value) {\n                if (utils.isEmpty(value)) {\n                    return true;\n                }\n\n                if (value.length !== 17) {\n                    return false;\n                }\n                var i, n, d, f, cd, cdv,//eslint-disable-line vars-on-top\n                    LL = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'],//eslint-disable-line max-len\n                    VL = [1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 7, 9, 2, 3, 4, 5, 6, 7, 8, 9],\n                    FL = [8, 7, 6, 5, 4, 3, 2, 10, 0, 9, 8, 7, 6, 5, 4, 3, 2],\n                    rs = 0;\n\n                for (i = 0; i < 17; i++) {\n                    f = FL[i];\n                    d = value.slice(i, i + 1);\n\n                    if (i === 8) {\n                        cdv = d;\n                    }\n\n                    if (!isNaN(d)) {\n                        d *= f;\n                    } else {\n                        for (n = 0; n < LL.length; n++) {//eslint-disable-line max-depth\n                            if (d.toUpperCase() === LL[n]) {//eslint-disable-line max-depth\n                                d = VL[n];\n                                d *= f;\n\n                                if (isNaN(cdv) && n === 8) {//eslint-disable-line max-depth\n                                    cdv = LL[n];\n                                }\n                                break;\n                            }\n                        }\n                    }\n                    rs += d;\n                }\n                cd = rs % 11;\n\n                if (cd === 10) {\n                    cd = 'X';\n                }\n\n                if (cd === cdv) {\n                    return true;\n                }\n\n                return false;\n            },\n            $.mage.__('The specified vehicle identification number (VIN) is invalid.')\n        ],\n        'dateITA': [\n            function (value) {\n                var check = false,\n                    re = /^\\d{1,2}\\/\\d{1,2}\\/\\d{4}$/,\n                    adata, gg, mm, aaaa, xdata;\n\n                if (re.test(value)) {\n                    adata = value.split('/');\n                    gg = parseInt(adata[0], 10);\n                    mm = parseInt(adata[1], 10);\n                    aaaa = parseInt(adata[2], 10);\n                    xdata = new Date(aaaa, mm - 1, gg);\n\n                    if (xdata.getFullYear() === aaaa &&\n                        xdata.getMonth() === mm - 1 &&\n                        xdata.getDate() === gg\n                    ) {\n                        check = true;\n                    } else {\n                        check = false;\n                    }\n                } else {\n                    check = false;\n                }\n\n                return check;\n            },\n            $.mage.__('Please enter a correct date')\n        ],\n        'dateNL': [\n            function (value) {\n                return /^\\d\\d?[\\.\\/-]\\d\\d?[\\.\\/-]\\d\\d\\d?\\d?$/.test(value);\n            },\n            $.mage.__('Vul hier een geldige datum in.')\n        ],\n        'time': [\n            function (value) {\n                return utils.isEmpty(value) || /^([01]\\d|2[0-3])(:[0-5]\\d){0,2}$/.test(value);\n            },\n            $.mage.__('Please enter a valid time, between 00:00 and 23:59')\n        ],\n        'time12h': [\n            function (value) {\n                return utils.isEmpty(value) || /^((0?[1-9]|1[012])(:[0-5]\\d){0,2}(\\s[AP]M))$/i.test(value);\n            },\n            $.mage.__('Please enter a valid time, between 00:00 am and 12:00 pm')\n        ],\n        'phoneUS': [\n            function (value) {\n                value = value.replace(/\\s+/g, '');\n\n                return utils.isEmpty(value) || value.length > 9 &&\n                    value.match(/^(1-?)?(\\([2-9]\\d{2}\\)|[2-9]\\d{2})-?[2-9]\\d{2}-?\\d{4}$/);\n            },\n            $.mage.__('Please specify a valid phone number')\n        ],\n        'phoneUK': [\n            function (value) {\n                return utils.isEmpty(value) || value.length > 9 &&\n                    value.match(/^(\\(?(0|\\+44)[1-9]{1}\\d{1,4}?\\)?\\s?\\d{3,4}\\s?\\d{3,4})$/);\n            },\n            $.mage.__('Please specify a valid phone number')\n        ],\n        'mobileUK': [\n            function (value) {\n                return utils.isEmpty(value) || value.length > 9 && value.match(/^((0|\\+44)7\\d{3}\\s?\\d{6})$/);\n            },\n            $.mage.__('Please specify a valid mobile number')\n        ],\n        'stripped-min-length': [\n            function (value, param) {\n                return _.isUndefined(value) || value.length === 0 || utils.stripHtml(value).length >= param;\n            },\n            $.mage.__('Please enter at least {0} characters')\n        ],\n        'email2': [\n            function (value) {\n                return utils.isEmpty(value) || /^((([a-z]|\\d|[!#\\$%&\\u0027\\*\\+\\-\\/=\\?\\^_`{\\|}~]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z]|\\d|[!#\\$%&\\u0027\\*\\+\\-\\/=\\?\\^_`{\\|}~]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*)|((\\u0022)((((\\x20|\\x09)*(\\x0d\\x0a))?(\\x20|\\x09)+)?(([\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x7f]|\\x21|[\\x23-\\x5b]|[\\x5d-\\x7e]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(\\\\([\\x01-\\x09\\x0b\\x0c\\x0d-\\x7f]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]))))*(((\\x20|\\x09)*(\\x0d\\x0a))?(\\x20|\\x09)+)?(\\u0022)))@((([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.)*(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.?$/i.test(value);//eslint-disable-line max-len\n            },\n            $.validator.messages.email\n        ],\n        'url2': [\n            function (value) {\n                return utils.isEmpty(value) || /^(https?|ftp):\\/\\/(((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&\\u0027\\(\\)\\*\\+,;=]|:)*@)?(((\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5]))|((([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.)*(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.?)(:\\d*)?)(\\/((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&\\u0027\\(\\)\\*\\+,;=]|:|@)+(\\/(([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&\\u0027\\(\\)\\*\\+,;=]|:|@)*)*)?)?(\\?((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&\\u0027\\(\\)\\*\\+,;=]|:|@)|[\\uE000-\\uF8FF]|\\/|\\?)*)?(\\#((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&\\u0027\\(\\)\\*\\+,;=]|:|@)|\\/|\\?)*)?$/i.test(value);//eslint-disable-line max-len\n            },\n            $.validator.messages.url\n        ],\n        'credit-card-types': [\n            function (value, param) {\n                var validTypes;\n\n                if (utils.isEmpty(value)) {\n                    return true;\n                }\n\n                if (/[^0-9-]+/.test(value)) {\n                    return false;\n                }\n                value = value.replace(/\\D/g, '');\n                validTypes = 0x0000;\n\n                if (param.mastercard) {\n                    validTypes |= 0x0001;\n                }\n\n                if (param.visa) {\n                    validTypes |= 0x0002;\n                }\n\n                if (param.amex) {\n                    validTypes |= 0x0004;\n                }\n\n                if (param.dinersclub) {\n                    validTypes |= 0x0008;\n                }\n\n                if (param.enroute) {\n                    validTypes |= 0x0010;\n                }\n\n                if (param.discover) {\n                    validTypes |= 0x0020;\n                }\n\n                if (param.jcb) {\n                    validTypes |= 0x0040;\n                }\n\n                if (param.unknown) {\n                    validTypes |= 0x0080;\n                }\n\n                if (param.all) {\n                    validTypes = 0x0001 | 0x0002 | 0x0004 | 0x0008 | 0x0010 | 0x0020 | 0x0040 | 0x0080;\n                }\n\n                if (validTypes & 0x0001 && /^(51|52|53|54|55)/.test(value)) { //mastercard\n                    return value.length === 16;\n                }\n\n                if (validTypes & 0x0002 && /^(4)/.test(value)) { //visa\n                    return value.length === 16;\n                }\n\n                if (validTypes & 0x0004 && /^(34|37)/.test(value)) { //amex\n                    return value.length === 15;\n                }\n\n                if (validTypes & 0x0008 && /^(300|301|302|303|304|305|36|38)/.test(value)) { //dinersclub\n                    return value.length === 14;\n                }\n\n                if (validTypes & 0x0010 && /^(2014|2149)/.test(value)) { //enroute\n                    return value.length === 15;\n                }\n\n                if (validTypes & 0x0020 && /^(6011)/.test(value)) { //discover\n                    return value.length === 16;\n                }\n\n                if (validTypes & 0x0040 && /^(3)/.test(value)) { //jcb\n                    return value.length === 16;\n                }\n\n                if (validTypes & 0x0040 && /^(2131|1800)/.test(value)) { //jcb\n                    return value.length === 15;\n                }\n\n                if (validTypes & 0x0080) { //unknown\n                    return true;\n                }\n\n                return false;\n            },\n            $.mage.__('Please enter a valid credit card number.')\n        ],\n        'ipv4': [\n            function (value) {\n                return utils.isEmpty(value) || /^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/i.test(value);//eslint-disable-line max-len\n            },\n            $.mage.__('Please enter a valid IP v4 address.')\n        ],\n        'ipv6': [\n            function (value) {\n                return utils.isEmpty(value) || /^((([0-9A-Fa-f]{1,4}:){7}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}:[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){5}:([0-9A-Fa-f]{1,4}:)?[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){4}:([0-9A-Fa-f]{1,4}:){0,2}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){3}:([0-9A-Fa-f]{1,4}:){0,3}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){2}:([0-9A-Fa-f]{1,4}:){0,4}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}((\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b)\\.){3}(\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b))|(([0-9A-Fa-f]{1,4}:){0,5}:((\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b)\\.){3}(\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b))|(::([0-9A-Fa-f]{1,4}:){0,5}((\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b)\\.){3}(\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b))|([0-9A-Fa-f]{1,4}::([0-9A-Fa-f]{1,4}:){0,5}[0-9A-Fa-f]{1,4})|(::([0-9A-Fa-f]{1,4}:){0,6}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){1,7}:))$/i.test(value);//eslint-disable-line max-len\n            },\n            $.mage.__('Please enter a valid IP v6 address.')\n        ],\n        'pattern': [\n            function (value, param) {\n                return utils.isEmpty(value) || new RegExp(param).test(value);\n            },\n            $.mage.__('Invalid format.')\n        ],\n        'validate-no-html-tags': [\n            function (value) {\n                return !/<(\\/)?\\w+/.test(value);\n            },\n            $.mage.__('HTML tags are not allowed.')\n        ],\n        'validate-select': [\n            function (value) {\n                return value !== 'none' && value != null && value.length !== 0;\n            },\n            $.mage.__('Please select an option.')\n        ],\n        'validate-no-empty': [\n            function (value) {\n                return !utils.isEmpty(value);\n            },\n            $.mage.__('Empty Value.')\n        ],\n        'validate-alphanum-with-spaces': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^[a-zA-Z0-9 ]+$/.test(value);\n            },\n            $.mage.__('Please use only letters (a-z or A-Z), numbers (0-9) or spaces only in this field.')\n        ],\n        'validate-data': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^[A-Za-z]+[A-Za-z0-9_]+$/.test(value);\n            },\n            $.mage.__('Please use only letters (a-z or A-Z), numbers (0-9) or underscore (_) in this field, and the first character should be a letter.')//eslint-disable-line max-len\n        ],\n        'validate-street': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) ||\n                    /^[ \\w]{3,}([A-Za-z]\\.)?([ \\w]*\\#\\d+)?(\\r\\n| )[ \\w]{3,}/.test(value);\n            },\n            $.mage.__('Please use only letters (a-z or A-Z), numbers (0-9), spaces and \"#\" in this field.')\n        ],\n        'validate-phoneStrict': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^(\\()?\\d{3}(\\))?(-|\\s)?\\d{3}(-|\\s)\\d{4}$/.test(value);\n            },\n            $.mage.__('Please enter a valid phone number. For example (123) 456-7890 or 123-456-7890.')\n        ],\n        'validate-phoneLax': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) ||\n                    /^((\\d[\\-. ]?)?((\\(\\d{3}\\))|\\d{3}))?[\\-. ]?\\d{3}[\\-. ]?\\d{4}$/.test(value);\n            },\n            $.mage.__('Please enter a valid phone number. For example (123) 456-7890 or 123-456-7890.')\n        ],\n        'validate-fax': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^(\\()?\\d{3}(\\))?(-|\\s)?\\d{3}(-|\\s)\\d{4}$/.test(value);\n            },\n            $.mage.__('Please enter a valid fax number (Ex: 123-456-7890).')\n        ],\n        'validate-email': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^([a-z0-9,!\\#\\$%&'\\*\\+\\/=\\?\\^_`\\{\\|\\}~-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z0-9,!\\#\\$%&'\\*\\+\\/=\\?\\^_`\\{\\|\\}~-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*@([a-z0-9-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z0-9-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*\\.(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]){2,})$/i.test(value);//eslint-disable-line max-len\n            },\n            $.mage.__('Please enter a valid email address (Ex: johndoe@domain.com).')\n        ],\n        'validate-emailSender': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^[\\S ]+$/.test(value);\n            },\n            $.mage.__('Please enter a valid email address (Ex: johndoe@domain.com).')\n        ],\n        'validate-password': [\n            function (value) {\n                var pass;\n\n                if (value == null) {\n                    return false;\n                }\n\n                pass = $.trim(value);\n\n                if (!pass.length) {\n                    return true;\n                }\n\n                return !(pass.length > 0 && pass.length < 6);\n            },\n            $.mage.__('Please enter 6 or more characters. Leading and trailing spaces will be ignored.')\n        ],\n        'validate-admin-password': [\n            function (value) {\n                var pass;\n\n                if (value == null) {\n                    return false;\n                }\n\n                pass = $.trim(value);\n\n                if (pass.length === 0) {\n                    return true;\n                }\n\n                if (!/[a-z]/i.test(value) || !/[0-9]/.test(value)) {\n                    return false;\n                }\n\n                if (pass.length < 7) {\n                    return false;\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter 7 or more characters, using both numeric and alphabetic.')\n        ],\n        'validate-customer-password': [\n            function (v, elm) {\n                var validator = this,\n                    counter = 0,\n                    passwordMinLength = $(elm).data('password-min-length'),\n                    passwordMinCharacterSets = $(elm).data('password-min-character-sets'),\n                    pass = $.trim(v),\n                    result = pass.length >= passwordMinLength;\n\n                if (result === false) {\n                    validator.passwordErrorMessage = $.mage.__('Minimum length of this field must be equal or greater than %1 symbols. Leading and trailing spaces will be ignored.').replace('%1', passwordMinLength);//eslint-disable-line max-len\n\n                    return result;\n                }\n\n                if (pass.match(/\\d+/)) {\n                    counter++;\n                }\n\n                if (pass.match(/[a-z]+/)) {\n                    counter++;\n                }\n\n                if (pass.match(/[A-Z]+/)) {\n                    counter++;\n                }\n\n                if (pass.match(/[^a-zA-Z0-9]+/)) {\n                    counter++;\n                }\n\n                if (counter < passwordMinCharacterSets) {\n                    result = false;\n                    validator.passwordErrorMessage = $.mage.__('Minimum of different classes of characters in password is %1. Classes of characters: Lower Case, Upper Case, Digits, Special Characters.').replace('%1', passwordMinCharacterSets);//eslint-disable-line max-len\n                }\n\n                return result;\n            }, function () {\n                return this.passwordErrorMessage;\n            }\n        ],\n        'validate-url': [\n            function (value) {\n                if (utils.isEmptyNoTrim(value)) {\n                    return true;\n                }\n                value = (value || '').replace(/^\\s+/, '').replace(/\\s+$/, '');\n\n                return (/^(http|https|ftp):\\/\\/(([A-Z0-9]([A-Z0-9_-]*[A-Z0-9]|))(\\.[A-Z0-9]([A-Z0-9_-]*[A-Z0-9]|))*)(:(\\d+))?(\\/[A-Z0-9~](([A-Z0-9_~-]|\\.)*[A-Z0-9~]|))*\\/?(.*)?$/i).test(value);//eslint-disable-line max-len\n\n            },\n            $.mage.__('Please enter a valid URL. Protocol is required (http://, https:// or ftp://).')\n        ],\n        'validate-clean-url': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^(http|https|ftp):\\/\\/(([A-Z0-9][A-Z0-9_-]*)(\\.[A-Z0-9][A-Z0-9_-]*)+.(com|org|net|dk|at|us|tv|info|uk|co.uk|biz|se)$)(:(\\d+))?\\/?/i.test(value) || /^(www)((\\.[A-Z0-9][A-Z0-9_-]*)+.(com|org|net|dk|at|us|tv|info|uk|co.uk|biz|se)$)(:(\\d+))?\\/?/i.test(value);//eslint-disable-line max-len\n\n            },\n            $.mage.__('Please enter a valid URL. For example http://www.example.com or www.example.com.')\n        ],\n        'validate-xml-identifier': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^[A-Z][A-Z0-9_\\/-]*$/i.test(value);\n\n            },\n            $.mage.__('Please enter a valid XML-identifier (Ex: something_1, block5, id-4).')\n        ],\n        'validate-ssn': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^\\d{3}-?\\d{2}-?\\d{4}$/.test(value);\n\n            },\n            $.mage.__('Please enter a valid social security number (Ex: 123-45-6789).')\n        ],\n        'validate-zip-us': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /(^\\d{5}$)|(^\\d{5}-\\d{4}$)/.test(value);\n\n            },\n            $.mage.__('Please enter a valid zip code (Ex: 90602 or 90602-1234).')\n        ],\n        'validate-date-au': [\n            function (value) {\n                var regex = /^(\\d{2})\\/(\\d{2})\\/(\\d{4})$/,\n                    d;\n\n                if (utils.isEmptyNoTrim(value)) {\n                    return true;\n                }\n\n                if (utils.isEmpty(value) || !regex.test(value)) {\n                    return false;\n                }\n                d = new Date(value.replace(regex, '$2/$1/$3'));\n\n                return parseInt(RegExp.$2, 10) === 1 + d.getMonth() &&\n                    parseInt(RegExp.$1, 10) === d.getDate() &&\n                    parseInt(RegExp.$3, 10) === d.getFullYear();\n\n            },\n            $.mage.__('Please use this date format: dd/mm/yyyy. For example 17/03/2006 for the 17th of March, 2006.')\n        ],\n        'validate-currency-dollar': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^\\$?\\-?([1-9]{1}[0-9]{0,2}(\\,[0-9]{3})*(\\.[0-9]{0,2})?|[1-9]{1}\\d*(\\.[0-9]{0,2})?|0(\\.[0-9]{0,2})?|(\\.[0-9]{1,2})?)$/.test(value);//eslint-disable-line max-len\n\n            },\n            $.mage.__('Please enter a valid $ amount. For example $100.00.')\n        ],\n        'validate-not-negative-number': [\n            function (value) {\n                if (utils.isEmptyNoTrim(value)) {\n                    return true;\n                }\n                value = utils.parseNumber(value);\n\n                return !isNaN(value) && value >= 0;\n\n            },\n            $.mage.__('Please enter a number 0 or greater in this field.')\n        ],\n        // validate-not-negative-number should be replaced in all places with this one and then removed\n        'validate-zero-or-greater': [\n            function (value) {\n                if (utils.isEmptyNoTrim(value)) {\n                    return true;\n                }\n                value = utils.parseNumber(value);\n\n                return !isNaN(value) && value >= 0;\n            },\n            $.mage.__('Please enter a number 0 or greater in this field.')\n        ],\n        'validate-greater-than-zero': [\n            function (value) {\n                if (utils.isEmptyNoTrim(value)) {\n                    return true;\n                }\n                value = utils.parseNumber(value);\n\n                return !isNaN(value) && value > 0;\n            },\n            $.mage.__('Please enter a number greater than 0 in this field.')\n        ],\n        'validate-css-length': [\n            function (value) {\n                if (value !== '') {\n                    return (/^[0-9]*\\.*[0-9]+(px|pc|pt|ex|em|mm|cm|in|%)?$/).test(value);\n                }\n\n                return true;\n            },\n            $.mage.__('Please input a valid CSS-length (Ex: 100px, 77pt, 20em, .5ex or 50%).')\n        ],\n        'validate-number': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) ||\n                    !isNaN(utils.parseNumber(value)) && /^\\s*-?\\d*(,\\d*)*(\\.\\d*)?\\s*$/.test(value);\n            },\n            $.mage.__('Please enter a valid number in this field.')\n        ],\n        'validate-integer': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || !isNaN(utils.parseNumber(value)) && /^\\s*-?\\d*\\s*$/.test(value);\n            },\n            $.mage.__('Please enter a valid integer in this field.')\n        ],\n        'validate-number-range': [\n            function (value, param) {\n                var numValue, dataAttrRange, result, range, m;\n\n                if (utils.isEmptyNoTrim(value)) {\n                    return true;\n                }\n\n                numValue = utils.parseNumber(value);\n\n                if (isNaN(numValue)) {\n                    return false;\n                }\n\n                dataAttrRange = /^(-?[\\d.,]+)?-(-?[\\d.,]+)?$/;\n                result = true;\n                range = param;\n\n                if (range) {\n                    m = dataAttrRange.exec(range);\n\n                    if (m) {\n                        result = result && utils.isBetween(numValue, m[1], m[2]);\n                    }\n                }\n\n                return result;\n            },\n            $.mage.__('The value is not within the specified range.')\n        ],\n        'validate-positive-percent-decimal': [\n            function (value) {\n                var numValue;\n\n                if (utils.isEmptyNoTrim(value) || !/^\\s*-?\\d*(\\.\\d*)?\\s*$/.test(value)) {\n                    return false;\n                }\n\n                numValue = utils.parseNumber(value);\n\n                if (isNaN(numValue)) {\n                    return false;\n                }\n\n                return utils.isBetween(numValue, 0.01, 100);\n            },\n            $.mage.__('Please enter a valid percentage discount value greater than 0.')\n        ],\n        'validate-digits': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || !/[^\\d]/.test(value);\n            },\n            $.mage.__('Please enter a valid number in this field.')\n        ],\n        'validate-digits-range': [\n            function (value, param) {\n                var numValue, dataAttrRange, result, range, m;\n\n                if (utils.isEmptyNoTrim(value)) {\n                    return true;\n                }\n\n                numValue = utils.parseNumber(value);\n\n                if (isNaN(numValue)) {\n                    return false;\n                }\n\n                dataAttrRange = /^(-?\\d+)?-(-?\\d+)?$/;\n                result = true;\n                range = param;\n\n                if (range) {\n                    m = dataAttrRange.exec(range);\n\n                    if (m) {\n                        result = result && utils.isBetween(numValue, m[1], m[2]);\n                    }\n                }\n\n                return result;\n            },\n            $.mage.__('The value is not within the specified range.')\n        ],\n        'validate-range': [\n            function (value) {\n                var minValue, maxValue, ranges;\n\n                if (utils.isEmptyNoTrim(value)) {\n                    return true;\n                } else if ($.validator.methods['validate-digits'] && $.validator.methods['validate-digits'](value)) {\n                    minValue = maxValue = utils.parseNumber(value);\n                } else {\n                    ranges = /^(-?\\d+)?-(-?\\d+)?$/.exec(value);\n\n                    if (ranges) {\n                        minValue = utils.parseNumber(ranges[1]);\n                        maxValue = utils.parseNumber(ranges[2]);\n\n                        if (minValue > maxValue) {//eslint-disable-line max-depth\n                            return false;\n                        }\n                    } else {\n                        return false;\n                    }\n                }\n            },\n            $.mage.__('The value is not within the specified range.')\n        ],\n        'validate-alpha': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^[a-zA-Z]+$/.test(value);\n            },\n            $.mage.__('Please use letters only (a-z or A-Z) in this field.')\n        ],\n        'validate-code': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^[a-z]+[a-z0-9_]+$/.test(value);\n            },\n            $.mage.__('Please use only lowercase letters (a-z), numbers (0-9) or underscore (_) in this field, and the first character should be a letter.')//eslint-disable-line max-len\n        ],\n        'validate-alphanum': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^[a-zA-Z0-9]+$/.test(value);\n            },\n            $.mage.__('Please use only letters (a-z or A-Z) or numbers (0-9) in this field. No spaces or other characters are allowed.')//eslint-disable-line max-len\n        ],\n        'validate-not-number-first': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^[^0-9-\\.].*$/.test(value.trim());\n            },\n            $.mage.__('First character must be letter.')\n        ],\n        'validate-date': [\n            function (value, params, additionalParams) {\n                var test = moment(value, additionalParams.dateFormat);\n\n                return utils.isEmptyNoTrim(value) || test.isValid();\n            },\n            $.mage.__('Please enter a valid date.')\n        ],\n        'validate-date-range': [\n            function (value, params) {\n                var fromDate = $('input[name*=\"' + params + '\"]').val();\n\n                return moment.utc(value).unix() >= moment.utc(fromDate).unix() || isNaN(moment.utc(value).unix());\n            },\n            $.mage.__('Make sure the To Date is later than or the same as the From Date.')\n        ],\n        'validate-identifier': [\n            function (value) {\n                return utils.isEmptyNoTrim(value) || /^[a-z0-9][a-z0-9_\\/-]+(\\.[a-z0-9_-]+)?$/.test(value);\n            },\n            $.mage.__('Please enter a valid URL Key (Ex: \"example-page\", \"example-page.html\" or \"anotherlevel/example-page\").')//eslint-disable-line max-len\n        ],\n        'validate-zip-international': [\n\n            /*function(v) {\n            // @TODO: Cleanup\n            return Validation.get('IsEmpty').test(v) || /(^[A-z0-9]{2,10}([\\s]{0,1}|[\\-]{0,1})[A-z0-9]{2,10}$)/.test(v);\n            }*/\n            function () {\n                return true;\n            },\n            $.mage.__('Please enter a valid zip code.')\n        ],\n        'validate-state': [\n            function (value) {\n                return value !== 0 || value === '';\n            },\n            $.mage.__('Please select State/Province.')\n        ],\n        'less-than-equals-to': [\n            function (value, params) {\n                if ($.isNumeric(params) && $.isNumeric(value)) {\n                    return parseFloat(value) <= parseFloat(params);\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter a value less than or equal to {0}.')\n        ],\n        'greater-than-equals-to': [\n            function (value, params) {\n                if ($.isNumeric(params) && $.isNumeric(value)) {\n                    return parseFloat(value) >= parseFloat(params);\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter a value greater than or equal to {0}.')\n        ],\n        'validate-emails': [\n            function (value) {\n                var validRegexp, emails, i;\n\n                if (utils.isEmpty(value)) {\n                    return true;\n                }\n                validRegexp = /^[a-z0-9\\._-]{1,30}@([a-z0-9_-]{1,30}\\.){1,5}[a-z]{2,4}$/i;\n                emails = value.split(/[\\s\\n\\,]+/g);\n\n                for (i = 0; i < emails.length; i++) {\n                    if (!validRegexp.test(emails[i].strip())) {\n                        return false;\n                    }\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter valid email addresses, separated by commas. For example, johndoe@domain.com, johnsmith@domain.com.')//eslint-disable-line max-len\n        ],\n        'validate-cc-number': [\n\n            /**\n             * Validate credit card number based on mod 10.\n             *\n             * @param {String} value - credit card number\n             * @return {Boolean}\n             */\n            function (value) {\n                if (value) {\n                    return validateCreditCard(value);\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter a valid credit card number.')\n        ],\n        'validate-cc-ukss': [\n\n            /**\n             * Validate Switch/Solo/Maestro issue number and start date is filled.\n             *\n             * @param {String} value - input field value\n             * @return {*}\n             */\n            function (value) {\n                return value;\n            },\n            $.mage.__('Please enter issue number or start date for switch/solo card type.')\n        ],\n        'required-entry': [\n            function (value) {\n                return !utils.isEmpty(value);\n            },\n            $.mage.__('This is a required field.')\n        ],\n        'checked': [\n            function (value) {\n                return value;\n            },\n            $.mage.__('This is a required field.')\n        ],\n        'not-negative-amount': [\n            function (value) {\n                if (value.length) {\n                    return (/^\\s*\\d+([,.]\\d+)*\\s*%?\\s*$/).test(value);\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter positive number in this field.')\n        ],\n        'validate-per-page-value-list': [\n            function (value) {\n                var isValid = true,\n                    values = value.split(','),\n                    i;\n\n                if (utils.isEmpty(value)) {\n                    return isValid;\n                }\n\n                for (i = 0; i < values.length; i++) {\n                    if (!/^[0-9]+$/.test(values[i])) {\n                        isValid = false;\n                    }\n                }\n\n                return isValid;\n            },\n            $.mage.__('Please enter a valid value, ex: 10,20,30')\n        ],\n        'validate-new-password': [\n            function (value) {\n                if ($.validator.methods['validate-password'] && !$.validator.methods['validate-password'](value)) {\n                    return false;\n                }\n\n                if (utils.isEmpty(value) && value !== '') {\n                    return false;\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter 6 or more characters. Leading and trailing spaces will be ignored.')\n        ],\n        'validate-item-quantity': [\n            function (value, params) {\n                var validator = this,\n                    result = false,\n                    // obtain values for validation\n                    qty = utils.parseNumber(value),\n                    isMinAllowedValid = typeof params.minAllowed === 'undefined' ||\n                        qty >= utils.parseNumber(params.minAllowed),\n                    isMaxAllowedValid = typeof params.maxAllowed === 'undefined' ||\n                        qty <= utils.parseNumber(params.maxAllowed),\n                    isQtyIncrementsValid = typeof params.qtyIncrements === 'undefined' ||\n                        qty % utils.parseNumber(params.qtyIncrements) === 0;\n\n                result = qty > 0;\n\n                if (result === false) {\n                    validator.itemQtyErrorMessage = $.mage.__('Please enter a quantity greater than 0.');//eslint-disable-line max-len\n\n                    return result;\n                }\n\n                result = isMinAllowedValid;\n\n                if (result === false) {\n                    validator.itemQtyErrorMessage = $.mage.__('The fewest you may purchase is %1.').replace('%1', params.minAllowed);//eslint-disable-line max-len\n\n                    return result;\n                }\n\n                result = isMaxAllowedValid;\n\n                if (result === false) {\n                    validator.itemQtyErrorMessage = $.mage.__('The maximum you may purchase is %1.').replace('%1', params.maxAllowed);//eslint-disable-line max-len\n\n                    return result;\n                }\n\n                result = isQtyIncrementsValid;\n\n                if (result === false) {\n                    validator.itemQtyErrorMessage = $.mage.__('You can buy this product only in quantities of %1 at a time.').replace('%1', params.qtyIncrements);//eslint-disable-line max-len\n\n                    return result;\n                }\n\n                return result;\n            }, function () {\n                return this.itemQtyErrorMessage;\n            }\n        ],\n        'equalTo': [\n            function (value, param) {\n                return value === $(param).val();\n            },\n            $.validator.messages.equalTo\n        ],\n        'validate-file-type': [\n            function (name, types) {\n                var extension = name.split('.').pop().toLowerCase();\n\n                if (types && typeof types === 'string') {\n                    types = types.split(' ');\n                }\n\n                return !types || !types.length || ~types.indexOf(extension);\n            },\n            $.mage.__('We don\\'t recognize or support this file extension type.')\n        ],\n        'validate-max-size': [\n            function (size, maxSize) {\n                return maxSize === false || size < maxSize;\n            },\n            $.mage.__('File you are trying to upload exceeds maximum file size limit.')\n        ],\n        'validate-if-tag-script-exist': [\n            function (value) {\n                return !value || (/<script\\b[^>]*>([\\s\\S]*?)<\\/script>$/ig).test(value);\n            },\n            $.mage.__('Please use tag SCRIPT with SRC attribute or with proper content to include JavaScript to the document.')//eslint-disable-line max-len\n        ],\n        'date_range_min': [\n            function (value, minValue, params) {\n                return moment.utc(value, params.dateFormat).unix() >= minValue;\n            },\n            $.mage.__('The date is not within the specified range.')\n        ],\n        'date_range_max': [\n            function (value, maxValue, params) {\n                return moment.utc(value, params.dateFormat).unix() <= maxValue;\n            },\n            $.mage.__('The date is not within the specified range.')\n        ],\n        'validate-color': [\n            function (value) {\n                if (value === '') {\n                    return true;\n                }\n\n                return tinycolor(value).isValid();\n            },\n            $.mage.__('Wrong color format. Please specify color in HEX, RGBa, HSVa, HSLa or use color name.')\n        ],\n        'blacklist-url': [\n            function (value, param) {\n                return new RegExp(param).test(value);\n            },\n            $.mage.__('This link is not allowed.')\n        ]\n    }, function (data) {\n        return {\n            handler: data[0],\n            message: data[1]\n        };\n    });\n});\n","Magento_Ui/js/lib/validation/utils.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine(function () {\n    'use strict';\n\n    var utils = {\n        /**\n         * Check if string is empty with trim.\n         *\n         * @param {String} value\n         * @return {Boolean}\n         */\n        isEmpty: function (value) {\n            return value === '' || value == null || value.length === 0 || /^\\s+$/.test(value);\n        },\n\n        /**\n         * Check if string is empty no trim.\n         *\n         * @param {String} value\n         * @return {Boolean}\n         */\n        isEmptyNoTrim: function (value) {\n            return value === '' || value == null || value.length === 0;\n        },\n\n        /**\n         * Checks if {value} is between numbers {from} and {to}.\n         *\n         * @param {String} value\n         * @param {String} from\n         * @param {String} to\n         * @return {Boolean}\n         */\n        isBetween: function (value, from, to) {\n            return (from === null || from === '' || value >= utils.parseNumber(from)) &&\n                   (to === null || to === '' || value <= utils.parseNumber(to));\n        },\n\n        /**\n         * Parse price string.\n         *\n         * @param {String} value\n         * @return {Number}\n         */\n        parseNumber: function (value) {\n            var isDot, isComa;\n\n            if (typeof value !== 'string') {\n                return parseFloat(value);\n            }\n            isDot = value.indexOf('.');\n            isComa = value.indexOf(',');\n\n            if (isDot !== -1 && isComa !== -1) {\n                if (isComa > isDot) {\n                    value = value.replace('.', '').replace(',', '.');\n                } else {\n                    value = value.replace(',', '');\n                }\n            } else if (isComa !== -1) {\n                value = value.replace(',', '.');\n            }\n\n            return parseFloat(value);\n        },\n\n        /**\n         * Removes HTML tags and space characters, numbers and punctuation.\n         *\n         * @param {String} value -  Value being stripped.\n         * @return {String}\n         */\n        stripHtml: function (value) {\n            return value.replace(/<.[^<>]*?>/g, ' ').replace(/&nbsp;|&#160;/gi, ' ')\n                .replace(/[0-9.(),;:!?%#$'\"_+=\\/-]*/g, '');\n        }\n    };\n\n    return utils;\n});\n","Magento_Ui/js/lib/validation/validator.js":"/*\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    './rules'\n], function (_, rulesList) {\n    'use strict';\n\n    /**\n     * Validates provided value be the specified rule.\n     *\n     * @param {String} id - Rule identifier.\n     * @param {*} value - Value to be checked.\n     * @param {*} [params]\n     * @param {*} additionalParams - additional validation params set by method caller\n     * @returns {Object}\n     */\n    function validate(id, value, params, additionalParams) {\n        var rule,\n            message,\n            valid,\n            result = {\n                rule: id,\n                passed: true,\n                message: ''\n            };\n\n        if (_.isObject(params)) {\n            message = params.message || '';\n        }\n\n        if (!rulesList[id]) {\n            return result;\n        }\n\n        rule    = rulesList[id];\n        message = message || rule.message;\n        valid   = rule.handler(value, params, additionalParams);\n\n        if (!valid) {\n            params = Array.isArray(params) ?\n                params :\n                [params];\n\n            if (typeof message === 'function') {\n                message = message.call(rule);\n            }\n\n            message = params.reduce(function (msg, param, idx) {\n                return msg.replace(new RegExp('\\\\{' + idx + '\\\\}', 'g'), param);\n            }, message);\n\n            result.passed = false;\n            result.message = message;\n        }\n\n        return result;\n    }\n\n    /**\n     * Validates provided value by a specified set of rules.\n     *\n     * @param {(String|Object)} rules - One or many validation rules.\n     * @param {*} value - Value to be checked.\n     * @param {*} additionalParams - additional validation params set by method caller\n     * @returns {Object}\n     */\n    function validator(rules, value, additionalParams) {\n        var result;\n\n        if (typeof rules === 'object') {\n            result = {\n                passed: true\n            };\n\n            _.every(rules, function (ruleParams, id) {\n                if (ruleParams.validate || ruleParams !== false || additionalParams) {\n                    result = validate(id, value, ruleParams, additionalParams);\n\n                    return result.passed;\n                }\n\n                return true;\n            });\n\n            return result;\n        }\n\n        return validate.apply(null, arguments);\n    }\n\n    /**\n     * Adds new validation rule.\n     *\n     * @param {String} id - Rule identifier.\n     * @param {Function} handler - Validation function.\n     * @param {String} message - Error message.\n     */\n    validator.addRule = function (id, handler, message) {\n        rulesList[id] = {\n            handler: handler,\n            message: message\n        };\n    };\n\n    /**\n     * Returns rule object found by provided identifier.\n     *\n     * @param {String} id - Rule identifier.\n     * @returns {Object}\n     */\n    validator.getRule = function (id) {\n        return rulesList[id];\n    };\n\n    return validator;\n});\n","Magento_Ui/js/form/provider.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'uiElement',\n    './client',\n    'mageUtils'\n], function (_, Element, Client, utils) {\n    'use strict';\n\n    return Element.extend({\n        defaults: {\n            clientConfig: {\n                urls: {\n                    save: '${ $.submit_url }',\n                    beforeSave: '${ $.validate_url }'\n                }\n            },\n            ignoreTmpls: {\n                data: true\n            }\n        },\n\n        /**\n         * Initializes provider component.\n         *\n         * @returns {Provider} Chainable.\n         */\n        initialize: function () {\n            this._super()\n                .initClient();\n\n            return this;\n        },\n\n        /**\n         * Initializes client component.\n         *\n         * @returns {Provider} Chainable.\n         */\n        initClient: function () {\n            this.client = new Client(this.clientConfig);\n\n            return this;\n        },\n\n        /**\n         * Saves currently available data.\n         *\n         * @param {Object} [options] - Addtitional request options.\n         * @returns {Provider} Chainable.\n         */\n        save: function (options) {\n            var data = this.get('data');\n\n            this.client.save(data, options);\n\n            return this;\n        },\n\n        /**\n         * Update data that stored in provider.\n         *\n         * @param {Boolean} isProvider\n         * @param {Object} newData\n         * @param {Object} oldData\n         *\n         * @returns {Provider}\n         */\n        updateConfig: function (isProvider, newData, oldData) {\n            if (isProvider === true) {\n                this.setData(oldData, newData, this);\n            }\n\n            return this;\n        },\n\n        /**\n         *  Set data to provider based on current data.\n         *\n         * @param {Object} oldData\n         * @param {Object} newData\n         * @param {Provider} current\n         * @param {String} parentPath\n         */\n        setData: function (oldData, newData, current, parentPath) {\n            _.each(newData, function (val, key) {\n                if (_.isObject(val) || _.isArray(val)) {\n                    this.setData(oldData[key], val, current[key], utils.fullPath(parentPath, key));\n                } else if (val != oldData[key] && oldData[key] == current[key]) {//eslint-disable-line eqeqeq\n                    this.set(utils.fullPath(parentPath, key), val);\n                }\n            }, this);\n        }\n    });\n});\n","Magento_Ui/js/form/client.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'mageUtils',\n    'uiClass'\n], function ($, _, utils, Class) {\n    'use strict';\n\n    /**\n     * Before save validate request.\n     *\n     * @param {Object} data\n     * @param {String} url\n     * @param {String} selectorPrefix\n     * @param {String} messagesClass\n     * @returns {*}\n     */\n    function beforeSave(data, url, selectorPrefix, messagesClass) {\n        var save = $.Deferred();\n\n        data = utils.serialize(utils.filterFormData(data));\n        data['form_key'] = window.FORM_KEY;\n\n        if (!url || url === 'undefined') {\n            return save.resolve();\n        }\n\n        $('body').trigger('processStart');\n\n        $.ajax({\n            url: url,\n            data: data,\n\n            /**\n             * Success callback.\n             * @param {Object} resp\n             * @returns {Boolean}\n             */\n            success: function (resp) {\n                if (!resp.error) {\n                    save.resolve();\n\n                    return true;\n                }\n\n                $('body').notification('clear');\n                $.each(resp.messages || [resp.message] || [], function (key, message) {\n                    $('body').notification('add', {\n                        error: resp.error,\n                        message: message,\n\n                        /**\n                         * Insert method.\n                         *\n                         * @param {String} msg\n                         */\n                        insertMethod: function (msg) {\n                            var $wrapper = $('<div/>').addClass(messagesClass).html(msg);\n\n                            $('.page-main-actions', selectorPrefix).after($wrapper);\n                        }\n                    });\n                });\n            },\n\n            /**\n             * Complete callback.\n             */\n            complete: function () {\n                $('body').trigger('processStop');\n            }\n        });\n\n        return save.promise();\n    }\n\n    return Class.extend({\n\n        /**\n         * Assembles data and submits it using 'utils.submit' method\n         */\n        save: function (data, options) {\n            var url = this.urls.beforeSave,\n                save = this._save.bind(this, data, options);\n\n            beforeSave(data, url, this.selectorPrefix, this.messagesClass).then(save);\n\n            return this;\n        },\n\n        /**\n         * Save data.\n         *\n         * @param {Object} data\n         * @param {Object} options\n         * @returns {Object}\n         * @private\n         */\n        _save: function (data, options) {\n            var url = this.urls.save;\n\n            $('body').trigger('processStart');\n            options = options || {};\n\n            if (!options.redirect) {\n                url += 'back/edit';\n            }\n\n            if (options.ajaxSave) {\n                utils.ajaxSubmit({\n                    url: url,\n                    data: data\n                }, options);\n\n                $('body').trigger('processStop');\n\n                return this;\n            }\n\n            utils.submit({\n                url: url,\n                data: data\n            }, options.attributes);\n\n            return this;\n        }\n    });\n});\n","Magento_Ui/js/form/button-adapter.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'uiClass',\n    'jquery',\n    'underscore',\n    'uiRegistry'\n], function (Class, $, _, registry) {\n    'use strict';\n\n    return Class.extend({\n\n        /**\n         * Initialize actions and adapter.\n         *\n         * @param {Object} config\n         * @param {Element} elem\n         * @returns {Object}\n         */\n        initialize: function (config, elem) {\n            return this._super()\n                .initActions()\n                .initAdapter(elem);\n        },\n\n        /**\n         * Creates callback from declared actions.\n         *\n         * @returns {Object}\n         */\n        initActions: function () {\n            var callbacks = [];\n\n            _.each(this.actions, function (action) {\n                callbacks.push({\n                    action: registry.async(action.targetName),\n                    args: _.union([action.actionName], action.params)\n                });\n            });\n\n            /**\n             * Callback function.\n             */\n            this.callback = function () {\n                _.each(callbacks, function (callback) {\n                    callback.action.apply(callback.action, callback.args);\n                });\n            };\n\n            return this;\n        },\n\n        /**\n         * Attach callback handler on button.\n         *\n         * @param {Element} elem\n         */\n        initAdapter: function (elem) {\n            $(elem).on('click', this.callback);\n\n            return this;\n        }\n    });\n});\n","Magento_Ui/js/form/switcher.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'uiRegistry',\n    'uiClass'\n], function (_, registry, Class) {\n    'use strict';\n\n    return Class.extend({\n        defaults: {\n            rules: []\n        },\n\n        /**\n         * Initializes instance of a DataSwitcher.\n         *\n         * @returns {DataSwitcher} Chainable.\n         */\n        initialize: function () {\n            this._super()\n                .initRules();\n\n            return this;\n        },\n\n        /**\n         *\n         * @returns {DataSwitcher} Chainable.\n         */\n        initRules: function () {\n            this.rules.forEach(this.initRule, this);\n\n            return this;\n        },\n\n        /**\n         *\n         * @param {Object} rule - Rule definition.\n         * @returns {DataSwitcher} Chainable.\n         */\n        initRule: function (rule) {\n            var handler = this.onValueChange.bind(this, rule);\n\n            if (!rule.target) {\n                rule.target = this.target;\n            }\n\n            if (!rule.property) {\n                rule.property = this.property;\n            }\n\n            registry.get(rule.target, function (target) {\n                this.applyRule(rule, target.get(rule.property));\n                target.on(rule.property, handler);\n            }.bind(this));\n\n            return this;\n        },\n\n        /**\n         *\n         * @param {Object} rule - Rule definition.\n         * @returns {DataSwitcher} Chainable.\n         */\n        addRule: function (rule) {\n            this.rules.push(rule);\n            this.initRule(rule);\n\n            return this;\n        },\n\n        /**\n         *\n         * @param {Object} rule - Rule object.\n         * @param {*} value - Current value associated with a rule.\n         */\n        applyRule: function (rule, value) {\n            var actions = rule.actions;\n\n            //TODO Refactor this logic in scope of MAGETWO-48585\n            /* eslint-disable eqeqeq */\n            if (rule.value != value) {\n                return;\n            } else if (rule.strict) {\n                return;\n            }\n\n            /* eslint-enable eqeqeq */\n            actions.forEach(this.applyAction, this);\n        },\n\n        /**\n         *\n         * @param {Object} action - Action object.\n         */\n        applyAction: function (action) {\n            registry.get(action.target, function (target) {\n                var callback = target[action.callback];\n\n                callback.apply(target, action.params || []);\n            });\n        },\n\n        /**\n         *\n         * @param {Object} rule - Rules object.\n         * @param {*} value - Current value associated with a rule.\n         */\n        onValueChange: function (rule, value) {\n            this.applyRule(rule, value);\n        }\n    });\n});\n","Magento_Ui/js/form/form.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'Magento_Ui/js/lib/spinner',\n    'rjsResolver',\n    './adapter',\n    'uiCollection',\n    'mageUtils',\n    'jquery',\n    'Magento_Ui/js/core/app',\n    'mage/validation'\n], function (_, loader, resolver, adapter, Collection, utils, $, app) {\n    'use strict';\n\n    /**\n     * Format params\n     *\n     * @param {Object} params\n     * @returns {Array}\n     */\n    function prepareParams(params) {\n        var result = '?';\n\n        _.each(params, function (value, key) {\n            result += key + '=' + value + '&';\n        });\n\n        return result.slice(0, -1);\n    }\n\n    /**\n     * Collect form data.\n     *\n     * @param {Array} items\n     * @returns {Object}\n     */\n    function collectData(items) {\n        var result = {},\n            name;\n\n        items = Array.prototype.slice.call(items);\n\n        items.forEach(function (item) {\n            switch (item.type) {\n                case 'checkbox':\n                    result[item.name] = +!!item.checked;\n                    break;\n\n                case 'radio':\n                    if (item.checked) {\n                        result[item.name] = item.value;\n                    }\n                    break;\n\n                case 'select-multiple':\n                    name = item.name.substring(0, item.name.length - 2); //remove [] from the name ending\n                    result[name] = _.pluck(item.selectedOptions, 'value');\n                    break;\n\n                default:\n                    result[item.name] = item.value;\n            }\n        });\n\n        return result;\n    }\n\n    /**\n     * Makes ajax request\n     *\n     * @param {Object} params\n     * @param {Object} data\n     * @param {String} url\n     * @returns {*}\n     */\n    function makeRequest(params, data, url) {\n        var save = $.Deferred();\n\n        data = utils.serialize(data);\n        data['form_key'] = window.FORM_KEY;\n\n        if (!url) {\n            save.resolve();\n        }\n\n        $('body').trigger('processStart');\n\n        $.ajax({\n            url: url + prepareParams(params),\n            data: data,\n            dataType: 'json',\n\n            /**\n             * Success callback.\n             * @param {Object} resp\n             * @returns {Boolean}\n             */\n            success: function (resp) {\n                if (resp.ajaxExpired) {\n                    window.location.href = resp.ajaxRedirect;\n                }\n\n                if (!resp.error) {\n                    save.resolve(resp);\n\n                    return true;\n                }\n\n                $('body').notification('clear');\n                $.each(resp.messages, function (key, message) {\n                    $('body').notification('add', {\n                        error: resp.error,\n                        message: message,\n\n                        /**\n                         * Inserts message on page\n                         * @param {String} msg\n                         */\n                        insertMethod: function (msg) {\n                            $('.page-main-actions').after(msg);\n                        }\n                    });\n                });\n            },\n\n            /**\n             * Complete callback.\n             */\n            complete: function () {\n                $('body').trigger('processStop');\n            }\n        });\n\n        return save.promise();\n    }\n\n    /**\n     * Check if fields is valid.\n     *\n     * @param {Array}items\n     * @returns {Boolean}\n     */\n    function isValidFields(items) {\n        var result = true;\n\n        _.each(items, function (item) {\n            if (!$.validator.validateSingleElement(item)) {\n                result = false;\n            }\n        });\n\n        return result;\n    }\n\n    return Collection.extend({\n        defaults: {\n            additionalFields: [],\n            additionalInvalid: false,\n            selectorPrefix: '.page-content',\n            messagesClass: 'messages',\n            errorClass: '.admin__field._error',\n            eventPrefix: '.${ $.index }',\n            ajaxSave: false,\n            ajaxSaveType: 'default',\n            imports: {\n                reloadUrl: '${ $.provider}:reloadUrl'\n            },\n            listens: {\n                selectorPrefix: 'destroyAdapter initAdapter',\n                '${ $.name }.${ $.reloadItem }': 'params.set reload'\n            },\n            exports: {\n                selectorPrefix: '${ $.provider }:client.selectorPrefix',\n                messagesClass: '${ $.provider }:client.messagesClass'\n            }\n        },\n\n        /** @inheritdoc */\n        initialize: function () {\n            this._super()\n                .initAdapter();\n\n            resolver(this.hideLoader, this);\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            return this._super()\n                .observe([\n                    'responseData',\n                    'responseStatus'\n                ]);\n        },\n\n        /** @inheritdoc */\n        initConfig: function () {\n            this._super();\n\n            this.selector = '[data-form-part=' + this.namespace + ']';\n\n            return this;\n        },\n\n        /**\n         * Initialize adapter handlers.\n         *\n         * @returns {Object}\n         */\n        initAdapter: function () {\n            adapter.on({\n                'reset': this.reset.bind(this),\n                'save': this.save.bind(this, true, {}),\n                'saveAndContinue': this.save.bind(this, false, {})\n            }, this.selectorPrefix, this.eventPrefix);\n\n            return this;\n        },\n\n        /**\n         * Destroy adapter handlers.\n         *\n         * @returns {Object}\n         */\n        destroyAdapter: function () {\n            adapter.off([\n                'reset',\n                'save',\n                'saveAndContinue'\n            ], this.eventPrefix);\n\n            return this;\n        },\n\n        /**\n         * Hide loader.\n         *\n         * @returns {Object}\n         */\n        hideLoader: function () {\n            loader.get(this.name).hide();\n\n            return this;\n        },\n\n        /**\n         * Validate and save form.\n         *\n         * @param {String} redirect\n         * @param {Object} data\n         */\n        save: function (redirect, data) {\n            this.validate();\n\n            if (!this.additionalInvalid && !this.source.get('params.invalid')) {\n                this.setAdditionalData(data)\n                    .submit(redirect);\n            } else {\n                this.focusInvalid();\n            }\n        },\n\n        /**\n         * Tries to set focus on first invalid form field.\n         *\n         * @returns {Object}\n         */\n        focusInvalid: function () {\n            var invalidField = _.find(this.delegate('checkInvalid'));\n\n            if (!_.isUndefined(invalidField) && _.isFunction(invalidField.focused)) {\n                invalidField.focused(true);\n            }\n\n            return this;\n        },\n\n        /**\n         * Set additional data to source before form submit and after validation.\n         *\n         * @param {Object} data\n         * @returns {Object}\n         */\n        setAdditionalData: function (data) {\n            _.each(data, function (value, name) {\n                this.source.set('data.' + name, value);\n            }, this);\n\n            return this;\n        },\n\n        /**\n         * Submits form\n         *\n         * @param {String} redirect\n         */\n        submit: function (redirect) {\n            var additional = collectData(this.additionalFields),\n                source = this.source;\n\n            _.each(additional, function (value, name) {\n                source.set('data.' + name, value);\n            });\n\n            source.save({\n                redirect: redirect,\n                ajaxSave: this.ajaxSave,\n                ajaxSaveType: this.ajaxSaveType,\n                response: {\n                    data: this.responseData,\n                    status: this.responseStatus\n                },\n                attributes: {\n                    id: this.namespace\n                }\n            });\n        },\n\n        /**\n         * Validates each element and returns true, if all elements are valid.\n         */\n        validate: function () {\n            this.additionalFields = document.querySelectorAll(this.selector);\n            this.source.set('params.invalid', false);\n            this.source.trigger('data.validate');\n            this.set('additionalInvalid', !isValidFields(this.additionalFields));\n        },\n\n        /**\n         * Trigger reset form data.\n         */\n        reset: function () {\n            this.source.trigger('data.reset');\n            $('[data-bind*=datepicker]').val('');\n        },\n\n        /**\n         * Trigger overload form data.\n         */\n        overload: function () {\n            this.source.trigger('data.overload');\n        },\n\n        /**\n         * Updates data from server.\n         */\n        reload: function () {\n            makeRequest(this.params, this.data, this.reloadUrl).then(function (data) {\n                app(data, true);\n            });\n        }\n    });\n});\n","Magento_Ui/js/form/adapter.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'underscore',\n    'Magento_Ui/js/form/adapter/buttons'\n], function ($, _, buttons) {\n    'use strict';\n\n    var selectorPrefix = '',\n        eventPrefix;\n\n    /**\n     * Initialize listener.\n     *\n     * @param {Function} callback\n     * @param {String} action\n     */\n    function initListener(callback, action) {\n        var selector    = selectorPrefix ? selectorPrefix + ' ' + buttons[action] : buttons[action],\n            elem        = $(selector)[0];\n\n        if (!elem) {\n            return;\n        }\n\n        if (elem.onclick) {\n            elem.onclick = null;\n        }\n\n        $(elem).on('click' + eventPrefix, callback);\n    }\n\n    /**\n     * Destroy listener.\n     *\n     * @param {String} action\n     */\n    function destroyListener(action) {\n        var selector    = selectorPrefix ? selectorPrefix + ' ' + buttons[action] : buttons[action],\n            elem        = $(selector)[0];\n\n        if (!elem) {\n            return;\n        }\n\n        if (elem.onclick) {\n            elem.onclick = null;\n        }\n\n        $(elem).off('click' + eventPrefix);\n    }\n\n    return {\n\n        /**\n         * Attaches events handlers.\n         *\n         * @param {Object} handlers\n         * @param {String} selectorPref\n         * @param {String} eventPref\n         */\n        on: function (handlers, selectorPref, eventPref) {\n            selectorPrefix = selectorPrefix || selectorPref;\n            eventPrefix = eventPref;\n            _.each(handlers, initListener);\n            selectorPrefix = '';\n        },\n\n        /**\n         * Removes events handlers.\n         *\n         * @param {Object} handlers\n         * @param {String} eventPref\n         */\n        off: function (handlers, eventPref) {\n            eventPrefix = eventPref;\n            _.each(handlers, destroyListener);\n        }\n    };\n});\n","Magento_Ui/js/form/element/multiselect.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'mageUtils',\n    './select'\n], function (_, utils, Select) {\n    'use strict';\n\n    return Select.extend({\n        defaults: {\n            size: 5,\n            elementTmpl: 'ui/form/element/multiselect',\n            listens: {\n                value: 'setDifferedFromDefault setPrepareToSendData'\n            }\n        },\n\n        /**\n         * @inheritdoc\n         */\n        setInitialValue: function () {\n            this._super();\n\n            this.initialValue = utils.copy(this.initialValue);\n\n            return this;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        normalizeData: function (value) {\n            if (utils.isEmpty(value)) {\n                value = [];\n            }\n\n            return _.isString(value) ? value.split(',') : value;\n        },\n\n        /**\n         * Sets the prepared data to dataSource\n         * by path, where key is component link to dataSource with\n         * suffix \"-prepared-for-send\"\n         *\n         * @param {Array} data - current component value\n         */\n        setPrepareToSendData: function (data) {\n            if (_.isUndefined(data) || !data.length) {\n                data = '';\n            }\n\n            this.source.set(this.dataScope + '-prepared-for-send', data);\n        },\n\n        /**\n         * @inheritdoc\n         */\n        getInitialValue: function () {\n            var values = [\n                    this.normalizeData(this.source.get(this.dataScope)),\n                    this.normalizeData(this.default)\n                ],\n                value;\n\n            values.some(function (v) {\n                return _.isArray(v) && (value = utils.copy(v)) && !_.isEmpty(v);\n            });\n\n            return value;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        hasChanged: function () {\n            var value = this.value(),\n                initial = this.initialValue;\n\n            return !utils.equalArrays(value, initial);\n        },\n\n        /**\n         * @inheritdoc\n         */\n        reset: function () {\n            this.value(utils.copy(this.initialValue));\n            this.error(false);\n\n            return this;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        clear: function () {\n            this.value([]);\n            this.error(false);\n\n            return this;\n        }\n    });\n});\n","Magento_Ui/js/form/element/country.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'uiRegistry',\n    './select'\n], function (_, registry, Select) {\n    'use strict';\n\n    return Select.extend({\n        defaults: {\n            imports: {\n                update: '${ $.parentName }.website_id:value'\n            }\n        },\n\n        /**\n         * Filters 'initialOptions' property by 'field' and 'value' passed,\n         * calls 'setOptions' passing the result to it\n         *\n         * @param {*} value\n         * @param {String} field\n         */\n        filter: function (value, field) {\n            var result, defaultCountry, defaultValue;\n\n            if (!field) { //validate field, if we are on update\n                field = this.filterBy.field;\n            }\n\n            this._super(value, field);\n            result = _.filter(this.initialOptions, function (item) {\n\n                if (item[field]) {\n                    return ~item[field].indexOf(value);\n                }\n\n                return false;\n            });\n\n            this.setOptions(result);\n            this.reset();\n\n            if (!this.value()) {\n                defaultCountry = _.filter(result, function (item) {\n                    return item['is_default'] && _.contains(item['is_default'], value);\n                });\n\n                if (defaultCountry.length) {\n                    defaultValue = defaultCountry.shift();\n                    this.value(defaultValue.value);\n                }\n            }\n        }\n    });\n});\n\n","Magento_Ui/js/form/element/website.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'uiRegistry',\n    './select'\n], function (_, registry, Select) {\n    'use strict';\n\n    return Select.extend({\n        defaults: {\n            customerId: null,\n            isGlobalScope: 0\n        },\n\n        /**\n         * Website component constructor.\n         * @returns {exports}\n         */\n        initialize: function () {\n            this._super();\n\n            if (this.customerId || this.isGlobalScope) {\n                this.disable(true);\n            }\n\n            return this;\n        }\n    });\n});\n","Magento_Ui/js/form/element/post-code.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'uiRegistry',\n    './abstract'\n], function (_, registry, Abstract) {\n    'use strict';\n\n    return Abstract.extend({\n        defaults: {\n            imports: {\n                update: '${ $.parentName }.country_id:value'\n            }\n        },\n\n        /**\n         * Initializes observable properties of instance\n         *\n         * @returns {Abstract} Chainable.\n         */\n        initObservable: function () {\n            this._super();\n\n            /**\n             * equalityComparer function\n             *\n             * @returns boolean.\n             */\n            this.value.equalityComparer = function (oldValue, newValue) {\n                return !oldValue && !newValue || oldValue === newValue;\n            };\n\n            return this;\n        },\n\n        /**\n         * @param {String} value\n         */\n        update: function (value) {\n            var country = registry.get(this.parentName + '.' + 'country_id'),\n                options = country.indexedOptions,\n                option = null;\n\n            if (!value) {\n                return;\n            }\n\n            option = options[value];\n\n            if (!option) {\n                return;\n            }\n\n            if (option['is_zipcode_optional']) {\n                this.error(false);\n                this.validation = _.omit(this.validation, 'required-entry');\n            } else {\n                this.validation['required-entry'] = true;\n            }\n\n            this.required(!option['is_zipcode_optional']);\n        }\n    });\n});\n","Magento_Ui/js/form/element/single-checkbox-toggle-notice.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Ui/js/form/element/single-checkbox'\n], function (SingleCheckbox) {\n    'use strict';\n\n    return SingleCheckbox.extend({\n        defaults: {\n            notices: [],\n            tracks: {\n                notice: true\n            }\n        },\n\n        /**\n         * Choose notice on initialization\n         *\n         * @returns {*|void|Element}\n         */\n        initialize: function () {\n            this._super()\n                .chooseNotice();\n\n            return this;\n        },\n\n        /**\n         * Choose notice function\n         *\n         * @returns void\n         */\n        chooseNotice: function () {\n            var checkedNoticeNumber = Number(this.checked());\n\n            this.notice = this.notices[checkedNoticeNumber];\n        },\n\n        /**\n         * Choose notice on update\n         *\n         * @returns void\n         */\n        onUpdate: function () {\n            this._super();\n            this.chooseNotice();\n        }\n    });\n});\n","Magento_Ui/js/form/element/wysiwyg.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'wysiwygAdapter',\n    'Magento_Ui/js/lib/view/utils/async',\n    'underscore',\n    'ko',\n    './abstract',\n    'mage/adminhtml/events',\n    'Magento_Variable/variables'\n], function (wysiwyg, $, _, ko, Abstract, varienGlobalEvents) {\n    'use strict';\n\n    return Abstract.extend({\n        currentWysiwyg: undefined,\n        defaults: {\n            elementSelector: 'textarea',\n            suffixRegExpPattern: '${ $.wysiwygUniqueSuffix }',\n            $wysiwygEditorButton: '',\n            links: {\n                value: '${ $.provider }:${ $.dataScope }'\n            },\n            template: 'ui/form/field',\n            elementTmpl: 'ui/form/element/wysiwyg',\n            content:        '',\n            showSpinner:    false,\n            loading:        false,\n            listens: {\n                disabled: 'setDisabled'\n            }\n        },\n\n        /**\n         *\n         * @returns {} Chainable.\n         */\n        initialize: function () {\n            this._super()\n                .initNodeListener();\n\n            $.async({\n                component: this,\n                selector: 'button'\n            }, function (element) {\n                this.$wysiwygEditorButton = this.$wysiwygEditorButton ?\n                    this.$wysiwygEditorButton.add($(element)) : $(element);\n            }.bind(this));\n\n            // disable editor completely after initialization is field is disabled\n            varienGlobalEvents.attachEventHandler('wysiwygEditorInitialized', function () {\n                if (!_.isUndefined(window.tinyMceEditors)) {\n                    this.currentWysiwyg = window.tinyMceEditors[this.wysiwygId];\n                }\n\n                if (this.disabled()) {\n                    this.setDisabled(true);\n                }\n            }.bind(this));\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initConfig: function (config) {\n            var pattern = config.suffixRegExpPattern || this.constructor.defaults.suffixRegExpPattern;\n\n            pattern = pattern.replace(/\\$/g, '\\\\$&');\n            config.content = config.content.replace(new RegExp(pattern, 'g'), this.getUniqueSuffix(config));\n            this._super();\n\n            return this;\n        },\n\n        /**\n         * Build unique id based on name, underscore separated.\n         *\n         * @param {Object} config\n         */\n        getUniqueSuffix: function (config) {\n            return config.name.replace(/(\\.|-)/g, '_');\n        },\n\n        /**\n         * @inheritdoc\n         */\n        destroy: function () {\n            this._super();\n            wysiwyg.removeEvents(this.wysiwygId);\n        },\n\n        /**\n         *\n         * @returns {exports}\n         */\n        initObservable: function () {\n            this._super()\n                .observe('value');\n\n            return this;\n        },\n\n        /**\n         *\n         * @returns {} Chainable.\n         */\n        initNodeListener: function () {\n            $.async({\n                component: this,\n                selector: this.elementSelector\n            }, this.setElementNode.bind(this));\n\n            return this;\n        },\n\n        /**\n         *\n         * @param {HTMLElement} node\n         */\n        setElementNode: function (node) {\n            $(node).bindings({\n                value: this.value\n            });\n        },\n\n        /**\n         * Set disabled property to wysiwyg component\n         *\n         * @param {Boolean} disabled\n         */\n        setDisabled: function (disabled) {\n            if (this.$wysiwygEditorButton && disabled) {\n                this.$wysiwygEditorButton.prop('disabled', 'disabled');\n            } else if (this.$wysiwygEditorButton) {\n                this.$wysiwygEditorButton.removeProp('disabled');\n            }\n\n            /* eslint-disable no-undef */\n            if (!_.isUndefined(this.currentWysiwyg) && this.currentWysiwyg.activeEditor()) {\n                this.currentWysiwyg.setEnabledStatus(!disabled);\n                this.currentWysiwyg.getPluginButtons().prop('disabled', disabled);\n            }\n        }\n    });\n});\n","Magento_Ui/js/form/element/select.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'mageUtils',\n    'uiRegistry',\n    './abstract',\n    'uiLayout'\n], function (_, utils, registry, Abstract, layout) {\n    'use strict';\n\n    var inputNode = {\n        parent: '${ $.$data.parentName }',\n        component: 'Magento_Ui/js/form/element/abstract',\n        template: '${ $.$data.template }',\n        provider: '${ $.$data.provider }',\n        name: '${ $.$data.index }_input',\n        dataScope: '${ $.$data.customEntry }',\n        customScope: '${ $.$data.customScope }',\n        sortOrder: {\n            after: '${ $.$data.name }'\n        },\n        displayArea: 'body',\n        label: '${ $.$data.label }'\n    };\n\n    /**\n     * Parses incoming options, considers options with undefined value property\n     *     as caption\n     *\n     * @param  {Array} nodes\n     * @return {Object}\n     */\n    function parseOptions(nodes, captionValue) {\n        var caption,\n            value;\n\n        nodes = _.map(nodes, function (node) {\n            value = node.value;\n\n            if (value === null || value === captionValue) {\n                if (_.isUndefined(caption)) {\n                    caption = node.label;\n                }\n            } else {\n                return node;\n            }\n        });\n\n        return {\n            options: _.compact(nodes),\n            caption: _.isString(caption) ? caption : false\n        };\n    }\n\n    /**\n     * Recursively loops over data to find non-undefined, non-array value\n     *\n     * @param  {Array} data\n     * @return {*} - first non-undefined value in array\n     */\n    function findFirst(data) {\n        var value;\n\n        data.some(function (node) {\n            value = node.value;\n\n            if (Array.isArray(value)) {\n                value = findFirst(value);\n            }\n\n            return !_.isUndefined(value);\n        });\n\n        return value;\n    }\n\n    /**\n     * Recursively set to object item like value and item.value like key.\n     *\n     * @param {Array} data\n     * @param {Object} result\n     * @returns {Object}\n     */\n    function indexOptions(data, result) {\n        var value;\n\n        result = result || {};\n\n        data.forEach(function (item) {\n            value = item.value;\n\n            if (Array.isArray(value)) {\n                indexOptions(value, result);\n            } else {\n                result[value] = item;\n            }\n        });\n\n        return result;\n    }\n\n    return Abstract.extend({\n        defaults: {\n            customName: '${ $.parentName }.${ $.index }_input',\n            elementTmpl: 'ui/form/element/select',\n            caption: '',\n            options: []\n        },\n\n        /**\n         * Extends instance with defaults, extends config with formatted values\n         *     and options, and invokes initialize method of AbstractElement class.\n         *     If instance's 'customEntry' property is set to true, calls 'initInput'\n         */\n        initialize: function () {\n            this._super();\n\n            if (this.customEntry) {\n                registry.get(this.name, this.initInput.bind(this));\n            }\n\n            if (this.filterBy) {\n                this.initFilter();\n            }\n\n            return this;\n        },\n\n        /**\n         * Calls 'initObservable' of parent, initializes 'options' and 'initialOptions'\n         *     properties, calls 'setOptions' passing options to it\n         *\n         * @returns {Object} Chainable.\n         */\n        initObservable: function () {\n            this._super();\n\n            this.initialOptions = this.options;\n\n            this.observe('options caption')\n                .setOptions(this.options());\n\n            return this;\n        },\n\n        /**\n         * Set link for filter.\n         *\n         * @returns {Object} Chainable\n         */\n        initFilter: function () {\n            var filter = this.filterBy;\n\n            this.filter(this.default, filter.field);\n            this.setLinks({\n                filter: filter.target\n            }, 'imports');\n\n            return this;\n        },\n\n        /**\n         * Creates input from template, renders it via renderer.\n         *\n         * @returns {Object} Chainable.\n         */\n        initInput: function () {\n            layout([utils.template(inputNode, this)]);\n\n            return this;\n        },\n\n        /**\n         * Matches specified value with existing options\n         * or, if value is not specified, returns value of the first option.\n         *\n         * @returns {*}\n         */\n        normalizeData: function () {\n            var value = this._super(),\n                option;\n\n            if (value !== '') {\n                option = this.getOption(value);\n\n                return option && option.value;\n            }\n\n            if (!this.caption()) {\n                return findFirst(this.options);\n            }\n        },\n\n        /**\n         * Filters 'initialOptions' property by 'field' and 'value' passed,\n         * calls 'setOptions' passing the result to it\n         *\n         * @param {*} value\n         * @param {String} field\n         */\n        filter: function (value, field) {\n            var source = this.initialOptions,\n                result;\n\n            field = field || this.filterBy.field;\n\n            result = _.filter(source, function (item) {\n                return item[field] === value || item.value === '';\n            });\n\n            this.setOptions(result);\n        },\n\n        /**\n         * Change visibility for input.\n         *\n         * @param {Boolean} isVisible\n         */\n        toggleInput: function (isVisible) {\n            registry.get(this.customName, function (input) {\n                input.setVisible(isVisible);\n            });\n        },\n\n        /**\n         * Sets 'data' to 'options' observable array, if instance has\n         * 'customEntry' property set to true, calls 'setHidden' method\n         *  passing !options.length as a parameter\n         *\n         * @param {Array} data\n         * @returns {Object} Chainable\n         */\n        setOptions: function (data) {\n            var captionValue = this.captionValue || '',\n                result = parseOptions(data, captionValue),\n                isVisible;\n\n            this.indexedOptions = indexOptions(result.options);\n\n            this.options(result.options);\n\n            if (!this.caption()) {\n                this.caption(result.caption);\n            }\n\n            if (this.customEntry) {\n                isVisible = !!result.options.length;\n\n                this.setVisible(isVisible);\n                this.toggleInput(!isVisible);\n            }\n\n            return this;\n        },\n\n        /**\n         * Processes preview for option by it's value, and sets the result\n         * to 'preview' observable\n         *\n         * @returns {Object} Chainable.\n         */\n        getPreview: function () {\n            var value = this.value(),\n                option = this.indexedOptions[value],\n                preview = option ? option.label : '';\n\n            this.preview(preview);\n\n            return preview;\n        },\n\n        /**\n         * Get option from indexedOptions list.\n         *\n         * @param {Number} value\n         * @returns {Object} Chainable\n         */\n        getOption: function (value) {\n            return this.indexedOptions[value];\n        },\n\n        /**\n         * Select first available option\n         *\n         * @returns {Object} Chainable.\n         */\n        clear: function () {\n            var value = this.caption() ? '' : findFirst(this.options);\n\n            this.value(value);\n\n            return this;\n        },\n\n        /**\n         * Initializes observable properties of instance\n         *\n         * @returns {Object} Chainable.\n         */\n        setInitialValue: function () {\n            if (_.isUndefined(this.value()) && !this.default) {\n                this.clear();\n            }\n\n            return this._super();\n        }\n    });\n});\n","Magento_Ui/js/form/element/single-checkbox-use-config.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Ui/js/form/element/single-checkbox'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            isUseDefault: false,\n            isUseConfig: false,\n            listens: {\n                'isUseConfig': 'toggleElement',\n                'isUseDefault': 'toggleElement'\n            }\n        },\n\n        /**\n         * @inheritdoc\n         */\n        initObservable: function () {\n\n            return this\n                ._super()\n                .observe('isUseConfig');\n        },\n\n        /**\n         * Toggle element\n         */\n        toggleElement: function () {\n            this.disabled(this.isUseDefault() || this.isUseConfig());\n\n            if (this.source) {\n                this.source.set('data.use_default.' + this.index, Number(this.isUseDefault()));\n            }\n        }\n    });\n});\n","Magento_Ui/js/form/element/image-uploader.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global Base64 */\ndefine([\n    'jquery',\n    'underscore',\n    'mageUtils',\n    'Magento_Ui/js/modal/alert',\n    'Magento_Ui/js/lib/validation/validator',\n    'Magento_Ui/js/form/element/file-uploader',\n    'mage/adminhtml/browser'\n], function ($, _, utils, uiAlert, validator, Element, browser) {\n    'use strict';\n\n    return Element.extend({\n        /**\n         * {@inheritDoc}\n         */\n        initialize: function () {\n            this._super();\n\n            // Listen for file deletions from the media browser\n            $(window).on('fileDeleted.mediabrowser', this.onDeleteFile.bind(this));\n        },\n\n        /**\n         * Assign uid for media gallery\n         *\n         * @return {ImageUploader} Chainable.\n         */\n        initConfig: function () {\n            var mediaGalleryUid = utils.uniqueid();\n\n            this._super();\n\n            _.extend(this, {\n                mediaGalleryUid: mediaGalleryUid\n            });\n\n            return this;\n        },\n\n        /**\n         * Add file event callback triggered from media gallery\n         *\n         * @param {ImageUploader} imageUploader - UI Class\n         * @param {Event} e\n         */\n        addFileFromMediaGallery: function (imageUploader, e) {\n            var $buttonEl = $(e.target),\n                fileSize = $buttonEl.data('size'),\n                fileMimeType = $buttonEl.data('mime-type'),\n                filePathname = $buttonEl.val(),\n                fileBasename = filePathname.split('/').pop();\n\n            this.addFile({\n                type: fileMimeType,\n                name: fileBasename,\n                size: fileSize,\n                url: filePathname\n            });\n        },\n\n        /**\n         * Open the media browser dialog\n         *\n         * @param {ImageUploader} imageUploader - UI Class\n         * @param {Event} e\n         */\n        openMediaBrowserDialog: function (imageUploader, e) {\n            var $buttonEl = $(e.target),\n                openDialogUrl = this.mediaGallery.openDialogUrl +\n                'target_element_id/' + $buttonEl.attr('id') +\n                '/store/' + this.mediaGallery.storeId +\n                '/type/image/?isAjax=true';\n\n            if (this.mediaGallery.initialOpenSubpath) {\n                openDialogUrl += '&current_tree_path=' + Base64.mageEncode(this.mediaGallery.initialOpenSubpath);\n            }\n\n            browser.openDialog(openDialogUrl, null, null, this.mediaGallery.openDialogTitle);\n        },\n\n        /**\n         * @param {jQuery.event} e\n         * @param {Object} data\n         * @returns {Object} Chainables\n         */\n        onDeleteFile: function (e, data) {\n            var fileId = this.getFileId(),\n                deletedFileIds = data.ids;\n\n            if (fileId && $.inArray(fileId, deletedFileIds) > -1) {\n                this.clear();\n            }\n\n            return this;\n        },\n\n        /**\n         * {@inheritDoc}\n         */\n        clear: function () {\n            this.value([]);\n\n            return this;\n        },\n\n        /**\n         * Gets the ID of the file used if set\n         *\n         * @return {String|Null} ID\n         */\n        getFileId: function () {\n            return this.hasData() ? this.value()[0].id : null;\n        },\n\n        /**\n         * Trigger native browser file upload UI via clicking on 'Upload' button\n         *\n         * @param {ImageUploader} imageUploader - UI Class\n         * @param {Event} e\n         */\n        triggerImageUpload: function (imageUploader, e) {\n            $(e.target).closest('.file-uploader').find('input[type=\"file\"]').click();\n        },\n\n        /**\n         * Get list of file extensions allowed in comma delimited format\n         *\n         * @return {String}\n         */\n        getAllowedFileExtensionsInCommaDelimitedFormat: function () {\n            var allowedExtensions = this.allowedExtensions.toUpperCase().split(' ');\n\n            // if jpg and jpeg in allowed extensions, remove jpeg from list\n            if (allowedExtensions.indexOf('JPG') !== -1 && allowedExtensions.indexOf('JPEG') !== -1) {\n                allowedExtensions.splice(allowedExtensions.indexOf('JPEG'), 1);\n            }\n\n            return allowedExtensions.join(', ');\n        }\n    });\n});\n","Magento_Ui/js/form/element/single-checkbox.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Ui/js/form/element/abstract',\n    'underscore',\n    'mage/translate'\n], function (AbstractField, _, $t) {\n    'use strict';\n\n    return AbstractField.extend({\n        defaults: {\n            template: 'ui/form/components/single/field',\n            checked: false,\n            initialChecked: false,\n            multiple: false,\n            prefer: 'checkbox', // 'radio' | 'checkbox' | 'toggle'\n            valueMap: {},\n\n            templates: {\n                radio: 'ui/form/components/single/radio',\n                checkbox: 'ui/form/components/single/checkbox',\n                toggle: 'ui/form/components/single/switcher'\n            },\n\n            listens: {\n                'checked': 'onCheckedChanged',\n                'value': 'onExtendedValueChanged'\n            }\n        },\n\n        /**\n         * @inheritdoc\n         */\n        initConfig: function (config) {\n            this._super();\n\n            if (!config.elementTmpl) {\n                if (!this.prefer && !this.multiple) {\n                    this.elementTmpl = this.templates.radio;\n                } else if (this.prefer === 'radio') {\n                    this.elementTmpl = this.templates.radio;\n                } else if (this.prefer === 'checkbox') {\n                    this.elementTmpl = this.templates.checkbox;\n                } else if (this.prefer === 'toggle') {\n                    this.elementTmpl = this.templates.toggle;\n                } else {\n                    this.elementTmpl = this.templates.checkbox;\n                }\n            }\n\n            if (this.prefer === 'toggle' && _.isEmpty(this.toggleLabels)) {\n                this.toggleLabels = {\n                    'on': $t('Yes'),\n                    'off': $t('No')\n                };\n            }\n\n            if (typeof this.default === 'undefined' || this.default === null) {\n                this.default = '';\n            }\n\n            if (typeof this.value === 'undefined' || this.value === null) {\n                this.value = _.isEmpty(this.valueMap) || this.default !== '' ? this.default : this.valueMap.false;\n                this.initialValue = this.value;\n            } else {\n                this.initialValue = this.value;\n            }\n\n            if (this.multiple && !_.isArray(this.value)) {\n                this.value = []; // needed for correct observable assignment\n            }\n\n            this.initialChecked = this.checked;\n\n            return this;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        initObservable: function () {\n            return this\n                ._super()\n                .observe('checked');\n        },\n\n        /**\n         * Get true/false key from valueMap by value.\n         *\n         * @param {*} value\n         * @returns {Boolean|undefined}\n         */\n        getReverseValueMap: function getReverseValueMap(value) {\n            var bool = false;\n\n            _.some(this.valueMap, function (iValue, iBool) {\n                if (iValue === value) {\n                    bool = iBool === 'true';\n\n                    return true;\n                }\n            });\n\n            return bool;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        setInitialValue: function () {\n            if (_.isEmpty(this.valueMap)) {\n                this.on('value', this.onUpdate.bind(this));\n            } else {\n                this._super();\n                this.checked(this.getReverseValueMap(this.value()));\n            }\n\n            return this;\n        },\n\n        /**\n         * Handle dataScope changes for checkbox / radio button.\n         *\n         * @param {*} newExportedValue\n         */\n        onExtendedValueChanged: function (newExportedValue) {\n            var isMappedUsed = !_.isEmpty(this.valueMap),\n                oldChecked = this.checked.peek(),\n                oldValue = this.initialValue,\n                newChecked;\n\n            if (this.multiple) {\n                newChecked = newExportedValue.indexOf(oldValue) !== -1;\n            } else if (isMappedUsed) {\n                newChecked = this.getReverseValueMap(newExportedValue);\n            } else if (typeof newExportedValue === 'boolean') {\n                newChecked = newExportedValue;\n            } else {\n                newChecked = newExportedValue === oldValue;\n            }\n\n            if (newChecked !== oldChecked) {\n                this.checked(newChecked);\n            }\n        },\n\n        /**\n         * Handle checked state changes for checkbox / radio button.\n         *\n         * @param {Boolean} newChecked\n         */\n        onCheckedChanged: function (newChecked) {\n            var isMappedUsed = !_.isEmpty(this.valueMap),\n                oldValue = this.initialValue,\n                newValue;\n\n            if (isMappedUsed) {\n                newValue = this.valueMap[newChecked];\n            } else {\n                newValue = oldValue;\n            }\n\n            if (!this.multiple && newChecked) {\n                this.value(newValue);\n            } else if (!this.multiple && !newChecked) {\n                if (typeof newValue === 'boolean') {\n                    this.value(newChecked);\n                } else if (newValue === this.value.peek()) {\n                    this.value('');\n                }\n\n                if (isMappedUsed) {\n                    this.value(newValue);\n                }\n            } else if (this.multiple && newChecked && this.value.indexOf(newValue) === -1) {\n                this.value.push(newValue);\n            } else if (this.multiple && !newChecked && this.value.indexOf(newValue) !== -1) {\n                this.value.splice(this.value.indexOf(newValue), 1);\n            }\n        },\n\n        /**\n         * @inheritdoc\n         */\n        onUpdate: function () {\n            if (this.hasUnique) {\n                this.setUnique();\n            }\n\n            return this._super();\n        },\n\n        /**\n         * @inheritdoc\n         */\n        reset: function () {\n            if (this.multiple && this.initialChecked) {\n                this.value.push(this.initialValue);\n            } else if (this.multiple && !this.initialChecked) {\n                this.value.splice(this.value.indexOf(this.initialValue), 1);\n            } else {\n                this.value(this.initialValue);\n            }\n\n            this.error(false);\n\n            return this;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        clear: function () {\n            if (this.multiple) {\n                this.value([]);\n            } else {\n                this.value('');\n            }\n\n            this.error(false);\n\n            return this;\n        }\n    });\n});\n","Magento_Ui/js/form/element/region.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'uiRegistry',\n    './select',\n    'Magento_Checkout/js/model/default-post-code-resolver'\n], function (_, registry, Select, defaultPostCodeResolver) {\n    'use strict';\n\n    return Select.extend({\n        defaults: {\n            skipValidation: false,\n            imports: {\n                update: '${ $.parentName }.country_id:value'\n            }\n        },\n\n        /**\n         * @param {String} value\n         */\n        update: function (value) {\n            var country = registry.get(this.parentName + '.' + 'country_id'),\n                options = country.indexedOptions,\n                isRegionRequired,\n                option;\n\n            if (!value) {\n                return;\n            }\n            option = options[value];\n\n            if (typeof option === 'undefined') {\n                return;\n            }\n\n            defaultPostCodeResolver.setUseDefaultPostCode(!option['is_zipcode_optional']);\n\n            if (this.skipValidation) {\n                this.validation['required-entry'] = false;\n                this.required(false);\n            } else {\n                if (option && !option['is_region_required']) {\n                    this.error(false);\n                    this.validation = _.omit(this.validation, 'required-entry');\n                    registry.get(this.customName, function (input) {\n                        input.validation['required-entry'] = false;\n                        input.required(false);\n                    });\n                } else {\n                    this.validation['required-entry'] = true;\n                }\n\n                if (option && !this.options().length) {\n                    registry.get(this.customName, function (input) {\n                        isRegionRequired = !!option['is_region_required'];\n                        input.validation['required-entry'] = isRegionRequired;\n                        input.validation['validate-not-number-first'] = true;\n                        input.required(isRegionRequired);\n                    });\n                }\n\n                this.required(!!option['is_region_required']);\n            }\n        },\n\n        /**\n         * Filters 'initialOptions' property by 'field' and 'value' passed,\n         * calls 'setOptions' passing the result to it\n         *\n         * @param {*} value\n         * @param {String} field\n         */\n        filter: function (value, field) {\n            var superFn = this._super;\n\n            registry.get(this.parentName + '.' + 'country_id', function (country) {\n                var option = country.indexedOptions[value];\n\n                superFn.call(this, value, field);\n\n                if (option && option['is_region_visible'] === false) {\n                    // hide select and corresponding text input field if region must not be shown for selected country\n                    this.setVisible(false);\n\n                    if (this.customEntry) {// eslint-disable-line max-depth\n                        this.toggleInput(false);\n                    }\n                }\n            }.bind(this));\n        }\n    });\n});\n\n","Magento_Ui/js/form/element/boolean.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    './abstract'\n], function (Abstract) {\n    'use strict';\n\n    return Abstract.extend({\n        defaults: {\n            checked: false,\n            links: {\n                checked: 'value'\n            }\n        },\n\n        /**\n         * @returns {*|void|Element}\n         */\n        initObservable: function () {\n            return this._super()\n                    .observe('checked');\n        },\n\n        /**\n         * Converts provided value to boolean.\n         *\n         * @returns {Boolean}\n         */\n        normalizeData: function () {\n            return !!+this._super();\n        },\n\n        /**\n         * Calls 'onUpdate' method of parent, if value is defined and instance's\n         *     'unique' property set to true, calls 'setUnique' method\n         *\n         * @return {Object} - reference to instance\n         */\n        onUpdate: function () {\n            if (this.hasUnique) {\n                this.setUnique();\n            }\n\n            return this._super();\n        }\n    });\n});\n","Magento_Ui/js/form/element/color-picker.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'mage/translate',\n    'Magento_Ui/js/form/element/abstract',\n    'Magento_Ui/js/form/element/color-picker-palette'\n], function ($t, Abstract, palette) {\n    'use strict';\n\n    return Abstract.extend({\n\n        defaults: {\n            colorPickerConfig: {\n                chooseText: $t('Apply'),\n                cancelText: $t('Cancel'),\n                maxSelectionSize: 8,\n                clickoutFiresChange: true,\n                allowEmpty: true,\n                localStorageKey: 'magento.spectrum',\n                palette: palette\n            }\n        },\n\n        /**\n         * Invokes initialize method of parent class,\n         * contains initialization logic\n         */\n        initialize: function () {\n            this._super();\n\n            this.colorPickerConfig.value = this.value;\n\n            return this;\n        }\n    });\n});\n","Magento_Ui/js/form/element/textarea.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    './abstract'\n], function (Abstract) {\n    'use strict';\n\n    return Abstract.extend({\n        defaults: {\n            cols: 15,\n            rows: 2,\n            elementTmpl: 'ui/form/element/textarea'\n        }\n    });\n});\n","Magento_Ui/js/form/element/url-input.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'uiLayout',\n    'mage/translate',\n    'Magento_Ui/js/form/element/abstract'\n], function (_, layout, $t, Abstract) {\n    'use strict';\n\n    return Abstract.extend({\n        defaults: {\n            linkedElement: {},\n            settingTemplate: 'ui/form/element/urlInput/setting',\n            typeSelectorTemplate: 'ui/form/element/urlInput/typeSelector',\n            options: [],\n            linkedElementInstances: {},\n            //checkbox\n            isDisplayAdditionalSettings: true,\n            settingValue: false,\n            settingLabel: $t('Open in new tab'),\n            tracks: {\n                linkedElement: true\n            },\n            baseLinkSetting: {\n                namePrefix: '${$.name}.',\n                dataScopePrefix: '${$.dataScope}.',\n                provider: '${$.provider}'\n            },\n            urlTypes: {},\n            listens: {\n                settingValue: 'checked',\n                disabled: 'hideLinkedElement',\n                linkType: 'createChildUrlInputComponent'\n            },\n            links: {\n                linkType: '${$.provider}:${$.dataScope}.type',\n                settingValue: '${$.provider}:${$.dataScope}.setting'\n            }\n        },\n\n        /** @inheritdoc */\n        initConfig: function (config) {\n            var processedLinkTypes = {},\n                baseLinkType = this.constructor.defaults.baseLinkSetting;\n\n            _.each(config.urlTypes, function (linkSettingsArray, linkName) {\n                //add link name by link type\n                linkSettingsArray.name = baseLinkType.namePrefix + linkName;\n                linkSettingsArray.dataScope = baseLinkType.dataScopePrefix + linkName;\n                linkSettingsArray.type = linkName;\n                linkSettingsArray.disabled = config.disabled;\n                linkSettingsArray.visible = config.visible;\n                processedLinkTypes[linkName] = {};\n                _.extend(processedLinkTypes[linkName], baseLinkType, linkSettingsArray);\n            });\n            _.extend(this.constructor.defaults.urlTypes, processedLinkTypes);\n\n            this._super();\n        },\n\n        /**\n         * Initializes observable properties of instance\n         *\n         * @returns {Abstract} Chainable.\n         */\n        initObservable: function () {\n            this._super()\n                .observe('componentTemplate options value linkType settingValue checked isDisplayAdditionalSettings')\n                .setOptions();\n\n            return this;\n        },\n\n        /**\n         * Set options to select based on link types configuration\n         *\n         * @return {Object}\n         */\n        setOptions: function () {\n            var result = [];\n\n            _.each(this.urlTypes, function (option, key) {\n                result.push({\n                    value: key,\n                    label: option.label,\n                    sortOrder: option.sortOrder || 0\n                });\n            });\n\n            //sort options by sortOrder\n            result.sort(function (a, b) {\n                return a.sortOrder > b.sortOrder ? 1 : -1;\n            });\n\n            this.options(result);\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        setPreview: function (visible) {\n            this.linkedElement().visible(visible);\n        },\n\n        /**\n         * Initializes observable properties of instance\n         *\n         * @param {Boolean} disabled\n         */\n        hideLinkedElement: function (disabled) {\n            this.linkedElement().disabled(disabled);\n        },\n\n        /** @inheritdoc */\n        destroy: function () {\n            _.each(this.linkedElementInstances, function (value) {\n                value().destroy();\n            });\n            this._super();\n        },\n\n        /**\n         * Create child component by value\n         *\n         * @param {String} value\n         * @return void\n         */\n        createChildUrlInputComponent: function (value) {\n            var elementConfig;\n\n            if (!_.isEmpty(value) && _.isUndefined(this.linkedElementInstances[value])) {\n                elementConfig = this.urlTypes[value];\n                layout([elementConfig]);\n                this.linkedElementInstances[value] = this.requestModule(elementConfig.name);\n            }\n            this.linkedElement = this.linkedElementInstances[value];\n\n        },\n\n        /**\n         * Returns linked element to display related field in template\n         * @return String\n         */\n        getLinkedElementName: function () {\n            return this.linkedElement;\n        },\n\n        /**\n         * Add ability to choose check box by clicking on label\n         */\n        checkboxClick: function () {\n            if (!this.disabled()) {\n                this.settingValue(!this.settingValue());\n            }\n        }\n    });\n});\n","Magento_Ui/js/form/element/checkbox-set.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'mageUtils',\n    './abstract'\n], function (_, utils, Abstract) {\n    'use strict';\n\n    return Abstract.extend({\n        defaults: {\n            template: 'ui/form/element/checkbox-set',\n            multiple: false,\n            multipleScopeValue: null\n        },\n\n        /**\n         * @inheritdoc\n         */\n        initConfig: function () {\n            this._super();\n\n            this.value = this.normalizeData(this.value);\n\n            return this;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        initLinks: function () {\n            var scope = this.source.get(this.dataScope);\n\n            this.multipleScopeValue = this.multiple && _.isArray(scope) ? utils.copy(scope) : undefined;\n\n            return this._super();\n        },\n\n        /**\n         * @inheritdoc\n         */\n        reset: function () {\n            this.value(utils.copy(this.initialValue));\n            this.error(false);\n\n            return this;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        clear: function () {\n            var value = this.multiple ? [] : '';\n\n            this.value(value);\n            this.error(false);\n\n            return this;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        normalizeData: function (value) {\n            if (!this.multiple) {\n                return this._super();\n            }\n\n            return _.isArray(value) ? utils.copy(value) : [];\n        },\n\n        /**\n         * @inheritdoc\n         */\n        setInitialValue: function () {\n            this._super();\n\n            this.initialValue = utils.copy(this.initialValue);\n\n            return this;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        getInitialValue: function () {\n            var values = [this.multipleScopeValue, this.default, this.value.peek(), []],\n                value;\n\n            if (!this.multiple) {\n                return this._super();\n            }\n\n            values.some(function (v) {\n                return _.isArray(v) && (value = utils.copy(v));\n            });\n\n            return value;\n        },\n\n        /**\n         * Returns labels which matches current value.\n         *\n         * @returns {String|Array}\n         */\n        getPreview: function () {\n            var option;\n\n            if (!this.multiple) {\n                option = this.getOption(this.value());\n\n                return option ? option.label : '';\n            }\n\n            return this.value.map(function (value) {\n                return this.getOption(value).label;\n            }, this);\n        },\n\n        /**\n         * Returns option object associated with provided value.\n         *\n         * @param {String} value\n         * @returns {Object}\n         */\n        getOption: function (value) {\n            return _.findWhere(this.options, {\n                value: value\n            });\n        },\n\n        /**\n         * @inheritdoc\n         */\n        hasChanged: function () {\n            var value = this.value(),\n                initial = this.initialValue;\n\n            return this.multiple ?\n                !utils.equalArrays(value, initial) :\n                this._super();\n        }\n    });\n});\n","Magento_Ui/js/form/element/abstract.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'mageUtils',\n    'uiLayout',\n    'uiElement',\n    'Magento_Ui/js/lib/validation/validator'\n], function (_, utils, layout, Element, validator) {\n    'use strict';\n\n    return Element.extend({\n        defaults: {\n            visible: true,\n            preview: '',\n            focused: false,\n            required: false,\n            disabled: false,\n            valueChangedByUser: false,\n            elementTmpl: 'ui/form/element/input',\n            tooltipTpl: 'ui/form/element/helper/tooltip',\n            fallbackResetTpl: 'ui/form/element/helper/fallback-reset',\n            'input_type': 'input',\n            placeholder: false,\n            description: '',\n            labelVisible: true,\n            label: '',\n            error: '',\n            warn: '',\n            notice: '',\n            customScope: '',\n            default: '',\n            isDifferedFromDefault: false,\n            showFallbackReset: false,\n            additionalClasses: {},\n            isUseDefault: '',\n            serviceDisabled: false,\n            valueUpdate: false, // ko binding valueUpdate\n\n            switcherConfig: {\n                component: 'Magento_Ui/js/form/switcher',\n                name: '${ $.name }_switcher',\n                target: '${ $.name }',\n                property: 'value'\n            },\n            listens: {\n                visible: 'setPreview',\n                value: 'setDifferedFromDefault',\n                '${ $.provider }:data.reset': 'reset',\n                '${ $.provider }:data.overload': 'overload',\n                '${ $.provider }:${ $.customScope ? $.customScope + \".\" : \"\"}data.validate': 'validate',\n                'isUseDefault': 'toggleUseDefault'\n            },\n            ignoreTmpls: {\n                value: true\n            },\n\n            links: {\n                value: '${ $.provider }:${ $.dataScope }'\n            }\n        },\n\n        /**\n         * Invokes initialize method of parent class,\n         * contains initialization logic\n         */\n        initialize: function () {\n            _.bindAll(this, 'reset');\n\n            this._super()\n                .setInitialValue()\n                ._setClasses()\n                .initSwitcher();\n\n            return this;\n        },\n\n        /**\n         * Checks if component has error.\n         *\n         * @returns {Object}\n         */\n        checkInvalid: function () {\n            return this.error() && this.error().length ? this : null;\n        },\n\n        /**\n         * Initializes observable properties of instance\n         *\n         * @returns {Abstract} Chainable.\n         */\n        initObservable: function () {\n            var rules = this.validation = this.validation || {};\n\n            this._super();\n\n            this.observe('error disabled focused preview visible value warn notice isDifferedFromDefault')\n                .observe('isUseDefault serviceDisabled')\n                .observe({\n                    'required': !!rules['required-entry']\n                });\n\n            return this;\n        },\n\n        /**\n         * Initializes regular properties of instance.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        initConfig: function () {\n            var uid = utils.uniqueid(),\n                name,\n                valueUpdate,\n                scope;\n\n            this._super();\n\n            scope = this.dataScope.split('.');\n            name = scope.length > 1 ? scope.slice(1) : scope;\n\n            valueUpdate = this.showFallbackReset ? 'afterkeydown' : this.valueUpdate;\n\n            _.extend(this, {\n                uid: uid,\n                noticeId: 'notice-' + uid,\n                errorId: 'error-' + uid,\n                inputName: utils.serializeName(name.join('.')),\n                valueUpdate: valueUpdate\n            });\n\n            return this;\n        },\n\n        /**\n         * Initializes switcher element instance.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        initSwitcher: function () {\n            if (this.switcherConfig.enabled) {\n                layout([this.switcherConfig]);\n            }\n\n            return this;\n        },\n\n        /**\n         * Sets initial value of the element and subscribes to it's changes.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        setInitialValue: function () {\n            this.initialValue = this.getInitialValue();\n\n            if (this.value.peek() !== this.initialValue) {\n                this.value(this.initialValue);\n            }\n\n            this.on('value', this.onUpdate.bind(this));\n            this.isUseDefault(this.disabled());\n\n            return this;\n        },\n\n        /**\n         * Extends 'additionalClasses' object.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        _setClasses: function () {\n            var additional = this.additionalClasses;\n\n            if (_.isString(additional)) {\n                this.additionalClasses = {};\n\n                if (additional.trim().length) {\n                    additional = additional.trim().split(' ');\n\n                    additional.forEach(function (name) {\n                        if (name.length) {\n                            this.additionalClasses[name] = true;\n                        }\n                    }, this);\n                }\n            }\n\n            _.extend(this.additionalClasses, {\n                _required: this.required,\n                _error: this.error,\n                _warn: this.warn,\n                _disabled: this.disabled\n            });\n\n            return this;\n        },\n\n        /**\n         * Gets initial value of element\n         *\n         * @returns {*} Elements' value.\n         */\n        getInitialValue: function () {\n            var values = [this.value(), this.default],\n                value;\n\n            values.some(function (v) {\n                if (v !== null && v !== undefined) {\n                    value = v;\n\n                    return true;\n                }\n\n                return false;\n            });\n\n            return this.normalizeData(value);\n        },\n\n        /**\n         * Sets 'value' as 'hidden' property's value, triggers 'toggle' event,\n         * sets instance's hidden identifier in params storage based on\n         * 'value'.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        setVisible: function (isVisible) {\n            this.visible(isVisible);\n\n            return this;\n        },\n\n        /**\n         * Show element.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        show: function () {\n            this.visible(true);\n\n            return this;\n        },\n\n        /**\n         * Hide element.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        hide: function () {\n            this.visible(false);\n\n            return this;\n        },\n\n        /**\n         * Disable element.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        disable: function () {\n            this.disabled(true);\n\n            return this;\n        },\n\n        /**\n         * Enable element.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        enable: function () {\n            this.disabled(false);\n\n            return this;\n        },\n\n        /**\n         *\n         * @param {(String|Object)} rule\n         * @param {(Object|Boolean)} [options]\n         * @returns {Abstract} Chainable.\n         */\n        setValidation: function (rule, options) {\n            var rules = utils.copy(this.validation),\n                changed;\n\n            if (_.isObject(rule)) {\n                _.extend(this.validation, rule);\n            } else {\n                this.validation[rule] = options;\n            }\n\n            changed = !utils.compare(rules, this.validation).equal;\n\n            if (changed) {\n                this.required(!!rules['required-entry']);\n                this.validate();\n            }\n\n            return this;\n        },\n\n        /**\n         * Returns unwrapped preview observable.\n         *\n         * @returns {String} Value of the preview observable.\n         */\n        getPreview: function () {\n            return this.value();\n        },\n\n        /**\n         * Checks if element has addons\n         *\n         * @returns {Boolean}\n         */\n        hasAddons: function () {\n            return this.addbefore || this.addafter;\n        },\n\n        /**\n         * Checks if element has service setting\n         *\n         * @returns {Boolean}\n         */\n        hasService: function () {\n            return this.service && this.service.template;\n        },\n\n        /**\n         * Defines if value has changed.\n         *\n         * @returns {Boolean}\n         */\n        hasChanged: function () {\n            var notEqual = this.value() !== this.initialValue;\n\n            return !this.visible() ? false : notEqual;\n        },\n\n        /**\n         * Checks if 'value' is not empty.\n         *\n         * @returns {Boolean}\n         */\n        hasData: function () {\n            return !utils.isEmpty(this.value());\n        },\n\n        /**\n         * Sets value observable to initialValue property.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        reset: function () {\n            this.value(this.initialValue);\n            this.error(false);\n\n            return this;\n        },\n\n        /**\n         * Sets current state as initial.\n         */\n        overload: function () {\n            this.setInitialValue();\n            this.bubble('update', this.hasChanged());\n        },\n\n        /**\n         * Clears 'value' property.\n         *\n         * @returns {Abstract} Chainable.\n         */\n        clear: function () {\n            this.value('');\n\n            return this;\n        },\n\n        /**\n         * Converts values like 'null' or 'undefined' to an empty string.\n         *\n         * @param {*} value - Value to be processed.\n         * @returns {*}\n         */\n        normalizeData: function (value) {\n            return utils.isEmpty(value) ? '' : value;\n        },\n\n        /**\n         * Validates itself by it's validation rules using validator object.\n         * If validation of a rule did not pass, writes it's message to\n         * 'error' observable property.\n         *\n         * @returns {Object} Validate information.\n         */\n        validate: function () {\n            var value = this.value(),\n                result = validator(this.validation, value, this.validationParams),\n                message = !this.disabled() && this.visible() ? result.message : '',\n                isValid = this.disabled() || !this.visible() || result.passed;\n\n            this.error(message);\n            this.error.valueHasMutated();\n            this.bubble('error', message);\n\n            //TODO: Implement proper result propagation for form\n            if (this.source && !isValid) {\n                this.source.set('params.invalid', true);\n            }\n\n            return {\n                valid: isValid,\n                target: this\n            };\n        },\n\n        /**\n         * Callback that fires when 'value' property is updated.\n         */\n        onUpdate: function () {\n            this.bubble('update', this.hasChanged());\n\n            this.validate();\n        },\n\n        /**\n         * Restore value to default\n         */\n        restoreToDefault: function () {\n            this.value(this.default);\n            this.focused(true);\n        },\n\n        /**\n         * Update whether value differs from default value\n         */\n        setDifferedFromDefault: function () {\n            var value = typeof this.value() != 'undefined' && this.value() !== null ? this.value() : '',\n                defaultValue = typeof this.default != 'undefined' && this.default !== null ? this.default : '';\n\n            this.isDifferedFromDefault(value !== defaultValue);\n        },\n\n        /**\n         * @param {Boolean} state\n         */\n        toggleUseDefault: function (state) {\n            this.disabled(state);\n\n            if (this.source && this.hasService()) {\n                this.source.set('data.use_default.' + this.index, Number(state));\n            }\n        },\n\n        /**\n         *  Callback when value is changed by user\n         */\n        userChanges: function () {\n            this.valueChangedByUser = true;\n        },\n\n        /**\n         * Returns correct id for 'aria-describedby' accessibility attribute\n         *\n         * @returns {Boolean|String}\n         */\n        getDescriptionId: function () {\n            var id = false;\n\n            if (this.error()) {\n                id = this.errorId;\n            } else if (this.notice()) {\n                id = this.noticeId;\n            }\n\n            return id;\n        }\n    });\n});\n","Magento_Ui/js/form/element/text.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'uiElement',\n    'mageUtils'\n], function (Element, utils) {\n    'use strict';\n\n    return Element.extend({\n        defaults: {\n            visible: true,\n            label: '',\n            error: '',\n            uid: utils.uniqueid(),\n            disabled: false,\n            links: {\n                value: '${ $.provider }:${ $.dataScope }'\n            }\n        },\n\n        /**\n         * Has service\n         *\n         * @returns {Boolean} false.\n         */\n        hasService: function () {\n            return false;\n        },\n\n        /**\n         * Has addons\n         *\n         * @returns {Boolean} false.\n         */\n        hasAddons: function () {\n            return false;\n        },\n\n        /**\n         * Calls 'initObservable' of parent\n         *\n         * @returns {Object} Chainable.\n         */\n        initObservable: function () {\n            this._super()\n                .observe('disabled visible value');\n\n            return this;\n        }\n    });\n});\n","Magento_Ui/js/form/element/color-picker-palette.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/**\n * @api\n */\ndefine([], function () {\n    'use strict';\n\n    return [\n        [\n            'rgb(0,0,0)', 'rgb(52,52,52)', 'rgb(83,83,83)', 'rgb(135,135,135)', 'rgb(193,193,193)',\n            'rgb(234,234,234)', 'rgb(240,240,240)', 'rgb(255,255,255)'\n        ],\n        [\n            'rgb(252,0,9)', 'rgb(253,135,10)', 'rgb(255,255,13)', 'rgb(35,255,9)', 'rgb(33,255,255)',\n            'rgb(0,0,254)', 'rgb(132,0,254)', 'rgb(251,0,255)'\n        ],\n        [\n            'rgb(240,192,194)', 'rgb(251,223,194)', 'rgb(255,241,193)', 'rgb(210,230,201)',\n            'rgb(199,217,220)', 'rgb(197,219,240)', 'rgb(208,200,227)', 'rgb(229,199,212)'\n        ],\n        [\n            'rgb(228,133,135)', 'rgb(246,193,139)', 'rgb(254,225,136)', 'rgb(168,208,152)',\n            'rgb(146,184,190)', 'rgb(143,184,227)', 'rgb(165,148,204)', 'rgb(202,147,175)'\n        ],\n        [\n            'rgb(214,78,83)', 'rgb(243,163,88)', 'rgb(254,211,83)', 'rgb(130,187,106)',\n            'rgb(99,149,159)', 'rgb(93,150,211)', 'rgb(123,100,182)', 'rgb(180,100,142)'\n        ],\n        [\n            'rgb(190,0,5)', 'rgb(222,126,44)', 'rgb(236,183,39)', 'rgb(89,155,61)', 'rgb(55,110,123)',\n            'rgb(49,112,185)', 'rgb(83,55,150)', 'rgb(147,55,101)'\n        ],\n        [\n            'rgb(133,0,3)', 'rgb(163,74,10)', 'rgb(177,127,7)', 'rgb(45,101,23)', 'rgb(18,62,74)',\n            'rgb(14,62,129)', 'rgb(40,15,97)', 'rgb(95,16,55)'\n        ],\n        [\n            'rgb(81,0,1)', 'rgb(100,48,7)', 'rgb(107,78,3)', 'rgb(31,63,16)',\n            'rgb(13,39,46)', 'rgb(10,40,79)', 'rgb(24,12,59)', 'rgb(59,10,36)'\n        ]\n    ];\n});\n","Magento_Ui/js/form/element/media.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'mageUtils',\n    './abstract'\n], function (utils, Abstract) {\n    'use strict';\n\n    return Abstract.extend({\n        defaults: {\n            links: {\n                value: ''\n            }\n        },\n\n        /**\n         * Initializes file component.\n         *\n         * @returns {Media} Chainable.\n         */\n        initialize: function () {\n            this._super()\n                .initFormId();\n\n            return this;\n        },\n\n        /**\n         * Defines form ID with which file input will be associated.\n         *\n         * @returns {Media} Chainable.\n         */\n        initFormId: function () {\n            var namespace;\n\n            if (this.formId) {\n                return this;\n            }\n\n            namespace   = this.name.split('.');\n            this.formId = namespace[0];\n\n            return this;\n        }\n    });\n});\n","Magento_Ui/js/form/element/date.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'moment',\n    'mageUtils',\n    './abstract',\n    'moment-timezone-with-data'\n], function (moment, utils, Abstract) {\n    'use strict';\n\n    return Abstract.extend({\n        defaults: {\n            options: {},\n\n            storeTimeZone: 'UTC',\n\n            validationParams: {\n                dateFormat: '${ $.outputDateFormat }'\n            },\n\n            /**\n             * Format of date that comes from the\n             * server (ICU Date Format).\n             *\n             * Used only in date picker mode\n             * (this.options.showsTime == false).\n             *\n             * @type {String}\n             */\n            inputDateFormat: 'y-MM-dd',\n\n            /**\n             * Format of date that should be sent to the\n             * server (ICU Date Format).\n             *\n             * Used only in date picker mode\n             * (this.options.showsTime == false).\n             *\n             * @type {String}\n             */\n            outputDateFormat: 'MM/dd/y',\n\n            /**\n             * Date/time format that is used to display date in\n             * the input field.\n             *\n             * @type {String}\n             */\n            pickerDateTimeFormat: '',\n\n            pickerDefaultDateFormat: 'MM/dd/y', // ICU Date Format\n            pickerDefaultTimeFormat: 'h:mm a', // ICU Time Format\n\n            elementTmpl: 'ui/form/element/date',\n\n            /**\n             * Format needed by moment timezone for conversion\n             */\n            timezoneFormat: 'YYYY-MM-DD HH:mm',\n\n            listens: {\n                'value': 'onValueChange',\n                'shiftedValue': 'onShiftedValueChange'\n            },\n\n            /**\n             * Date/time value shifted to corresponding timezone\n             * according to this.storeTimeZone property. This value\n             * will be sent to the server.\n             *\n             * @type {String}\n             */\n            shiftedValue: ''\n        },\n\n        /**\n         * Initializes regular properties of instance.\n         *\n         * @returns {Object} Chainable.\n         */\n        initConfig: function () {\n            this._super();\n\n            if (!this.options.dateFormat) {\n                this.options.dateFormat = this.pickerDefaultDateFormat;\n            }\n\n            if (!this.options.timeFormat) {\n                this.options.timeFormat = this.pickerDefaultTimeFormat;\n            }\n\n            this.prepareDateTimeFormats();\n\n            return this;\n        },\n\n        /**\n         * @inheritdoc\n         */\n        initObservable: function () {\n            return this._super().observe(['shiftedValue']);\n        },\n\n        /**\n         * Prepares and sets date/time value that will be displayed\n         * in the input field.\n         *\n         * @param {String} value\n         */\n        onValueChange: function (value) {\n            var shiftedValue;\n\n            if (value) {\n                if (this.options.showsTime) {\n                    shiftedValue = moment.tz(value, 'UTC').tz(this.storeTimeZone);\n                } else {\n                    shiftedValue = moment(value, this.outputDateFormat);\n                }\n\n                if (!shiftedValue.isValid()) {\n                    shiftedValue = moment(value, this.inputDateFormat);\n                }\n                shiftedValue = shiftedValue.format(this.pickerDateTimeFormat);\n            } else {\n                shiftedValue = '';\n            }\n\n            if (shiftedValue !== this.shiftedValue()) {\n                this.shiftedValue(shiftedValue);\n            }\n        },\n\n        /**\n         * Prepares and sets date/time value that will be sent\n         * to the server.\n         *\n         * @param {String} shiftedValue\n         */\n        onShiftedValueChange: function (shiftedValue) {\n            var value,\n                formattedValue,\n                momentValue;\n\n            if (shiftedValue) {\n                momentValue = moment(shiftedValue, this.pickerDateTimeFormat);\n\n                if (this.options.showsTime) {\n                    formattedValue = moment(momentValue).format(this.timezoneFormat);\n                    value = moment.tz(formattedValue, this.storeTimeZone).tz('UTC').toISOString();\n                } else {\n                    value = momentValue.format(this.outputDateFormat);\n                }\n            } else {\n                value = '';\n            }\n\n            if (value !== this.value()) {\n                this.value(value);\n            }\n        },\n\n        /**\n         * Prepares and converts all date/time formats to be compatible\n         * with moment.js library.\n         */\n        prepareDateTimeFormats: function () {\n            this.pickerDateTimeFormat = this.options.dateFormat;\n\n            if (this.options.showsTime) {\n                this.pickerDateTimeFormat += ' ' + this.options.timeFormat;\n            }\n\n            this.pickerDateTimeFormat = utils.convertToMomentFormat(this.pickerDateTimeFormat);\n\n            if (this.options.dateFormat) {\n                this.outputDateFormat = this.options.dateFormat;\n            }\n\n            this.inputDateFormat = utils.convertToMomentFormat(this.inputDateFormat);\n            this.outputDateFormat = utils.convertToMomentFormat(this.outputDateFormat);\n\n            this.validationParams.dateFormat = this.outputDateFormat;\n        }\n    });\n});\n","Magento_Ui/js/form/adapter/buttons.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine(function () {\n    'use strict';\n\n    return {\n        'reset': '#reset',\n        'save': '#save',\n        'saveAndContinue': '#save_and_continue'\n    };\n});\n","Magento_Ui/js/form/components/fieldset.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Ui/js/lib/collapsible',\n    'underscore'\n], function (Collapsible, _) {\n    'use strict';\n\n    return Collapsible.extend({\n        defaults: {\n            template: 'ui/form/fieldset',\n            collapsible: false,\n            changed: false,\n            loading: false,\n            error: false,\n            opened: false,\n            level: 0,\n            visible: true,\n            initializeFieldsetDataByDefault: false, /* Data in some fieldsets should be initialized before open */\n            disabled: false,\n            listens: {\n                'opened': 'onVisibilityChange'\n            },\n            additionalClasses: {}\n        },\n\n        /**\n         * Extends instance with defaults. Invokes parent initialize method.\n         * Calls initListeners and pushParams methods.\n         */\n        initialize: function () {\n            _.bindAll(this, 'onChildrenUpdate', 'onChildrenError', 'onContentLoading');\n\n            return this._super()\n                ._setClasses();\n        },\n\n        /**\n         * Initializes components' configuration.\n         *\n         * @returns {Fieldset} Chainable.\n         */\n        initConfig: function () {\n            this._super();\n            this._wasOpened = this.opened || !this.collapsible;\n\n            return this;\n        },\n\n        /**\n         * Calls initObservable of parent class.\n         * Defines observable properties of instance.\n         *\n         * @returns {Object} Reference to instance\n         */\n        initObservable: function () {\n            this._super()\n                .observe('changed loading error visible');\n\n            return this;\n        },\n\n        /**\n         * Calls parent's initElement method.\n         * Assigns callbacks on various events of incoming element.\n         *\n         * @param  {Object} elem\n         * @return {Object} - reference to instance\n         */\n        initElement: function (elem) {\n            elem.initContainer(this);\n\n            elem.on({\n                'update': this.onChildrenUpdate,\n                'loading': this.onContentLoading,\n                'error': this.onChildrenError\n            });\n\n            if (this.disabled) {\n                try {\n                    elem.disabled(true);\n                }\n                catch (e) {\n\n                }\n            }\n\n            return this;\n        },\n\n        /**\n         * Is being invoked on children update.\n         * Sets changed property to one incoming.\n         *\n         * @param  {Boolean} hasChanged\n         */\n        onChildrenUpdate: function (hasChanged) {\n            if (!hasChanged) {\n                hasChanged = _.some(this.delegate('hasChanged'));\n            }\n\n            this.bubble('update', hasChanged);\n            this.changed(hasChanged);\n        },\n\n        /**\n         * Extends 'additionalClasses' object.\n         *\n         * @returns {Group} Chainable.\n         */\n        _setClasses: function () {\n            var additional = this.additionalClasses,\n                classes;\n\n            if (_.isString(additional)) {\n                additional = this.additionalClasses.split(' ');\n                classes = this.additionalClasses = {};\n\n                additional.forEach(function (name) {\n                    classes[name] = true;\n                }, this);\n            }\n\n            _.extend(this.additionalClasses, {\n                'admin__collapsible-block-wrapper': this.collapsible,\n                _show: this.opened,\n                _hide: !this.opened,\n                _disabled: this.disabled\n            });\n\n            return this;\n        },\n\n        /**\n         * Handler of the \"opened\" property changes.\n         *\n         * @param {Boolean} isOpened\n         */\n        onVisibilityChange: function (isOpened) {\n            if (!this._wasOpened) {\n                this._wasOpened = isOpened;\n            }\n        },\n\n        /**\n         * Is being invoked on children validation error.\n         * Sets error property to one incoming.\n         *\n         * @param {String} message - error message.\n         */\n        onChildrenError: function (message) {\n            var hasErrors = false;\n\n            if (!message) {\n                hasErrors = this._isChildrenHasErrors(hasErrors, this);\n            }\n\n            this.error(hasErrors || message);\n\n            if (hasErrors || message) {\n                this.open();\n            }\n        },\n\n        /**\n         * Returns errors of children if exist\n         *\n         * @param {Boolean} hasErrors\n         * @param {*} container\n         * @return {Boolean}\n         * @private\n         */\n        _isChildrenHasErrors: function (hasErrors, container) {\n            var self = this;\n\n            if (hasErrors === false && container.hasOwnProperty('elems')) {\n                hasErrors = container.elems.some('error');\n\n                if (hasErrors === false && container.hasOwnProperty('_elems')) {\n                    container._elems.forEach(function (child) {\n\n                        if (hasErrors === false) {\n                            hasErrors = self._isChildrenHasErrors(hasErrors, child);\n                        }\n                    });\n                }\n            }\n\n            return hasErrors;\n        },\n\n        /**\n         * Callback that sets loading property to true.\n         */\n        onContentLoading: function (isLoading) {\n            this.loading(isLoading);\n        }\n    });\n});\n","Magento_Ui/js/form/components/area.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    './tab'\n], function (_, Tab) {\n    'use strict';\n\n    return Tab.extend({\n        defaults: {\n            uniqueNs:   'params.activeArea',\n            template:   'ui/area',\n            changed:    false,\n            loading:    false\n        },\n\n        /**\n         * Extends instance with defaults. Invokes parent initialize method.\n         * Calls initListeners and pushParams methods.\n         */\n        initialize: function () {\n            _.bindAll(this, 'onChildrenUpdate', 'onContentLoading');\n\n            return this._super();\n        },\n\n        /**\n         * Calls initObservable of parent class.\n         * Defines observable properties of instance.\n         * @return {Object} - reference to instance\n         */\n        initObservable: function () {\n            this._super()\n                .observe('changed loading');\n\n            return this;\n        },\n\n        /**\n         * Calls parent's initElement method.\n         * Assigns callbacks on various events of incoming element.\n         * @param  {Object} elem\n         * @return {Object} - reference to instance\n         */\n        initElement: function (elem) {\n            this._super();\n\n            elem.on({\n                'update':   this.onChildrenUpdate,\n                'loading':  this.onContentLoading\n            });\n\n            return this;\n        },\n\n        /**\n         * Is being invoked on children update.\n         * Sets changed property to one incoming.\n         * Invokes setActive method if settings\n         * contain makeVisible property set to true.\n         *\n         * @param  {Boolean} hasChanged\n         */\n        onChildrenUpdate: function (hasChanged) {\n            if (!hasChanged) {\n                hasChanged = _.some(this.delegate('hasChanged'));\n            }\n\n            this.changed(hasChanged);\n        },\n\n        /**\n         * Callback that sets loading property to true.\n         */\n        onContentLoading: function (isLoading) {\n            this.loading(isLoading);\n        }\n    });\n});\n","Magento_Ui/js/form/components/tab_group.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'Magento_Ui/js/lib/collapsible'\n], function (_, Collapsible) {\n    'use strict';\n\n    return Collapsible.extend({\n        defaults: {\n            listens: {\n                '${ $.provider }:data.validate': 'onValidate'\n            },\n            collapsible: false,\n            opened: true\n        },\n\n        /**\n         * Invokes initElement method of parent class, calls 'initActivation' method\n         * passing element to it.\n         * @param {Object} elem\n         * @returns {Object} - reference to instance\n         */\n        initElement: function (elem) {\n            this._super()\n                .initActivation(elem);\n\n            return this;\n        },\n\n        /**\n         * Activates element if one is first or if one has 'active' propert\n         * set to true.\n         *\n         * @param  {Object} elem\n         * @returns {Object} - reference to instance\n         */\n        initActivation: function (elem) {\n            var elems   = this.elems(),\n                isFirst = !elems.indexOf(elem);\n\n            if (isFirst || elem.active()) {\n                elem.activate();\n            }\n\n            return this;\n        },\n\n        /**\n         * Delegates 'validate' method on element, then reads 'invalid' property\n         * of params storage, and if defined, activates element, sets\n         * 'allValid' property of instance to false and sets invalid's\n         * 'focused' property to true.\n         *\n         * @param {Object} elem\n         */\n        validate: function (elem) {\n            var result  = elem.delegate('validate'),\n                invalid;\n\n            invalid = _.find(result, function (item) {\n                return typeof item !== 'undefined' && !item.valid;\n            });\n\n            if (invalid) {\n                elem.activate();\n                invalid.target.focused(true);\n            }\n\n            return invalid;\n        },\n\n        /**\n         * Sets 'allValid' property of instance to true, then calls 'validate' method\n         * of instance for each element.\n         */\n        onValidate: function () {\n            this.elems.sortBy(function (elem) {\n                return !elem.active();\n            }).some(this.validate, this);\n        }\n    });\n});\n","Magento_Ui/js/form/components/group.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'uiCollection'\n], function (_, Collection) {\n    'use strict';\n\n    return Collection.extend({\n        defaults: {\n            visible: true,\n            label: '',\n            showLabel: true,\n            required: false,\n            template: 'ui/group/group',\n            fieldTemplate: 'ui/form/field',\n            breakLine: true,\n            validateWholeGroup: false,\n            additionalClasses: {}\n        },\n\n        /**\n         * Extends this with defaults and config.\n         * Then calls initObservable, iniListenes and extractData methods.\n         */\n        initialize: function () {\n            this._super()\n                ._setClasses();\n\n            return this;\n        },\n\n        /**\n         * Calls initObservable of parent class.\n         * Defines observable properties of instance.\n         *\n         * @return {Object} - reference to instance\n         */\n        initObservable: function () {\n            this._super()\n                .observe('visible')\n                .observe({\n                    required: !!+this.required\n                });\n\n            return this;\n        },\n\n        /**\n         * Extends 'additionalClasses' object.\n         *\n         * @returns {Group} Chainable.\n         */\n        _setClasses: function () {\n            var additional = this.additionalClasses,\n                classes;\n\n            if (_.isString(additional)) {\n                additional = this.additionalClasses.split(' ');\n                classes = this.additionalClasses = {};\n\n                additional.forEach(function (name) {\n                    classes[name] = true;\n                }, this);\n            }\n\n            _.extend(this.additionalClasses, {\n                'admin__control-grouped': !this.breakLine,\n                'admin__control-fields': this.breakLine,\n                required:   this.required,\n                _error:     this.error,\n                _disabled:  this.disabled\n            });\n\n            return this;\n        },\n\n        /**\n         * Defines if group has only one element.\n         * @return {Boolean}\n         */\n        isSingle: function () {\n            return this.elems.getLength() === 1;\n        },\n\n        /**\n         * Defines if group has multiple elements.\n         * @return {Boolean}\n         */\n        isMultiple: function () {\n            return this.elems.getLength() > 1;\n        },\n\n        /**\n         * Returns an array of child components previews.\n         *\n         * @returns {Array}\n         */\n        getPreview: function () {\n            return this.elems.map('getPreview');\n        }\n    });\n});\n","Magento_Ui/js/form/components/button.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'uiElement',\n    'uiRegistry',\n    'uiLayout',\n    'mageUtils',\n    'underscore'\n], function (Element, registry, layout, utils, _) {\n    'use strict';\n\n    return Element.extend({\n        defaults: {\n            buttonClasses: {},\n            additionalClasses: {},\n            displayArea: 'outsideGroup',\n            displayAsLink: false,\n            elementTmpl: 'ui/form/element/button',\n            template: 'ui/form/components/button/simple',\n            visible: true,\n            disabled: false,\n            title: ''\n        },\n\n        /**\n         * Initializes component.\n         *\n         * @returns {Object} Chainable.\n         */\n        initialize: function () {\n            return this._super()\n                ._setClasses()\n                ._setButtonClasses();\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            return this._super()\n                .observe([\n                    'visible',\n                    'disabled',\n                    'title',\n                    'childError'\n                ]);\n        },\n\n        /**\n         * Performs configured actions\n         */\n        action: function () {\n            this.actions.forEach(this.applyAction, this);\n        },\n\n        /**\n         * Apply action on target component,\n         * but previously create this component from template if it is not existed\n         *\n         * @param {Object} action - action configuration\n         */\n        applyAction: function (action) {\n            var targetName = action.targetName,\n                params = utils.copy(action.params) || [],\n                actionName = action.actionName,\n                target;\n\n            if (!registry.has(targetName)) {\n                this.getFromTemplate(targetName);\n            }\n            target = registry.async(targetName);\n\n            if (target && typeof target === 'function' && actionName) {\n                params.unshift(actionName);\n                target.apply(target, params);\n            }\n        },\n\n        /**\n         * Create target component from template\n         *\n         * @param {Object} targetName - name of component,\n         * that supposed to be a template and need to be initialized\n         */\n        getFromTemplate: function (targetName) {\n            var parentName = targetName.split('.'),\n                index = parentName.pop(),\n                child;\n\n            parentName = parentName.join('.');\n            child = utils.template({\n                parent: parentName,\n                name: index,\n                nodeTemplate: targetName\n            });\n            layout([child]);\n        },\n\n        /**\n         * Extends 'additionalClasses' object.\n         *\n         * @returns {Object} Chainable.\n         */\n        _setClasses: function () {\n            if (typeof this.additionalClasses === 'string') {\n                if (this.additionalClasses === '') {\n                    this.additionalClasses = {};\n\n                    return this;\n                }\n\n                this.additionalClasses = this.additionalClasses\n                    .trim()\n                    .split(' ')\n                    .reduce(function (classes, name) {\n                        classes[name] = true;\n\n                        return classes;\n                    }, {}\n                );\n            }\n\n            return this;\n        },\n\n        /**\n         * Extends 'buttonClasses' object.\n         *\n         * @returns {Object} Chainable.\n         */\n        _setButtonClasses: function () {\n            var additional = this.buttonClasses;\n\n            if (_.isString(additional)) {\n                this.buttonClasses = {};\n\n                if (additional.trim().length) {\n                    additional = additional.trim().split(' ');\n\n                    additional.forEach(function (name) {\n                        if (name.length) {\n                            this.buttonClasses[name] = true;\n                        }\n                    }, this);\n                }\n            }\n\n            _.extend(this.buttonClasses, {\n                'action-basic': !this.displayAsLink,\n                'action-additional': this.displayAsLink\n            });\n\n            return this;\n        }\n    });\n});\n","Magento_Ui/js/form/components/insert-form.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    './insert',\n    'mageUtils',\n    'jquery'\n], function (Insert, utils, $) {\n    'use strict';\n\n    /**\n     * Get page actions element.\n     *\n     * @param {String} elem\n     * @param {String} actionsClass\n     * @returns {String}\n     */\n    function getPageActions(elem, actionsClass) {\n        var el = document.createElement('div');\n\n        el.innerHTML = elem;\n\n        return el.getElementsByClassName(actionsClass)[0];\n    }\n\n    /**\n     * Return element without page actions toolbar\n     *\n     * @param {String} elem\n     * @param {String} actionsClass\n     * @returns {String}\n     */\n    function removePageActions(elem, actionsClass) {\n        var el = document.createElement('div'),\n            actions;\n\n        el.innerHTML = elem;\n        actions = el.getElementsByClassName(actionsClass)[0];\n\n        if (actions) {\n            el.removeChild(actions);\n        }\n\n        return el.innerHTML;\n    }\n\n    return Insert.extend({\n        defaults: {\n            externalFormName: '${ $.ns }.${ $.ns }',\n            pageActionsClass: 'page-actions',\n            actionsContainerClass: 'page-main-actions',\n            exports: {\n                prefix: '${ $.externalFormName }:selectorPrefix'\n            },\n            imports: {\n                toolbarSection: '${ $.toolbarContainer }:toolbarSection',\n                prefix: '${ $.toolbarContainer }:rootSelector',\n                messagesClass: '${ $.externalFormName }:messagesClass'\n            },\n            settings: {\n                ajax: {\n                    ajaxSave: true,\n                    exports: {\n                        ajaxSave: '${ $.externalFormName }:ajaxSave'\n                    },\n                    imports: {\n                        responseStatus: '${ $.externalFormName }:responseStatus',\n                        responseData: '${ $.externalFormName }:responseData'\n                    }\n                }\n            },\n            modules: {\n                externalForm: '${ $.externalFormName }'\n            }\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            return this._super()\n                .observe('responseStatus');\n        },\n\n        /** @inheritdoc */\n        initConfig: function (config) {\n            var defaults = this.constructor.defaults;\n\n            utils.extend(defaults, defaults.settings[config.formSubmitType] || {});\n\n            return this._super();\n        },\n\n        /** @inheritdoc*/\n        destroyInserted: function () {\n            if (this.isRendered && this.externalForm()) {\n                this.externalForm().delegate('destroy');\n                this.removeActions();\n                this.responseStatus(undefined);\n                this.responseData = {};\n            }\n\n            return this._super();\n        },\n\n        /** @inheritdoc */\n        onRender: function (data) {\n            var actions = getPageActions(data, this.pageActionsClass);\n\n            if (!data.length) {\n                return this;\n            }\n            data = removePageActions(data, this.pageActionsClass);\n            this.renderActions(actions);\n            this._super(data);\n        },\n\n        /**\n         * Insert actions in toolbar.\n         *\n         * @param {String} actions\n         */\n        renderActions: function (actions) {\n            var $container = $('<div/>');\n\n            $container\n                .addClass(this.actionsContainerClass)\n                .append(actions);\n\n            this.formHeader = $container;\n\n            $(this.toolbarSection).append(this.formHeader);\n        },\n\n        /**\n         * Remove actions toolbar.\n         */\n        removeActions: function () {\n            $(this.formHeader).siblings('.' + this.messagesClass).remove();\n            $(this.formHeader).remove();\n            this.formHeader = $();\n        },\n\n        /**\n         * Reset external form data.\n         */\n        resetForm: function () {\n            if (this.externalSource()) {\n                this.externalSource().trigger('data.reset');\n                this.responseStatus(undefined);\n            }\n        }\n    });\n});\n","Magento_Ui/js/form/components/html.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'underscore',\n    'uiComponent'\n], function ($, _, Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            content:        '',\n            showSpinner:    false,\n            loading:        false,\n            visible:        true,\n            template:       'ui/content/content',\n            additionalClasses: {},\n            ignoreTmpls: {\n                content: true\n            }\n        },\n\n        /**\n         * Extends instance with default config, calls 'initialize' method of\n         *     parent, calls 'initAjaxConfig'\n         */\n        initialize: function () {\n            _.bindAll(this, 'onContainerToggle', 'onDataLoaded');\n\n            this._super()\n                ._setClasses()\n                .initAjaxConfig();\n\n            return this;\n        },\n\n        /**\n         * Calls 'initObservable' method of parent, initializes observable\n         * properties of instance\n         *\n         * @return {Object} - reference to instance\n         */\n        initObservable: function () {\n            this._super()\n                .observe('content loading visible');\n\n            return this;\n        },\n\n        /**\n         * Extends 'additionalClasses' object.\n         *\n         * @returns {Group} Chainable.\n         */\n        _setClasses: function () {\n            var additional = this.additionalClasses,\n                classes;\n\n            if (_.isString(additional)) {\n                additional = this.additionalClasses.split(' ');\n                classes = this.additionalClasses = {};\n\n                additional.forEach(function (name) {\n                    classes[name] = true;\n                }, this);\n            }\n\n            _.extend(this.additionalClasses, {\n                'admin__scope-old': !!additional\n            });\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initContainer: function (parent) {\n            this._super();\n\n            parent.on('active', this.onContainerToggle);\n\n            return this;\n        },\n\n        /**\n         * Initializes default ajax config on instance\n         *\n         * @return {Object} - reference to instance\n         */\n        initAjaxConfig: function () {\n            this.ajaxConfig = {\n                url: this.url,\n                data: {\n                    FORM_KEY: window.FORM_KEY\n                },\n                success:    this.onDataLoaded\n            };\n\n            return this;\n        },\n\n        /**\n         * Calls 'loadData' if both 'active' variable and 'shouldLoad'\n         * property are truthy\n         *\n         * @param  {Boolean} active\n         */\n        onContainerToggle: function (active) {\n            if (active && this.shouldLoad()) {\n                this.loadData();\n            }\n        },\n\n        /**\n         * Defines if instance has 'content' property defined.\n         *\n         * @return {Boolean} [description]\n         */\n        hasData: function () {\n            return !!this.content();\n        },\n\n        /**\n         * Defines if instance should load external data\n         *\n         * @return {Boolean}\n         */\n        shouldLoad: function () {\n            return this.url && !this.hasData() && !this.loading();\n        },\n\n        /**\n         * Sets loading property to true, makes ajax call\n         *\n         * @return {Object} - reference to instance\n         */\n        loadData: function () {\n            this.loading(true);\n\n            $.ajax(this.ajaxConfig);\n\n            return this;\n        },\n\n        /**\n         * Ajax's request success handler. Calls 'updateContent' passing 'data'\n         * to it, then sets 'loading' property to false.\n         *\n         * @param  {String} data\n         */\n        onDataLoaded: function (data) {\n            this.updateContent(data)\n                .loading(false);\n        },\n\n        /**\n         * Sets incoming data 'content' property's value\n         *\n         * @param  {String} content\n         * @return {Object} - reference to instance\n         */\n        updateContent: function (content) {\n            this.content(content);\n\n            return this;\n        }\n    });\n});\n","Magento_Ui/js/form/components/collection.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'mageUtils',\n    'uiRegistry',\n    'uiComponent',\n    'uiLayout',\n    'Magento_Ui/js/modal/confirm'\n], function (_, utils, registry, Component, layout, confirm) {\n    'use strict';\n\n    var childTemplate = {\n        parent: '${ $.$data.name }',\n        name: '${ $.$data.childIndex }',\n        dataScope: '${ $.name }',\n        nodeTemplate: '${ $.$data.name }.${ $.$data.itemTemplate }'\n    };\n\n    return Component.extend({\n        defaults: {\n            lastIndex: 0,\n            template: 'ui/form/components/collection'\n        },\n\n        /**\n         * Extends instance with default config, calls initialize of parent\n         * class, calls initChildren method.\n         */\n        initialize: function () {\n            this._super()\n                .initChildren();\n\n            return this;\n        },\n\n        /**\n         * Activates the incoming child and triggers the update event.\n         *\n         * @param {Object} elem - Incoming child.\n         */\n        initElement: function (elem) {\n            this._super();\n\n            elem.activate();\n\n            this.bubble('update');\n\n            return this;\n        },\n\n        /**\n         * Loops over corresponding data in data storage,\n         * creates child for each and pushes it's identifier to initialItems array.\n         *\n         * @returns {Collection} Chainable.\n         */\n        initChildren: function () {\n            var children = this.source.get(this.dataScope),\n                initial = this.initialItems = [];\n\n            _.each(children, function (item, index) {\n                initial.push(index);\n                this.addChild(index);\n            }, this);\n\n            return this;\n        },\n\n        /**\n         * Creates new item of collection, based on incoming 'index'.\n         * If not passed creates one with 'new_' prefix.\n         *\n         * @param {String|Object} [index] - Index of a child.\n         * @returns {Collection} Chainable.\n         */\n        addChild: function (index) {\n            this.childIndex = !_.isString(index) ?\n                'new_' + this.lastIndex++ :\n                index;\n\n            layout([utils.template(childTemplate, this)]);\n\n            return this;\n        },\n\n        /**\n         * Returns true if current set of items differ from initial one,\n         * or if some child has been changed.\n         *\n         * @returns {Boolean}\n         */\n        hasChanged: function () {\n            var initial = this.initialItems,\n                current = this.elems.pluck('index'),\n                changed = !utils.equalArrays(initial, current);\n\n            return changed || this.elems.some(function (elem) {\n                return _.some(elem.delegate('hasChanged'));\n            });\n        },\n\n        /**\n         * Initiates validation of its' children components.\n         *\n         * @returns {Array} An array of validation results.\n         */\n        validate: function () {\n            var elems;\n\n            this.allValid = true;\n\n            elems = this.elems.sortBy(function (elem) {\n                return !elem.active();\n            });\n\n            elems = elems.map(this._validate, this);\n\n            return _.flatten(elems);\n        },\n\n        /**\n         * Iterator function for components validation.\n         * Activates first invalid child component.\n         *\n         * @param {Object} elem - Element to run validation on.\n         * @returns {Array} An array of validation results.\n         */\n        _validate: function (elem) {\n            var result = elem.delegate('validate'),\n                invalid;\n\n            invalid = _.some(result, function (item) {\n                return !item.valid;\n            });\n\n            if (this.allValid && invalid) {\n                this.allValid = false;\n\n                elem.activate();\n            }\n\n            return result;\n        },\n\n        /**\n         * Creates function that removes element\n         * from collection using '_removeChild' method.\n         * @param  {Object} elem - Element that should be removed.\n         * @deprecated Not used anymore\n         */\n        removeAddress: function (elem) {\n            var self = this;\n\n            confirm({\n                content: this.removeMessage,\n                actions: {\n                    /** @inheritdoc */\n                    confirm: function () {\n                        self._removeAddress(elem);\n                    }\n                }\n            });\n        },\n\n        /**\n         * Removes element from both collection and data storage,\n         * activates first element if removed one was active,\n         * triggers 'update' event.\n         *\n         * @param {Object} elem - Element to remove.\n         */\n        _removeAddress: function (elem) {\n            var isActive = elem.active(),\n                first;\n\n            elem.destroy();\n\n            first = this.elems.first();\n\n            if (first && isActive) {\n                first.activate();\n            }\n\n            this.bubble('update');\n        }\n    });\n});\n","Magento_Ui/js/form/components/tab.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'uiCollection'\n], function (Collection) {\n    'use strict';\n\n    return Collection.extend({\n        defaults: {\n            uniqueProp:     'active',\n            active:         false,\n            wasActivated:   false\n        },\n\n        /**\n         * Extends instance with defaults. Invokes parent initialize method.\n         * Calls initListeners and pushParams methods.\n         */\n        initialize: function () {\n            this._super()\n                .setUnique();\n        },\n\n        /**\n         * Calls initObservable of parent class.\n         * Defines observable properties of instance.\n         * @return {Object} - reference to instance\n         */\n        initObservable: function () {\n            this._super()\n                .observe('active wasActivated');\n\n            return this;\n        },\n\n        /**\n         * Sets active property to true, then invokes pushParams method.\n         */\n        activate: function () {\n            this.active(true);\n            this.wasActivated(true);\n\n            this.setUnique();\n\n            return true;\n        }\n    });\n});\n","Magento_Ui/js/form/components/collection/item.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'mageUtils',\n    '../tab'\n], function (_, utils, Tab) {\n    'use strict';\n\n    var previewConfig = {\n        separator: ' ',\n        prefix: ''\n    };\n\n    /**\n     * Parses incoming data and returns result merged with default preview config\n     *\n     * @param  {Object|String} data\n     * @return {Object}\n     */\n    function parsePreview(data) {\n        if (typeof data == 'string') {\n            data = {\n                items: data\n            };\n        }\n\n        data.items = utils.stringToArray(data.items);\n\n        return _.defaults(data, previewConfig);\n    }\n\n    return Tab.extend({\n        defaults: {\n            label: '',\n            uniqueNs: 'activeCollectionItem',\n            previewTpl: 'ui/form/components/collection/preview'\n        },\n\n        /**\n         * Extends instance with default config, calls initializes of parent class\n         */\n        initialize: function () {\n            _.bindAll(this, 'buildPreview', 'hasPreview');\n\n            return this._super();\n        },\n\n        /**\n         * Calls initProperties of parent class, initializes properties\n         * of instance.\n         *\n         * @return {Object} - reference to instance\n         */\n        initConfig: function () {\n            this._super();\n\n            this.displayed = [];\n\n            return this;\n        },\n\n        /**\n         * Calls initObservable of parent class, initializes observable\n         * properties of instance.\n         *\n         * @return {Object} - reference to instance\n         */\n        initObservable: function () {\n            this._super()\n                .observe({\n                    noPreview: true,\n                    indexed: {}\n                });\n\n            return this;\n        },\n\n        /**\n         * Is being called when child element has been initialized,\n         *     calls initElement of parent class, binds to element's update event,\n         *     calls insertToArea and insertToIndexed methods passing element to it\n         *\n         * @param  {Object} elem\n         */\n        initElement: function (elem) {\n            this._super()\n                .insertToIndexed(elem);\n\n            return this;\n        },\n\n        /**\n         * Adds element to observable indexed object of instance\n         *\n         * @param  {Object} elem\n         * @return {Object} - reference to instance\n         */\n        insertToIndexed: function (elem) {\n            var indexed = this.indexed();\n\n            indexed[elem.index] = elem;\n\n            this.indexed(indexed);\n\n            return this;\n        },\n\n        /**\n         * Destroys current instance along with all of its' children.\n         * Overrides base method to clear data when this method is called.\n         */\n        destroy: function () {\n            this._super();\n            this._clearData();\n        },\n\n        /**\n         * Clears all data associated with component.\n         * @private\n         *\n         * @returns {Item} Chainable.\n         */\n        _clearData: function () {\n            this.source.remove(this.dataScope);\n\n            return this;\n        },\n\n        /**\n         * Formats incoming previews array via parsePreview function.\n         *\n         * @param  {Array} previews\n         * @return {Array} - formatted previews\n         */\n        formatPreviews: function (previews) {\n            return previews.map(parsePreview);\n        },\n\n        /**\n         * Creates string view of previews\n         *\n         * @param  {Object} data\n         * @return {Strict} - formatted preview string\n         */\n        buildPreview: function (data) {\n            var preview = this.getPreview(data.items),\n                prefix = data.prefix;\n\n            return prefix + preview.join(data.separator);\n        },\n\n        /**\n         * Defines if instance has preview for incoming data\n         *\n         * @param  {Object}  data\n         * @return {Boolean}\n         */\n        hasPreview: function (data) {\n            return !!this.getPreview(data.items).length;\n        },\n\n        /**\n         * Creates an array of previews for elements specified in incoming\n         * items array, calls updatePreview afterwards.\n         *\n         * @param  {Array} items - An array of element's indexes.\n         * @returns {Array} An array of previews.\n         */\n        getPreview: function (items) {\n            var elems = this.indexed(),\n                displayed = this.displayed,\n                preview;\n\n            items = items.map(function (index) {\n                var elem = elems[index];\n\n                preview = elem && elem.visible() ? elem.getPreview() : '';\n\n                preview = Array.isArray(preview) ?\n                    _.compact(preview).join(', ') :\n                    preview;\n\n                utils.toggle(displayed, index, !!preview);\n\n                return preview;\n            });\n\n            this.noPreview(!displayed.length);\n\n            return _.compact(items);\n        }\n    });\n});\n","Magento_Ui/js/core/app.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    './renderer/types',\n    './renderer/layout',\n    '../lib/knockout/bootstrap'\n], function (types, layout) {\n    'use strict';\n\n    return function (data, merge) {\n        types.set(data.types);\n        layout(data.components, undefined, true, merge);\n    };\n});\n","Magento_Ui/js/core/renderer/layout.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'jquery',\n    'mageUtils',\n    'uiRegistry',\n    './types',\n    '../../lib/logger/console-logger'\n], function (_, $, utils, registry, types, consoleLogger) {\n    'use strict';\n\n    var templates = registry.create(),\n        layout = {},\n        cachedConfig = {};\n\n    /**\n     * Build name from parent name and node name\n     *\n     * @param {Object} parent\n     * @param {Object} node\n     * @param {String} [name]\n     * @returns {String}\n     */\n    function getNodeName(parent, node, name) {\n        var parentName = parent && parent.name;\n\n        if (typeof name !== 'string') {\n            name = node.name || name;\n        }\n\n        return utils.fullPath(parentName, name);\n    }\n\n    /**\n     * Get node type from node or parent.\n     *\n     * @param {Object} parent\n     * @param {Object} node\n     * @returns {String}\n     */\n    function getNodeType(parent, node) {\n        return node.type || parent && parent.childType;\n    }\n\n    /**\n     * Get data scope based on parent data scope and node data scope.\n     *\n     * @param {Object} parent\n     * @param {Object} node\n     * @returns {String}\n     */\n    function getDataScope(parent, node) {\n        var dataScope = node.dataScope,\n            parentScope = parent && parent.dataScope;\n\n        return !utils.isEmpty(parentScope) ?\n            !utils.isEmpty(dataScope) ?\n                parentScope + '.' + dataScope :\n                parentScope :\n            dataScope || '';\n    }\n\n    /**\n     * Load node dependencies on other instances.\n     *\n     * @param {Object} node\n     * @returns {jQueryPromise}\n     */\n    function loadDeps(node) {\n        var loaded = $.Deferred(),\n            loggerUtils = consoleLogger.utils;\n\n        if (node.deps) {\n            consoleLogger.utils.asyncLog(\n                loaded,\n                {\n                    data: {\n                        component: node.name,\n                        deps: node.deps\n                    },\n                    messages: loggerUtils.createMessages(\n                        'depsStartRequesting',\n                        'depsFinishRequesting',\n                        'depsLoadingFail'\n                    )\n                }\n            );\n        }\n\n        registry.get(node.deps, function (deps) {\n            node.provider = node.extendProvider ? deps && deps.name : node.provider;\n            loaded.resolve(node);\n        });\n\n        return loaded.promise();\n    }\n\n    /**\n     * Load node component file via requirejs.\n     *\n     * @param {Object} node\n     * @returns {jQueryPromise}\n     */\n    function loadSource(node) {\n        var loaded = $.Deferred(),\n            source = node.component;\n\n        consoleLogger.info('componentStartLoading', {\n            component: node.component\n        });\n\n        require([source], function (constr) {\n            consoleLogger.info('componentFinishLoading', {\n                component: node.component\n            });\n            loaded.resolve(node, constr);\n        }, function () {\n            consoleLogger.error('componentLoadingFail', {\n                component: node.component\n            });\n        });\n\n        return loaded.promise();\n    }\n\n    /**\n     * Create a new component instance and set it to the registry.\n     *\n     * @param {Object} node\n     * @param {Function} Constr\n     */\n    function initComponent(node, Constr) {\n        var component = new Constr(_.omit(node, 'children'));\n\n        consoleLogger.info('componentStartInitialization', {\n            component: node.component,\n            componentName: node.name\n        });\n\n        registry.set(node.name, component);\n    }\n\n    /**\n     * Application entry point.\n     *\n     * @param {Object} nodes\n     * @param {Object} parent\n     * @param {Boolean} cached\n     * @param {Boolean} merge\n     * @returns {Boolean|undefined}\n     */\n    function run(nodes, parent, cached, merge) {\n        if (_.isBoolean(merge) && merge) {\n            layout.merge(nodes);\n\n            return false;\n        }\n\n        if (cached) {\n            cachedConfig[_.keys(nodes)[0]] = JSON.parse(JSON.stringify(nodes));\n        }\n\n        _.each(nodes || [], layout.iterator.bind(layout, parent));\n    }\n\n    _.extend(layout, {\n        /**\n         * Determines if node ready to be added or process it.\n         *\n         * @param {Object} parent\n         * @param {Object|String} node\n         */\n        iterator: function (parent, node) {\n            var action = _.isString(node) ?\n                this.addChild :\n                this.process;\n\n            action.apply(this, arguments);\n        },\n\n        /**\n         * Prepare component.\n         *\n         * @param {Object} parent\n         * @param {Object} node\n         * @param {String} name\n         * @returns {Object}\n         */\n        process: function (parent, node, name) {\n            if (!parent && node.parent) {\n                return this.waitParent(node, name);\n            }\n\n            if (node.nodeTemplate) {\n                return this.waitTemplate.apply(this, arguments);\n            }\n\n            node = this.build.apply(this, arguments);\n\n            if (!registry.has(node.name)) {\n                this.addChild(parent, node)\n                    .manipulate(node)\n                    .initComponent(node);\n            }\n\n            if (node) {\n                run(node.children, node);\n            }\n\n            return this;\n        },\n\n        /**\n         * Detailed processing of component config.\n         *\n         * @param {Object} parent\n         * @param {Object} node\n         * @param {String} name\n         * @returns {Boolean|Object}\n         */\n        build: function (parent, node, name) {\n            var defaults    = parent && parent.childDefaults || {},\n                children    = this.filterDisabledChildren(node.children),\n                type        = getNodeType(parent, node),\n                dataScope   = getDataScope(parent, node),\n                component,\n                extendDeps  = true,\n                nodeName;\n\n            node.children = false;\n            node.extendProvider = true;\n\n            if (node.config && node.config.provider || node.provider) {\n                node.extendProvider = false;\n            }\n\n            if (node.config && node.config.deps || node.deps) {\n                extendDeps = false;\n            }\n\n            node = utils.extend({\n            }, types.get(type), defaults, node);\n\n            nodeName = getNodeName(parent, node, name);\n\n            if (registry.has(nodeName)) {\n                component = registry.get(nodeName);\n                component.children = children;\n\n                return component;\n            }\n\n            if (extendDeps && parent && parent.deps && type) {\n                node.deps = parent.deps;\n            }\n\n            _.extend(node, node.config || {}, {\n                index: node.name || name,\n                name: nodeName,\n                dataScope: dataScope,\n                parentName: utils.getPart(nodeName, -2),\n                parentScope: utils.getPart(dataScope, -2)\n            });\n\n            node.children = children;\n            node.componentType = node.type;\n\n            delete node.type;\n            delete node.config;\n\n            if (children) {\n                node.initChildCount = _.size(children);\n            }\n\n            if (node.isTemplate) {\n                node.isTemplate = false;\n\n                templates.set(node.name, node);\n                registry.get(node.parentName, function (parentComp) {\n                    parentComp.childTemplate = node;\n                });\n\n                return false;\n            }\n\n            if (node.componentDisabled === true) {\n                return false;\n            }\n\n            return node;\n        },\n\n        /**\n         * Filter out all disabled components.\n         *\n         * @param {Object} children\n         * @returns {*}\n         */\n        filterDisabledChildren: function (children) {\n            var cIds;\n\n            //cleanup children config.componentDisabled = true\n            if (children && typeof children === 'object') {\n                cIds = Object.keys(children);\n\n                if (cIds) {\n                    _.each(cIds, function (cId) {\n                        if (typeof children[cId] === 'object' &&\n                            children[cId].hasOwnProperty('config') &&\n                            typeof children[cId].config === 'object' &&\n                            children[cId].config.hasOwnProperty('componentDisabled') &&\n                            children[cId].config.componentDisabled === true) {\n                            delete children[cId];\n                        }\n                    });\n                }\n            }\n\n            return children;\n        },\n\n        /**\n         * Init component.\n         *\n         * @param {Object} node\n         * @returns {Object}\n         */\n        initComponent: function (node) {\n            if (!node.component) {\n                return this;\n            }\n\n            loadDeps(node)\n                .then(loadSource)\n                .done(initComponent);\n\n            return this;\n        }\n    });\n\n    _.extend(layout, {\n        /**\n         * Loading component marked as isTemplate.\n         *\n         * @param {Object} parent\n         * @param {Object} node\n         * @returns {Object}\n         */\n        waitTemplate: function (parent, node) {\n            var args = _.toArray(arguments);\n\n            templates.get(node.nodeTemplate, function () {\n                this.applyTemplate.apply(this, args);\n            }.bind(this));\n\n            return this;\n        },\n\n        /**\n         * Waiting for parent component and process provided component.\n         *\n         * @param {Object} node\n         * @param {String} name\n         * @returns {Object}\n         */\n        waitParent: function (node, name) {\n            var process = this.process.bind(this);\n\n            registry.get(node.parent, function (parent) {\n                process(parent, node, name);\n            });\n\n            return this;\n        },\n\n        /**\n         * Processing component marked as isTemplate.\n         *\n         * @param {Object} parent\n         * @param {Object} node\n         * @param {String} name\n         */\n        applyTemplate: function (parent, node, name) {\n            var template = templates.get(node.nodeTemplate);\n\n            node = utils.extend({}, template, node);\n\n            delete node.nodeTemplate;\n\n            this.process(parent, node, name);\n        }\n    });\n\n    _.extend(layout, {\n        /**\n         * Determines inserting strategy.\n         *\n         * @param {Object} node\n         * @returns {Object}\n         */\n        manipulate: function (node) {\n            var name = node.name;\n\n            if (node.appendTo) {\n                this.insert(name, node.appendTo, -1);\n            }\n\n            if (node.prependTo) {\n                this.insert(name, node.prependTo, 0);\n            }\n\n            if (node.insertTo) {\n                this.insertTo(name, node.insertTo);\n            }\n\n            return this;\n        },\n\n        /**\n         * Insert component to provide target and position.\n         *\n         * @param {Object|String} item\n         * @param {Object} target\n         * @param {Number} position\n         * @returns {Object}\n         */\n        insert: function (item, target, position) {\n            registry.get(target, function (container) {\n                container.insertChild(item, position);\n            });\n\n            return this;\n        },\n\n        /**\n         * Insert component into multiple targets.\n         *\n         * @param {Object} item\n         * @param {Array} targets\n         * @returns {Object}\n         */\n        insertTo: function (item, targets) {\n            _.each(targets, function (info, target) {\n                this.insert(item, target, info.position);\n            }, this);\n\n            return this;\n        },\n\n        /**\n         * Add provided child to parent.\n         *\n         * @param {Object} parent\n         * @param {Object|String} child\n         * @returns {Object}\n         */\n        addChild: function (parent, child) {\n            var name;\n\n            if (parent && parent.component) {\n                name = child.name || child;\n\n                this.insert(name, parent.name, child.sortOrder);\n            }\n\n            return this;\n        },\n\n        /**\n         * Merge components configuration with cached configuration.\n         *\n         * @param {Array} components\n         */\n        merge: function (components) {\n            var cachedKey = _.keys(components)[0],\n                compared = utils.compare(cachedConfig[cachedKey], components),\n                remove = this.filterComponents(this.getByProperty(compared.changes, 'type', 'remove'), true),\n                update = this.getByProperty(compared.changes, 'type', 'update'),\n                dataSources = this.getDataSources(components),\n                names, index, name, component;\n\n            _.each(dataSources, function (val, key) {\n                name = key.replace(/\\.children|\\.config/g, '');\n                component = registry.get(name);\n\n                component.cacheData();\n                component.updateConfig(\n                    true,\n                    this.getFullConfig(key, components),\n                    this.getFullConfig(key, cachedConfig[cachedKey])\n                );\n            }, this);\n\n            _.each(remove, function (val) {\n                component = registry.get(val.path);\n\n                if (component) {\n                    component.cleanData().destroy();\n                }\n            });\n\n            update = _.compact(_.filter(update, function (val) {\n                return !_.isEqual(val.oldValue, val.value);\n            }));\n\n            _.each(update, function (val) {\n                names = val.path.split('.');\n                index = Math.max(_.lastIndexOf(names, 'config'), _.lastIndexOf(names, 'children') + 2);\n                name = _.without(names.splice(0, index), 'children', 'config').join('.');\n                component = registry.get(name);\n\n                if (val.name === 'sortOrder' && component) {\n                    registry.get(component.parentName).insertChild(component, val.value);\n                } else if (component) {\n                    component.updateConfig(\n                        val.oldValue,\n                        val.value,\n                        val.path\n                    );\n                }\n            }, this);\n\n            run(components, undefined, true);\n        },\n\n        /**\n         * Recursive dataSource assignment.\n         *\n         * @param {Object} config\n         * @param {String} parentPath\n         * @returns {Object}\n         */\n        getDataSources: function (config, parentPath) {\n            var dataSources = {},\n                key, obj;\n\n            /* eslint-disable no-loop-func, max-depth */\n            for (key in config) {\n                if (config.hasOwnProperty(key)) {\n                    if (\n                        key === 'type' &&\n                        config[key] === 'dataSource' &&\n                        config.hasOwnProperty('config')\n                    ) {\n                        dataSources[parentPath + '.config'] = config.config;\n                    } else if (_.isObject(config[key])) {\n                        obj = this.getDataSources(config[key], utils.fullPath(parentPath, key));\n\n                        _.each(obj, function (value, path) {\n                            dataSources[path] = value;\n                        });\n                    }\n                }\n            }\n\n            /* eslint-enable no-loop-func, max-depth */\n\n            return dataSources;\n        },\n\n        /**\n         * Configuration getter.\n         *\n         * @param {String} path\n         * @param {Object} config\n         * @returns {Boolean|Object}\n         */\n        getFullConfig: function (path, config) {\n            var index;\n\n            path = path.split('.');\n            index = _.lastIndexOf(path, 'config');\n\n            if (!~index) {\n                return false;\n            }\n            path = path.splice(0, index);\n\n            _.each(path, function (val) {\n                config = config[val];\n            });\n\n            return config.config;\n        },\n\n        /**\n         * Filter data by property and value.\n         *\n         * @param {Object} data\n         * @param {String} prop\n         * @param {*} propValue\n         */\n        getByProperty: function (data, prop, propValue) {\n            return _.filter(data, function (value) {\n                return value[prop] === propValue;\n            });\n        },\n\n        /**\n         * Filter components.\n         *\n         * @param {Array} data\n         * @param {Boolean} splitPath\n         * @param {Number} index\n         * @param {String} separator\n         * @param {String} keyName\n         * @returns {Array}\n         */\n        filterComponents: function (data, splitPath, index, separator, keyName) {\n            var result = [],\n                names, length;\n\n            index = -2;\n            separator = '.' || separator;\n            keyName = 'children' || keyName;\n\n            _.each(data, function (val) {\n                names = val.path.split(separator);\n                length  = names.length;\n\n                if (names[length + index] === keyName) {\n                    val.path = splitPath ? _.without(names, keyName).join(separator) : val.path;\n                    result.push(val);\n                }\n            });\n\n            return result;\n        }\n    });\n\n    return run;\n});\n","Magento_Ui/js/core/renderer/types.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'mageUtils'\n], function (_, utils) {\n    'use strict';\n\n    var store = {};\n\n    /**\n     * Flatten a nested data.\n     *\n     * @param {Object} data\n     * @returns {Object}\n     */\n    function flatten(data) {\n        var extender = data.extends || [],\n            result = {};\n\n        extender = utils.stringToArray(extender);\n\n        extender.push(data);\n\n        extender.forEach(function (item) {\n            if (_.isString(item)) {\n                item = store[item] || {};\n            }\n\n            utils.extend(result, item);\n        });\n\n        delete result.extends;\n\n        return result;\n    }\n\n    return {\n        /**\n         * Set types to store object.\n         *\n         * @param {Object} types\n         */\n        set: function (types) {\n            types = types || {};\n\n            utils.extend(store, types);\n\n            _.each(types, function (data, type) {\n                store[type] = flatten(data);\n            });\n        },\n\n        /**\n         * Get type from store object.\n         *\n         * @param {String} type\n         * @returns {*|{}}\n         */\n        get: function (type) {\n            return store[type] || {};\n        }\n    };\n});\n","Magento_Ui/js/modal/modal-component.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Ui/js/lib/view/utils/async',\n    'uiCollection',\n    'uiRegistry',\n    'underscore',\n    './modal'\n], function ($, Collection, registry, _) {\n    'use strict';\n\n    return Collection.extend({\n        defaults: {\n            template: 'ui/modal/modal-component',\n            title: '',\n            subTitle: '',\n            options: {\n                modalClass: '',\n                title: '',\n                subTitle: '',\n                buttons: [],\n                keyEventHandlers: {}\n            },\n            valid: true,\n            links: {\n                title: 'options.title',\n                subTitle: 'options.subTitle'\n            },\n            listens: {\n                state: 'onState',\n                title: 'setTitle',\n                'options.subTitle': 'setSubTitle'\n            },\n            modalClass: 'modal-component',\n            onCancel: 'closeModal'\n        },\n\n        /**\n         * Initializes component.\n         *\n         * @returns {Object} Chainable.\n         */\n        initialize: function () {\n            this._super();\n            _.bindAll(this,\n                'initModal',\n                'openModal',\n                'closeModal',\n                'toggleModal',\n                'setPrevValues',\n                'validate');\n            this.initializeContent();\n\n            return this;\n        },\n\n        /**\n         * Initializes modal configuration\n         *\n         * @returns {Object} Chainable.\n         */\n        initConfig: function () {\n            return this._super()\n                .initSelector()\n                .initModalEvents();\n        },\n\n        /**\n         * Configure modal selector\n         *\n         * @returns {Object} Chainable.\n         */\n        initSelector: function () {\n            var modalClass = this.name.replace(/\\./g, '_');\n\n            this.contentSelector = '.' + this.modalClass;\n            this.options.modalClass = this.options.modalClass + ' ' + modalClass;\n            this.rootSelector = '.' + modalClass;\n\n            return this;\n        },\n\n        /**\n         * Configure modal keyboard handlers\n         * and outer click\n         *\n         * @returns {Object} Chainable.\n         */\n        initModalEvents: function () {\n            this.options.keyEventHandlers.escapeKey = this.options.outerClickHandler = this[this.onCancel].bind(this);\n\n            return this;\n        },\n\n        /**\n         * Initialize modal's content components\n         */\n        initializeContent: function () {\n            $.async({\n                component: this.name\n            }, this.initModal);\n        },\n\n        /**\n         * Init toolbar section so other components will be able to place something in it\n         */\n        initToolbarSection: function () {\n            this.set('toolbarSection', this.modal.data('mage-modal').modal.find('header').get(0));\n        },\n\n        /**\n         * Initializes observable properties.\n         *\n         * @returns {Object} Chainable.\n         */\n        initObservable: function () {\n            this._super();\n            this.observe(['state', 'focused']);\n\n            return this;\n        },\n\n        /**\n         * Wrap content in a modal of certain type\n         *\n         * @param {HTMLElement} element\n         * @returns {Object} Chainable.\n         */\n        initModal: function (element) {\n            if (!this.modal) {\n                this.overrideModalButtonCallback();\n                this.options.modalCloseBtnHandler = this[this.onCancel].bind(this);\n                this.modal = $(element).modal(this.options);\n                this.initToolbarSection();\n\n                if (this.waitCbk) {\n                    this.waitCbk();\n                    this.waitCbk = null;\n                }\n            }\n\n            return this;\n        },\n\n        /**\n         * Open modal\n         */\n        openModal: function () {\n            if (this.modal) {\n                this.state(true);\n            } else {\n                this.waitCbk = this.openModal;\n            }\n        },\n\n        /**\n         * Close modal\n         */\n        closeModal: function () {\n            if (this.modal) {\n                this.state(false);\n            } else {\n                this.waitCbk = this.closeModal;\n            }\n        },\n\n        /**\n         * Toggle modal\n         */\n        toggleModal: function () {\n            if (this.modal) {\n                this.state(!this.state());\n            } else {\n                this.waitCbk = this.toggleModal;\n            }\n        },\n\n        /**\n         * Sets title for modal\n         *\n         * @param {String} title\n         */\n        setTitle: function (title) {\n            if (this.title !== title) {\n                this.title = title;\n            }\n\n            if (this.modal) {\n                this.modal.modal('setTitle', title);\n            }\n        },\n\n        /**\n         * Sets subTitle for modal\n         *\n         * @param {String} subTitle\n         */\n        setSubTitle: function (subTitle) {\n            if (this.subTitle !== subTitle) {\n                this.subTitle = subTitle;\n            }\n\n            if (this.modal) {\n                this.modal.modal('setSubTitle', subTitle);\n            }\n        },\n\n        /**\n         * Wrap content in a modal of certain type\n         *\n         * @param {Boolean} state\n         */\n        onState: function (state) {\n            if (state) {\n                this.modal.modal('openModal');\n                this.applyData();\n            } else {\n                this.modal.modal('closeModal');\n            }\n        },\n\n        /**\n         * Validate everything validatable in modal\n         */\n        validate: function (elem) {\n            if (typeof elem === 'undefined') {\n                return;\n            }\n\n            if (typeof elem.validate === 'function') {\n                this.valid = this.valid & elem.validate().valid;\n            } else if (elem.elems) {\n                elem.elems().forEach(this.validate, this);\n            }\n        },\n\n        /**\n         * Reset data from provider\n         */\n        resetData: function () {\n            this.elems().forEach(this.resetValue, this);\n        },\n\n        /**\n         * Update 'applied' property with data from modal content\n         */\n        applyData: function () {\n            var applied = {};\n\n            this.elems().forEach(this.gatherValues.bind(this, applied), this);\n            this.applied = applied;\n        },\n\n        /**\n         * Gather values from modal content\n         *\n         * @param {Array} applied\n         * @param {HTMLElement} elem\n         */\n        gatherValues: function (applied, elem) {\n            if (typeof elem.value === 'function') {\n                applied[elem.name] = elem.value();\n            } else if (elem.elems) {\n                elem.elems().forEach(this.gatherValues.bind(this, applied), this);\n            }\n        },\n\n        /**\n         * Set to previous values from modal content\n         *\n         * @param {HTMLElement} elem\n         */\n        setPrevValues: function (elem) {\n            if (typeof elem.value === 'function') {\n                this.modal.focus();\n                elem.value(this.applied[elem.name]);\n            } else if (elem.elems) {\n                elem.elems().forEach(this.setPrevValues, this);\n            }\n        },\n\n        /**\n         * Triggers some method in every modal child elem, if this method is defined\n         *\n         * @param {Object} action - action configuration,\n         * must contain actionName and targetName and\n         * can contain params\n         */\n        triggerAction: function (action) {\n            var targetName = action.targetName,\n                params = action.params || [],\n                actionName = action.actionName,\n                target;\n\n            target = registry.async(targetName);\n\n            if (target && typeof target === 'function' && actionName) {\n                params.unshift(actionName);\n                target.apply(target, params);\n            }\n        },\n\n        /**\n         * Override modal buttons callback placeholders with real callbacks\n         */\n        overrideModalButtonCallback: function () {\n            var buttons = this.options.buttons;\n\n            if (buttons && buttons.length) {\n                buttons.forEach(function (button) {\n                    button.click = this.getButtonClickHandler(button.actions);\n                }, this);\n            }\n        },\n\n        /**\n         * Generate button click handler based on button's 'actions' configuration\n         */\n        getButtonClickHandler: function (actionsConfig) {\n            var actions = actionsConfig.map(\n                function (actionConfig) {\n                    if (_.isObject(actionConfig)) {\n                        return this.triggerAction.bind(this, actionConfig);\n                    }\n\n                    return this[actionConfig] ? this[actionConfig].bind(this) : function () {};\n                }, this);\n\n            return function () {\n                actions.forEach(\n                    function (action) {\n                        action();\n                    }\n                );\n            };\n        },\n\n        /**\n         * Cancels changes in modal:\n         * returning elems values to the previous state,\n         * and close modal\n         */\n        actionCancel: function () {\n            this.elems().forEach(this.setPrevValues, this);\n            this.closeModal();\n        },\n\n        /**\n         * Accept changes in modal by not preventing them.\n         * Can be extended by exporting 'gatherValues' result somewhere\n         */\n        actionDone: function () {\n            this.valid = true;\n            this.elems().forEach(this.validate, this);\n\n            if (this.valid) {\n                this.closeModal();\n            }\n        }\n    });\n});\n","Magento_Ui/js/modal/alert.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'underscore',\n    'jquery-ui-modules/widget',\n    'Magento_Ui/js/modal/confirm',\n    'mage/translate'\n], function ($, _) {\n    'use strict';\n\n    $.widget('mage.alert', $.mage.confirm, {\n        options: {\n            modalClass: 'confirm',\n            title: $.mage.__('Attention'),\n            actions: {\n\n                /**\n                 * Callback always - called on all actions.\n                 */\n                always: function () {}\n            },\n            buttons: [{\n                text: $.mage.__('OK'),\n                class: 'action-primary action-accept',\n\n                /**\n                 * Click handler.\n                 */\n                click: function () {\n                    this.closeModal(true);\n                }\n            }]\n        },\n\n        /**\n         * Create widget.\n         */\n        _create: function () {\n            this.options.actions.always();\n            this._super();\n        },\n\n        /**\n         * Close modal window.\n         */\n        closeModal: function () {\n            this.element.bind('alertclosed', _.bind(this._remove, this));\n\n            return this._super();\n        }\n    });\n\n    return function (config) {\n        return $('<div></div>').html(config.content).alert(config);\n    };\n});\n","Magento_Ui/js/modal/modalToggle.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/modal'\n], function ($) {\n    'use strict';\n\n    return function (config, el) {\n        var widget,\n            content;\n\n        if (config.contentSelector) {\n            content = $(config.contentSelector);\n        } else if (config.content) {\n            content = $('<div />').html(config.content);\n        } else {\n            content = $('<div />');\n        }\n\n        widget = content.modal(config);\n\n        $(el).on(config.toggleEvent, function () {\n            var state = widget.data('mage-modal').options.isOpen;\n\n            if (state) {\n                widget.modal('closeModal');\n            } else {\n                widget.modal('openModal');\n            }\n\n            return false;\n        });\n\n        return widget;\n    };\n});\n","Magento_Ui/js/modal/prompt.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'underscore',\n    'mage/template',\n    'text!ui/template/modal/modal-prompt-content.html',\n    'jquery-ui-modules/widget',\n    'Magento_Ui/js/modal/modal',\n    'mage/translate'\n], function ($, _, template, promptContentTmpl) {\n    'use strict';\n\n    $.widget('mage.prompt', $.mage.modal, {\n        options: {\n            modalClass: 'prompt',\n            promptContentTmpl: promptContentTmpl,\n            promptField: '[data-role=\"promptField\"]',\n            attributesForm: {},\n            attributesField: {},\n            value: '',\n            validation: false,\n            validationRules: [],\n            actions: {\n\n                /**\n                 * Callback always - called on all actions.\n                 */\n                always: function () {},\n\n                /**\n                 * Callback confirm.\n                 */\n                confirm: function () {},\n\n                /**\n                 * Callback cancel.\n                 */\n                cancel: function () {}\n            },\n            buttons: [{\n                text: $.mage.__('Cancel'),\n                class: 'action-secondary action-dismiss',\n\n                /**\n                 * Click handler.\n                 */\n                click: function () {\n                    this.closeModal();\n                }\n            }, {\n                text: $.mage.__('OK'),\n                class: 'action-primary action-accept',\n\n                /**\n                 * Click handler.\n                 */\n                click: function () {\n                    this.closeModal(true);\n                }\n            }]\n        },\n\n        /**\n         * Create widget.\n         */\n        _create: function () {\n            this.options.focus = this.options.promptField;\n            this.options.validation = this.options.validation && this.options.validationRules.length;\n            this._super();\n            this.modal.find(this.options.modalContent).append(this.getFormTemplate());\n            this.modal.find(this.options.modalCloseBtn).off().on('click',  _.bind(this.closeModal, this, false));\n\n            if (this.options.validation) {\n                this.setValidationClasses();\n            }\n\n            this.openModal();\n        },\n\n        /**\n         * Form template getter.\n         *\n         * @returns {Object} Form template.\n         */\n        getFormTemplate: function () {\n            var formTemplate,\n                formAttr = '',\n                inputAttr = '',\n                attributeName;\n\n            for (attributeName in this.options.attributesForm) {\n                if (this.options.attributesForm.hasOwnProperty(attributeName)) {\n                    formAttr = formAttr + ' ' + attributeName + '=\"' +\n                        this.options.attributesForm[attributeName] + '\"';\n                }\n            }\n\n            for (attributeName in this.options.attributesField) {\n                if (this.options.attributesField.hasOwnProperty(attributeName)) {\n                    inputAttr = inputAttr + ' ' + attributeName + '=\"' +\n                        this.options.attributesField[attributeName] + '\"';\n                }\n            }\n\n            formTemplate = $(template(this.options.promptContentTmpl, {\n                data: this.options,\n                formAttr: formAttr,\n                inputAttr: inputAttr\n            }));\n\n            return formTemplate;\n        },\n\n        /**\n         * Remove widget\n         */\n        _remove: function () {\n            this.modal.remove();\n        },\n\n        /**\n         * Validate prompt field\n         */\n        validate: function () {\n            return $.validator.validateSingleElement(this.options.promptField);\n        },\n\n        /**\n         * Add validation classes to prompt field\n         */\n        setValidationClasses: function () {\n            this.modal.find(this.options.promptField).attr('class', $.proxy(function (i, val) {\n                return val + ' ' + this.options.validationRules.join(' ');\n            }, this));\n        },\n\n        /**\n         * Open modal window\n         */\n        openModal: function () {\n            this._super();\n            this.modal.find(this.options.promptField).val(this.options.value);\n        },\n\n        /**\n         * Close modal window\n         */\n        closeModal: function (result) {\n            var value;\n\n            if (result) {\n                if (this.options.validation && !this.validate()) {\n                    return false;\n                }\n\n                value = this.modal.find(this.options.promptField).val();\n                this.options.actions.confirm.call(this, value);\n            } else {\n                this.options.actions.cancel.call(this, result);\n            }\n\n            this.options.actions.always();\n            this.element.bind('promptclosed', _.bind(this._remove, this));\n\n            return this._super();\n        }\n    });\n\n    return function (config) {\n        return $('<div class=\"prompt-message\"></div>').html(config.content).prompt(config);\n    };\n});\n","Magento_Ui/js/modal/confirm.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'underscore',\n    'mage/translate',\n    'jquery-ui-modules/widget',\n    'Magento_Ui/js/modal/modal'\n], function ($, _, $t) {\n    'use strict';\n\n    $.widget('mage.confirm', $.mage.modal, {\n        options: {\n            modalClass: 'confirm',\n            title: '',\n            focus: '.action-accept',\n            actions: {\n\n                /**\n                 * Callback always - called on all actions.\n                 */\n                always: function () {},\n\n                /**\n                 * Callback confirm.\n                 */\n                confirm: function () {},\n\n                /**\n                 * Callback cancel.\n                 */\n                cancel: function () {}\n            },\n            buttons: [{\n                text: $t('Cancel'),\n                class: 'action-secondary action-dismiss',\n\n                /**\n                 * Click handler.\n                 */\n                click: function (event) {\n                    this.closeModal(event);\n                }\n            }, {\n                text: $t('OK'),\n                class: 'action-primary action-accept',\n\n                /**\n                 * Click handler.\n                 */\n                click: function (event) {\n                    this.closeModal(event, true);\n                }\n            }]\n        },\n\n        /**\n         * Create widget.\n         */\n        _create: function () {\n            this._super();\n            this.modal.find(this.options.modalCloseBtn).off().on('click', _.bind(this.closeModal, this));\n            this.openModal();\n        },\n\n        /**\n         * Remove modal window.\n         */\n        _remove: function () {\n            this.modal.remove();\n        },\n\n        /**\n         * Open modal window.\n         */\n        openModal: function () {\n            return this._super();\n        },\n\n        /**\n         * Close modal window.\n         */\n        closeModal: function (event, result) {\n            result = result || false;\n\n            if (result) {\n                this.options.actions.confirm(event);\n            } else {\n                this.options.actions.cancel(event);\n            }\n            this.options.actions.always(event);\n            this.element.bind('confirmclosed', _.bind(this._remove, this));\n\n            return this._super();\n        }\n    });\n\n    return function (config) {\n        return $('<div></div>').html(config.content).confirm(config);\n    };\n});\n","Magento_Ui/js/modal/modal.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'underscore',\n    'mage/template',\n    'text!ui/template/modal/modal-popup.html',\n    'text!ui/template/modal/modal-slide.html',\n    'text!ui/template/modal/modal-custom.html',\n    'Magento_Ui/js/lib/key-codes',\n    'jquery-ui-modules/widget',\n    'mage/translate'\n], function ($, _, template, popupTpl, slideTpl, customTpl, keyCodes) {\n    'use strict';\n\n    /**\n     * Detect browser transition end event.\n     * @return {String|undefined} - transition event.\n     */\n    var transitionEvent =  (function () {\n        var transition,\n            elementStyle = document.createElement('div').style,\n            transitions = {\n                'transition': 'transitionend',\n                'OTransition': 'oTransitionEnd',\n                'MozTransition': 'transitionend',\n                'WebkitTransition': 'webkitTransitionEnd'\n            };\n\n        for (transition in transitions) {\n            if (elementStyle[transition] !== undefined && transitions.hasOwnProperty(transition)) {\n                return transitions[transition];\n            }\n        }\n    })();\n\n    /**\n     * Modal Window Widget\n     */\n    $.widget('mage.modal', {\n        options: {\n            id: null,\n            type: 'popup',\n            title: '',\n            subTitle: '',\n            modalClass: '',\n            focus: '[data-role=\"closeBtn\"]',\n            autoOpen: false,\n            clickableOverlay: true,\n            popupTpl: popupTpl,\n            slideTpl: slideTpl,\n            customTpl: customTpl,\n            modalVisibleClass: '_show',\n            parentModalClass: '_has-modal',\n            innerScrollClass: '_inner-scroll',\n            responsive: false,\n            innerScroll: false,\n            modalTitle: '[data-role=\"title\"]',\n            modalSubTitle: '[data-role=\"subTitle\"]',\n            modalBlock: '[data-role=\"modal\"]',\n            modalCloseBtn: '[data-role=\"closeBtn\"]',\n            modalContent: '[data-role=\"content\"]',\n            modalAction: '[data-role=\"action\"]',\n            focusableScope: '[data-role=\"focusable-scope\"]',\n            focusableStart: '[data-role=\"focusable-start\"]',\n            focusableEnd: '[data-role=\"focusable-end\"]',\n            appendTo: 'body',\n            wrapperClass: 'modals-wrapper',\n            overlayClass: 'modals-overlay',\n            responsiveClass: 'modal-slide',\n            trigger: '',\n            modalLeftMargin: 45,\n            closeText: $.mage.__('Close'),\n            buttons: [{\n                text: $.mage.__('Ok'),\n                class: '',\n                attr: {},\n\n                /**\n                 * Default action on button click\n                 */\n                click: function (event) {\n                    this.closeModal(event);\n                }\n            }],\n            keyEventHandlers: {\n\n                /**\n                 * Tab key press handler,\n                 * set focus to elements\n                 */\n                tabKey: function () {\n                    if (document.activeElement === this.modal[0]) {\n                        this._setFocus('start');\n                    }\n                },\n\n                /**\n                 * Escape key press handler,\n                 * close modal window\n                 * @param {Object} event - event\n                 */\n                escapeKey: function (event) {\n                    if (this.options.isOpen && this.modal.find(document.activeElement).length ||\n                        this.options.isOpen && this.modal[0] === document.activeElement) {\n                        this.closeModal(event);\n                    }\n                }\n            }\n        },\n\n        /**\n         * Creates modal widget.\n         */\n        _create: function () {\n            _.bindAll(\n                this,\n                'keyEventSwitcher',\n                '_tabSwitcher',\n                'closeModal'\n            );\n\n            this.options.id = this.uuid;\n            this.options.transitionEvent = transitionEvent;\n            this._createWrapper();\n            this._renderModal();\n            this._createButtons();\n            $(this.options.trigger).on('click', _.bind(this.toggleModal, this));\n            this._on(this.modal.find(this.options.modalCloseBtn), {\n                'click': this.options.modalCloseBtnHandler ? this.options.modalCloseBtnHandler : this.closeModal\n            });\n            this._on(this.element, {\n                'openModal': this.openModal,\n                'closeModal': this.closeModal\n            });\n            this.options.autoOpen ? this.openModal() : false;\n        },\n\n        /**\n         * Returns element from modal node.\n         * @return {Object} - element.\n         */\n        _getElem: function (elem) {\n            return this.modal.find(elem);\n        },\n\n        /**\n         * Gets visible modal count.\n         * * @return {Number} - visible modal count.\n         */\n        _getVisibleCount: function () {\n            var modals = this.modalWrapper.find(this.options.modalBlock);\n\n            return modals.filter('.' + this.options.modalVisibleClass).length;\n        },\n\n        /**\n         * Gets count of visible modal by slide type.\n         * * @return {Number} - visible modal count.\n         */\n        _getVisibleSlideCount: function () {\n            var elems = this.modalWrapper.find('[data-type=\"slide\"]');\n\n            return elems.filter('.' + this.options.modalVisibleClass).length;\n        },\n\n        /**\n         * Listener key events.\n         * Call handler function if it exists\n         */\n        keyEventSwitcher: function (event) {\n            var key = keyCodes[event.keyCode];\n\n            if (this.options.keyEventHandlers.hasOwnProperty(key)) {\n                this.options.keyEventHandlers[key].apply(this, arguments);\n            }\n        },\n\n        /**\n         * Set title for modal.\n         *\n         * @param {String} title\n         */\n        setTitle: function (title) {\n            var $title = $(this.options.modalTitle),\n                $subTitle = this.modal.find(this.options.modalSubTitle);\n\n            $title.text(title);\n            $title.append($subTitle);\n        },\n\n        /**\n         * Set sub title for modal.\n         *\n         * @param {String} subTitle\n         */\n        setSubTitle: function (subTitle) {\n            this.options.subTitle = subTitle;\n            this.modal.find(this.options.modalSubTitle).html(subTitle);\n        },\n\n        /**\n         * Toggle modal.\n         * * @return {Element} - current element.\n         */\n        toggleModal: function () {\n            if (this.options.isOpen === true) {\n                this.closeModal();\n            } else {\n                this.openModal();\n            }\n        },\n\n        /**\n         * Open modal.\n         * * @return {Element} - current element.\n         */\n        openModal: function () {\n            this.options.isOpen = true;\n            this.focussedElement = document.activeElement;\n            this._createOverlay();\n            this._setActive();\n            this._setKeyListener();\n            this.modal.one(this.options.transitionEvent, _.bind(this._setFocus, this, 'end', 'opened'));\n            this.modal.one(this.options.transitionEvent, _.bind(this._trigger, this, 'opened'));\n            this.modal.addClass(this.options.modalVisibleClass);\n\n            if (!this.options.transitionEvent) {\n                this._trigger('opened');\n            }\n\n            return this.element;\n        },\n\n        /**\n         * Set focus to element.\n         * @param {String} position - can be \"start\" and \"end\"\n         *      positions.\n         *      If position is \"end\" - sets focus to first\n         *      focusable element in modal window scope.\n         *      If position is \"start\" - sets focus to last\n         *      focusable element in modal window scope\n         *\n         *  @param {String} type - can be \"opened\" or false\n         *      If type is \"opened\" - looks to \"this.options.focus\"\n         *      property and sets focus\n         */\n        _setFocus: function (position, type) {\n            var focusableElements,\n                infelicity;\n\n            if (type === 'opened' && this.options.focus) {\n                this.modal.find($(this.options.focus)).focus();\n            } else if (type === 'opened' && !this.options.focus) {\n                this.modal.find(this.options.focusableScope).focus();\n            } else if (position === 'end') {\n                this.modal.find(this.options.modalCloseBtn).focus();\n            } else if (position === 'start') {\n                infelicity = 2; //Constant for find last focusable element\n                focusableElements = this.modal.find(':focusable');\n                focusableElements.eq(focusableElements.length - infelicity).focus();\n            }\n        },\n\n        /**\n         * Set events listener when modal is opened.\n         */\n        _setKeyListener: function () {\n            this.modal.find(this.options.focusableStart).bind('focusin', this._tabSwitcher);\n            this.modal.find(this.options.focusableEnd).bind('focusin', this._tabSwitcher);\n            this.modal.bind('keydown', this.keyEventSwitcher);\n        },\n\n        /**\n         * Remove events listener when modal is closed.\n         */\n        _removeKeyListener: function () {\n            this.modal.find(this.options.focusableStart).unbind('focusin', this._tabSwitcher);\n            this.modal.find(this.options.focusableEnd).unbind('focusin', this._tabSwitcher);\n            this.modal.unbind('keydown', this.keyEventSwitcher);\n        },\n\n        /**\n         * Switcher for focus event.\n         * @param {Object} e - event\n         */\n        _tabSwitcher: function (e) {\n            var target = $(e.target);\n\n            if (target.is(this.options.focusableStart)) {\n                this._setFocus('start');\n            } else if (target.is(this.options.focusableEnd)) {\n                this._setFocus('end');\n            }\n        },\n\n        /**\n         * Close modal.\n         * * @return {Element} - current element.\n         */\n        closeModal: function () {\n            var that = this;\n\n            this._removeKeyListener();\n            this.options.isOpen = false;\n            this.modal.one(this.options.transitionEvent, function () {\n                that._close();\n            });\n            this.modal.removeClass(this.options.modalVisibleClass);\n\n            if (!this.options.transitionEvent) {\n                that._close();\n            }\n\n            return this.element;\n        },\n\n        /**\n         * Helper for closeModal function.\n         */\n        _close: function () {\n            var trigger = _.bind(this._trigger, this, 'closed', this.modal);\n\n            $(this.focussedElement).focus();\n            this._destroyOverlay();\n            this._unsetActive();\n            _.defer(trigger, this);\n        },\n\n        /**\n         * Set z-index and margin for modal and overlay.\n         */\n        _setActive: function () {\n            var zIndex = this.modal.zIndex(),\n                baseIndex = zIndex + this._getVisibleCount();\n\n            if (this.modal.data('active')) {\n                return;\n            }\n\n            this.modal.data('active', true);\n\n            this.overlay.zIndex(++baseIndex);\n            this.prevOverlayIndex = this.overlay.zIndex();\n            this.modal.zIndex(this.overlay.zIndex() + 1);\n\n            if (this._getVisibleSlideCount()) {\n                this.modal.css('marginLeft', this.options.modalLeftMargin * this._getVisibleSlideCount());\n            }\n        },\n\n        /**\n         * Unset styles for modal and set z-index for previous modal.\n         */\n        _unsetActive: function () {\n            this.modal.removeAttr('style');\n            this.modal.data('active', false);\n\n            if (this.overlay) {\n                this.overlay.zIndex(this.prevOverlayIndex - 1);\n            }\n        },\n\n        /**\n         * Creates wrapper to hold all modals.\n         */\n        _createWrapper: function () {\n            this.modalWrapper = $(this.options.appendTo).find('.' + this.options.wrapperClass);\n\n            if (!this.modalWrapper.length) {\n                this.modalWrapper = $('<div></div>')\n                    .addClass(this.options.wrapperClass)\n                    .appendTo(this.options.appendTo);\n            }\n        },\n\n        /**\n         * Compile template and append to wrapper.\n         */\n        _renderModal: function () {\n            $(template(\n                this.options[this.options.type + 'Tpl'],\n                {\n                    data: this.options\n                })).appendTo(this.modalWrapper);\n            this.modal = this.modalWrapper.find(this.options.modalBlock).last();\n            this.element.appendTo(this._getElem(this.options.modalContent));\n\n            if (this.element.is(':hidden')) {\n                this.element.show();\n            }\n        },\n\n        /**\n         * Creates buttons pane.\n         */\n        _createButtons: function () {\n            this.buttons = this._getElem(this.options.modalAction);\n            _.each(this.options.buttons, function (btn, key) {\n                var button = this.buttons[key];\n\n                if (btn.attr) {\n                    $(button).attr(btn.attr);\n                }\n\n                if (btn.class) {\n                    $(button).addClass(btn.class);\n                }\n\n                if (!btn.click) {\n                    btn.click = this.closeModal;\n                }\n                $(button).on('click', _.bind(btn.click, this));\n            }, this);\n        },\n\n        /**\n         * Creates overlay, append it to wrapper, set previous click event on overlay.\n         */\n        _createOverlay: function () {\n            var events,\n                outerClickHandler = this.options.outerClickHandler || this.closeModal;\n\n            this.overlay = $('.' + this.options.overlayClass);\n\n            if (!this.overlay.length) {\n                $(this.options.appendTo).addClass(this.options.parentModalClass);\n                this.overlay = $('<div></div>')\n                    .addClass(this.options.overlayClass)\n                    .appendTo(this.modalWrapper);\n            }\n            events = $._data(this.overlay.get(0), 'events');\n            events ? this.prevOverlayHandler = events.click[0].handler : false;\n            this.options.clickableOverlay ? this.overlay.unbind().on('click', outerClickHandler) : false;\n        },\n\n        /**\n         * Destroy overlay.\n         */\n        _destroyOverlay: function () {\n            if (this._getVisibleCount()) {\n                this.overlay.unbind().on('click', this.prevOverlayHandler);\n            } else {\n                $(this.options.appendTo).removeClass(this.options.parentModalClass);\n                this.overlay.remove();\n                this.overlay = null;\n            }\n        }\n    });\n\n    return $.mage.modal;\n});\n","Magento_Ui/js/view/messages.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'ko',\n    'jquery',\n    'uiComponent',\n    '../model/messageList',\n    'jquery-ui-modules/effect-blind'\n], function (ko, $, Component, globalMessages) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Ui/messages',\n            selector: '[data-role=checkout-messages]',\n            isHidden: false,\n            listens: {\n                isHidden: 'onHiddenChange'\n            }\n        },\n\n        /** @inheritdoc */\n        initialize: function (config, messageContainer) {\n            this._super()\n                .initObservable();\n\n            this.messageContainer = messageContainer || config.messageContainer || globalMessages;\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super()\n                .observe('isHidden');\n\n            return this;\n        },\n\n        /**\n         * Checks visibility.\n         *\n         * @return {Boolean}\n         */\n        isVisible: function () {\n            return this.isHidden(this.messageContainer.hasMessages());\n        },\n\n        /**\n         * Remove all messages.\n         */\n        removeAll: function () {\n            this.messageContainer.clear();\n        },\n\n        /**\n         * @param {Boolean} isHidden\n         */\n        onHiddenChange: function (isHidden) {\n            var self = this;\n\n            // Hide message block if needed\n            if (isHidden) {\n                setTimeout(function () {\n                    $(self.selector).hide('blind', {}, 500);\n                }, 5000);\n            }\n        }\n    });\n});\n","Magento_Ui/js/model/messageList.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    './messages'\n], function (Messages) {\n    'use strict';\n\n    return new Messages();\n});\n","Magento_Ui/js/model/messages.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'ko',\n    'uiClass'\n], function (ko, Class) {\n    'use strict';\n\n    return Class.extend({\n        /** @inheritdoc */\n        initialize: function () {\n            this._super()\n                .initObservable();\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this.errorMessages = ko.observableArray([]);\n            this.successMessages = ko.observableArray([]);\n\n            return this;\n        },\n\n        /**\n         * Add  message to list.\n         * @param {Object} messageObj\n         * @param {Object} type\n         * @returns {Boolean}\n         */\n        add: function (messageObj, type) {\n            var expr = /([%])\\w+/g,\n                message;\n\n            if (!messageObj.hasOwnProperty('parameters')) {\n                this.clear();\n                type.push(messageObj.message);\n\n                return true;\n            }\n            message = messageObj.message.replace(expr, function (varName) {\n                varName = varName.substr(1);\n\n                if (messageObj.parameters.hasOwnProperty(varName)) {\n                    return messageObj.parameters[varName];\n                }\n\n                return messageObj.parameters.shift();\n            });\n            this.clear();\n            type.push(message);\n\n            return true;\n        },\n\n        /**\n         * Add success message.\n         *\n         * @param {Object} message\n         * @return {*|Boolean}\n         */\n        addSuccessMessage: function (message) {\n            return this.add(message, this.successMessages);\n        },\n\n        /**\n         * Add error message.\n         *\n         * @param {Object} message\n         * @return {*|Boolean}\n         */\n        addErrorMessage: function (message) {\n            return this.add(message, this.errorMessages);\n        },\n\n        /**\n         * Get error messages.\n         *\n         * @return {Array}\n         */\n        getErrorMessages: function () {\n            return this.errorMessages;\n        },\n\n        /**\n         * Get success messages.\n         *\n         * @return {Array}\n         */\n        getSuccessMessages: function () {\n            return this.successMessages;\n        },\n\n        /**\n         * Checks if an instance has stored messages.\n         *\n         * @return {Boolean}\n         */\n        hasMessages: function () {\n            return this.errorMessages().length > 0 || this.successMessages().length > 0;\n        },\n\n        /**\n         * Removes stored messages.\n         */\n        clear: function () {\n            this.errorMessages.removeAll();\n            this.successMessages.removeAll();\n        }\n    });\n});\n","Magento_SalesRule/js/form/element/manage-coupon-codes.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'uiRegistry',\n    'Magento_Ui/js/form/components/fieldset',\n    'Magento_Ui/js/lib/view/utils/async'\n], function (_, uiRegistry, fieldset, async) {\n    'use strict';\n\n    return fieldset.extend({\n\n        /*eslint-disable no-unused-vars*/\n        /**\n         * Initialize element\n         *\n         * @returns {Abstract} Chainable\n         */\n        initialize: function (elems, position) {\n            var obj = this;\n\n            this._super();\n\n            async.async('#sales-rule-form-tab-coupons', document.getElementById('container'), function (node) {\n                var useAutoGeneration = uiRegistry.get(\n                    'sales_rule_form.sales_rule_form.rule_information.use_auto_generation'\n                );\n\n                useAutoGeneration.on('checked', function () {\n                    obj.enableDisableFields();\n                });\n                obj.enableDisableFields();\n            });\n\n            return this;\n        },\n\n        /*eslint-enable no-unused-vars*/\n        /*eslint-disable lines-around-comment*/\n\n        /**\n         * Enable/disable fields on Coupons tab\n         */\n        enableDisableFields: function () {\n            var selector,\n                isUseAutoGenerationChecked,\n                couponType,\n                disableAuto;\n\n            selector = '[id=sales-rule-form-tab-coupons] input, [id=sales-rule-form-tab-coupons] select, ' +\n                    '[id=sales-rule-form-tab-coupons] button';\n            isUseAutoGenerationChecked = uiRegistry\n                    .get('sales_rule_form.sales_rule_form.rule_information.use_auto_generation')\n                    .checked();\n            couponType = uiRegistry\n                .get('sales_rule_form.sales_rule_form.rule_information.coupon_type')\n                .value();\n            /**\n             * \\Magento\\Rule\\Model\\AbstractModel::COUPON_TYPE_AUTO\n             */\n            disableAuto = couponType === 3 || isUseAutoGenerationChecked;\n            _.each(\n                document.querySelectorAll(selector),\n                function (element) {\n                    element.disabled = !disableAuto;\n                }\n            );\n        }\n    });\n});\n","Magento_SalesRule/js/form/element/coupon-type.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'uiRegistry',\n    'Magento_Ui/js/form/element/select'\n], function (_, uiRegistry, select) {\n    'use strict';\n\n    return select.extend({\n\n        /**\n         * Hide fields on coupon tab\n         */\n        onUpdate: function () {\n\n            /* eslint-disable eqeqeq */\n            if (this.value() != this.displayOnlyForCouponType) {\n                uiRegistry.get('sales_rule_form.sales_rule_form.rule_information.use_auto_generation').checked(false);\n            }\n\n            this.enableDisableFields();\n        },\n\n        /**\n         * Enable/disable fields on Coupons tab\n         */\n        enableDisableFields: function () {\n            var selector,\n                isUseAutoGenerationChecked,\n                couponType,\n                disableAuto;\n\n            selector = '[id=sales-rule-form-tab-coupons] input, [id=sales-rule-form-tab-coupons] select, ' +\n                    '[id=sales-rule-form-tab-coupons] button';\n            isUseAutoGenerationChecked = uiRegistry\n                    .get('sales_rule_form.sales_rule_form.rule_information.use_auto_generation')\n                    .checked();\n            couponType = uiRegistry\n                .get('sales_rule_form.sales_rule_form.rule_information.coupon_type')\n                .value();\n            disableAuto = couponType === 3 || isUseAutoGenerationChecked;\n            _.each(\n                document.querySelectorAll(selector),\n                function (element) {\n                    element.disabled = !disableAuto;\n                }\n            );\n        }\n    });\n});\n","Magento_SalesRule/js/action/set-coupon-code.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * Customer store credit(balance) application\n */\ndefine([\n    'ko',\n    'jquery',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/resource-url-manager',\n    'Magento_Checkout/js/model/error-processor',\n    'Magento_SalesRule/js/model/payment/discount-messages',\n    'mage/storage',\n    'mage/translate',\n    'Magento_Checkout/js/action/get-payment-information',\n    'Magento_Checkout/js/model/totals',\n    'Magento_Checkout/js/model/full-screen-loader'\n], function (ko, $, quote, urlManager, errorProcessor, messageContainer, storage, $t, getPaymentInformationAction,\n    totals, fullScreenLoader\n) {\n    'use strict';\n\n    var dataModifiers = [],\n        successCallbacks = [],\n        failCallbacks = [],\n        action;\n\n    /**\n     * Apply provided coupon.\n     *\n     * @param {String} couponCode\n     * @param {Boolean}isApplied\n     * @returns {Deferred}\n     */\n    action = function (couponCode, isApplied) {\n        var quoteId = quote.getQuoteId(),\n            url = urlManager.getApplyCouponUrl(couponCode, quoteId),\n            message = $t('Your coupon was successfully applied.'),\n            data = {},\n            headers = {};\n\n        //Allowing to modify coupon-apply request\n        dataModifiers.forEach(function (modifier) {\n            modifier(headers, data);\n        });\n        fullScreenLoader.startLoader();\n\n        return storage.put(\n            url,\n            data,\n            false,\n            null,\n            headers\n        ).done(function (response) {\n            var deferred;\n\n            if (response) {\n                deferred = $.Deferred();\n\n                isApplied(true);\n                totals.isLoading(true);\n                getPaymentInformationAction(deferred);\n                $.when(deferred).done(function () {\n                    fullScreenLoader.stopLoader();\n                    totals.isLoading(false);\n                });\n                messageContainer.addSuccessMessage({\n                    'message': message\n                });\n                //Allowing to tap into apply-coupon process.\n                successCallbacks.forEach(function (callback) {\n                    callback(response);\n                });\n            }\n        }).fail(function (response) {\n            fullScreenLoader.stopLoader();\n            totals.isLoading(false);\n            errorProcessor.process(response, messageContainer);\n            //Allowing to tap into apply-coupon process.\n            failCallbacks.forEach(function (callback) {\n                callback(response);\n            });\n        });\n    };\n\n    /**\n     * Modifying data to be sent.\n     *\n     * @param {Function} modifier\n     */\n    action.registerDataModifier = function (modifier) {\n        dataModifiers.push(modifier);\n    };\n\n    /**\n     * When successfully added a coupon.\n     *\n     * @param {Function} callback\n     */\n    action.registerSuccessCallback = function (callback) {\n        successCallbacks.push(callback);\n    };\n\n    /**\n     * When failed to add a coupon.\n     *\n     * @param {Function} callback\n     */\n    action.registerFailCallback = function (callback) {\n        failCallbacks.push(callback);\n    };\n\n    return action;\n});\n","Magento_SalesRule/js/action/cancel-coupon.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * Customer store credit(balance) application\n */\ndefine([\n    'jquery',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/resource-url-manager',\n    'Magento_Checkout/js/model/error-processor',\n    'Magento_SalesRule/js/model/payment/discount-messages',\n    'mage/storage',\n    'Magento_Checkout/js/action/get-payment-information',\n    'Magento_Checkout/js/model/totals',\n    'mage/translate',\n    'Magento_Checkout/js/model/full-screen-loader'\n], function ($, quote, urlManager, errorProcessor, messageContainer, storage, getPaymentInformationAction, totals, $t,\n  fullScreenLoader\n) {\n    'use strict';\n\n    var successCallbacks = [],\n        action,\n        callSuccessCallbacks;\n\n    /**\n     * Execute callbacks when a coupon is successfully canceled.\n     */\n    callSuccessCallbacks = function () {\n        successCallbacks.forEach(function (callback) {\n            callback();\n        });\n    };\n\n    /**\n     * Cancel applied coupon.\n     *\n     * @param {Boolean} isApplied\n     * @returns {Deferred}\n     */\n    action =  function (isApplied) {\n        var quoteId = quote.getQuoteId(),\n            url = urlManager.getCancelCouponUrl(quoteId),\n            message = $t('Your coupon was successfully removed.');\n\n        messageContainer.clear();\n        fullScreenLoader.startLoader();\n\n        return storage.delete(\n            url,\n            false\n        ).done(function () {\n            var deferred = $.Deferred();\n\n            totals.isLoading(true);\n            getPaymentInformationAction(deferred);\n            $.when(deferred).done(function () {\n                isApplied(false);\n                totals.isLoading(false);\n                fullScreenLoader.stopLoader();\n                //Allowing to tap into coupon-cancel process.\n                callSuccessCallbacks();\n            });\n            messageContainer.addSuccessMessage({\n                'message': message\n            });\n        }).fail(function (response) {\n            totals.isLoading(false);\n            fullScreenLoader.stopLoader();\n            errorProcessor.process(response, messageContainer);\n        });\n    };\n\n    /**\n     * Callback for when the cancel-coupon process is finished.\n     *\n     * @param {Function} callback\n     */\n    action.registerSuccessCallback = function (callback) {\n        successCallbacks.push(callback);\n    };\n\n    return action;\n});\n","Magento_SalesRule/js/view/summary/discount.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/view/summary/abstract-total',\n    'Magento_Checkout/js/model/quote'\n], function (Component, quote) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_SalesRule/summary/discount'\n        },\n        totals: quote.getTotals(),\n\n        /**\n         * @return {*|Boolean}\n         */\n        isDisplayed: function () {\n            return this.isFullMode() && this.getPureValue() != 0; //eslint-disable-line eqeqeq\n        },\n\n        /**\n         * @return {*}\n         */\n        getCouponCode: function () {\n            if (!this.totals()) {\n                return null;\n            }\n\n            return this.totals()['coupon_code'];\n        },\n\n        /**\n         * @return {*}\n         */\n        getCouponLabel: function () {\n            if (!this.totals()) {\n                return null;\n            }\n\n            return this.totals()['coupon_label'];\n        },\n\n        /**\n         * Get discount title\n         *\n         * @returns {null|String}\n         */\n        getTitle: function () {\n            var discountSegments;\n\n            if (!this.totals()) {\n                return null;\n            }\n\n            discountSegments = this.totals()['total_segments'].filter(function (segment) {\n                return segment.code.indexOf('discount') !== -1;\n            });\n\n            return discountSegments.length ? discountSegments[0].title : null;\n        },\n\n        /**\n         * @return {Number}\n         */\n        getPureValue: function () {\n            var price = 0;\n\n            if (this.totals() && this.totals()['discount_amount']) {\n                price = parseFloat(this.totals()['discount_amount']);\n            }\n\n            return price;\n        },\n\n        /**\n         * @return {*|String}\n         */\n        getValue: function () {\n            return this.getFormattedPrice(this.getPureValue());\n        }\n    });\n});\n","Magento_SalesRule/js/view/payment/discount-messages.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Ui/js/view/messages',\n    '../../model/payment/discount-messages'\n], function (Component, messageContainer) {\n    'use strict';\n\n    return Component.extend({\n        /** @inheritdoc */\n        initialize: function (config) {\n            return this._super(config, messageContainer);\n        }\n    });\n});\n","Magento_SalesRule/js/view/payment/discount.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'ko',\n    'uiComponent',\n    'Magento_Checkout/js/model/quote',\n    'Magento_SalesRule/js/action/set-coupon-code',\n    'Magento_SalesRule/js/action/cancel-coupon'\n], function ($, ko, Component, quote, setCouponCodeAction, cancelCouponAction) {\n    'use strict';\n\n    var totals = quote.getTotals(),\n        couponCode = ko.observable(null),\n        isApplied;\n\n    if (totals()) {\n        couponCode(totals()['coupon_code']);\n    }\n    isApplied = ko.observable(couponCode() != null);\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_SalesRule/payment/discount'\n        },\n        couponCode: couponCode,\n\n        /**\n         * Applied flag\n         */\n        isApplied: isApplied,\n\n        /**\n         * Coupon code application procedure\n         */\n        apply: function () {\n            if (this.validate()) {\n                setCouponCodeAction(couponCode(), isApplied);\n            }\n        },\n\n        /**\n         * Cancel using coupon\n         */\n        cancel: function () {\n            if (this.validate()) {\n                couponCode('');\n                cancelCouponAction(isApplied);\n            }\n        },\n\n        /**\n         * Coupon form validation\n         *\n         * @returns {Boolean}\n         */\n        validate: function () {\n            var form = '#discount-form';\n\n            return $(form).validation() && $(form).validation('isValid');\n        }\n    });\n});\n","Magento_SalesRule/js/view/payment/captcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n        'Magento_Captcha/js/view/checkout/defaultCaptcha',\n        'Magento_Captcha/js/model/captchaList',\n        'Magento_SalesRule/js/action/set-coupon-code',\n        'Magento_SalesRule/js/action/cancel-coupon',\n        'Magento_Checkout/js/model/quote',\n        'ko'\n    ],\n    function (defaultCaptcha, captchaList, setCouponCodeAction, cancelCouponAction, quote, ko) {\n        'use strict';\n\n        var totals = quote.getTotals(),\n            couponCode = ko.observable(null),\n            isApplied;\n\n        if (totals()) {\n            couponCode(totals()['coupon_code']);\n        }\n        //Captcha can only be required for adding a coupon so we need to know if one was added already.\n        isApplied = ko.observable(couponCode() != null);\n\n        return defaultCaptcha.extend({\n            /** @inheritdoc */\n            initialize: function () {\n                var self = this,\n                    currentCaptcha;\n\n                this._super();\n                //Getting coupon captcha model.\n                currentCaptcha = captchaList.getCaptchaByFormId(this.formId);\n\n                if (currentCaptcha != null) {\n                    if (!isApplied()) {\n                        //Show captcha if we don't have a coupon applied.\n                        currentCaptcha.setIsVisible(true);\n                    }\n                    this.setCurrentCaptcha(currentCaptcha);\n                    //Add captcha code to coupon-apply request.\n                    setCouponCodeAction.registerDataModifier(function (headers) {\n                        if (self.isRequired()) {\n                            headers['X-Captcha'] = self.captchaValue()();\n                        }\n                    });\n                    //Refresh captcha after failed request.\n                    setCouponCodeAction.registerFailCallback(function () {\n                        if (self.isRequired()) {\n                            self.refresh();\n                        }\n                    });\n                    //Hide captcha when a coupon has been applied.\n                    setCouponCodeAction.registerSuccessCallback(function () {\n                        self.setIsVisible(false);\n                    });\n                    //Show captcha again if it was canceled.\n                    cancelCouponAction.registerSuccessCallback(function () {\n                        if (self.isRequired()) {\n                            self.setIsVisible(true);\n                        }\n                    });\n                }\n            }\n        });\n    });\n","Magento_SalesRule/js/view/cart/totals/discount.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_SalesRule/js/view/summary/discount'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_SalesRule/cart/totals/discount'\n        },\n\n        /**\n         * @override\n         *\n         * @returns {Boolean}\n         */\n        isDisplayed: function () {\n            return this.getPureValue() != 0; //eslint-disable-line eqeqeq\n        }\n    });\n});\n","Magento_SalesRule/js/model/payment/discount-messages.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Ui/js/model/messages'\n], function (Messages) {\n    'use strict';\n\n    return new Messages();\n});\n","Magento_Captcha/js/captcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    /**\n     * @api\n     */\n    $.widget('mage.captcha', {\n        options: {\n            refreshClass: 'refreshing',\n            reloadSelector: '.captcha-reload',\n            imageSelector: '.captcha-img',\n            imageLoader: ''\n        },\n\n        /**\n         * Method binds click event to reload image\n         * @private\n         */\n        _create: function () {\n            this.element.on('click', this.options.reloadSelector, $.proxy(this.refresh, this));\n        },\n\n        /**\n         * Method triggers an AJAX request to refresh the CAPTCHA image\n         */\n        refresh: function () {\n            var imageLoader = this.options.imageLoader;\n\n            if (imageLoader) {\n                this.element.find(this.options.imageSelector).attr('src', imageLoader);\n            }\n            this.element.addClass(this.options.refreshClass);\n\n            $.ajax({\n                url: this.options.url,\n                type: 'post',\n                async: false,\n                dataType: 'json',\n                context: this,\n                data: {\n                    'formId': this.options.type\n                },\n\n                /**\n                 * @param {Object} response\n                 */\n                success: function (response) {\n                    if (response.imgSrc) {\n                        this.element.find(this.options.imageSelector).attr('src', response.imgSrc);\n                    }\n                },\n\n                /** Complete callback. */\n                complete: function () {\n                    this.element.removeClass(this.options.refreshClass);\n                }\n            });\n        }\n    });\n\n    return $.mage.captcha;\n});\n","Magento_Captcha/js/action/refresh.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'mage/storage'\n], function (storage) {\n    'use strict';\n\n    return function (refreshUrl, formId, imageSource) {\n        return storage.post(\n            refreshUrl,\n            JSON.stringify({\n                'formId': formId\n            }),\n            false\n        ).done(\n            function (response) {\n                if (response.imgSrc) {\n                    imageSource(response.imgSrc);\n                }\n            }\n        );\n    };\n});\n","Magento_Captcha/js/view/checkout/defaultCaptcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'uiComponent',\n    'Magento_Captcha/js/model/captcha',\n    'Magento_Captcha/js/model/captchaList',\n    'Magento_Customer/js/customer-data',\n    'underscore'\n], function ($, Component, Captcha, captchaList, customerData, _) {\n    'use strict';\n\n    var captchaConfig;\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Captcha/checkout/captcha'\n        },\n        dataScope: 'global',\n        currentCaptcha: null,\n\n        /**\n         * @return {*}\n         */\n        captchaValue: function () {\n            return this.currentCaptcha.getCaptchaValue();\n        },\n\n        /** @inheritdoc */\n        initialize: function () {\n            this._super();\n\n            if (window[this.configSource] && window[this.configSource].captcha) {\n                captchaConfig = window[this.configSource].captcha;\n                $.each(captchaConfig, function (formId, captchaData) {\n                    var captcha;\n\n                    captchaData.formId = formId;\n                    captcha = Captcha(captchaData);\n                    this.checkCustomerData(formId, customerData.get('captcha')(), captcha);\n                    this.subscribeCustomerData(formId, captcha);\n                    captchaList.add(captcha);\n                }.bind(this));\n            }\n        },\n\n        /**\n         * Check customer data for captcha configuration.\n         *\n         * @param {String} formId\n         * @param {Object} captchaData\n         * @param {Object} captcha\n         */\n        checkCustomerData: function (formId, captchaData, captcha) {\n            if (!_.isEmpty(captchaData) &&\n                !_.isEmpty(captchaData)[formId] &&\n                captchaData[formId].timestamp > captcha.timestamp\n            ) {\n                if (!captcha.isRequired() && captchaData[formId].isRequired) {\n                    captcha.refresh();\n                }\n                captcha.isRequired(captchaData[formId].isRequired);\n                captcha.timestamp = captchaData[formId].timestamp;\n            }\n        },\n\n        /**\n         * Subscribe for customer data updates.\n         *\n         * @param {String} formId\n         * @param {Object} captcha\n         */\n        subscribeCustomerData: function (formId, captcha) {\n            customerData.get('captcha').subscribe(function (captchaData) {\n                this.checkCustomerData(formId, captchaData, captcha);\n            }.bind(this));\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        getIsLoading: function () {\n            return this.currentCaptcha !== null ? this.currentCaptcha.isLoading : false;\n        },\n\n        /**\n         * @return {null|Object}\n         */\n        getCurrentCaptcha: function () {\n            return this.currentCaptcha;\n        },\n\n        /**\n         * @param {Object} captcha\n         */\n        setCurrentCaptcha: function (captcha) {\n            this.currentCaptcha = captcha;\n        },\n\n        /**\n         * @return {String|null}\n         */\n        getFormId: function () {\n            return this.currentCaptcha !== null ? this.currentCaptcha.getFormId() : null;\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        getIsVisible: function () {\n            return this.currentCaptcha !== null ? this.currentCaptcha.getIsVisible() : false;\n        },\n\n        /**\n         * @param {Boolean} flag\n         */\n        setIsVisible: function (flag) {\n            this.currentCaptcha.setIsVisible(flag);\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        isRequired: function () {\n            return this.currentCaptcha !== null ? this.currentCaptcha.getIsRequired() : false;\n        },\n\n        /**\n         * Set isRequired on current captcha model.\n         *\n         * @param {Boolean} flag\n         */\n        setIsRequired: function (flag) {\n            this.currentCaptcha.setIsRequired(flag);\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        isCaseSensitive: function () {\n            return this.currentCaptcha !== null ? this.currentCaptcha.getIsCaseSensitive() : false;\n        },\n\n        /**\n         * @return {String|Number|null}\n         */\n        imageHeight: function () {\n            return this.currentCaptcha !== null ? this.currentCaptcha.getImageHeight() : null;\n        },\n\n        /**\n         * @return {String|null}\n         */\n        getImageSource: function () {\n            return this.currentCaptcha !== null ? this.currentCaptcha.getImageSource() : null;\n        },\n\n        /**\n         * Refresh captcha.\n         */\n        refresh: function () {\n            this.currentCaptcha.refresh();\n        }\n    });\n});\n","Magento_Captcha/js/view/checkout/loginCaptcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Captcha/js/view/checkout/defaultCaptcha',\n    'Magento_Captcha/js/model/captchaList',\n    'Magento_Customer/js/action/login',\n    'underscore'\n],\nfunction (defaultCaptcha, captchaList, loginAction, _) {\n    'use strict';\n\n    return defaultCaptcha.extend({\n        /** @inheritdoc */\n        initialize: function () {\n            var self = this,\n                currentCaptcha;\n\n            this._super();\n            currentCaptcha = captchaList.getCaptchaByFormId(this.formId);\n\n            if (currentCaptcha != null) {\n                currentCaptcha.setIsVisible(true);\n                this.setCurrentCaptcha(currentCaptcha);\n\n                loginAction.registerLoginCallback(function (loginData) {\n                    if (loginData['captcha_form_id'] &&\n                        loginData['captcha_form_id'] === self.formId &&\n                        self.isRequired()\n                    ) {\n                        _.defer(self.refresh.bind(self));\n                    }\n                });\n            }\n        }\n    });\n});\n","Magento_Captcha/js/model/captchaList.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['jquery'], function ($) {\n    'use strict';\n\n    var captchaList = [];\n\n    return {\n        /**\n         * @param {Object} captcha\n         */\n        add: function (captcha) {\n            captchaList.push(captcha);\n        },\n\n        /**\n         * @param {String} formId\n         * @return {Object}\n         */\n        getCaptchaByFormId: function (formId) {\n            var captcha = null;\n\n            $.each(captchaList, function (key, item) {\n                if (formId === item.formId) {\n                    captcha = item;\n\n                    return false;\n                }\n            });\n\n            return captcha;\n        },\n\n        /**\n         * @return {Array}\n         */\n        getCaptchaList: function () {\n            return captchaList;\n        }\n    };\n});\n","Magento_Captcha/js/model/captcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/*global alert*/\ndefine([\n    'jquery',\n    'ko',\n    'Magento_Captcha/js/action/refresh'\n], function ($, ko, refreshAction) {\n    'use strict';\n\n    return function (captchaData) {\n        return {\n            formId: captchaData.formId,\n            imageSource: ko.observable(captchaData.imageSrc),\n            visibility: ko.observable(false),\n            captchaValue: ko.observable(null),\n            isRequired: ko.observable(captchaData.isRequired),\n            isCaseSensitive: captchaData.isCaseSensitive,\n            imageHeight: captchaData.imageHeight,\n            refreshUrl: captchaData.refreshUrl,\n            isLoading: ko.observable(false),\n            timestamp: null,\n\n            /**\n             * @return {String}\n             */\n            getFormId: function () {\n                return this.formId;\n            },\n\n            /**\n             * @param {String} formId\n             */\n            setFormId: function (formId) {\n                this.formId = formId;\n            },\n\n            /**\n             * @return {Boolean}\n             */\n            getIsVisible: function () {\n                return this.visibility();\n            },\n\n            /**\n             * @param {Boolean} flag\n             */\n            setIsVisible: function (flag) {\n                this.visibility(flag);\n            },\n\n            /**\n             * @return {Boolean}\n             */\n            getIsRequired: function () {\n                return this.isRequired();\n            },\n\n            /**\n             * @param {Boolean} flag\n             */\n            setIsRequired: function (flag) {\n                this.isRequired(flag);\n            },\n\n            /**\n             * @return {Boolean}\n             */\n            getIsCaseSensitive: function () {\n                return this.isCaseSensitive;\n            },\n\n            /**\n             * @param {Boolean} flag\n             */\n            setIsCaseSensitive: function (flag) {\n                this.isCaseSensitive = flag;\n            },\n\n            /**\n             * @return {String|Number}\n             */\n            getImageHeight: function () {\n                return this.imageHeight;\n            },\n\n            /**\n             * @param {String|Number}height\n             */\n            setImageHeight: function (height) {\n                this.imageHeight = height;\n            },\n\n            /**\n             * @return {String}\n             */\n            getImageSource: function () {\n                return this.imageSource;\n            },\n\n            /**\n             * @param {String} imageSource\n             */\n            setImageSource: function (imageSource) {\n                this.imageSource(imageSource);\n            },\n\n            /**\n             * @return {String}\n             */\n            getRefreshUrl: function () {\n                return this.refreshUrl;\n            },\n\n            /**\n             * @param {String} url\n             */\n            setRefreshUrl: function (url) {\n                this.refreshUrl = url;\n            },\n\n            /**\n             * @return {*}\n             */\n            getCaptchaValue: function () {\n                return this.captchaValue;\n            },\n\n            /**\n             * @param {*} value\n             */\n            setCaptchaValue: function (value) {\n                this.captchaValue(value);\n            },\n\n            /**\n             * Refresh captcha.\n             */\n            refresh: function () {\n                var refresh,\n                    self = this;\n\n                this.isLoading(true);\n\n                refresh = refreshAction(this.getRefreshUrl(), this.getFormId(), this.getImageSource());\n                $.when(refresh).done(function () {\n                    self.isLoading(false);\n                });\n            }\n        };\n    };\n});\n","Magento_Downloadable/js/downloadable.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget',\n    'Magento_Catalog/js/price-box'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.downloadable', {\n        options: {\n            priceHolderSelector: '.price-box'\n        },\n\n        /** @inheritdoc */\n        _create: function () {\n            var self = this;\n\n            this.element.find(this.options.linkElement).on('change', $.proxy(function () {\n                this._reloadPrice();\n            }, this));\n\n            this.element.find(this.options.allElements).on('change', function () {\n                if (this.checked) {\n                    $('label[for=\"' + this.id + '\"] > span').text($(this).attr('data-checked'));\n                    self.element.find(self.options.linkElement + ':not(:checked)').each(function () {\n                        $(this).trigger('click');\n                    });\n                } else {\n                    $('[for=\"' + this.id + '\"] > span').text($(this).attr('data-notchecked'));\n                    self.element.find(self.options.linkElement + ':checked').each(function () {\n                        $(this).trigger('click');\n                    });\n                }\n            });\n        },\n\n        /**\n         * Reload product price with selected link price included\n         * @private\n         */\n        _reloadPrice: function () {\n            var finalPrice = 0,\n                basePrice = 0;\n\n            this.element.find(this.options.linkElement + ':checked').each($.proxy(function (index, element) {\n                finalPrice += this.options.config.links[$(element).val()].finalPrice;\n                basePrice += this.options.config.links[$(element).val()].basePrice;\n            }, this));\n\n            $(this.options.priceHolderSelector).trigger('updatePrice', {\n                'prices': {\n                    'finalPrice': {\n                        'amount': finalPrice\n                    },\n                    'basePrice': {\n                        'amount': basePrice\n                    }\n                }\n            });\n        }\n    });\n\n    return $.mage.downloadable;\n});\n","Magento_OfflinePayments/js/view/payment/offline-payments.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/payment/renderer-list'\n], function (Component, rendererList) {\n    'use strict';\n\n    rendererList.push(\n        {\n            type: 'checkmo',\n            component: 'Magento_OfflinePayments/js/view/payment/method-renderer/checkmo-method'\n        },\n        {\n            type: 'banktransfer',\n            component: 'Magento_OfflinePayments/js/view/payment/method-renderer/banktransfer-method'\n        },\n        {\n            type: 'cashondelivery',\n            component: 'Magento_OfflinePayments/js/view/payment/method-renderer/cashondelivery-method'\n        },\n        {\n            type: 'purchaseorder',\n            component: 'Magento_OfflinePayments/js/view/payment/method-renderer/purchaseorder-method'\n        }\n    );\n\n    /** Add view logic here if needed */\n    return Component.extend({});\n});\n","Magento_OfflinePayments/js/view/payment/method-renderer/checkmo-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'Magento_Checkout/js/view/payment/default'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_OfflinePayments/payment/checkmo'\n        },\n\n        /**\n         * Returns send check to info.\n         *\n         * @return {*}\n         */\n        getMailingAddress: function () {\n            return window.checkoutConfig.payment.checkmo.mailingAddress;\n        },\n\n        /**\n         * Returns payable to info.\n         *\n         * @return {*}\n         */\n        getPayableTo: function () {\n            return window.checkoutConfig.payment.checkmo.payableTo;\n        }\n    });\n});\n","Magento_OfflinePayments/js/view/payment/method-renderer/banktransfer-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'ko',\n    'Magento_Checkout/js/view/payment/default'\n], function (ko, Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_OfflinePayments/payment/banktransfer'\n        },\n\n        /**\n         * Get value of instruction field.\n         * @returns {String}\n         */\n        getInstructions: function () {\n            return window.checkoutConfig.payment.instructions[this.item.method];\n        }\n    });\n});\n","Magento_OfflinePayments/js/view/payment/method-renderer/cashondelivery-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'Magento_Checkout/js/view/payment/default'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_OfflinePayments/payment/cashondelivery'\n        },\n\n        /**\n         * Returns payment method instructions.\n         *\n         * @return {*}\n         */\n        getInstructions: function () {\n            return window.checkoutConfig.payment.instructions[this.item.method];\n        }\n    });\n});\n","Magento_OfflinePayments/js/view/payment/method-renderer/purchaseorder-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'Magento_Checkout/js/view/payment/default',\n    'jquery',\n    'mage/validation'\n], function (Component, $) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_OfflinePayments/payment/purchaseorder-form',\n            purchaseOrderNumber: ''\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super()\n                .observe('purchaseOrderNumber');\n\n            return this;\n        },\n\n        /**\n         * @return {Object}\n         */\n        getData: function () {\n            return {\n                method: this.item.method,\n                'po_number': this.purchaseOrderNumber(),\n                'additional_data': null\n            };\n        },\n\n        /**\n         * @return {jQuery}\n         */\n        validate: function () {\n            var form = 'form[data-role=purchaseorder-form]';\n\n            return $(form).validation() && $(form).validation('isValid');\n        }\n    });\n});\n","Magento_PageCache/js/page-cache.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'domReady',\n    'consoleLogger',\n    'jquery-ui-modules/widget',\n    'mage/cookies'\n], function ($, domReady, consoleLogger) {\n    'use strict';\n\n    /**\n     * Helper. Generate random string\n     * TODO: Merge with mage/utils\n     * @param {String} chars - list of symbols\n     * @param {Number} length - length for need string\n     * @returns {String}\n     */\n    function generateRandomString(chars, length) {\n        var result = '';\n\n        length = length > 0 ? length : 1;\n\n        while (length--) {\n            result += chars[Math.round(Math.random() * (chars.length - 1))];\n        }\n\n        return result;\n    }\n\n    /**\n     * Nodes tree to flat list converter\n     * @returns {Array}\n     */\n    $.fn.comments = function () {\n        var elements = [],\n            contents,\n            elementContents;\n\n        /**\n         * @param {jQuery} element - Comment holder\n         */\n        (function lookup(element) {\n            var iframeHostName;\n\n            // prevent cross origin iframe content reading\n            if ($(element).prop('tagName') === 'IFRAME') {\n                iframeHostName = $('<a>').prop('href', $(element).prop('src'))\n                    .prop('hostname');\n\n                if (window.location.hostname !== iframeHostName) {\n                    return [];\n                }\n            }\n\n            /**\n             * Rewrite jQuery contents().\n             *\n             * @param {jQuery} elem\n             */\n            contents = function (elem) {\n                return $.map(elem, function (el) {\n                    try {\n                        return $.nodeName(el, 'iframe') ?\n                            el.contentDocument || (el.contentWindow ? el.contentWindow.document : []) :\n                            $.merge([], el.childNodes);\n                    } catch (e) {\n                        consoleLogger.error(e);\n\n                        return [];\n                    }\n                });\n            };\n\n            elementContents = contents($(element));\n\n            $.each(elementContents, function (index, el) {\n                switch (el.nodeType) {\n                    case 1: // ELEMENT_NODE\n                        lookup(el);\n                        break;\n\n                    case 8: // COMMENT_NODE\n                        elements.push(el);\n                        break;\n\n                    case 9: // DOCUMENT_NODE\n                        lookup($(el).find('body'));\n                        break;\n                }\n            });\n        })(this);\n\n        return elements;\n    };\n\n    /**\n     * FormKey Widget - this widget is generating from key, saves it to cookie and\n     */\n    $.widget('mage.formKey', {\n        options: {\n            inputSelector: 'input[name=\"form_key\"]',\n            allowedCharacters: '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ',\n            length: 16\n        },\n\n        /**\n         * Creates widget 'mage.formKey'\n         * @private\n         */\n        _create: function () {\n            var formKey = $.mage.cookies.get('form_key');\n\n            if (!formKey) {\n                formKey = generateRandomString(this.options.allowedCharacters, this.options.length);\n                $.mage.cookies.set('form_key', formKey);\n            }\n            $(this.options.inputSelector).val(formKey);\n        }\n    });\n\n    /**\n     * PageCache Widget\n     * Handles additional ajax request for rendering user private content.\n     */\n    $.widget('mage.pageCache', {\n        options: {\n            url: '/',\n            patternPlaceholderOpen: /^ BLOCK (.+) $/,\n            patternPlaceholderClose: /^ \\/BLOCK (.+) $/,\n            versionCookieName: 'private_content_version',\n            handles: []\n        },\n\n        /**\n         * Creates widget 'mage.pageCache'\n         * @private\n         */\n        _create: function () {\n            var placeholders,\n                version = $.mage.cookies.get(this.options.versionCookieName);\n\n            if (!version) {\n                return;\n            }\n            placeholders = this._searchPlaceholders(this.element.comments());\n\n            if (placeholders && placeholders.length) {\n                this._ajax(placeholders, version);\n            }\n        },\n\n        /**\n         * Parse page for placeholders.\n         * @param {Array} elements\n         * @returns {Array}\n         * @private\n         */\n        _searchPlaceholders: function (elements) {\n            var placeholders = [],\n                tmp = {},\n                ii,\n                len,\n                el, matches, name;\n\n            if (!(elements && elements.length)) {\n                return placeholders;\n            }\n\n            for (ii = 0, len = elements.length; ii < len; ii++) {\n                el = elements[ii];\n                matches = this.options.patternPlaceholderOpen.exec(el.nodeValue);\n                name = null;\n\n                if (matches) {\n                    name = matches[1];\n                    tmp[name] = {\n                        name: name,\n                        openElement: el\n                    };\n                } else {\n                    matches = this.options.patternPlaceholderClose.exec(el.nodeValue);\n\n                    if (matches) { //eslint-disable-line max-depth\n                        name = matches[1];\n\n                        if (tmp[name]) { //eslint-disable-line max-depth\n                            tmp[name].closeElement = el;\n                            placeholders.push(tmp[name]);\n                            delete tmp[name];\n                        }\n                    }\n                }\n            }\n\n            return placeholders;\n        },\n\n        /**\n         * Parse for page and replace placeholders\n         * @param {Object} placeholder\n         * @param {Object} html\n         * @protected\n         */\n        _replacePlaceholder: function (placeholder, html) {\n            var startReplacing = false,\n                prevSibling = null,\n                parent, contents, yy, len, element;\n\n            if (!placeholder || !html) {\n                return;\n            }\n\n            parent = $(placeholder.openElement).parent();\n            contents = parent.contents();\n\n            for (yy = 0, len = contents.length; yy < len; yy++) {\n                element = contents[yy];\n\n                if (element == placeholder.openElement) { //eslint-disable-line eqeqeq\n                    startReplacing = true;\n                }\n\n                if (startReplacing) {\n                    $(element).remove();\n                } else if (element.nodeType != 8) { //eslint-disable-line eqeqeq\n                    //due to comment tag doesn't have siblings we try to find it manually\n                    prevSibling = element;\n                }\n\n                if (element == placeholder.closeElement) { //eslint-disable-line eqeqeq\n                    break;\n                }\n            }\n\n            if (prevSibling) {\n                $(prevSibling).after(html);\n            } else {\n                $(parent).prepend(html);\n            }\n\n            // trigger event to use mage-data-init attribute\n            $(parent).trigger('contentUpdated');\n        },\n\n        /**\n         * AJAX helper\n         * @param {Object} placeholders\n         * @param {String} version\n         * @private\n         */\n        _ajax: function (placeholders, version) {\n            var ii,\n                data = {\n                    blocks: [],\n                    handles: this.options.handles,\n                    originalRequest: this.options.originalRequest,\n                    version: version\n                };\n\n            for (ii = 0; ii < placeholders.length; ii++) {\n                data.blocks.push(placeholders[ii].name);\n            }\n            data.blocks = JSON.stringify(data.blocks.sort());\n            data.handles = JSON.stringify(data.handles);\n            data.originalRequest = JSON.stringify(data.originalRequest);\n            $.ajax({\n                url: this.options.url,\n                data: data,\n                type: 'GET',\n                cache: true,\n                dataType: 'json',\n                context: this,\n\n                /**\n                 * Response handler\n                 * @param {Object} response\n                 */\n                success: function (response) {\n                    var placeholder, i;\n\n                    for (i = 0; i < placeholders.length; i++) {\n                        placeholder = placeholders[i];\n\n                        if (response.hasOwnProperty(placeholder.name)) {\n                            this._replacePlaceholder(placeholder, response[placeholder.name]);\n                        }\n                    }\n                }\n            });\n        }\n    });\n\n    domReady(function () {\n        $('body')\n            .formKey();\n    });\n\n    return {\n        'pageCache': $.mage.pageCache,\n        'formKey': $.mage.formKey\n    };\n});\n","fotorama/fotorama.js":"/*!\n * Fotorama 4.6.4 | http://fotorama.io/license/\n */\nfotoramaVersion = '4.6.4';\n(function (window, document, location, $, undefined) {\n    \"use strict\";\n    var _fotoramaClass = 'fotorama',\n        _fullscreenClass = 'fotorama__fullscreen',\n\n        wrapClass = _fotoramaClass + '__wrap',\n        wrapCss2Class = wrapClass + '--css2',\n        wrapCss3Class = wrapClass + '--css3',\n        wrapVideoClass = wrapClass + '--video',\n        wrapFadeClass = wrapClass + '--fade',\n        wrapSlideClass = wrapClass + '--slide',\n        wrapNoControlsClass = wrapClass + '--no-controls',\n        wrapNoShadowsClass = wrapClass + '--no-shadows',\n        wrapPanYClass = wrapClass + '--pan-y',\n        wrapRtlClass = wrapClass + '--rtl',\n        wrapOnlyActiveClass = wrapClass + '--only-active',\n        wrapNoCaptionsClass = wrapClass + '--no-captions',\n        wrapToggleArrowsClass = wrapClass + '--toggle-arrows',\n\n        stageClass = _fotoramaClass + '__stage',\n        stageFrameClass = stageClass + '__frame',\n        stageFrameVideoClass = stageFrameClass + '--video',\n        stageShaftClass = stageClass + '__shaft',\n\n        grabClass = _fotoramaClass + '__grab',\n        pointerClass = _fotoramaClass + '__pointer',\n\n        arrClass = _fotoramaClass + '__arr',\n        arrDisabledClass = arrClass + '--disabled',\n        arrPrevClass = arrClass + '--prev',\n        arrNextClass = arrClass + '--next',\n\n        navClass = _fotoramaClass + '__nav',\n        navWrapClass = navClass + '-wrap',\n        navShaftClass = navClass + '__shaft',\n        navShaftVerticalClass = navWrapClass + '--vertical',\n        navShaftListClass = navWrapClass + '--list',\n        navShafthorizontalClass = navWrapClass + '--horizontal',\n        navDotsClass = navClass + '--dots',\n        navThumbsClass = navClass + '--thumbs',\n        navFrameClass = navClass + '__frame',\n\n        fadeClass = _fotoramaClass + '__fade',\n        fadeFrontClass = fadeClass + '-front',\n        fadeRearClass = fadeClass + '-rear',\n\n        shadowClass = _fotoramaClass + '__shadow',\n        shadowsClass = shadowClass + 's',\n        shadowsLeftClass = shadowsClass + '--left',\n        shadowsRightClass = shadowsClass + '--right',\n        shadowsTopClass = shadowsClass + '--top',\n        shadowsBottomClass = shadowsClass + '--bottom',\n\n        activeClass = _fotoramaClass + '__active',\n        selectClass = _fotoramaClass + '__select',\n\n        hiddenClass = _fotoramaClass + '--hidden',\n\n        fullscreenClass = _fotoramaClass + '--fullscreen',\n        fullscreenIconClass = _fotoramaClass + '__fullscreen-icon',\n\n        errorClass = _fotoramaClass + '__error',\n        loadingClass = _fotoramaClass + '__loading',\n        loadedClass = _fotoramaClass + '__loaded',\n        loadedFullClass = loadedClass + '--full',\n        loadedImgClass = loadedClass + '--img',\n\n        grabbingClass = _fotoramaClass + '__grabbing',\n\n        imgClass = _fotoramaClass + '__img',\n        imgFullClass = imgClass + '--full',\n\n        thumbClass = _fotoramaClass + '__thumb',\n        thumbArrLeft = thumbClass + '__arr--left',\n        thumbArrRight = thumbClass + '__arr--right',\n        thumbBorderClass = thumbClass + '-border',\n\n        htmlClass = _fotoramaClass + '__html',\n\n        videoContainerClass = _fotoramaClass + '-video-container',\n        videoClass = _fotoramaClass + '__video',\n        videoPlayClass = videoClass + '-play',\n        videoCloseClass = videoClass + '-close',\n\n\n        horizontalImageClass = _fotoramaClass + '_horizontal_ratio',\n        verticalImageClass = _fotoramaClass + '_vertical_ratio',\n        fotoramaSpinnerClass = _fotoramaClass + '__spinner',\n        spinnerShowClass = fotoramaSpinnerClass + '--show';\n    var JQUERY_VERSION = $ && $.fn.jquery.split('.');\n\n    if (!JQUERY_VERSION\n        || JQUERY_VERSION[0] < 1\n        || (JQUERY_VERSION[0] == 1 && JQUERY_VERSION[1] < 8)) {\n        throw 'Fotorama requires jQuery 1.8 or later and will not run without it.';\n    }\n\n    var _ = {};\n    /* Modernizr 2.8.3 (Custom Build) | MIT & BSD\n     * Build: http://modernizr.com/download/#-csstransforms3d-csstransitions-touch-prefixed\n     */\n\n    var Modernizr = (function (window, document, undefined) {\n        var version = '2.8.3',\n            Modernizr = {},\n\n\n            docElement = document.documentElement,\n\n            mod = 'modernizr',\n            modElem = document.createElement(mod),\n            mStyle = modElem.style,\n            inputElem,\n\n\n            toString = {}.toString,\n\n            prefixes = ' -webkit- -moz- -o- -ms- '.split(' '),\n\n\n            omPrefixes = 'Webkit Moz O ms',\n\n            cssomPrefixes = omPrefixes.split(' '),\n\n            domPrefixes = omPrefixes.toLowerCase().split(' '),\n\n\n            tests = {},\n            inputs = {},\n            attrs = {},\n\n            classes = [],\n\n            slice = classes.slice,\n\n            featureName,\n\n\n            injectElementWithStyles = function (rule, callback, nodes, testnames) {\n\n                var style, ret, node, docOverflow,\n                    div = document.createElement('div'),\n                    body = document.body,\n                    fakeBody = body || document.createElement('body');\n\n                if (parseInt(nodes, 10)) {\n                    while (nodes--) {\n                        node = document.createElement('div');\n                        node.id = testnames ? testnames[nodes] : mod + (nodes + 1);\n                        div.appendChild(node);\n                    }\n                }\n\n                style = ['&#173;', '<style id=\"s', mod, '\">', rule, '</style>'].join('');\n                div.id = mod;\n                (body ? div : fakeBody).innerHTML += style;\n                fakeBody.appendChild(div);\n                if (!body) {\n                    fakeBody.style.background = '';\n                    fakeBody.style.overflow = 'hidden';\n                    docOverflow = docElement.style.overflow;\n                    docElement.style.overflow = 'hidden';\n                    docElement.appendChild(fakeBody);\n                }\n\n                ret = callback(div, rule);\n                if (!body) {\n                    fakeBody.parentNode.removeChild(fakeBody);\n                    docElement.style.overflow = docOverflow;\n                } else {\n                    div.parentNode.removeChild(div);\n                }\n\n                return !!ret;\n\n            },\n            _hasOwnProperty = ({}).hasOwnProperty, hasOwnProp;\n\n        if (!is(_hasOwnProperty, 'undefined') && !is(_hasOwnProperty.call, 'undefined')) {\n            hasOwnProp = function (object, property) {\n                return _hasOwnProperty.call(object, property);\n            };\n        }\n        else {\n            hasOwnProp = function (object, property) {\n                return ((property in object) && is(object.constructor.prototype[property], 'undefined'));\n            };\n        }\n\n\n        if (!Function.prototype.bind) {\n            Function.prototype.bind = function bind(that) {\n\n                var target = this;\n\n                if (typeof target != \"function\") {\n                    throw new TypeError();\n                }\n\n                var args = slice.call(arguments, 1),\n                    bound = function () {\n\n                        if (this instanceof bound) {\n\n                            var F = function () {\n                            };\n                            F.prototype = target.prototype;\n                            var self = new F();\n\n                            var result = target.apply(\n                                self,\n                                args.concat(slice.call(arguments))\n                            );\n                            if (Object(result) === result) {\n                                return result;\n                            }\n                            return self;\n\n                        } else {\n\n                            return target.apply(\n                                that,\n                                args.concat(slice.call(arguments))\n                            );\n\n                        }\n\n                    };\n\n                return bound;\n            };\n        }\n\n        function setCss(str) {\n            mStyle.cssText = str;\n        }\n\n        function setCssAll(str1, str2) {\n            return setCss(prefixes.join(str1 + ';') + ( str2 || '' ));\n        }\n\n        function is(obj, type) {\n            return typeof obj === type;\n        }\n\n        function contains(str, substr) {\n            return !!~('' + str).indexOf(substr);\n        }\n\n        function testProps(props, prefixed) {\n            for (var i in props) {\n                var prop = props[i];\n                if (!contains(prop, \"-\") && mStyle[prop] !== undefined) {\n                    return prefixed == 'pfx' ? prop : true;\n                }\n            }\n            return false;\n        }\n\n        function testDOMProps(props, obj, elem) {\n            for (var i in props) {\n                var item = obj[props[i]];\n                if (item !== undefined) {\n\n                    if (elem === false) return props[i];\n\n                    if (is(item, 'function')) {\n                        return item.bind(elem || obj);\n                    }\n\n                    return item;\n                }\n            }\n            return false;\n        }\n\n        function testPropsAll(prop, prefixed, elem) {\n\n            var ucProp = prop.charAt(0).toUpperCase() + prop.slice(1),\n                props = (prop + ' ' + cssomPrefixes.join(ucProp + ' ') + ucProp).split(' ');\n\n            if (is(prefixed, \"string\") || is(prefixed, \"undefined\")) {\n                return testProps(props, prefixed);\n\n            } else {\n                props = (prop + ' ' + (domPrefixes).join(ucProp + ' ') + ucProp).split(' ');\n                return testDOMProps(props, prefixed, elem);\n            }\n        }\n\n        tests['touch'] = function () {\n            var bool;\n\n            if (('ontouchstart' in window) || window.DocumentTouch && document instanceof DocumentTouch) {\n                bool = true;\n            } else {\n                injectElementWithStyles(['@media (', prefixes.join('touch-enabled),('), mod, ')', '{#modernizr{top:9px;position:absolute}}'].join(''), function (node) {\n                    bool = node.offsetTop === 9;\n                });\n            }\n\n            return bool;\n        };\n        tests['csstransforms3d'] = function () {\n\n            var ret = !!testPropsAll('perspective');\n\n            if (ret && 'webkitPerspective' in docElement.style) {\n\n                injectElementWithStyles('@media (transform-3d),(-webkit-transform-3d){#modernizr{left:9px;position:absolute;height:3px;}}', function (node, rule) {\n                    ret = node.offsetLeft === 9 && node.offsetHeight === 3;\n                });\n            }\n            return ret;\n        };\n\n\n        tests['csstransitions'] = function () {\n            return testPropsAll('transition');\n        };\n\n\n        for (var feature in tests) {\n            if (hasOwnProp(tests, feature)) {\n                featureName = feature.toLowerCase();\n                Modernizr[featureName] = tests[feature]();\n\n                classes.push((Modernizr[featureName] ? '' : 'no-') + featureName);\n            }\n        }\n\n\n        Modernizr.addTest = function (feature, test) {\n            if (typeof feature == 'object') {\n                for (var key in feature) {\n                    if (hasOwnProp(feature, key)) {\n                        Modernizr.addTest(key, feature[key]);\n                    }\n                }\n            } else {\n\n                feature = feature.toLowerCase();\n\n                if (Modernizr[feature] !== undefined) {\n                    return Modernizr;\n                }\n\n                test = typeof test == 'function' ? test() : test;\n\n                if (typeof enableClasses !== \"undefined\" && enableClasses) {\n                    docElement.className += ' ' + (test ? '' : 'no-') + feature;\n                }\n                Modernizr[feature] = test;\n\n            }\n\n            return Modernizr;\n        };\n\n\n        setCss('');\n        modElem = inputElem = null;\n\n\n        Modernizr._version = version;\n\n        Modernizr._prefixes = prefixes;\n        Modernizr._domPrefixes = domPrefixes;\n        Modernizr._cssomPrefixes = cssomPrefixes;\n\n\n        Modernizr.testProp = function (prop) {\n            return testProps([prop]);\n        };\n\n        Modernizr.testAllProps = testPropsAll;\n        Modernizr.testStyles = injectElementWithStyles;\n        Modernizr.prefixed = function (prop, obj, elem) {\n            if (!obj) {\n                return testPropsAll(prop, 'pfx');\n            } else {\n                return testPropsAll(prop, obj, elem);\n            }\n        };\n        return Modernizr;\n    })(window, document);\n\n    var fullScreenApi = {\n            ok: false,\n            is: function () {\n                return false;\n            },\n            request: function () {\n            },\n            cancel: function () {\n            },\n            event: '',\n            prefix: ''\n        },\n        browserPrefixes = 'webkit moz o ms khtml'.split(' ');\n\n// check for native support\n    if (typeof document.cancelFullScreen != 'undefined') {\n        fullScreenApi.ok = true;\n    } else {\n        // check for fullscreen support by vendor prefix\n        for (var i = 0, il = browserPrefixes.length; i < il; i++) {\n            fullScreenApi.prefix = browserPrefixes[i];\n            if (typeof document[fullScreenApi.prefix + 'CancelFullScreen'] != 'undefined') {\n                fullScreenApi.ok = true;\n                break;\n            }\n        }\n    }\n\n// update methods to do something useful\n    if (fullScreenApi.ok) {\n        fullScreenApi.event = fullScreenApi.prefix + 'fullscreenchange';\n        fullScreenApi.is = function () {\n            switch (this.prefix) {\n                case '':\n                    return document.fullScreen;\n                case 'webkit':\n                    return document.webkitIsFullScreen;\n                default:\n                    return document[this.prefix + 'FullScreen'];\n            }\n        };\n        fullScreenApi.request = function (el) {\n            return (this.prefix === '') ? el.requestFullScreen() : el[this.prefix + 'RequestFullScreen']();\n        };\n        fullScreenApi.cancel = function (el) {\n            return (this.prefix === '') ? document.cancelFullScreen() : document[this.prefix + 'CancelFullScreen']();\n        };\n    }\n    /* Bez v1.0.10-g5ae0136\n     * http://github.com/rdallasgray/bez\n     *\n     * A plugin to convert CSS3 cubic-bezier co-ordinates to jQuery-compatible easing functions\n     *\n     * With thanks to Nikolay Nemshilov for clarification on the cubic-bezier maths\n     * See http://st-on-it.blogspot.com/2011/05/calculating-cubic-bezier-function.html\n     *\n     * Copyright 2011 Robert Dallas Gray. All rights reserved.\n     * Provided under the FreeBSD license: https://github.com/rdallasgray/bez/blob/master/LICENSE.txt\n     */\n    function bez(coOrdArray) {\n        var encodedFuncName = \"bez_\" + $.makeArray(arguments).join(\"_\").replace(\".\", \"p\");\n        if (typeof $['easing'][encodedFuncName] !== \"function\") {\n            var polyBez = function (p1, p2) {\n                var A = [null, null],\n                    B = [null, null],\n                    C = [null, null],\n                    bezCoOrd = function (t, ax) {\n                        C[ax] = 3 * p1[ax];\n                        B[ax] = 3 * (p2[ax] - p1[ax]) - C[ax];\n                        A[ax] = 1 - C[ax] - B[ax];\n                        return t * (C[ax] + t * (B[ax] + t * A[ax]));\n                    },\n                    xDeriv = function (t) {\n                        return C[0] + t * (2 * B[0] + 3 * A[0] * t);\n                    },\n                    xForT = function (t) {\n                        var x = t, i = 0, z;\n                        while (++i < 14) {\n                            z = bezCoOrd(x, 0) - t;\n                            if (Math.abs(z) < 1e-3) break;\n                            x -= z / xDeriv(x);\n                        }\n                        return x;\n                    };\n                return function (t) {\n                    return bezCoOrd(xForT(t), 1);\n                }\n            };\n            $['easing'][encodedFuncName] = function (x, t, b, c, d) {\n                return c * polyBez([coOrdArray[0], coOrdArray[1]], [coOrdArray[2], coOrdArray[3]])(t / d) + b;\n            }\n        }\n        return encodedFuncName;\n    }\n\n    var $WINDOW = $(window),\n        $DOCUMENT = $(document),\n        $HTML,\n        $BODY,\n\n        QUIRKS_FORCE = location.hash.replace('#', '') === 'quirks',\n        TRANSFORMS3D = Modernizr.csstransforms3d,\n        CSS3 = TRANSFORMS3D && !QUIRKS_FORCE,\n        COMPAT = TRANSFORMS3D || document.compatMode === 'CSS1Compat',\n        FULLSCREEN = fullScreenApi.ok,\n\n        MOBILE = navigator.userAgent.match(/Android|webOS|iPhone|iPad|iPod|BlackBerry|Windows Phone/i),\n        SLOW = !CSS3 || MOBILE,\n\n        MS_POINTER = navigator.msPointerEnabled,\n\n        WHEEL = \"onwheel\" in document.createElement(\"div\") ? \"wheel\" : document.onmousewheel !== undefined ? \"mousewheel\" : \"DOMMouseScroll\",\n\n        TOUCH_TIMEOUT = 250,\n        TRANSITION_DURATION = 300,\n\n        SCROLL_LOCK_TIMEOUT = 1400,\n\n        AUTOPLAY_INTERVAL = 5000,\n        MARGIN = 2,\n        THUMB_SIZE = 64,\n\n        WIDTH = 500,\n        HEIGHT = 333,\n\n        STAGE_FRAME_KEY = '$stageFrame',\n        NAV_DOT_FRAME_KEY = '$navDotFrame',\n        NAV_THUMB_FRAME_KEY = '$navThumbFrame',\n\n        AUTO = 'auto',\n\n        BEZIER = bez([.1, 0, .25, 1]),\n\n        MAX_WIDTH = 1200,\n\n        /**\n         * Number of thumbnails in slide. Calculated only on setOptions and resize.\n         * @type {number}\n         */\n        thumbsPerSlide = 1,\n\n        OPTIONS = {\n\n            /**\n             * Set width for gallery.\n             * Default value - width of first image\n             * Number - set value in px\n             * String - set value in quotes\n             *\n             */\n            width: null,\n\n            /**\n             * Set min-width for gallery\n             *\n             */\n            minwidth: null,\n\n            /**\n             * Set max-width for gallery\n             *\n             */\n            maxwidth: '100%',\n\n            /**\n             * Set height for gallery\n             * Default value - height of first image\n             * Number - set value in px\n             * String - set value in quotes\n             *\n             */\n            height: null,\n\n            /**\n             * Set min-height for gallery\n             *\n             */\n            minheight: null,\n\n            /**\n             * Set max-height for gallery\n             *\n             */\n            maxheight: null,\n\n            /**\n             * Set proportion ratio for gallery depends of image\n             *\n             */\n            ratio: null, // '16/9' || 500/333 || 1.5\n\n            margin: MARGIN,\n\n            nav: 'dots', // 'thumbs' || false\n            navposition: 'bottom', // 'top'\n            navwidth: null,\n            thumbwidth: THUMB_SIZE,\n            thumbheight: THUMB_SIZE,\n            thumbmargin: MARGIN,\n            thumbborderwidth: MARGIN,\n\n            allowfullscreen: false, // true || 'native'\n\n            transition: 'slide', // 'crossfade' || 'dissolve'\n            clicktransition: null,\n            transitionduration: TRANSITION_DURATION,\n\n            captions: true,\n\n            startindex: 0,\n\n            loop: false,\n\n            autoplay: false,\n            stopautoplayontouch: true,\n\n            keyboard: false,\n\n            arrows: true,\n            click: true,\n            swipe: false,\n            trackpad: false,\n\n            shuffle: false,\n\n            direction: 'ltr', // 'rtl'\n\n            shadows: true,\n\n            showcaption: true,\n\n            /**\n             * Set type of thumbnail navigation\n             */\n            navdir: 'horizontal',\n\n            /**\n             * Set configuration to show or hide arrows in thumb navigation\n             */\n            navarrows: true,\n\n            /**\n             * Set type of navigation. Can be thumbs or slides\n             */\n            navtype: 'thumbs'\n\n        },\n\n        KEYBOARD_OPTIONS = {\n            left: true,\n            right: true,\n            down: true,\n            up: true,\n            space: false,\n            home: false,\n            end: false\n        };\n\n    function noop() {\n    }\n\n    function minMaxLimit(value, min, max) {\n        return Math.max(isNaN(min) ? -Infinity : min, Math.min(isNaN(max) ? Infinity : max, value));\n    }\n\n    function readTransform(css, dir) {\n        return css.match(/ma/) && css.match(/-?\\d+(?!d)/g)[css.match(/3d/) ?\n                (dir === 'vertical' ? 13 : 12) : (dir === 'vertical' ? 5 : 4)\n                ]\n    }\n\n    function readPosition($el, dir) {\n        if (CSS3) {\n            return +readTransform($el.css('transform'), dir);\n        } else {\n            return +$el.css(dir === 'vertical' ? 'top' : 'left').replace('px', '');\n        }\n    }\n\n    function getTranslate(pos, direction) {\n        var obj = {};\n\n        if (CSS3) {\n\n            switch (direction) {\n                case 'vertical':\n                    obj.transform = 'translate3d(0, ' + (pos) + 'px,0)';\n                    break;\n                case 'list':\n                    break;\n                default :\n                    obj.transform = 'translate3d(' + (pos) + 'px,0,0)';\n                    break;\n            }\n        } else {\n            direction === 'vertical' ?\n                obj.top = pos :\n                obj.left = pos;\n        }\n        return obj;\n    }\n\n    function getDuration(time) {\n        return {'transition-duration': time + 'ms'};\n    }\n\n    function unlessNaN(value, alternative) {\n        return isNaN(value) ? alternative : value;\n    }\n\n    function numberFromMeasure(value, measure) {\n        return unlessNaN(+String(value).replace(measure || 'px', ''));\n    }\n\n    function numberFromPercent(value) {\n        return /%$/.test(value) ? numberFromMeasure(value, '%') : undefined;\n    }\n\n    function numberFromWhatever(value, whole) {\n        return unlessNaN(numberFromPercent(value) / 100 * whole, numberFromMeasure(value));\n    }\n\n    function measureIsValid(value) {\n        return (!isNaN(numberFromMeasure(value)) || !isNaN(numberFromMeasure(value, '%'))) && value;\n    }\n\n    function getPosByIndex(index, side, margin, baseIndex) {\n\n        return (index - (baseIndex || 0)) * (side + (margin || 0));\n    }\n\n    function getIndexByPos(pos, side, margin, baseIndex) {\n        return -Math.round(pos / (side + (margin || 0)) - (baseIndex || 0));\n    }\n\n    function bindTransitionEnd($el) {\n        var elData = $el.data();\n\n        if (elData.tEnd) return;\n\n        var el = $el[0],\n            transitionEndEvent = {\n                WebkitTransition: 'webkitTransitionEnd',\n                MozTransition: 'transitionend',\n                OTransition: 'oTransitionEnd otransitionend',\n                msTransition: 'MSTransitionEnd',\n                transition: 'transitionend'\n            };\n        addEvent(el, transitionEndEvent[Modernizr.prefixed('transition')], function (e) {\n            elData.tProp && e.propertyName.match(elData.tProp) && elData.onEndFn();\n        });\n        elData.tEnd = true;\n    }\n\n    function afterTransition($el, property, fn, time) {\n        var ok,\n            elData = $el.data();\n\n        if (elData) {\n            elData.onEndFn = function () {\n                if (ok) return;\n                ok = true;\n                clearTimeout(elData.tT);\n                fn();\n            };\n            elData.tProp = property;\n\n            // Passive call, just in case of fail of native transition-end event\n            clearTimeout(elData.tT);\n            elData.tT = setTimeout(function () {\n                elData.onEndFn();\n            }, time * 1.5);\n\n            bindTransitionEnd($el);\n        }\n    }\n\n\n    function stop($el, pos/*, _001*/) {\n        var dir = $el.navdir || 'horizontal';\n        if ($el.length) {\n            var elData = $el.data();\n            if (CSS3) {\n                $el.css(getDuration(0));\n                elData.onEndFn = noop;\n                clearTimeout(elData.tT);\n            } else {\n                $el.stop();\n            }\n            var lockedPos = getNumber(pos, function () {\n                return readPosition($el, dir);\n            });\n\n            $el.css(getTranslate(lockedPos, dir/*, _001*/));//.width(); // `.width()` for reflow\n            return lockedPos;\n        }\n    }\n\n    function getNumber() {\n        var number;\n        for (var _i = 0, _l = arguments.length; _i < _l; _i++) {\n            number = _i ? arguments[_i]() : arguments[_i];\n            if (typeof number === 'number') {\n                break;\n            }\n        }\n\n        return number;\n    }\n\n    function edgeResistance(pos, edge) {\n        return Math.round(pos + ((edge - pos) / 1.5));\n    }\n\n    function getProtocol() {\n        getProtocol.p = getProtocol.p || (location.protocol === 'https:' ? 'https://' : 'http://');\n        return getProtocol.p;\n    }\n\n    function parseHref(href) {\n        var a = document.createElement('a');\n        a.href = href;\n        return a;\n    }\n\n    function findVideoId(href, forceVideo) {\n        if (typeof href !== 'string') return href;\n        href = parseHref(href);\n\n        var id,\n            type;\n\n        if (href.host.match(/youtube\\.com/) && href.search) {\n            //.log();\n            id = href.search.split('v=')[1];\n            if (id) {\n                var ampersandPosition = id.indexOf('&');\n                if (ampersandPosition !== -1) {\n                    id = id.substring(0, ampersandPosition);\n                }\n                type = 'youtube';\n            }\n        } else if (href.host.match(/youtube\\.com|youtu\\.be|youtube-nocookie.com/)) {\n            id = href.pathname.replace(/^\\/(embed\\/|v\\/)?/, '').replace(/\\/.*/, '');\n            type = 'youtube';\n        } else if (href.host.match(/vimeo\\.com/)) {\n            type = 'vimeo';\n            id = href.pathname.replace(/^\\/(video\\/)?/, '').replace(/\\/.*/, '');\n        }\n\n        if ((!id || !type) && forceVideo) {\n            id = href.href;\n            type = 'custom';\n        }\n\n        return id ? {id: id, type: type, s: href.search.replace(/^\\?/, ''), p: getProtocol()} : false;\n    }\n\n    function getVideoThumbs(dataFrame, data, fotorama) {\n        var img, thumb, video = dataFrame.video;\n        if (video.type === 'youtube') {\n            thumb = getProtocol() + 'img.youtube.com/vi/' + video.id + '/default.jpg';\n            img = thumb.replace(/\\/default.jpg$/, '/hqdefault.jpg');\n            dataFrame.thumbsReady = true;\n        } else if (video.type === 'vimeo') {\n            $.ajax({\n                url: getProtocol() + 'vimeo.com/api/v2/video/' + video.id + '.json',\n                dataType: 'jsonp',\n                success: function (json) {\n                    dataFrame.thumbsReady = true;\n                    updateData(data, {\n                        img: json[0].thumbnail_large,\n                        thumb: json[0].thumbnail_small\n                    }, dataFrame.i, fotorama);\n                }\n            });\n        } else {\n            dataFrame.thumbsReady = true;\n        }\n\n        return {\n            img: img,\n            thumb: thumb\n        }\n    }\n\n    function updateData(data, _dataFrame, i, fotorama) {\n        for (var _i = 0, _l = data.length; _i < _l; _i++) {\n            var dataFrame = data[_i];\n\n            if (dataFrame.i === i && dataFrame.thumbsReady) {\n                var clear = {videoReady: true};\n                clear[STAGE_FRAME_KEY] = clear[NAV_THUMB_FRAME_KEY] = clear[NAV_DOT_FRAME_KEY] = false;\n\n                fotorama.splice(_i, 1, $.extend(\n                    {},\n                    dataFrame,\n                    clear,\n                    _dataFrame\n                ));\n\n                break;\n            }\n        }\n    }\n\n    function getDataFromHtml($el) {\n        var data = [];\n\n        function getDataFromImg($img, imgData, checkVideo) {\n            var $child = $img.children('img').eq(0),\n                _imgHref = $img.attr('href'),\n                _imgSrc = $img.attr('src'),\n                _thumbSrc = $child.attr('src'),\n                _video = imgData.video,\n                video = checkVideo ? findVideoId(_imgHref, _video === true) : false;\n\n            if (video) {\n                _imgHref = false;\n            } else {\n                video = _video;\n            }\n\n            getDimensions($img, $child, $.extend(imgData, {\n                video: video,\n                img: imgData.img || _imgHref || _imgSrc || _thumbSrc,\n                thumb: imgData.thumb || _thumbSrc || _imgSrc || _imgHref\n            }));\n        }\n\n        function getDimensions($img, $child, imgData) {\n            var separateThumbFLAG = imgData.thumb && imgData.img !== imgData.thumb,\n                width = numberFromMeasure(imgData.width || $img.attr('width')),\n                height = numberFromMeasure(imgData.height || $img.attr('height'));\n\n            $.extend(imgData, {\n                width: width,\n                height: height,\n                thumbratio: getRatio(imgData.thumbratio || (numberFromMeasure(imgData.thumbwidth || ($child && $child.attr('width')) || separateThumbFLAG || width) / numberFromMeasure(imgData.thumbheight || ($child && $child.attr('height')) || separateThumbFLAG || height)))\n            });\n        }\n\n        $el.children().each(function () {\n            var $this = $(this),\n                dataFrame = optionsToLowerCase($.extend($this.data(), {id: $this.attr('id')}));\n            if ($this.is('a, img')) {\n                getDataFromImg($this, dataFrame, true);\n            } else if (!$this.is(':empty')) {\n                getDimensions($this, null, $.extend(dataFrame, {\n                    html: this,\n                    _html: $this.html() // Because of IE\n                }));\n            } else return;\n\n            data.push(dataFrame);\n        });\n\n        return data;\n    }\n\n    function isHidden(el) {\n        return el.offsetWidth === 0 && el.offsetHeight === 0;\n    }\n\n    function isDetached(el) {\n        return !$.contains(document.documentElement, el);\n    }\n\n    function waitFor(test, fn, timeout, i) {\n        if (!waitFor.i) {\n            waitFor.i = 1;\n            waitFor.ii = [true];\n        }\n\n        i = i || waitFor.i;\n\n        if (typeof waitFor.ii[i] === 'undefined') {\n            waitFor.ii[i] = true;\n        }\n\n        if (test()) {\n            fn();\n        } else {\n            waitFor.ii[i] && setTimeout(function () {\n                waitFor.ii[i] && waitFor(test, fn, timeout, i);\n            }, timeout || 100);\n        }\n\n        return waitFor.i++;\n    }\n\n    waitFor.stop = function (i) {\n        waitFor.ii[i] = false;\n    };\n\n    function fit($el, measuresToFit) {\n        var elData = $el.data(),\n            measures = elData.measures;\n\n        if (measures && (!elData.l ||\n            elData.l.W !== measures.width ||\n            elData.l.H !== measures.height ||\n            elData.l.r !== measures.ratio ||\n            elData.l.w !== measuresToFit.w ||\n            elData.l.h !== measuresToFit.h)) {\n\n            var height = minMaxLimit(measuresToFit.h, 0, measures.height),\n                width = height * measures.ratio;\n\n            UTIL.setRatio($el, width, height);\n\n            elData.l = {\n                W: measures.width,\n                H: measures.height,\n                r: measures.ratio,\n                w: measuresToFit.w,\n                h: measuresToFit.h\n            };\n        }\n\n        return true;\n    }\n\n    function setStyle($el, style) {\n        var el = $el[0];\n        if (el.styleSheet) {\n            el.styleSheet.cssText = style;\n        } else {\n            $el.html(style);\n        }\n    }\n\n    function findShadowEdge(pos, min, max, dir) {\n        return min === max ? false :\n            dir === 'vertical' ?\n                (pos <= min ? 'top' : pos >= max ? 'bottom' : 'top bottom') :\n                (pos <= min ? 'left' : pos >= max ? 'right' : 'left right');\n    }\n\n    function smartClick($el, fn, _options) {\n        _options = _options || {};\n\n        $el.each(function () {\n            var $this = $(this),\n                thisData = $this.data(),\n                startEvent;\n\n            if (thisData.clickOn) return;\n\n            thisData.clickOn = true;\n\n            $.extend(touch($this, {\n                onStart: function (e) {\n                    startEvent = e;\n                    (_options.onStart || noop).call(this, e);\n                },\n                onMove: _options.onMove || noop,\n                onTouchEnd: _options.onTouchEnd || noop,\n                onEnd: function (result) {\n                    if (result.moved) return;\n                    fn.call(this, startEvent);\n                }\n            }), {noMove: true});\n        });\n    }\n\n    function div(classes, child) {\n        return '<div class=\"' + classes + '\">' + (child || '') + '</div>';\n    }\n\n\n    /**\n     * Function transforming into valid classname\n     * @param className - name of the class\n     * @returns {string} - dom format of class name\n     */\n    function cls(className) {\n        return \".\" + className;\n    }\n\n    /**\n     *\n     * @param {json-object} videoItem Parsed object from data.video item or href from link a in input dates\n     * @returns {string} DOM view of video iframe\n     */\n    function createVideoFrame(videoItem) {\n        var frame = '<iframe src=\"' + videoItem.p + videoItem.type + '.com/embed/' + videoItem.id + '\" frameborder=\"0\" allowfullscreen></iframe>';\n        return frame;\n    }\n\n// Fisher\u2013Yates Shuffle\n// http://bost.ocks.org/mike/shuffle/\n    function shuffle(array) {\n        // While there remain elements to shuffle\n        var l = array.length;\n        while (l) {\n            // Pick a remaining element\n            var i = Math.floor(Math.random() * l--);\n\n            // And swap it with the current element\n            var t = array[l];\n            array[l] = array[i];\n            array[i] = t;\n        }\n\n        return array;\n    }\n\n    function clone(array) {\n        return Object.prototype.toString.call(array) == '[object Array]'\n            && $.map(array, function (frame) {\n                return $.extend({}, frame);\n            });\n    }\n\n    function lockScroll($el, left, top) {\n        $el\n            .scrollLeft(left || 0)\n            .scrollTop(top || 0);\n    }\n\n    function optionsToLowerCase(options) {\n        if (options) {\n            var opts = {};\n            $.each(options, function (key, value) {\n                opts[key.toLowerCase()] = value;\n            });\n\n            return opts;\n        }\n    }\n\n    function getRatio(_ratio) {\n        if (!_ratio) return;\n        var ratio = +_ratio;\n        if (!isNaN(ratio)) {\n            return ratio;\n        } else {\n            ratio = _ratio.split('/');\n            return +ratio[0] / +ratio[1] || undefined;\n        }\n    }\n\n    function addEvent(el, e, fn, bool) {\n        if (!e) return;\n        el.addEventListener ? el.addEventListener(e, fn, !!bool) : el.attachEvent('on' + e, fn);\n    }\n\n    /**\n     *\n     * @param position guess position for navShaft\n     * @param restriction object contains min and max values for position\n     * @returns {*} filtered value of position\n     */\n    function validateRestrictions(position, restriction) {\n        if (position > restriction.max) {\n            position = restriction.max;\n        } else {\n            if (position < restriction.min) {\n                position = restriction.min;\n            }\n        }\n        return position;\n    }\n\n    function validateSlidePos(opt, navShaftTouchTail, guessIndex, offsetNav, $guessNavFrame, $navWrap, dir) {\n        var position,\n            size,\n            wrapSize;\n        if (dir === 'horizontal') {\n            size = opt.thumbwidth;\n            wrapSize = $navWrap.width();\n        } else {\n            size = opt.thumbheight;\n            wrapSize = $navWrap.height();\n        }\n        if ( (size + opt.margin) * (guessIndex + 1) >= (wrapSize - offsetNav) ) {\n            if (dir === 'horizontal') {\n                position = -$guessNavFrame.position().left;\n            } else {\n                position = -$guessNavFrame.position().top;\n            }\n        } else {\n            if ((size + opt.margin) * (guessIndex) <= Math.abs(offsetNav)) {\n                if (dir === 'horizontal') {\n                    position = -$guessNavFrame.position().left + wrapSize - (size + opt.margin);\n                } else {\n                    position = -$guessNavFrame.position().top + wrapSize - (size + opt.margin);\n                }\n            } else {\n                position = offsetNav;\n            }\n        }\n        position = validateRestrictions(position, navShaftTouchTail);\n\n        return position || 0;\n    }\n\n    function elIsDisabled(el) {\n        return !!el.getAttribute('disabled');\n    }\n\n    function disableAttr(FLAG, disable) {\n        if (disable) {\n            return {disabled: FLAG};\n        } else {\n            return {tabindex: FLAG * -1 + '', disabled: FLAG};\n\n        }\n    }\n\n    function addEnterUp(el, fn) {\n        addEvent(el, 'keyup', function (e) {\n            elIsDisabled(el) || e.keyCode == 13 && fn.call(el, e);\n        });\n    }\n\n    function addFocus(el, fn) {\n        addEvent(el, 'focus', el.onfocusin = function (e) {\n            fn.call(el, e);\n        }, true);\n    }\n\n    function stopEvent(e, stopPropagation) {\n        e.preventDefault ? e.preventDefault() : (e.returnValue = false);\n        stopPropagation && e.stopPropagation && e.stopPropagation();\n    }\n\n    function stubEvent($el, eventType) {\n        var isIOS = /ip(ad|hone|od)/i.test(window.navigator.userAgent);\n\n        if (isIOS && eventType === 'touchend') {\n            $el.on('touchend', function(e){\n                $DOCUMENT.trigger('mouseup', e);\n            })\n        }\n\n        $el.on(eventType, function (e) {\n            stopEvent(e, true);\n\n            return false;\n        });\n    }\n\n    function getDirectionSign(forward) {\n        return forward ? '>' : '<';\n    }\n\n    var UTIL = (function () {\n\n        function setRatioClass($el, wh, ht) {\n            var rateImg = wh / ht;\n\n            if (rateImg <= 1) {\n                $el.parent().removeClass(horizontalImageClass);\n                $el.parent().addClass(verticalImageClass);\n            } else {\n                $el.parent().removeClass(verticalImageClass);\n                $el.parent().addClass(horizontalImageClass);\n            }\n        }\n\n        /**\n         * Set specific attribute in thumbnail template\n         * @param $frame DOM item of specific thumbnail\n         * @param value Value which must be setted into specific attribute\n         * @param searchAttr Name of attribute where value must be included\n         */\n        function setThumbAttr($frame, value, searchAttr) {\n            var attr = searchAttr;\n\n            if (!$frame.attr(attr) && $frame.attr(attr) !== undefined) {\n                $frame.attr(attr, value);\n            }\n\n            if ($frame.find(\"[\" + attr + \"]\").length) {\n                $frame.find(\"[\" + attr + \"]\")\n                    .each(function () {\n                        $(this).attr(attr, value);\n                    });\n            }\n        }\n\n        /**\n         * Method describe behavior need to render caption on preview or not\n         * @param frameItem specific item from data\n         * @param isExpected {bool} if items with caption need render them or not\n         * @returns {boolean} if true then caption should be rendered\n         */\n        function isExpectedCaption(frameItem, isExpected, undefined) {\n            var expected = false,\n                frameExpected;\n\n            frameItem.showCaption === undefined || frameItem.showCaption === true ? frameExpected = true : frameExpected = false;\n\n            if (!isExpected) {\n                return false;\n            }\n\n            if (frameItem.caption && frameExpected) {\n                expected = true;\n            }\n\n            return expected;\n        }\n\n        return {\n            setRatio: setRatioClass,\n            setThumbAttr: setThumbAttr,\n            isExpectedCaption: isExpectedCaption\n        };\n\n    }(UTIL || {}, jQuery));\n\n    function slide($el, options) {\n        var elData = $el.data(),\n            elPos = Math.round(options.pos),\n            onEndFn = function () {\n                if (elData && elData.sliding) {\n                    elData.sliding = false;\n                }\n                (options.onEnd || noop)();\n            };\n\n        if (typeof options.overPos !== 'undefined' && options.overPos !== options.pos) {\n            elPos = options.overPos;\n        }\n\n        var translate = $.extend(getTranslate(elPos, options.direction), options.width && {width: options.width}, options.height && {height: options.height});\n        if (elData && elData.sliding) {\n            elData.sliding = true;\n        }\n\n        if (CSS3) {\n            $el.css($.extend(getDuration(options.time), translate));\n\n            if (options.time > 10) {\n                afterTransition($el, 'transform', onEndFn, options.time);\n            } else {\n                onEndFn();\n            }\n        } else {\n            $el.stop().animate(translate, options.time, BEZIER, onEndFn);\n        }\n    }\n\n    function fade($el1, $el2, $frames, options, fadeStack, chain) {\n        var chainedFLAG = typeof chain !== 'undefined';\n        if (!chainedFLAG) {\n            fadeStack.push(arguments);\n            Array.prototype.push.call(arguments, fadeStack.length);\n            if (fadeStack.length > 1) return;\n        }\n\n        $el1 = $el1 || $($el1);\n        $el2 = $el2 || $($el2);\n\n        var _$el1 = $el1[0],\n            _$el2 = $el2[0],\n            crossfadeFLAG = options.method === 'crossfade',\n            onEndFn = function () {\n                if (!onEndFn.done) {\n                    onEndFn.done = true;\n                    var args = (chainedFLAG || fadeStack.shift()) && fadeStack.shift();\n                    args && fade.apply(this, args);\n                    (options.onEnd || noop)(!!args);\n                }\n            },\n            time = options.time / (chain || 1);\n\n        $frames.removeClass(fadeRearClass + ' ' + fadeFrontClass);\n\n        $el1\n            .stop()\n            .addClass(fadeRearClass);\n        $el2\n            .stop()\n            .addClass(fadeFrontClass);\n\n        crossfadeFLAG && _$el2 && $el1.fadeTo(0, 0);\n\n        $el1.fadeTo(crossfadeFLAG ? time : 0, 1, crossfadeFLAG && onEndFn);\n        $el2.fadeTo(time, 0, onEndFn);\n\n        (_$el1 && crossfadeFLAG) || _$el2 || onEndFn();\n    }\n\n    var lastEvent,\n        moveEventType,\n        preventEvent,\n        preventEventTimeout,\n        dragDomEl;\n\n    function extendEvent(e) {\n        var touch = (e.touches || [])[0] || e;\n        e._x = touch.pageX || touch.originalEvent.pageX;\n        e._y = touch.clientY || touch.originalEvent.clientY;\n        e._now = $.now();\n    }\n\n    function touch($el, options) {\n        var el = $el[0],\n            tail = {},\n            touchEnabledFLAG,\n            startEvent,\n            $target,\n            controlTouch,\n            touchFLAG,\n            targetIsSelectFLAG,\n            targetIsLinkFlag,\n            isDisabledSwipe,\n            tolerance,\n            moved;\n\n        function onStart(e) {\n            $target = $(e.target);\n            tail.checked = targetIsSelectFLAG = targetIsLinkFlag = isDisabledSwipe = moved = false;\n\n            if (touchEnabledFLAG\n                || tail.flow\n                || (e.touches && e.touches.length > 1)\n                || e.which > 1\n                || (lastEvent && lastEvent.type !== e.type && preventEvent)\n                || (targetIsSelectFLAG = options.select && $target.is(options.select, el))) return targetIsSelectFLAG;\n\n            touchFLAG = e.type === 'touchstart';\n            targetIsLinkFlag = $target.is('a, a *', el);\n            isDisabledSwipe = $target.hasClass('disableSwipe');\n            controlTouch = tail.control;\n\n            tolerance = (tail.noMove || tail.noSwipe || controlTouch) ? 16 : !tail.snap ? 4 : 0;\n\n            extendEvent(e);\n\n            startEvent = lastEvent = e;\n            moveEventType = e.type.replace(/down|start/, 'move').replace(/Down/, 'Move');\n\n            (options.onStart || noop).call(el, e, {control: controlTouch, $target: $target});\n\n            touchEnabledFLAG = tail.flow = true;\n\n            if (!isDisabledSwipe && (!touchFLAG || tail.go)) stopEvent(e);\n        }\n\n        function onMove(e) {\n            if ((e.touches && e.touches.length > 1)\n                || (MS_POINTER && !e.isPrimary)\n                || moveEventType !== e.type\n                || !touchEnabledFLAG) {\n                touchEnabledFLAG && onEnd();\n                (options.onTouchEnd || noop)();\n                return;\n            }\n\n            isDisabledSwipe = $(e.target).hasClass('disableSwipe');\n\n            if (isDisabledSwipe) {\n                return;\n            }\n\n            extendEvent(e);\n\n            var xDiff = Math.abs(e._x - startEvent._x), // opt _x \u2192 _pageX\n                yDiff = Math.abs(e._y - startEvent._y),\n                xyDiff = xDiff - yDiff,\n                xWin = (tail.go || tail.x || xyDiff >= 0) && !tail.noSwipe,\n                yWin = xyDiff < 0;\n\n            if (touchFLAG && !tail.checked) {\n                if (touchEnabledFLAG = xWin) {\n                    stopEvent(e);\n                }\n            } else {\n                stopEvent(e);\n                if (movedEnough(xDiff,yDiff)) {\n                    (options.onMove || noop).call(el, e, {touch: touchFLAG});\n                }\n            }\n\n            if (!moved && movedEnough(xDiff, yDiff) && Math.sqrt(Math.pow(xDiff, 2) + Math.pow(yDiff, 2)) > tolerance) {\n                moved = true;\n            }\n\n            tail.checked = tail.checked || xWin || yWin;\n        }\n\n        function movedEnough(xDiff, yDiff) {\n            return xDiff > yDiff && xDiff > 1.5;\n        }\n\n        function onEnd(e) {\n            (options.onTouchEnd || noop)();\n\n            var _touchEnabledFLAG = touchEnabledFLAG;\n            tail.control = touchEnabledFLAG = false;\n\n            if (_touchEnabledFLAG) {\n                tail.flow = false;\n            }\n\n            if (!_touchEnabledFLAG || (targetIsLinkFlag && !tail.checked)) return;\n\n            e && stopEvent(e);\n\n            preventEvent = true;\n            clearTimeout(preventEventTimeout);\n            preventEventTimeout = setTimeout(function () {\n                preventEvent = false;\n            }, 1000);\n\n            (options.onEnd || noop).call(el, {\n                moved: moved,\n                $target: $target,\n                control: controlTouch,\n                touch: touchFLAG,\n                startEvent: startEvent,\n                aborted: !e || e.type === 'MSPointerCancel'\n            });\n        }\n\n        function onOtherStart() {\n            if (tail.flow) return;\n            tail.flow = true;\n        }\n\n        function onOtherEnd() {\n            if (!tail.flow) return;\n            tail.flow = false;\n        }\n\n        if (MS_POINTER) {\n            addEvent(el, 'MSPointerDown', onStart);\n            addEvent(document, 'MSPointerMove', onMove);\n            addEvent(document, 'MSPointerCancel', onEnd);\n            addEvent(document, 'MSPointerUp', onEnd);\n        } else {\n            addEvent(el, 'touchstart', onStart);\n            addEvent(el, 'touchmove', onMove);\n            addEvent(el, 'touchend', onEnd);\n\n            addEvent(document, 'touchstart', onOtherStart);\n            addEvent(document, 'touchend', onOtherEnd);\n            addEvent(document, 'touchcancel', onOtherEnd);\n\n            $WINDOW.on('scroll', onOtherEnd);\n\n            $el.on('mousedown pointerdown', onStart);\n            $DOCUMENT\n                .on('mousemove pointermove', onMove)\n                .on('mouseup pointerup', onEnd);\n        }\n        if (Modernizr.touch) {\n            dragDomEl = 'a';\n        } else {\n            dragDomEl = 'div';\n        }\n        $el.on('click', dragDomEl, function (e) {\n            tail.checked && stopEvent(e);\n        });\n\n        return tail;\n    }\n\n    function moveOnTouch($el, options) {\n        var el = $el[0],\n            elData = $el.data(),\n            tail = {},\n            startCoo,\n            coo,\n            startElPos,\n            moveElPos,\n            edge,\n            moveTrack,\n            startTime,\n            endTime,\n            min,\n            max,\n            snap,\n            dir,\n            slowFLAG,\n            controlFLAG,\n            moved,\n            tracked;\n\n        function startTracking(e, noStop) {\n            tracked = true;\n            startCoo = coo = (dir === 'vertical') ? e._y : e._x;\n            startTime = e._now;\n\n            moveTrack = [\n                [startTime, startCoo]\n            ];\n\n            startElPos = moveElPos = tail.noMove || noStop ? 0 : stop($el, (options.getPos || noop)()/*, options._001*/);\n\n            (options.onStart || noop).call(el, e);\n        }\n\n        function onStart(e, result) {\n            min = tail.min;\n            max = tail.max;\n            snap = tail.snap,\n                dir = tail.direction || 'horizontal',\n                $el.navdir = dir;\n\n            slowFLAG = e.altKey;\n            tracked = moved = false;\n\n            controlFLAG = result.control;\n\n            if (!controlFLAG && !elData.sliding) {\n                startTracking(e);\n            }\n        }\n\n        function onMove(e, result) {\n            if (!tail.noSwipe) {\n                if (!tracked) {\n                    startTracking(e);\n                }\n                coo = (dir === 'vertical') ? e._y : e._x;\n\n                moveTrack.push([e._now, coo]);\n\n                moveElPos = startElPos - (startCoo - coo);\n\n                edge = findShadowEdge(moveElPos, min, max, dir);\n\n                if (moveElPos <= min) {\n                    moveElPos = edgeResistance(moveElPos, min);\n                } else if (moveElPos >= max) {\n                    moveElPos = edgeResistance(moveElPos, max);\n                }\n\n                if (!tail.noMove) {\n                    $el.css(getTranslate(moveElPos, dir));\n                    if (!moved) {\n                        moved = true;\n                        // only for mouse\n                        result.touch || MS_POINTER || $el.addClass(grabbingClass);\n                    }\n\n                    (options.onMove || noop).call(el, e, {pos: moveElPos, edge: edge});\n                }\n            }\n        }\n\n        function onEnd(result) {\n            if (tail.noSwipe && result.moved) return;\n\n            if (!tracked) {\n                startTracking(result.startEvent, true);\n            }\n\n            result.touch || MS_POINTER || $el.removeClass(grabbingClass);\n\n            endTime = $.now();\n\n            var _backTimeIdeal = endTime - TOUCH_TIMEOUT,\n                _backTime,\n                _timeDiff,\n                _timeDiffLast,\n                backTime = null,\n                backCoo,\n                virtualPos,\n                limitPos,\n                newPos,\n                overPos,\n                time = TRANSITION_DURATION,\n                speed,\n                friction = options.friction;\n\n            for (var _i = moveTrack.length - 1; _i >= 0; _i--) {\n                _backTime = moveTrack[_i][0];\n                _timeDiff = Math.abs(_backTime - _backTimeIdeal);\n                if (backTime === null || _timeDiff < _timeDiffLast) {\n                    backTime = _backTime;\n                    backCoo = moveTrack[_i][1];\n                } else if (backTime === _backTimeIdeal || _timeDiff > _timeDiffLast) {\n                    break;\n                }\n                _timeDiffLast = _timeDiff;\n            }\n\n            newPos = minMaxLimit(moveElPos, min, max);\n\n            var cooDiff = backCoo - coo,\n                forwardFLAG = cooDiff >= 0,\n                timeDiff = endTime - backTime,\n                longTouchFLAG = timeDiff > TOUCH_TIMEOUT,\n                swipeFLAG = !longTouchFLAG && moveElPos !== startElPos && newPos === moveElPos;\n\n            if (snap) {\n                newPos = minMaxLimit(Math[swipeFLAG ? (forwardFLAG ? 'floor' : 'ceil') : 'round'](moveElPos / snap) * snap, min, max);\n                min = max = newPos;\n            }\n\n            if (swipeFLAG && (snap || newPos === moveElPos)) {\n                speed = -(cooDiff / timeDiff);\n                time *= minMaxLimit(Math.abs(speed), options.timeLow, options.timeHigh);\n                virtualPos = Math.round(moveElPos + speed * time / friction);\n\n                if (!snap) {\n                    newPos = virtualPos;\n                }\n\n                if (!forwardFLAG && virtualPos > max || forwardFLAG && virtualPos < min) {\n                    limitPos = forwardFLAG ? min : max;\n                    overPos = virtualPos - limitPos;\n                    if (!snap) {\n                        newPos = limitPos;\n                    }\n                    overPos = minMaxLimit(newPos + overPos * .03, limitPos - 50, limitPos + 50);\n                    time = Math.abs((moveElPos - overPos) / (speed / friction));\n                }\n            }\n\n            time *= slowFLAG ? 10 : 1;\n\n            (options.onEnd || noop).call(el, $.extend(result, {\n                moved: result.moved || longTouchFLAG && snap,\n                pos: moveElPos,\n                newPos: newPos,\n                overPos: overPos,\n                time: time,\n                dir: dir\n            }));\n        }\n\n        tail = $.extend(touch(options.$wrap, $.extend({}, options, {\n            onStart: onStart,\n            onMove: onMove,\n            onEnd: onEnd\n        })), tail);\n\n        return tail;\n    }\n\n    function wheel($el, options) {\n        var el = $el[0],\n            lockFLAG,\n            lastDirection,\n            lastNow,\n            tail = {\n                prevent: {}\n            };\n\n        addEvent(el, WHEEL, function (e) {\n            var yDelta = e.wheelDeltaY || -1 * e.deltaY || 0,\n                xDelta = e.wheelDeltaX || -1 * e.deltaX || 0,\n                xWin = Math.abs(xDelta) && !Math.abs(yDelta),\n                direction = getDirectionSign(xDelta < 0),\n                sameDirection = lastDirection === direction,\n                now = $.now(),\n                tooFast = now - lastNow < TOUCH_TIMEOUT;\n\n            lastDirection = direction;\n            lastNow = now;\n\n            if (!xWin || !tail.ok || tail.prevent[direction] && !lockFLAG) {\n                return;\n            } else {\n                stopEvent(e, true);\n                if (lockFLAG && sameDirection && tooFast) {\n                    return;\n                }\n            }\n\n            if (options.shift) {\n                lockFLAG = true;\n                clearTimeout(tail.t);\n                tail.t = setTimeout(function () {\n                    lockFLAG = false;\n                }, SCROLL_LOCK_TIMEOUT);\n            }\n\n            (options.onEnd || noop)(e, options.shift ? direction : xDelta);\n\n        });\n\n        return tail;\n    }\n\n    jQuery.Fotorama = function ($fotorama, opts) {\n        $HTML = $('html');\n        $BODY = $('body');\n\n        var that = this,\n            stamp = $.now(),\n            stampClass = _fotoramaClass + stamp,\n            fotorama = $fotorama[0],\n            data,\n            dataFrameCount = 1,\n            fotoramaData = $fotorama.data(),\n            size,\n\n            $style = $('<style></style>'),\n\n            $anchor = $(div(hiddenClass)),\n            $wrap = $fotorama.find(cls(wrapClass)),\n            $stage = $wrap.find(cls(stageClass)),\n            stage = $stage[0],\n\n            $stageShaft = $fotorama.find(cls(stageShaftClass)),\n            $stageFrame = $(),\n            $arrPrev = $fotorama.find(cls(arrPrevClass)),\n            $arrNext = $fotorama.find(cls(arrNextClass)),\n            $arrs = $fotorama.find(cls(arrClass)),\n            $navWrap = $fotorama.find(cls(navWrapClass)),\n            $nav = $navWrap.find(cls(navClass)),\n            $navShaft = $nav.find(cls(navShaftClass)),\n            $navFrame,\n            $navDotFrame = $(),\n            $navThumbFrame = $(),\n\n            stageShaftData = $stageShaft.data(),\n            navShaftData = $navShaft.data(),\n\n            $thumbBorder = $fotorama.find(cls(thumbBorderClass)),\n            $thumbArrLeft = $fotorama.find(cls(thumbArrLeft)),\n            $thumbArrRight = $fotorama.find(cls(thumbArrRight)),\n\n            $fullscreenIcon = $fotorama.find(cls(fullscreenIconClass)),\n            fullscreenIcon = $fullscreenIcon[0],\n            $videoPlay = $(div(videoPlayClass)),\n            $videoClose = $fotorama.find(cls(videoCloseClass)),\n            videoClose = $videoClose[0],\n\n            $spinner = $fotorama.find(cls(fotoramaSpinnerClass)),\n\n            $videoPlaying,\n\n            activeIndex = false,\n            activeFrame,\n            activeIndexes,\n            repositionIndex,\n            dirtyIndex,\n            lastActiveIndex,\n            prevIndex,\n            nextIndex,\n            nextAutoplayIndex,\n            startIndex,\n\n            o_loop,\n            o_nav,\n            o_navThumbs,\n            o_navTop,\n            o_allowFullScreen,\n            o_nativeFullScreen,\n            o_fade,\n            o_thumbSide,\n            o_thumbSide2,\n            o_transitionDuration,\n            o_transition,\n            o_shadows,\n            o_rtl,\n            o_keyboard,\n            lastOptions = {},\n\n            measures = {},\n            measuresSetFLAG,\n\n            stageShaftTouchTail = {},\n            stageWheelTail = {},\n            navShaftTouchTail = {},\n            navWheelTail = {},\n\n            scrollTop,\n            scrollLeft,\n\n            showedFLAG,\n            pausedAutoplayFLAG,\n            stoppedAutoplayFLAG,\n\n            toDeactivate = {},\n            toDetach = {},\n\n            measuresStash,\n\n            touchedFLAG,\n\n            hoverFLAG,\n\n            navFrameKey,\n            stageLeft = 0,\n\n            fadeStack = [];\n\n        $wrap[STAGE_FRAME_KEY] = $('<div class=\"' + stageFrameClass + '\"></div>');\n        $wrap[NAV_THUMB_FRAME_KEY] = $($.Fotorama.jst.thumb());\n        $wrap[NAV_DOT_FRAME_KEY] = $($.Fotorama.jst.dots());\n\n        toDeactivate[STAGE_FRAME_KEY] = [];\n        toDeactivate[NAV_THUMB_FRAME_KEY] = [];\n        toDeactivate[NAV_DOT_FRAME_KEY] = [];\n        toDetach[STAGE_FRAME_KEY] = {};\n\n        $wrap.addClass(CSS3 ? wrapCss3Class : wrapCss2Class);\n\n        fotoramaData.fotorama = this;\n\n        /**\n         * Search video items in incoming data and transform object for video layout.\n         *\n         */\n        function checkForVideo() {\n            $.each(data, function (i, dataFrame) {\n                if (!dataFrame.i) {\n                    dataFrame.i = dataFrameCount++;\n                    var video = findVideoId(dataFrame.video, true);\n                    if (video) {\n                        var thumbs = {};\n                        dataFrame.video = video;\n                        if (!dataFrame.img && !dataFrame.thumb) {\n                            thumbs = getVideoThumbs(dataFrame, data, that);\n                        } else {\n                            dataFrame.thumbsReady = true;\n                        }\n                        updateData(data, {img: thumbs.img, thumb: thumbs.thumb}, dataFrame.i, that);\n                    }\n                }\n            });\n        }\n\n        function allowKey(key) {\n            return o_keyboard[key];\n        }\n\n        function setStagePosition() {\n            if ($stage !== undefined) {\n\n                if (opts.navdir == 'vertical') {\n                    var padding = opts.thumbwidth + opts.thumbmargin;\n\n                    $stage.css('left', padding);\n                    $arrNext.css('right', padding);\n                    $fullscreenIcon.css('right', padding);\n                    $wrap.css('width', $wrap.css('width') + padding);\n                    $stageShaft.css('max-width', $wrap.width() - padding);\n                } else {\n                    $stage.css('left', '');\n                    $arrNext.css('right', '');\n                    $fullscreenIcon.css('right', '');\n                    $wrap.css('width', $wrap.css('width') + padding);\n                    $stageShaft.css('max-width', '');\n                }\n            }\n        }\n\n        function bindGlobalEvents(FLAG) {\n            var keydownCommon = 'keydown.' + _fotoramaClass,\n                localStamp = _fotoramaClass + stamp,\n                keydownLocal = 'keydown.' + localStamp,\n                keyupLocal = 'keyup.' + localStamp,\n                resizeLocal = 'resize.' + localStamp + ' ' + 'orientationchange.' + localStamp,\n                showParams;\n\n            if (FLAG) {\n                $DOCUMENT\n                    .on(keydownLocal, function (e) {\n                        var catched,\n                            index;\n\n                        if ($videoPlaying && e.keyCode === 27) {\n                            catched = true;\n                            unloadVideo($videoPlaying, true, true);\n                        } else if (that.fullScreen || (opts.keyboard && !that.index)) {\n                            if (e.keyCode === 27) {\n                                catched = true;\n                                that.cancelFullScreen();\n                            } else if ((e.shiftKey && e.keyCode === 32 && allowKey('space')) || (!e.altKey && !e.metaKey && e.keyCode === 37 && allowKey('left')) || (e.keyCode === 38 && allowKey('up') && $(':focus').attr('data-gallery-role'))) {\n                                that.longPress.progress();\n                                index = '<';\n                            } else if ((e.keyCode === 32 && allowKey('space')) || (!e.altKey && !e.metaKey && e.keyCode === 39 && allowKey('right')) || (e.keyCode === 40 && allowKey('down') && $(':focus').attr('data-gallery-role'))) {\n                                that.longPress.progress();\n                                index = '>';\n                            } else if (e.keyCode === 36 && allowKey('home')) {\n                                that.longPress.progress();\n                                index = '<<';\n                            } else if (e.keyCode === 35 && allowKey('end')) {\n                                that.longPress.progress();\n                                index = '>>';\n                            }\n                        }\n\n                        (catched || index) && stopEvent(e);\n                        showParams = {index: index, slow: e.altKey, user: true};\n                        index && (that.longPress.inProgress ?\n                            that.showWhileLongPress(showParams) :\n                            that.show(showParams));\n                    });\n\n                if (FLAG) {\n                    $DOCUMENT\n                        .on(keyupLocal, function (e) {\n                            if (that.longPress.inProgress) {\n                                that.showEndLongPress({user: true});\n                            }\n                            that.longPress.reset();\n                        });\n                }\n\n                if (!that.index) {\n                    $DOCUMENT\n                        .off(keydownCommon)\n                        .on(keydownCommon, 'textarea, input, select', function (e) {\n                            !$BODY.hasClass(_fullscreenClass) && e.stopPropagation();\n                        });\n                }\n\n                $WINDOW.on(resizeLocal, that.resize);\n            } else {\n                $DOCUMENT.off(keydownLocal);\n                $WINDOW.off(resizeLocal);\n            }\n        }\n\n        function appendElements(FLAG) {\n            if (FLAG === appendElements.f) return;\n\n            if (FLAG) {\n                $fotorama\n                    .addClass(_fotoramaClass + ' ' + stampClass)\n                    .before($anchor)\n                    .before($style);\n                addInstance(that);\n            } else {\n                $anchor.detach();\n                $style.detach();\n                $fotorama\n                    .html(fotoramaData.urtext)\n                    .removeClass(stampClass);\n\n                hideInstance(that);\n            }\n\n            bindGlobalEvents(FLAG);\n            appendElements.f = FLAG;\n        }\n\n        /**\n         * Set and install data from incoming @param {JSON} options or takes data attr from data-\"name\"=... values.\n         */\n        function setData() {\n            data = that.data = data || clone(opts.data) || getDataFromHtml($fotorama);\n            size = that.size = data.length;\n\n            ready.ok && opts.shuffle && shuffle(data);\n\n            checkForVideo();\n\n            activeIndex = limitIndex(activeIndex);\n\n            size && appendElements(true);\n        }\n\n        function stageNoMove() {\n            var _noMove = size < 2 || $videoPlaying;\n            stageShaftTouchTail.noMove = _noMove || o_fade;\n            stageShaftTouchTail.noSwipe = _noMove || !opts.swipe;\n\n            !o_transition && $stageShaft.toggleClass(grabClass, !opts.click && !stageShaftTouchTail.noMove && !stageShaftTouchTail.noSwipe);\n            MS_POINTER && $wrap.toggleClass(wrapPanYClass, !stageShaftTouchTail.noSwipe);\n        }\n\n        function setAutoplayInterval(interval) {\n            if (interval === true) interval = '';\n            opts.autoplay = Math.max(+interval || AUTOPLAY_INTERVAL, o_transitionDuration * 1.5);\n        }\n\n        function updateThumbArrow(opt) {\n            if (opt.navarrows && opt.nav === 'thumbs') {\n                $thumbArrLeft.show();\n                $thumbArrRight.show();\n            } else {\n                $thumbArrLeft.hide();\n                $thumbArrRight.hide();\n            }\n\n        }\n\n        function getThumbsInSlide($el, opts) {\n            return Math.floor($wrap.width() / (opts.thumbwidth + opts.thumbmargin));\n        }\n\n        /**\n         * Options on the fly\n         * */\n        function setOptions() {\n            if (!opts.nav || opts.nav === 'dots') {\n                opts.navdir = 'horizontal'\n            }\n\n            that.options = opts = optionsToLowerCase(opts);\n            thumbsPerSlide = getThumbsInSlide($wrap, opts);\n\n            o_fade = (opts.transition === 'crossfade' || opts.transition === 'dissolve');\n\n            o_loop = opts.loop && (size > 2 || (o_fade && (!o_transition || o_transition !== 'slide')));\n\n            o_transitionDuration = +opts.transitionduration || TRANSITION_DURATION;\n\n            o_rtl = opts.direction === 'rtl';\n\n            o_keyboard = $.extend({}, opts.keyboard && KEYBOARD_OPTIONS, opts.keyboard);\n            updateThumbArrow(opts);\n            var classes = {add: [], remove: []};\n\n            function addOrRemoveClass(FLAG, value) {\n                classes[FLAG ? 'add' : 'remove'].push(value);\n            }\n\n            if (size > 1) {\n                o_nav = opts.nav;\n                o_navTop = opts.navposition === 'top';\n                classes.remove.push(selectClass);\n\n                $arrs.toggle(!!opts.arrows);\n            } else {\n                o_nav = false;\n                $arrs.hide();\n            }\n\n            arrsUpdate();\n            stageWheelUpdate();\n            thumbArrUpdate();\n            if (opts.autoplay) setAutoplayInterval(opts.autoplay);\n\n            o_thumbSide = numberFromMeasure(opts.thumbwidth) || THUMB_SIZE;\n            o_thumbSide2 = numberFromMeasure(opts.thumbheight) || THUMB_SIZE;\n\n            stageWheelTail.ok = navWheelTail.ok = opts.trackpad && !SLOW;\n\n            stageNoMove();\n\n            extendMeasures(opts, [measures]);\n\n            o_navThumbs = o_nav === 'thumbs';\n\n            if ($navWrap.filter(':hidden') && !!o_nav) {\n                $navWrap.show();\n            }\n            if (o_navThumbs) {\n                frameDraw(size, 'navThumb');\n\n                $navFrame = $navThumbFrame;\n                navFrameKey = NAV_THUMB_FRAME_KEY;\n\n                setStyle($style, $.Fotorama.jst.style({\n                    w: o_thumbSide,\n                    h: o_thumbSide2,\n                    b: opts.thumbborderwidth,\n                    m: opts.thumbmargin,\n                    s: stamp,\n                    q: !COMPAT\n                }));\n\n                $nav\n                    .addClass(navThumbsClass)\n                    .removeClass(navDotsClass);\n            } else if (o_nav === 'dots') {\n                frameDraw(size, 'navDot');\n\n                $navFrame = $navDotFrame;\n                navFrameKey = NAV_DOT_FRAME_KEY;\n\n                $nav\n                    .addClass(navDotsClass)\n                    .removeClass(navThumbsClass);\n            } else {\n                $navWrap.hide();\n                o_nav = false;\n                $nav.removeClass(navThumbsClass + ' ' + navDotsClass);\n            }\n\n            if (o_nav) {\n                if (o_navTop) {\n                    $navWrap.insertBefore($stage);\n                } else {\n                    $navWrap.insertAfter($stage);\n                }\n                frameAppend.nav = false;\n\n                frameAppend($navFrame, $navShaft, 'nav');\n            }\n\n            o_allowFullScreen = opts.allowfullscreen;\n\n            if (o_allowFullScreen) {\n                $fullscreenIcon.prependTo($stage);\n                o_nativeFullScreen = FULLSCREEN && o_allowFullScreen === 'native';\n\n                // Due 300ms click delay on mobile devices\n                // we stub touchend and fallback to click.\n                // MAGETWO-69567\n                stubEvent($fullscreenIcon, 'touchend');\n            } else {\n                $fullscreenIcon.detach();\n                o_nativeFullScreen = false;\n            }\n\n            addOrRemoveClass(o_fade, wrapFadeClass);\n            addOrRemoveClass(!o_fade, wrapSlideClass);\n            addOrRemoveClass(!opts.captions, wrapNoCaptionsClass);\n            addOrRemoveClass(o_rtl, wrapRtlClass);\n            addOrRemoveClass(opts.arrows, wrapToggleArrowsClass);\n\n            o_shadows = opts.shadows && !SLOW;\n            addOrRemoveClass(!o_shadows, wrapNoShadowsClass);\n\n            $wrap\n                .addClass(classes.add.join(' '))\n                .removeClass(classes.remove.join(' '));\n\n            lastOptions = $.extend({}, opts);\n            setStagePosition();\n        }\n\n        function normalizeIndex(index) {\n            return index < 0 ? (size + (index % size)) % size : index >= size ? index % size : index;\n        }\n\n        function limitIndex(index) {\n            return minMaxLimit(index, 0, size - 1);\n        }\n\n        function edgeIndex(index) {\n            return o_loop ? normalizeIndex(index) : limitIndex(index);\n        }\n\n        function getPrevIndex(index) {\n            return index > 0 || o_loop ? index - 1 : false;\n        }\n\n        function getNextIndex(index) {\n            return index < size - 1 || o_loop ? index + 1 : false;\n        }\n\n        function setStageShaftMinmaxAndSnap() {\n            stageShaftTouchTail.min = o_loop ? -Infinity : -getPosByIndex(size - 1, measures.w, opts.margin, repositionIndex);\n            stageShaftTouchTail.max = o_loop ? Infinity : -getPosByIndex(0, measures.w, opts.margin, repositionIndex);\n            stageShaftTouchTail.snap = measures.w + opts.margin;\n        }\n\n        function setNavShaftMinMax() {\n\n            var isVerticalDir = (opts.navdir === 'vertical');\n            var param = isVerticalDir ? $navShaft.height() : $navShaft.width();\n            var mainParam = isVerticalDir ? measures.h : measures.nw;\n            navShaftTouchTail.min = Math.min(0, mainParam - param);\n            navShaftTouchTail.max = 0;\n            navShaftTouchTail.direction = opts.navdir;\n            $navShaft.toggleClass(grabClass, !(navShaftTouchTail.noMove = navShaftTouchTail.min === navShaftTouchTail.max));\n        }\n\n        function eachIndex(indexes, type, fn) {\n            if (typeof indexes === 'number') {\n                indexes = new Array(indexes);\n                var rangeFLAG = true;\n            }\n            return $.each(indexes, function (i, index) {\n                if (rangeFLAG) index = i;\n                if (typeof index === 'number') {\n                    var dataFrame = data[normalizeIndex(index)];\n\n                    if (dataFrame) {\n                        var key = '$' + type + 'Frame',\n                            $frame = dataFrame[key];\n\n                        fn.call(this, i, index, dataFrame, $frame, key, $frame && $frame.data());\n                    }\n                }\n            });\n        }\n\n        function setMeasures(width, height, ratio, index) {\n            if (!measuresSetFLAG || (measuresSetFLAG === '*' && index === startIndex)) {\n\n                width = measureIsValid(opts.width) || measureIsValid(width) || WIDTH;\n                height = measureIsValid(opts.height) || measureIsValid(height) || HEIGHT;\n                that.resize({\n                    width: width,\n                    ratio: opts.ratio || ratio || width / height\n                }, 0, index !== startIndex && '*');\n            }\n        }\n\n        function loadImg(indexes, type, specialMeasures, again) {\n\n            eachIndex(indexes, type, function (i, index, dataFrame, $frame, key, frameData) {\n\n                if (!$frame) return;\n\n                var fullFLAG = that.fullScreen && !frameData.$full && type === 'stage';\n\n                if (frameData.$img && !again && !fullFLAG) return;\n\n                var img = new Image(),\n                    $img = $(img),\n                    imgData = $img.data();\n\n                frameData[fullFLAG ? '$full' : '$img'] = $img;\n\n                var srcKey = type === 'stage' ? (fullFLAG ? 'full' : 'img') : 'thumb',\n                    src = dataFrame[srcKey],\n                    dummy = fullFLAG ? dataFrame['img'] : dataFrame[type === 'stage' ? 'thumb' : 'img'];\n\n                if (type === 'navThumb') $frame = frameData.$wrap;\n\n                function triggerTriggerEvent(event) {\n                    var _index = normalizeIndex(index);\n                    triggerEvent(event, {\n                        index: _index,\n                        src: src,\n                        frame: data[_index]\n                    });\n                }\n\n                function error() {\n                    $img.remove();\n\n                    $.Fotorama.cache[src] = 'error';\n\n                    if ((!dataFrame.html || type !== 'stage') && dummy && dummy !== src) {\n                        dataFrame[srcKey] = src = dummy;\n                        frameData.$full = null;\n                        loadImg([index], type, specialMeasures, true);\n                    } else {\n                        if (src && !dataFrame.html && !fullFLAG) {\n                            $frame\n                                .trigger('f:error')\n                                .removeClass(loadingClass)\n                                .addClass(errorClass);\n\n                            triggerTriggerEvent('error');\n                        } else if (type === 'stage') {\n                            $frame\n                                .trigger('f:load')\n                                .removeClass(loadingClass + ' ' + errorClass)\n                                .addClass(loadedClass);\n\n                            triggerTriggerEvent('load');\n                            setMeasures();\n                        }\n\n                        frameData.state = 'error';\n\n                        if (size > 1 && data[index] === dataFrame && !dataFrame.html && !dataFrame.deleted && !dataFrame.video && !fullFLAG) {\n                            dataFrame.deleted = true;\n                            that.splice(index, 1);\n                        }\n                    }\n                }\n\n                function loaded() {\n                    $.Fotorama.measures[src] = imgData.measures = $.Fotorama.measures[src] || {\n                            width: img.width,\n                            height: img.height,\n                            ratio: img.width / img.height\n                        };\n\n                    setMeasures(imgData.measures.width, imgData.measures.height, imgData.measures.ratio, index);\n\n                    $img\n                        .off('load error')\n                        .addClass('' + (fullFLAG ? imgFullClass: imgClass))\n                        .attr('aria-hidden', 'false')\n                        .prependTo($frame);\n\n                    if ($frame.hasClass(stageFrameClass) && !$frame.hasClass(videoContainerClass)) {\n                        $frame.attr(\"href\", $img.attr(\"src\"));\n                    }\n\n                    fit($img, (\n                            $.isFunction(specialMeasures) ? specialMeasures() : specialMeasures) || measures);\n\n                    $.Fotorama.cache[src] = frameData.state = 'loaded';\n\n                    setTimeout(function () {\n                        $frame\n                            .trigger('f:load')\n                            .removeClass(loadingClass + ' ' + errorClass)\n                            .addClass(loadedClass + ' ' + (fullFLAG ? loadedFullClass : loadedImgClass));\n\n                        if (type === 'stage') {\n                            triggerTriggerEvent('load');\n                        } else if (dataFrame.thumbratio === AUTO || !dataFrame.thumbratio && opts.thumbratio === AUTO) {\n                            // danger! reflow for all thumbnails\n                            dataFrame.thumbratio = imgData.measures.ratio;\n                            reset();\n                        }\n                    }, 0);\n                }\n\n                if (!src) {\n                    error();\n                    return;\n                }\n\n                function waitAndLoad() {\n                    var _i = 10;\n                    waitFor(function () {\n                        return !touchedFLAG || !_i-- && !SLOW;\n                    }, function () {\n                        loaded();\n                    });\n                }\n\n                if (!$.Fotorama.cache[src]) {\n                    $.Fotorama.cache[src] = '*';\n\n                    $img\n                        .on('load', waitAndLoad)\n                        .on('error', error);\n                } else {\n                    (function justWait() {\n                        if ($.Fotorama.cache[src] === 'error') {\n                            error();\n                        } else if ($.Fotorama.cache[src] === 'loaded') {\n                            setTimeout(waitAndLoad, 0);\n                        } else {\n                            setTimeout(justWait, 100);\n                        }\n                    })();\n                }\n\n                frameData.state = '';\n                img.src = src;\n\n                if (frameData.data.caption) {\n                    img.alt = frameData.data.caption || \"\";\n                }\n\n                if (frameData.data.full) {\n                    $(img).data('original', frameData.data.full);\n                }\n\n                if (UTIL.isExpectedCaption(dataFrame, opts.showcaption)) {\n                    $(img).attr('aria-labelledby', dataFrame.labelledby);\n                }\n            });\n        }\n\n        function updateFotoramaState() {\n            var $frame = activeFrame[STAGE_FRAME_KEY];\n\n            if ($frame && !$frame.data().state) {\n                $spinner.addClass(spinnerShowClass);\n                $frame.on('f:load f:error', function () {\n                    $frame.off('f:load f:error');\n                    $spinner.removeClass(spinnerShowClass);\n                });\n            }\n        }\n\n        function addNavFrameEvents(frame) {\n            addEnterUp(frame, onNavFrameClick);\n            addFocus(frame, function () {\n\n                setTimeout(function () {\n                    lockScroll($nav);\n                }, 0);\n                slideNavShaft({time: o_transitionDuration, guessIndex: $(this).data().eq, minMax: navShaftTouchTail});\n            });\n        }\n\n        function frameDraw(indexes, type) {\n            eachIndex(indexes, type, function (i, index, dataFrame, $frame, key, frameData) {\n                if ($frame) return;\n\n                $frame = dataFrame[key] = $wrap[key].clone();\n                frameData = $frame.data();\n                frameData.data = dataFrame;\n                var frame = $frame[0],\n                    labelledbyValue = \"labelledby\" + $.now();\n\n                if (type === 'stage') {\n\n                    if (dataFrame.html) {\n                        $('<div class=\"' + htmlClass + '\"></div>')\n                            .append(\n                            dataFrame._html ? $(dataFrame.html)\n                                .removeAttr('id')\n                                .html(dataFrame._html) // Because of IE\n                                : dataFrame.html\n                        )\n                            .appendTo($frame);\n                    }\n\n                    if (dataFrame.id) {\n                        labelledbyValue = dataFrame.id || labelledbyValue;\n                    }\n                    dataFrame.labelledby = labelledbyValue;\n\n                    if (UTIL.isExpectedCaption(dataFrame, opts.showcaption)) {\n                        $($.Fotorama.jst.frameCaption({\n                            caption: dataFrame.caption,\n                            labelledby: labelledbyValue\n                        })).appendTo($frame);\n                    }\n\n                    dataFrame.video && $frame\n                        .addClass(stageFrameVideoClass)\n                        .append($videoPlay.clone());\n\n                    // This solves tabbing problems\n                    addFocus(frame, function (e) {\n                        setTimeout(function () {\n                            lockScroll($stage);\n                        }, 0);\n                        clickToShow({index: frameData.eq, user: true}, e);\n                    });\n\n                    $stageFrame = $stageFrame.add($frame);\n                } else if (type === 'navDot') {\n                    addNavFrameEvents(frame);\n                    $navDotFrame = $navDotFrame.add($frame);\n                } else if (type === 'navThumb') {\n                    addNavFrameEvents(frame);\n                    frameData.$wrap = $frame.children(':first');\n\n                    $navThumbFrame = $navThumbFrame.add($frame);\n                    if (dataFrame.video) {\n                        frameData.$wrap.append($videoPlay.clone());\n                    }\n                }\n            });\n        }\n\n        function callFit($img, measuresToFit) {\n            return $img && $img.length && fit($img, measuresToFit);\n        }\n\n        function stageFramePosition(indexes) {\n            eachIndex(indexes, 'stage', function (i, index, dataFrame, $frame, key, frameData) {\n                if (!$frame) return;\n\n                var normalizedIndex = normalizeIndex(index);\n                frameData.eq = normalizedIndex;\n\n                toDetach[STAGE_FRAME_KEY][normalizedIndex] = $frame.css($.extend({left: o_fade ? 0 : getPosByIndex(index, measures.w, opts.margin, repositionIndex)}, o_fade && getDuration(0)));\n\n                if (isDetached($frame[0])) {\n                    $frame.appendTo($stageShaft);\n                    unloadVideo(dataFrame.$video);\n                }\n\n                callFit(frameData.$img, measures);\n                callFit(frameData.$full, measures);\n\n                if ($frame.hasClass(stageFrameClass) && !($frame.attr('aria-hidden') === \"false\" && $frame.hasClass(activeClass))) {\n                    $frame.attr('aria-hidden', 'true');\n                }\n            });\n        }\n\n        function thumbsDraw(pos, loadFLAG) {\n            var leftLimit,\n                rightLimit,\n                exceedLimit;\n\n\n            if (o_nav !== 'thumbs' || isNaN(pos)) return;\n\n            leftLimit = -pos;\n            rightLimit = -pos + measures.nw;\n\n            if (opts.navdir === 'vertical') {\n                pos = pos - opts.thumbheight;\n                rightLimit = -pos + measures.h;\n            }\n\n            $navThumbFrame.each(function () {\n                var $this = $(this),\n                    thisData = $this.data(),\n                    eq = thisData.eq,\n                    getSpecialMeasures = function () {\n                        return {\n                            h: o_thumbSide2,\n                            w: thisData.w\n                        }\n                    },\n                    specialMeasures = getSpecialMeasures(),\n                    exceedLimit = opts.navdir === 'vertical' ?\n                    thisData.t > rightLimit : thisData.l > rightLimit;\n                specialMeasures.w = thisData.w;\n\n                if ((opts.navdir !== 'vertical' && thisData.l + thisData.w < leftLimit)\n                    || exceedLimit\n                    || callFit(thisData.$img, specialMeasures)) return;\n\n                loadFLAG && loadImg([eq], 'navThumb', getSpecialMeasures);\n            });\n        }\n\n        function frameAppend($frames, $shaft, type) {\n            if (!frameAppend[type]) {\n\n                var thumbsFLAG = type === 'nav' && o_navThumbs,\n                    left = 0,\n                    top = 0;\n\n                $shaft.append(\n                    $frames\n                        .filter(function () {\n                            var actual,\n                                $this = $(this),\n                                frameData = $this.data();\n                            for (var _i = 0, _l = data.length; _i < _l; _i++) {\n                                if (frameData.data === data[_i]) {\n                                    actual = true;\n                                    frameData.eq = _i;\n                                    break;\n                                }\n                            }\n                            return actual || $this.remove() && false;\n                        })\n                        .sort(function (a, b) {\n                            return $(a).data().eq - $(b).data().eq;\n                        })\n                        .each(function () {\n                            var $this = $(this),\n                                frameData = $this.data();\n                            UTIL.setThumbAttr($this, frameData.data.caption, \"aria-label\");\n                        })\n                        .each(function () {\n\n                            if (!thumbsFLAG) return;\n\n                            var $this = $(this),\n                                frameData = $this.data(),\n                                thumbwidth = Math.round(o_thumbSide2 * frameData.data.thumbratio) || o_thumbSide,\n                                thumbheight = Math.round(o_thumbSide / frameData.data.thumbratio) || o_thumbSide2;\n                            frameData.t = top;\n                            frameData.h = thumbheight;\n                            frameData.l = left;\n                            frameData.w = thumbwidth;\n\n                            $this.css({width: thumbwidth});\n\n                            top += thumbheight + opts.thumbmargin;\n                            left += thumbwidth + opts.thumbmargin;\n                        })\n                );\n\n                frameAppend[type] = true;\n            }\n        }\n\n        function getDirection(x) {\n            return x - stageLeft > measures.w / 3;\n        }\n\n        function disableDirrection(i) {\n            return !o_loop && (!(activeIndex + i) || !(activeIndex - size + i)) && !$videoPlaying;\n        }\n\n        function arrsUpdate() {\n            var disablePrev = disableDirrection(0),\n                disableNext = disableDirrection(1);\n            $arrPrev\n                .toggleClass(arrDisabledClass, disablePrev)\n                .attr(disableAttr(disablePrev, false));\n            $arrNext\n                .toggleClass(arrDisabledClass, disableNext)\n                .attr(disableAttr(disableNext, false));\n        }\n\n        function thumbArrUpdate() {\n            var isLeftDisable = false,\n                isRightDisable = false;\n            if (opts.navtype === 'thumbs' && !opts.loop) {\n                (activeIndex == 0) ? isLeftDisable = true : isLeftDisable = false;\n                (activeIndex == opts.data.length - 1) ? isRightDisable = true : isRightDisable = false;\n            }\n            if (opts.navtype === 'slides') {\n                var pos = readPosition($navShaft, opts.navdir);\n                pos >= navShaftTouchTail.max ? isLeftDisable = true : isLeftDisable = false;\n                pos <= navShaftTouchTail.min ? isRightDisable = true : isRightDisable = false;\n            }\n            $thumbArrLeft\n                .toggleClass(arrDisabledClass, isLeftDisable)\n                .attr(disableAttr(isLeftDisable, true));\n            $thumbArrRight\n                .toggleClass(arrDisabledClass, isRightDisable)\n                .attr(disableAttr(isRightDisable, true));\n        }\n\n        function stageWheelUpdate() {\n            if (stageWheelTail.ok) {\n                stageWheelTail.prevent = {'<': disableDirrection(0), '>': disableDirrection(1)};\n            }\n        }\n\n        function getNavFrameBounds($navFrame) {\n            var navFrameData = $navFrame.data(),\n                left,\n                top,\n                width,\n                height;\n\n            if (o_navThumbs) {\n                left = navFrameData.l;\n                top = navFrameData.t;\n                width = navFrameData.w;\n                height = navFrameData.h;\n            } else {\n                left = $navFrame.position().left;\n                width = $navFrame.width();\n            }\n\n            var horizontalBounds = {\n                c: left + width / 2,\n                min: -left + opts.thumbmargin * 10,\n                max: -left + measures.w - width - opts.thumbmargin * 10\n            };\n\n            var verticalBounds = {\n                c: top + height / 2,\n                min: -top + opts.thumbmargin * 10,\n                max: -top + measures.h - height - opts.thumbmargin * 10\n            };\n\n            return opts.navdir === 'vertical' ? verticalBounds : horizontalBounds;\n        }\n\n        function slideThumbBorder(time) {\n            var navFrameData = activeFrame[navFrameKey].data();\n            slide($thumbBorder, {\n                time: time * 1.2,\n                pos: (opts.navdir === 'vertical' ? navFrameData.t : navFrameData.l),\n                width: navFrameData.w,\n                height: navFrameData.h,\n                direction: opts.navdir\n            });\n        }\n\n        function slideNavShaft(options) {\n            var $guessNavFrame = data[options.guessIndex][navFrameKey],\n                typeOfAnimation = opts.navtype;\n\n            var overflowFLAG,\n                time,\n                minMax,\n                boundTop,\n                boundLeft,\n                l,\n                pos,\n                x;\n\n            if ($guessNavFrame) {\n                if (typeOfAnimation === 'thumbs') {\n                    overflowFLAG = navShaftTouchTail.min !== navShaftTouchTail.max;\n                    minMax = options.minMax || overflowFLAG && getNavFrameBounds(activeFrame[navFrameKey]);\n                    boundTop = overflowFLAG && (options.keep && slideNavShaft.t ? slideNavShaft.l : minMaxLimit((options.coo || measures.nw / 2) - getNavFrameBounds($guessNavFrame).c, minMax.min, minMax.max));\n                    boundLeft = overflowFLAG && (options.keep && slideNavShaft.l ? slideNavShaft.l : minMaxLimit((options.coo || measures.nw / 2) - getNavFrameBounds($guessNavFrame).c, minMax.min, minMax.max));\n                    l = (opts.navdir === 'vertical' ? boundTop : boundLeft);\n                    pos = overflowFLAG && minMaxLimit(l, navShaftTouchTail.min, navShaftTouchTail.max) || 0;\n                    time = options.time * 1.1;\n                    slide($navShaft, {\n                        time: time,\n                        pos: pos,\n                        direction: opts.navdir,\n                        onEnd: function () {\n                            thumbsDraw(pos, true);\n                            thumbArrUpdate();\n                        }\n                    });\n\n                    setShadow($nav, findShadowEdge(pos, navShaftTouchTail.min, navShaftTouchTail.max, opts.navdir));\n                    slideNavShaft.l = l;\n                } else {\n                    x = readPosition($navShaft, opts.navdir);\n                    time = options.time * 1.11;\n\n                    pos = validateSlidePos(opts, navShaftTouchTail, options.guessIndex, x, $guessNavFrame, $navWrap, opts.navdir);\n\n                    slide($navShaft, {\n                        time: time,\n                        pos: pos,\n                        direction: opts.navdir,\n                        onEnd: function () {\n                            thumbsDraw(pos, true);\n                            thumbArrUpdate();\n                        }\n                    });\n                    setShadow($nav, findShadowEdge(pos, navShaftTouchTail.min, navShaftTouchTail.max, opts.navdir));\n                }\n            }\n        }\n\n        function navUpdate() {\n            deactivateFrames(navFrameKey);\n            toDeactivate[navFrameKey].push(activeFrame[navFrameKey].addClass(activeClass).attr('data-active', true));\n        }\n\n        function deactivateFrames(key) {\n            var _toDeactivate = toDeactivate[key];\n\n            while (_toDeactivate.length) {\n                _toDeactivate.shift().removeClass(activeClass).attr('data-active', false);\n            }\n        }\n\n        function detachFrames(key) {\n            var _toDetach = toDetach[key];\n\n            $.each(activeIndexes, function (i, index) {\n                delete _toDetach[normalizeIndex(index)];\n            });\n\n            $.each(_toDetach, function (index, $frame) {\n                delete _toDetach[index];\n                $frame.detach();\n            });\n        }\n\n        function stageShaftReposition(skipOnEnd) {\n\n            repositionIndex = dirtyIndex = activeIndex;\n\n            var $frame = activeFrame[STAGE_FRAME_KEY];\n\n            if ($frame) {\n                deactivateFrames(STAGE_FRAME_KEY);\n                toDeactivate[STAGE_FRAME_KEY].push($frame.addClass(activeClass).attr('data-active', true));\n\n                if ($frame.hasClass(stageFrameClass)) {\n                    $frame.attr('aria-hidden', 'false');\n                }\n\n                skipOnEnd || that.showStage.onEnd(true);\n                stop($stageShaft, 0, true);\n\n                detachFrames(STAGE_FRAME_KEY);\n                stageFramePosition(activeIndexes);\n                setStageShaftMinmaxAndSnap();\n                setNavShaftMinMax();\n                addEnterUp($stageShaft[0], function () {\n                    if (!$fotorama.hasClass(fullscreenClass)) {\n                        that.requestFullScreen();\n                        $fullscreenIcon.focus();\n                    }\n                });\n            }\n        }\n\n        function extendMeasures(options, measuresArray) {\n            if (!options) return;\n\n            $.each(measuresArray, function (i, measures) {\n                if (!measures) return;\n\n                $.extend(measures, {\n                    width: options.width || measures.width,\n                    height: options.height,\n                    minwidth: options.minwidth,\n                    maxwidth: options.maxwidth,\n                    minheight: options.minheight,\n                    maxheight: options.maxheight,\n                    ratio: getRatio(options.ratio)\n                })\n            });\n        }\n\n        function triggerEvent(event, extra) {\n            $fotorama.trigger(_fotoramaClass + ':' + event, [that, extra]);\n        }\n\n        function onTouchStart() {\n            clearTimeout(onTouchEnd.t);\n            touchedFLAG = 1;\n\n            if (opts.stopautoplayontouch) {\n                that.stopAutoplay();\n            } else {\n                pausedAutoplayFLAG = true;\n            }\n        }\n\n        function onTouchEnd() {\n            if (!touchedFLAG) return;\n            if (!opts.stopautoplayontouch) {\n                releaseAutoplay();\n                changeAutoplay();\n            }\n\n            onTouchEnd.t = setTimeout(function () {\n                touchedFLAG = 0;\n            }, TRANSITION_DURATION + TOUCH_TIMEOUT);\n        }\n\n        function releaseAutoplay() {\n            pausedAutoplayFLAG = !!($videoPlaying || stoppedAutoplayFLAG);\n        }\n\n        function changeAutoplay() {\n\n            clearTimeout(changeAutoplay.t);\n            waitFor.stop(changeAutoplay.w);\n\n            if (!opts.autoplay || pausedAutoplayFLAG) {\n                if (that.autoplay) {\n                    that.autoplay = false;\n                    triggerEvent('stopautoplay');\n                }\n\n                return;\n            }\n\n            if (!that.autoplay) {\n                that.autoplay = true;\n                triggerEvent('startautoplay');\n            }\n\n            var _activeIndex = activeIndex;\n\n\n            var frameData = activeFrame[STAGE_FRAME_KEY].data();\n            changeAutoplay.w = waitFor(function () {\n                return frameData.state || _activeIndex !== activeIndex;\n            }, function () {\n                changeAutoplay.t = setTimeout(function () {\n\n                    if (pausedAutoplayFLAG || _activeIndex !== activeIndex) return;\n\n                    var _nextAutoplayIndex = nextAutoplayIndex,\n                        nextFrameData = data[_nextAutoplayIndex][STAGE_FRAME_KEY].data();\n\n                    changeAutoplay.w = waitFor(function () {\n\n                        return nextFrameData.state || _nextAutoplayIndex !== nextAutoplayIndex;\n                    }, function () {\n                        if (pausedAutoplayFLAG || _nextAutoplayIndex !== nextAutoplayIndex) return;\n                        that.show(o_loop ? getDirectionSign(!o_rtl) : nextAutoplayIndex);\n                    });\n                }, opts.autoplay);\n            });\n\n        }\n\n        that.startAutoplay = function (interval) {\n            if (that.autoplay) return this;\n            pausedAutoplayFLAG = stoppedAutoplayFLAG = false;\n            setAutoplayInterval(interval || opts.autoplay);\n            changeAutoplay();\n\n            return this;\n        };\n\n        that.stopAutoplay = function () {\n            if (that.autoplay) {\n                pausedAutoplayFLAG = stoppedAutoplayFLAG = true;\n                changeAutoplay();\n            }\n            return this;\n        };\n\n        that.showSlide = function (slideDir) {\n            var currentPosition = readPosition($navShaft, opts.navdir),\n                pos,\n                time = 500 * 1.1,\n                size = opts.navdir === 'horizontal' ? opts.thumbwidth : opts.thumbheight,\n                onEnd = function () {\n                    thumbArrUpdate();\n                };\n            if (slideDir === 'next') {\n                pos = currentPosition - (size + opts.margin) * thumbsPerSlide;\n            }\n            if (slideDir === 'prev') {\n                pos = currentPosition + (size + opts.margin) * thumbsPerSlide;\n            }\n            pos = validateRestrictions(pos, navShaftTouchTail);\n            thumbsDraw(pos, true);\n            slide($navShaft, {\n                time: time,\n                pos: pos,\n                direction: opts.navdir,\n                onEnd: onEnd\n            });\n        };\n\n        that.showWhileLongPress = function (options) {\n            if (that.longPress.singlePressInProgress) {\n                return;\n            }\n\n            var index = calcActiveIndex(options);\n            calcGlobalIndexes(index);\n            var time = calcTime(options) / 50;\n            var _activeFrame = activeFrame;\n            that.activeFrame = activeFrame = data[activeIndex];\n            var silent = _activeFrame === activeFrame && !options.user;\n\n            that.showNav(silent, options, time);\n\n            return this;\n        };\n\n        that.showEndLongPress = function (options) {\n            if (that.longPress.singlePressInProgress) {\n                return;\n            }\n\n            var index = calcActiveIndex(options);\n            calcGlobalIndexes(index);\n            var time = calcTime(options) / 50;\n            var _activeFrame = activeFrame;\n            that.activeFrame = activeFrame = data[activeIndex];\n\n            var silent = _activeFrame === activeFrame && !options.user;\n\n            that.showStage(silent, options, time);\n\n            showedFLAG = typeof lastActiveIndex !== 'undefined' && lastActiveIndex !== activeIndex;\n            lastActiveIndex = activeIndex;\n            return this;\n        };\n\n        function calcActiveIndex (options) {\n            var index;\n\n            if (typeof options !== 'object') {\n                index = options;\n                options = {};\n            } else {\n                index = options.index;\n            }\n\n            index = index === '>' ? dirtyIndex + 1 : index === '<' ? dirtyIndex - 1 : index === '<<' ? 0 : index === '>>' ? size - 1 : index;\n            index = isNaN(index) ? undefined : index;\n            index = typeof index === 'undefined' ? activeIndex || 0 : index;\n\n            return index;\n        }\n\n        function calcGlobalIndexes (index) {\n            that.activeIndex = activeIndex = edgeIndex(index);\n            prevIndex = getPrevIndex(activeIndex);\n            nextIndex = getNextIndex(activeIndex);\n            nextAutoplayIndex = normalizeIndex(activeIndex + (o_rtl ? -1 : 1));\n            activeIndexes = [activeIndex, prevIndex, nextIndex];\n\n            dirtyIndex = o_loop ? index : activeIndex;\n        }\n\n        function calcTime (options) {\n            var diffIndex = Math.abs(lastActiveIndex - dirtyIndex),\n                time = getNumber(options.time, function () {\n                    return Math.min(o_transitionDuration * (1 + (diffIndex - 1) / 12), o_transitionDuration * 2);\n                });\n\n            if (options.slow) {\n                time *= 10;\n            }\n\n            return time;\n        }\n\n        that.showStage = function (silent, options, time, e) {\n            if (e !== undefined && e.target.tagName == 'IFRAME') {\n                return;\n            }\n            unloadVideo($videoPlaying, activeFrame.i !== data[normalizeIndex(repositionIndex)].i);\n            frameDraw(activeIndexes, 'stage');\n            stageFramePosition(SLOW ? [dirtyIndex] : [dirtyIndex, getPrevIndex(dirtyIndex), getNextIndex(dirtyIndex)]);\n            updateTouchTails('go', true);\n\n            silent || triggerEvent('show', {\n                user: options.user,\n                time: time\n            });\n\n            pausedAutoplayFLAG = true;\n\n            var overPos = options.overPos;\n            var onEnd = that.showStage.onEnd = function (skipReposition) {\n                if (onEnd.ok) return;\n                onEnd.ok = true;\n\n                skipReposition || stageShaftReposition(true);\n\n                if (!silent) {\n                    triggerEvent('showend', {\n                        user: options.user\n                    });\n                }\n\n                if (!skipReposition && o_transition && o_transition !== opts.transition) {\n                    that.setOptions({transition: o_transition});\n                    o_transition = false;\n                    return;\n                }\n\n                updateFotoramaState();\n                loadImg(activeIndexes, 'stage');\n\n                updateTouchTails('go', false);\n                stageWheelUpdate();\n\n                stageCursor();\n                releaseAutoplay();\n                changeAutoplay();\n\n                if (that.fullScreen) {\n                    activeFrame[STAGE_FRAME_KEY].find('.' + imgFullClass).attr('aria-hidden', false);\n                    activeFrame[STAGE_FRAME_KEY].find('.' + imgClass).attr('aria-hidden', true)\n                } else {\n                    activeFrame[STAGE_FRAME_KEY].find('.' + imgFullClass).attr('aria-hidden', true);\n                    activeFrame[STAGE_FRAME_KEY].find('.' + imgClass).attr('aria-hidden', false)\n                }\n            };\n\n            if (!o_fade) {\n                slide($stageShaft, {\n                    pos: -getPosByIndex(dirtyIndex, measures.w, opts.margin, repositionIndex),\n                    overPos: overPos,\n                    time: time,\n                    onEnd: onEnd\n                });\n            } else {\n                var $activeFrame = activeFrame[STAGE_FRAME_KEY],\n                    $prevActiveFrame = data[lastActiveIndex] && activeIndex !== lastActiveIndex ? data[lastActiveIndex][STAGE_FRAME_KEY] : null;\n\n                fade($activeFrame, $prevActiveFrame, $stageFrame, {\n                    time: time,\n                    method: opts.transition,\n                    onEnd: onEnd\n                }, fadeStack);\n            }\n\n            arrsUpdate();\n        };\n\n        that.showNav = function(silent, options, time){\n            thumbArrUpdate();\n            if (o_nav) {\n                navUpdate();\n\n                var guessIndex = limitIndex(activeIndex + minMaxLimit(dirtyIndex - lastActiveIndex, -1, 1));\n                slideNavShaft({\n                    time: time,\n                    coo: guessIndex !== activeIndex && options.coo,\n                    guessIndex: typeof options.coo !== 'undefined' ? guessIndex : activeIndex,\n                    keep: silent\n                });\n                if (o_navThumbs) slideThumbBorder(time);\n            }\n        };\n\n        that.show = function (options, e) {\n            that.longPress.singlePressInProgress = true;\n\n            var index = calcActiveIndex(options);\n            calcGlobalIndexes(index);\n            var time = calcTime(options);\n            var _activeFrame = activeFrame;\n            that.activeFrame = activeFrame = data[activeIndex];\n\n            var silent = _activeFrame === activeFrame && !options.user;\n\n            that.showStage(silent, options, time, e);\n            that.showNav(silent, options, time);\n\n            showedFLAG = typeof lastActiveIndex !== 'undefined' && lastActiveIndex !== activeIndex;\n            lastActiveIndex = activeIndex;\n            that.longPress.singlePressInProgress = false;\n\n            return this;\n        };\n\n        that.requestFullScreen = function () {\n            if (o_allowFullScreen && !that.fullScreen) {\n\n                //check that this is not video\n                var isVideo = $((that.activeFrame || {}).$stageFrame || {}).hasClass('fotorama-video-container');\n                if(isVideo) {\n                    return;\n                }\n\n                scrollTop = $WINDOW.scrollTop();\n                scrollLeft = $WINDOW.scrollLeft();\n\n                lockScroll($WINDOW);\n\n                updateTouchTails('x', true);\n\n                measuresStash = $.extend({}, measures);\n\n                $fotorama\n                    .addClass(fullscreenClass)\n                    .appendTo($BODY.addClass(_fullscreenClass));\n\n                $HTML.addClass(_fullscreenClass);\n\n                unloadVideo($videoPlaying, true, true);\n\n                that.fullScreen = true;\n\n                if (o_nativeFullScreen) {\n                    fullScreenApi.request(fotorama);\n                }\n\n                that.resize();\n                loadImg(activeIndexes, 'stage');\n                updateFotoramaState();\n                triggerEvent('fullscreenenter');\n\n                if (!('ontouchstart' in window)) {\n                    $fullscreenIcon.focus();\n                }\n            }\n\n            return this;\n        };\n\n        function cancelFullScreen() {\n            if (that.fullScreen) {\n                that.fullScreen = false;\n\n                if (FULLSCREEN) {\n                    fullScreenApi.cancel(fotorama);\n                }\n\n                $BODY.removeClass(_fullscreenClass);\n                $HTML.removeClass(_fullscreenClass);\n\n                $fotorama\n                    .removeClass(fullscreenClass)\n                    .insertAfter($anchor);\n\n                measures = $.extend({}, measuresStash);\n\n                unloadVideo($videoPlaying, true, true);\n\n                updateTouchTails('x', false);\n\n                that.resize();\n                loadImg(activeIndexes, 'stage');\n\n                lockScroll($WINDOW, scrollLeft, scrollTop);\n\n                triggerEvent('fullscreenexit');\n            }\n        }\n\n        that.cancelFullScreen = function () {\n            if (o_nativeFullScreen && fullScreenApi.is()) {\n                fullScreenApi.cancel(document);\n            } else {\n                cancelFullScreen();\n            }\n\n            return this;\n        };\n\n        that.toggleFullScreen = function () {\n            return that[(that.fullScreen ? 'cancel' : 'request') + 'FullScreen']();\n        };\n\n        that.resize = function (options) {\n            if (!data) return this;\n\n            var time = arguments[1] || 0,\n                setFLAG = arguments[2];\n\n            thumbsPerSlide = getThumbsInSlide($wrap, opts);\n            extendMeasures(!that.fullScreen ? optionsToLowerCase(options) : {\n                width: $(window).width(),\n                maxwidth: null,\n                minwidth: null,\n                height: $(window).height(),\n                maxheight: null,\n                minheight: null\n            }, [measures, setFLAG || that.fullScreen || opts]);\n\n            var width = measures.width,\n                height = measures.height,\n                ratio = measures.ratio,\n                windowHeight = $WINDOW.height() - (o_nav ? $nav.height() : 0);\n\n            if (measureIsValid(width)) {\n                $wrap.css({width: ''});\n                $wrap.css({height: ''});\n                $stage.css({width: ''});\n                $stage.css({height: ''});\n                $stageShaft.css({width: ''});\n                $stageShaft.css({height: ''});\n                $nav.css({width: ''});\n                $nav.css({height: ''});\n                $wrap.css({minWidth: measures.minwidth || 0, maxWidth: measures.maxwidth || MAX_WIDTH});\n\n                if (o_nav === 'dots') {\n                    $navWrap.hide();\n                }\n                width = measures.W = measures.w = $wrap.width();\n                measures.nw = o_nav && numberFromWhatever(opts.navwidth, width) || width;\n\n                $stageShaft.css({width: measures.w, marginLeft: (measures.W - measures.w) / 2});\n\n                height = numberFromWhatever(height, windowHeight);\n\n                height = height || (ratio && width / ratio);\n\n                if (height) {\n                    width = Math.round(width);\n                    height = measures.h = Math.round(minMaxLimit(height, numberFromWhatever(measures.minheight, windowHeight), numberFromWhatever(measures.maxheight, windowHeight)));\n                    $stage.css({'width': width, 'height': height});\n\n                    if (opts.navdir === 'vertical' && !that.fullscreen) {\n                        $nav.width(opts.thumbwidth + opts.thumbmargin * 2);\n                    }\n\n                    if (opts.navdir === 'horizontal' && !that.fullscreen) {\n                        $nav.height(opts.thumbheight + opts.thumbmargin * 2);\n                    }\n\n                    if (o_nav === 'dots') {\n                        $nav.width(width)\n                            .height('auto');\n                        $navWrap.show();\n                    }\n\n                    if (opts.navdir === 'vertical' && that.fullScreen) {\n                        $stage.css('height', $WINDOW.height());\n                    }\n\n                    if (opts.navdir === 'horizontal' && that.fullScreen) {\n                        $stage.css('height', $WINDOW.height() - $nav.height());\n                    }\n\n                    if (o_nav) {\n                        switch (opts.navdir) {\n                            case 'vertical':\n                                $navWrap.removeClass(navShafthorizontalClass);\n                                $navWrap.removeClass(navShaftListClass);\n                                $navWrap.addClass(navShaftVerticalClass);\n                                $nav\n                                    .stop()\n                                    .animate({height: measures.h, width: opts.thumbwidth}, time);\n                                break;\n                            case 'list':\n                                $navWrap.removeClass(navShaftVerticalClass);\n                                $navWrap.removeClass(navShafthorizontalClass);\n                                $navWrap.addClass(navShaftListClass);\n                                break;\n                            default:\n                                $navWrap.removeClass(navShaftVerticalClass);\n                                $navWrap.removeClass(navShaftListClass);\n                                $navWrap.addClass(navShafthorizontalClass);\n                                $nav\n                                    .stop()\n                                    .animate({width: measures.nw}, time);\n                                break;\n                        }\n\n                        stageShaftReposition();\n                        slideNavShaft({guessIndex: activeIndex, time: time, keep: true});\n                        if (o_navThumbs && frameAppend.nav) slideThumbBorder(time);\n                    }\n\n                    measuresSetFLAG = setFLAG || true;\n\n                    ready.ok = true;\n                    ready();\n                }\n            }\n\n            stageLeft = $stage.offset().left;\n            setStagePosition();\n\n            return this;\n        };\n\n        that.setOptions = function (options) {\n            $.extend(opts, options);\n            reset();\n            return this;\n        };\n\n        that.shuffle = function () {\n            data && shuffle(data) && reset();\n            return this;\n        };\n\n        function setShadow($el, edge) {\n            if (o_shadows) {\n                $el.removeClass(shadowsLeftClass + ' ' + shadowsRightClass);\n                $el.removeClass(shadowsTopClass + ' ' + shadowsBottomClass);\n                edge && !$videoPlaying && $el.addClass(edge.replace(/^|\\s/g, ' ' + shadowsClass + '--'));\n            }\n        }\n\n        that.longPress = {\n            threshold: 1,\n            count: 0,\n            thumbSlideTime: 20,\n            progress: function(){\n                if (!this.inProgress) {\n                    this.count++;\n                    this.inProgress = this.count > this.threshold;\n                }\n            },\n            end: function(){\n                if(this.inProgress) {\n                    this.isEnded = true\n                }\n            },\n            reset: function(){\n                this.count = 0;\n                this.inProgress = false;\n                this.isEnded = false;\n            }\n        };\n\n        that.destroy = function () {\n            that.cancelFullScreen();\n            that.stopAutoplay();\n\n            data = that.data = null;\n\n            appendElements();\n\n            activeIndexes = [];\n            detachFrames(STAGE_FRAME_KEY);\n\n            reset.ok = false;\n\n            return this;\n        };\n\n        /**\n         *\n         * @returns {jQuery.Fotorama}\n         */\n        that.playVideo = function () {\n            var dataFrame = activeFrame,\n                video = dataFrame.video,\n                _activeIndex = activeIndex;\n\n            if (typeof video === 'object' && dataFrame.videoReady) {\n                o_nativeFullScreen && that.fullScreen && that.cancelFullScreen();\n\n                waitFor(function () {\n                    return !fullScreenApi.is() || _activeIndex !== activeIndex;\n                }, function () {\n                    if (_activeIndex === activeIndex) {\n                        dataFrame.$video = dataFrame.$video || $(div(videoClass)).append(createVideoFrame(video));\n                        dataFrame.$video.appendTo(dataFrame[STAGE_FRAME_KEY]);\n\n                        $wrap.addClass(wrapVideoClass);\n                        $videoPlaying = dataFrame.$video;\n\n                        stageNoMove();\n\n                        $arrs.blur();\n                        $fullscreenIcon.blur();\n\n                        triggerEvent('loadvideo');\n                    }\n                });\n            }\n\n            return this;\n        };\n\n        that.stopVideo = function () {\n            unloadVideo($videoPlaying, true, true);\n            return this;\n        };\n\n        that.spliceByIndex = function (index, newImgObj) {\n            newImgObj.i = index + 1;\n            newImgObj.img && $.ajax({\n                url: newImgObj.img,\n                type: 'HEAD',\n                success: function () {\n                    data.splice(index, 1, newImgObj);\n                    reset();\n                }\n            });\n        };\n\n        function unloadVideo($video, unloadActiveFLAG, releaseAutoplayFLAG) {\n            if (unloadActiveFLAG) {\n                $wrap.removeClass(wrapVideoClass);\n                $videoPlaying = false;\n\n                stageNoMove();\n            }\n\n            if ($video && $video !== $videoPlaying) {\n                $video.remove();\n                triggerEvent('unloadvideo');\n            }\n\n            if (releaseAutoplayFLAG) {\n                releaseAutoplay();\n                changeAutoplay();\n            }\n        }\n\n        function toggleControlsClass(FLAG) {\n            $wrap.toggleClass(wrapNoControlsClass, FLAG);\n        }\n\n        function stageCursor(e) {\n            if (stageShaftTouchTail.flow) return;\n\n            var x = e ? e.pageX : stageCursor.x,\n                pointerFLAG = x && !disableDirrection(getDirection(x)) && opts.click;\n\n            if (stageCursor.p !== pointerFLAG\n                && $stage.toggleClass(pointerClass, pointerFLAG)) {\n                stageCursor.p = pointerFLAG;\n                stageCursor.x = x;\n            }\n        }\n\n        $stage.on('mousemove', stageCursor);\n\n        function clickToShow(showOptions, e) {\n            clearTimeout(clickToShow.t);\n\n            if (opts.clicktransition && opts.clicktransition !== opts.transition) {\n                setTimeout(function () {\n                    var _o_transition = opts.transition;\n\n                    that.setOptions({transition: opts.clicktransition});\n\n                    // now safe to pass base transition to o_transition, so that.show will restor it\n                    o_transition = _o_transition;\n                    // this timeout is here to prevent jerking in some browsers\n                    clickToShow.t = setTimeout(function () {\n                        that.show(showOptions);\n                    }, 10);\n                }, 0);\n            } else {\n                that.show(showOptions, e);\n            }\n        }\n\n        function onStageTap(e, toggleControlsFLAG) {\n            var target = e.target,\n                $target = $(target);\n            if ($target.hasClass(videoPlayClass)) {\n                that.playVideo();\n            } else if (target === fullscreenIcon) {\n                that.toggleFullScreen();\n            } else if ($videoPlaying) {\n                target === videoClose && unloadVideo($videoPlaying, true, true);\n            } else if (!$fotorama.hasClass(fullscreenClass)) {\n                that.requestFullScreen();\n            }\n        }\n\n        function updateTouchTails(key, value) {\n            stageShaftTouchTail[key] = navShaftTouchTail[key] = value;\n        }\n\n        stageShaftTouchTail = moveOnTouch($stageShaft, {\n            onStart: onTouchStart,\n            onMove: function (e, result) {\n                setShadow($stage, result.edge);\n            },\n            onTouchEnd: onTouchEnd,\n            onEnd: function (result) {\n                var toggleControlsFLAG;\n\n                setShadow($stage);\n                toggleControlsFLAG = (MS_POINTER && !hoverFLAG || result.touch) &&\n                    opts.arrows;\n\n                if ((result.moved || (toggleControlsFLAG && result.pos !== result.newPos && !result.control)) && result.$target[0] !== $fullscreenIcon[0]) {\n                    var index = getIndexByPos(result.newPos, measures.w, opts.margin, repositionIndex);\n\n                    that.show({\n                        index: index,\n                        time: o_fade ? o_transitionDuration : result.time,\n                        overPos: result.overPos,\n                        user: true\n                    });\n                } else if (!result.aborted && !result.control) {\n                    onStageTap(result.startEvent, toggleControlsFLAG);\n                }\n            },\n            timeLow: 1,\n            timeHigh: 1,\n            friction: 2,\n            select: '.' + selectClass + ', .' + selectClass + ' *',\n            $wrap: $stage,\n            direction: 'horizontal'\n\n        });\n\n        navShaftTouchTail = moveOnTouch($navShaft, {\n            onStart: onTouchStart,\n            onMove: function (e, result) {\n                setShadow($nav, result.edge);\n            },\n            onTouchEnd: onTouchEnd,\n            onEnd: function (result) {\n\n                function onEnd() {\n                    slideNavShaft.l = result.newPos;\n                    releaseAutoplay();\n                    changeAutoplay();\n                    thumbsDraw(result.newPos, true);\n                    thumbArrUpdate();\n                }\n\n                if (!result.moved) {\n                    var target = result.$target.closest('.' + navFrameClass, $navShaft)[0];\n                    target && onNavFrameClick.call(target, result.startEvent);\n                } else if (result.pos !== result.newPos) {\n                    pausedAutoplayFLAG = true;\n                    slide($navShaft, {\n                        time: result.time,\n                        pos: result.newPos,\n                        overPos: result.overPos,\n                        direction: opts.navdir,\n                        onEnd: onEnd\n                    });\n                    thumbsDraw(result.newPos);\n                    o_shadows && setShadow($nav, findShadowEdge(result.newPos, navShaftTouchTail.min, navShaftTouchTail.max, result.dir));\n                } else {\n                    onEnd();\n                }\n            },\n            timeLow: .5,\n            timeHigh: 2,\n            friction: 5,\n            $wrap: $nav,\n            direction: opts.navdir\n        });\n\n        stageWheelTail = wheel($stage, {\n            shift: true,\n            onEnd: function (e, direction) {\n                onTouchStart();\n                onTouchEnd();\n                that.show({index: direction, slow: e.altKey})\n            }\n        });\n\n        navWheelTail = wheel($nav, {\n            onEnd: function (e, direction) {\n                onTouchStart();\n                onTouchEnd();\n                var newPos = stop($navShaft) + direction * .25;\n                $navShaft.css(getTranslate(minMaxLimit(newPos, navShaftTouchTail.min, navShaftTouchTail.max), opts.navdir));\n                o_shadows && setShadow($nav, findShadowEdge(newPos, navShaftTouchTail.min, navShaftTouchTail.max, opts.navdir));\n                navWheelTail.prevent = {'<': newPos >= navShaftTouchTail.max, '>': newPos <= navShaftTouchTail.min};\n                clearTimeout(navWheelTail.t);\n                navWheelTail.t = setTimeout(function () {\n                    slideNavShaft.l = newPos;\n                    thumbsDraw(newPos, true)\n                }, TOUCH_TIMEOUT);\n                thumbsDraw(newPos);\n            }\n        });\n\n        $wrap.hover(\n            function () {\n                setTimeout(function () {\n                    if (touchedFLAG) return;\n                    toggleControlsClass(!(hoverFLAG = true));\n                }, 0);\n            },\n            function () {\n                if (!hoverFLAG) return;\n                toggleControlsClass(!(hoverFLAG = false));\n            }\n        );\n\n        function onNavFrameClick(e) {\n            var index = $(this).data().eq;\n\n            if (opts.navtype === 'thumbs') {\n                clickToShow({index: index, slow: e.altKey, user: true, coo: e._x - $nav.offset().left});\n            } else {\n                clickToShow({index: index, slow: e.altKey, user: true});\n            }\n        }\n\n        function onArrClick(e) {\n            clickToShow({index: $arrs.index(this) ? '>' : '<', slow: e.altKey, user: true});\n        }\n\n        smartClick($arrs, function (e) {\n            stopEvent(e);\n            onArrClick.call(this, e);\n        }, {\n            onStart: function () {\n                onTouchStart();\n                stageShaftTouchTail.control = true;\n            },\n            onTouchEnd: onTouchEnd\n        });\n\n        smartClick($thumbArrLeft, function (e) {\n            stopEvent(e);\n            if (opts.navtype === 'thumbs') {\n\n                that.show('<');\n            } else {\n                that.showSlide('prev')\n            }\n        });\n\n        smartClick($thumbArrRight, function (e) {\n            stopEvent(e);\n            if (opts.navtype === 'thumbs') {\n                that.show('>');\n            } else {\n                that.showSlide('next')\n            }\n\n        });\n\n\n        function addFocusOnControls(el) {\n            addFocus(el, function () {\n                setTimeout(function () {\n                    lockScroll($stage);\n                }, 0);\n                toggleControlsClass(false);\n            });\n        }\n\n        $arrs.each(function () {\n            addEnterUp(this, function (e) {\n                onArrClick.call(this, e);\n            });\n            addFocusOnControls(this);\n        });\n\n        addEnterUp(fullscreenIcon, function () {\n            if ($fotorama.hasClass(fullscreenClass)) {\n                that.cancelFullScreen();\n                $stageShaft.focus();\n            } else {\n                that.requestFullScreen();\n                $fullscreenIcon.focus();\n            }\n\n        });\n        addFocusOnControls(fullscreenIcon);\n\n        function reset() {\n            setData();\n            setOptions();\n\n            if (!reset.i) {\n                reset.i = true;\n                // Only once\n                var _startindex = opts.startindex;\n                activeIndex = repositionIndex = dirtyIndex = lastActiveIndex = startIndex = edgeIndex(_startindex) || 0;\n                /*(o_rtl ? size - 1 : 0)*///;\n            }\n\n            if (size) {\n                if (changeToRtl()) return;\n\n                if ($videoPlaying) {\n                    unloadVideo($videoPlaying, true);\n                }\n\n                activeIndexes = [];\n                detachFrames(STAGE_FRAME_KEY);\n\n                reset.ok = true;\n\n                that.show({index: activeIndex, time: 0});\n                that.resize();\n            } else {\n                that.destroy();\n            }\n        }\n\n        function changeToRtl() {\n\n            if (!changeToRtl.f === o_rtl) {\n                changeToRtl.f = o_rtl;\n                activeIndex = size - 1 - activeIndex;\n                that.reverse();\n\n                return true;\n            }\n        }\n\n        $.each('load push pop shift unshift reverse sort splice'.split(' '), function (i, method) {\n            that[method] = function () {\n                data = data || [];\n                if (method !== 'load') {\n                    Array.prototype[method].apply(data, arguments);\n                } else if (arguments[0] && typeof arguments[0] === 'object' && arguments[0].length) {\n                    data = clone(arguments[0]);\n                }\n                reset();\n                return that;\n            }\n        });\n\n        function ready() {\n            if (ready.ok) {\n                ready.ok = false;\n                triggerEvent('ready');\n            }\n        }\n\n        reset();\n    };\n    $.fn.fotorama = function (opts) {\n        return this.each(function () {\n            var that = this,\n                $fotorama = $(this),\n                fotoramaData = $fotorama.data(),\n                fotorama = fotoramaData.fotorama;\n\n            if (!fotorama) {\n                waitFor(function () {\n                    return !isHidden(that);\n                }, function () {\n                    fotoramaData.urtext = $fotorama.html();\n                    new $.Fotorama($fotorama,\n                        $.extend(\n                            {},\n                            OPTIONS,\n                            window.fotoramaDefaults,\n                            opts,\n                            fotoramaData\n                        )\n                    );\n                });\n            } else {\n                fotorama.setOptions(opts, true);\n            }\n        });\n    };\n    $.Fotorama.instances = [];\n\n    function calculateIndexes() {\n        $.each($.Fotorama.instances, function (index, instance) {\n            instance.index = index;\n        });\n    }\n\n    function addInstance(instance) {\n        $.Fotorama.instances.push(instance);\n        calculateIndexes();\n    }\n\n    function hideInstance(instance) {\n        $.Fotorama.instances.splice(instance.index, 1);\n        calculateIndexes();\n    }\n\n    $.Fotorama.cache = {};\n    $.Fotorama.measures = {};\n    $ = $ || {};\n    $.Fotorama = $.Fotorama || {};\n    $.Fotorama.jst = $.Fotorama.jst || {};\n\n    $.Fotorama.jst.dots = function (v) {\n        var __t, __p = '', __e = _.escape;\n        __p += '<div class=\"fotorama__nav__frame fotorama__nav__frame--dot\" tabindex=\"0\" role=\"button\" data-gallery-role=\"nav-frame\" data-nav-type=\"thumb\" aria-label>\\r\\n    <div class=\"fotorama__dot\"></div>\\r\\n</div>';\n        return __p\n    };\n\n    $.Fotorama.jst.frameCaption = function (v) {\n        var __t, __p = '', __e = _.escape;\n        __p += '<div class=\"fotorama__caption\" aria-hidden=\"true\">\\r\\n    <div class=\"fotorama__caption__wrap\" id=\"' +\n            ((__t = ( v.labelledby )) == null ? '' : __t) +\n            '\">' +\n            ((__t = ( v.caption )) == null ? '' : __t) +\n            '</div>\\r\\n</div>\\r\\n';\n        return __p\n    };\n\n    $.Fotorama.jst.style = function (v) {\n        var __t, __p = '', __e = _.escape;\n        __p += '.fotorama' +\n            ((__t = ( v.s )) == null ? '' : __t) +\n            ' .fotorama__nav--thumbs .fotorama__nav__frame{\\r\\npadding:' +\n            ((__t = ( v.m )) == null ? '' : __t) +\n            'px;\\r\\nheight:' +\n            ((__t = ( v.h )) == null ? '' : __t) +\n            'px}\\r\\n.fotorama' +\n            ((__t = ( v.s )) == null ? '' : __t) +\n            ' .fotorama__thumb-border{\\r\\nheight:' +\n            ((__t = ( v.h )) == null ? '' : __t) +\n            'px;\\r\\nborder-width:' +\n            ((__t = ( v.b )) == null ? '' : __t) +\n            'px;\\r\\nmargin-top:' +\n            ((__t = ( v.m )) == null ? '' : __t) +\n            'px}';\n        return __p\n    };\n\n    $.Fotorama.jst.thumb = function (v) {\n        var __t, __p = '', __e = _.escape;\n        __p += '<div class=\"fotorama__nav__frame fotorama__nav__frame--thumb\" tabindex=\"0\" role=\"button\" data-gallery-role=\"nav-frame\" data-nav-type=\"thumb\" aria-label>\\r\\n    <div class=\"fotorama__thumb\">\\r\\n    </div>\\r\\n</div>';\n        return __p\n    };\n})(window, document, location, typeof jQuery !== 'undefined' && jQuery);\n\n","Magento_Braintree/js/validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\ndefine([\n    'underscore'\n], function (_) {\n    'use strict';\n\n    return {\n        config: {},\n\n        /**\n         * Set configuration\n         * @param {Object} config\n         */\n        setConfig: function (config) {\n            this.config = config;\n        },\n\n        /**\n         * Get List of available card types\n         * @returns {*|exports.defaults.availableCardTypes|{}}\n         */\n        getAvailableCardTypes: function () {\n            return this.config.availableCardTypes;\n        },\n\n        /**\n         * Get list of card types\n         * @returns {Object}\n         */\n        getCcTypesMapper: function () {\n            return this.config.ccTypesMapper;\n        },\n\n        /**\n         * Find mage card type by Braintree type\n         * @param {String} type\n         * @param {Object} availableTypes\n         * @returns {*}\n         */\n        getMageCardType: function (type, availableTypes) {\n            var storedCardType = null,\n                mapper = this.getCcTypesMapper();\n\n            if (type && typeof mapper[type] !== 'undefined') {\n                storedCardType = mapper[type];\n\n                if (_.indexOf(availableTypes, storedCardType) !== -1) {\n                    return storedCardType;\n                }\n            }\n\n            return null;\n        },\n\n        /**\n         * Filter list of available card types\n         * @param {Object} availableTypes\n         * @param {Object} countrySpecificCardTypes\n         * @returns {Object}\n         */\n        collectTypes: function (availableTypes, countrySpecificCardTypes) {\n            var key,\n                filteredTypes = [];\n\n            for (key in availableTypes) {\n                if (_.indexOf(countrySpecificCardTypes, availableTypes[key]) !== -1) {\n                    filteredTypes.push(availableTypes[key]);\n                }\n            }\n\n            return filteredTypes;\n        },\n\n        /**\n         * Get list of card types for country\n         * @param {String} countryId\n         * @returns {*}\n         */\n        getCountrySpecificCardTypes: function (countryId) {\n            if (typeof this.config.countrySpecificCardTypes[countryId] !== 'undefined') {\n                return this.config.countrySpecificCardTypes[countryId];\n            }\n\n            return false;\n        }\n    };\n});\n","Magento_Braintree/js/paypal/form-builder.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine(\n    [\n        'jquery',\n        'underscore',\n        'mage/template'\n    ],\n    function ($, _, mageTemplate) {\n        'use strict';\n\n        return {\n\n            /**\n             * @param {Object} formData\n             * @returns {*|jQuery}\n             */\n            build: function (formData) {\n                var formTmpl = mageTemplate('<form action=\"<%= data.action %>\"' +\n                    ' method=\"POST\" hidden enctype=\"application/x-www-form-urlencoded\">' +\n                        '<% _.each(data.fields, function(val, key){ %>' +\n                            '<input value=\\'<%= val %>\\' name=\"<%= key %>\" type=\"hidden\">' +\n                        '<% }); %>' +\n                    '</form>');\n\n                return $(formTmpl({\n                    data: {\n                        action: formData.action,\n                        fields: formData.fields\n                    }\n                })).appendTo($('[data-container=\"body\"]'));\n            }\n        };\n    }\n);\n","Magento_Braintree/js/paypal/button.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine(\n    [\n        'rjsResolver',\n        'uiRegistry',\n        'uiComponent',\n        'underscore',\n        'jquery',\n        'braintreeClient',\n        'braintreePayPal',\n        'braintreePayPalCheckout',\n        'Magento_Braintree/js/paypal/form-builder',\n        'domReady!'\n    ],\n    function (\n        resolver,\n        registry,\n        Component,\n        _,\n        $,\n        braintreeClient,\n        braintreePayPal,\n        braintreePayPalCheckout,\n        formBuilder\n    ) {\n        'use strict';\n\n        return Component.extend({\n\n            defaults: {\n                displayName: null,\n                clientToken: null,\n                paypalCheckoutInstance: null\n            },\n\n            /**\n             * @returns {Object}\n             */\n            initialize: function () {\n                var self = this;\n\n                self._super();\n\n                braintreeClient.create({\n                    authorization: self.clientToken\n                })\n                    .then(function (clientInstance) {\n                        return braintreePayPal.create({\n                            client: clientInstance\n                        });\n                    })\n                    .then(function (paypalCheckoutInstance) {\n                        self.paypalCheckoutInstance = paypalCheckoutInstance;\n\n                        return self.paypalCheckoutInstance;\n                    });\n\n                self.initComponent();\n\n                return this;\n            },\n\n            /**\n             * @returns {Object}\n             */\n            initComponent: function () {\n                var self = this,\n                    selector = '#' + self.id,\n                    $this = $(selector),\n                    data = {\n                        amount: $this.data('amount'),\n                        locale: $this.data('locale'),\n                        currency: $this.data('currency')\n                    };\n\n                $this.html('');\n                braintreePayPalCheckout.Button.render({\n                    env: self.environment,\n                    style: {\n                        color: 'blue',\n                        shape: 'rect',\n                        size: 'medium',\n                        label: 'pay',\n                        tagline: false\n                    },\n\n                    /**\n                     * Payment setup\n                     */\n                    payment: function () {\n                        return self.paypalCheckoutInstance.createPayment(self.getClientConfig(data));\n                    },\n\n                    /**\n                     * Triggers on `onAuthorize` event\n                     *\n                     * @param {Object} response\n                     */\n                    onAuthorize: function (response) {\n                        return self.paypalCheckoutInstance.tokenizePayment(response)\n                            .then(function (payload) {\n                                $('body').trigger('processStart');\n\n                                formBuilder.build(\n                                    {\n                                        action: self.actionSuccess,\n                                        fields: {\n                                            result: JSON.stringify(payload)\n                                        }\n                                    }\n                                ).submit();\n                            });\n                    }\n                }, selector);\n\n                return this;\n            },\n\n            /**\n             * @returns {Object}\n             * @private\n             */\n            getClientConfig: function (data) {\n                var config = {\n                    flow: 'checkout',\n                    amount: data.amount,\n                    currency: data.currency,\n                    locale: data.locale,\n                    enableShippingAddress: true\n                };\n\n                if (this.displayName) {\n                    config.displayName = this.displayName;\n                }\n\n                return config;\n            }\n        });\n    }\n);\n","Magento_Braintree/js/view/payment/validator-handler.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\n\ndefine([\n    'jquery',\n    'Magento_Braintree/js/view/payment/3d-secure'\n], function ($, verify3DSecure) {\n    'use strict';\n\n    return {\n        initialized: false,\n        validators: [],\n\n        /**\n         * Inits list of validators\n         */\n        initialize: function () {\n            var config = this.getConfig();\n\n            if (this.initialized) {\n                return;\n            }\n\n            this.initialized = true;\n\n            if (config[verify3DSecure.getCode()].enabled) {\n                verify3DSecure.setConfig(config[verify3DSecure.getCode()]);\n                this.add(verify3DSecure);\n            }\n        },\n\n        /**\n         * Gets payment config\n         *\n         * @returns {Object}\n         */\n        getConfig: function () {\n            return window.checkoutConfig.payment;\n        },\n\n        /**\n         * Adds new validator\n         *\n         * @param {Object} validator\n         */\n        add: function (validator) {\n            this.validators.push(validator);\n        },\n\n        /**\n         * Runs pull of validators\n         *\n         * @param {Object} context\n         * @param {Function} successCallback\n         * @param {Function} errorCallback\n         */\n        validate: function (context, successCallback, errorCallback) {\n            var self = this,\n                deferred;\n\n            self.initialize();\n\n            // no available validators\n            if (!self.validators.length) {\n                successCallback();\n\n                return;\n            }\n\n            // get list of deferred validators\n            deferred = $.map(self.validators, function (current) {\n                return current.validate(context);\n            });\n\n            $.when.apply($, deferred)\n                .done(function () {\n                    successCallback();\n                }).fail(function (error) {\n                    errorCallback(error);\n                });\n        }\n    };\n});\n","Magento_Braintree/js/view/payment/kount.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\n\ndefine([\n    'jquery',\n    'braintreeDataCollector',\n    'Magento_Braintree/js/view/payment/adapter'\n], function (\n    $,\n    braintreeDataCollector,\n    braintreeAdapter\n) {\n    'use strict';\n\n    return {\n        paymentCode: 'braintree',\n\n        /**\n         * Returns information about a customer's device on checkout page for passing to Kount for review.\n         *\n         * @returns {Object}\n         */\n        getDeviceData: function () {\n            var state = $.Deferred();\n\n            if (this.hasFraudProtection()) {\n                braintreeAdapter.getApiClient()\n                    .then(function (clientInstance) {\n                        return braintreeDataCollector.create({\n                            client: clientInstance,\n                            kount: true\n                        });\n                    })\n                    .then(function (dataCollectorInstance) {\n                        var deviceData = dataCollectorInstance.deviceData;\n\n                        state.resolve(deviceData);\n                    })\n                    .catch(function (err) {\n                        state.reject(err);\n                    });\n            }\n\n            return state.promise();\n        },\n\n        /**\n         * Returns setting value.\n         *\n         * @returns {Boolean}\n         * @private\n         */\n        hasFraudProtection: function () {\n            return window.checkoutConfig.payment[this.paymentCode].hasFraudProtection;\n        }\n    };\n});\n","Magento_Braintree/js/view/payment/3d-secure.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\n\ndefine([\n    'jquery',\n    'braintree3DSecure',\n    'Magento_Braintree/js/view/payment/adapter',\n    'Magento_Checkout/js/model/quote',\n    'mage/translate',\n    'Magento_Ui/js/modal/modal',\n    'Magento_Checkout/js/model/full-screen-loader'\n], function (\n    $,\n    braintree3DSecure,\n    braintreeAdapter,\n    quote,\n    $t,\n    Modal,\n    fullScreenLoader\n) {\n    'use strict';\n\n    return {\n        config: null,\n        modal: null,\n        threeDSecureInstance: null,\n        state: null,\n\n        /**\n         * Initializes component\n         */\n        initialize: function () {\n            var self = this,\n                promise = $.Deferred();\n\n            self.state = $.Deferred();\n            braintreeAdapter.getApiClient()\n                .then(function (clientInstance) {\n                    return braintree3DSecure.create({\n                        version: 2, // Will use 3DS 2 whenever possible\n                        client: clientInstance\n                    });\n                })\n                .then(function (threeDSecureInstance) {\n                    self.threeDSecureInstance = threeDSecureInstance;\n                    promise.resolve(self.threeDSecureInstance);\n                })\n                .catch(function (err) {\n                    fullScreenLoader.stopLoader();\n                    promise.reject(err);\n                });\n\n            return promise.promise();\n        },\n\n        /**\n         * Sets 3D Secure config\n         *\n         * @param {Object} config\n         */\n        setConfig: function (config) {\n            this.config = config;\n            this.config.thresholdAmount = parseFloat(config.thresholdAmount);\n        },\n\n        /**\n         * Gets code\n         *\n         * @returns {String}\n         */\n        getCode: function () {\n            return 'three_d_secure';\n        },\n\n        /**\n         * Validates 3D Secure\n         *\n         * @param {Object} context\n         * @returns {Object}\n         */\n        validate: function (context) {\n            var self = this,\n                totalAmount = quote.totals()['base_grand_total'],\n                billingAddress = quote.billingAddress(),\n                shippingAddress = quote.shippingAddress(),\n                options = {\n                    amount: totalAmount,\n                    nonce: context.paymentPayload.nonce,\n                    billingAddress: {\n                        givenName: billingAddress.firstname,\n                        surname: billingAddress.lastname,\n                        phoneNumber: billingAddress.telephone,\n                        streetAddress: billingAddress.street[0],\n                        extendedAddress: billingAddress.street[1],\n                        locality: billingAddress.city,\n                        region: billingAddress.regionCode,\n                        postalCode: billingAddress.postcode,\n                        countryCodeAlpha2: billingAddress.countryId\n                    },\n\n                    /**\n                     * Will be called after receiving ThreeDSecure response, before completing the flow.\n                     *\n                     * @param {Object} data - ThreeDSecure data to consume before continuing\n                     * @param {Function} next - callback to continue flow\n                     */\n                    onLookupComplete: function (data, next) {\n                        next();\n                    }\n                };\n\n            if (context.paymentPayload.details) {\n                options.bin = context.paymentPayload.details.bin;\n            }\n\n            if (shippingAddress) {\n                options.additionalInformation = {\n                    shippingGivenName: shippingAddress.firstname,\n                    shippingSurname: shippingAddress.lastname,\n                    shippingPhone: shippingAddress.telephone,\n                    shippingAddress: {\n                        streetAddress: shippingAddress.street[0],\n                        extendedAddress: shippingAddress.street[1],\n                        locality: shippingAddress.city,\n                        region: shippingAddress.regionCode,\n                        postalCode: shippingAddress.postcode,\n                        countryCodeAlpha2: shippingAddress.countryId\n                    }\n                };\n            }\n\n            if (!this.isAmountAvailable(totalAmount) || !this.isCountryAvailable(billingAddress.countryId)) {\n                self.state = $.Deferred();\n                self.state.resolve();\n\n                return self.state.promise();\n            }\n\n            fullScreenLoader.startLoader();\n            this.initialize()\n                .then(function () {\n                    self.threeDSecureInstance.verifyCard(options, function (err, payload) {\n                        if (err) {\n                            fullScreenLoader.stopLoader();\n                            self.state.reject(err.message);\n\n                            return;\n                        }\n\n                        // `liabilityShifted` indicates that 3DS worked and authentication succeeded\n                        // if `liabilityShifted` and `liabilityShiftPossible` are false - card is ineligible for 3DS\n                        if (payload.liabilityShifted || !payload.liabilityShifted && !payload.liabilityShiftPossible) {\n                            context.paymentPayload.nonce = payload.nonce;\n                            self.state.resolve();\n                        } else {\n                            fullScreenLoader.stopLoader();\n                            self.state.reject($t('Please try again with another form of payment.'));\n                        }\n                    });\n                })\n                .fail(function () {\n                    fullScreenLoader.stopLoader();\n                    self.state.reject($t('Please try again with another form of payment.'));\n                });\n\n            return self.state.promise();\n        },\n\n        /**\n         * Checks minimal amount for 3D Secure activation\n         *\n         * @param {Number} amount\n         * @returns {Boolean}\n         * @private\n         */\n        isAmountAvailable: function (amount) {\n            amount = parseFloat(amount);\n\n            return amount >= this.config.thresholdAmount;\n        },\n\n        /**\n         * Checks if current country is available for 3D Secure\n         *\n         * @param {String} countryId\n         * @returns {Boolean}\n         * @private\n         */\n        isCountryAvailable: function (countryId) {\n            var key,\n                specificCountries = this.config.specificCountries;\n\n            // all countries are available\n            if (!specificCountries.length) {\n                return true;\n            }\n\n            for (key in specificCountries) {\n                if (countryId === specificCountries[key]) {\n                    return true;\n                }\n            }\n\n            return false;\n        }\n    };\n});\n","Magento_Braintree/js/view/payment/adapter.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\ndefine([\n    'jquery',\n    'braintreeClient'\n], function ($, braintreeClient) {\n    'use strict';\n\n    return {\n        apiClient: null,\n        checkout: null,\n        code: 'braintree',\n\n        /**\n         * Returns Braintree API client\n         * @returns {Object}\n         */\n        getApiClient: function () {\n            return braintreeClient.create({\n                authorization: this.getClientToken()\n            });\n        },\n\n        /**\n         * Returns payment code\n         *\n         * @returns {String}\n         */\n        getCode: function () {\n            return this.code;\n        },\n\n        /**\n         * Returns client token\n         *\n         * @returns {String}\n         * @private\n         */\n        getClientToken: function () {\n            return window.checkoutConfig.payment[this.code].clientToken;\n        }\n    };\n});\n","Magento_Braintree/js/view/payment/braintree.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\ndefine(\n    [\n        'uiComponent',\n        'Magento_Checkout/js/model/payment/renderer-list'\n    ],\n    function (\n        Component,\n        rendererList\n    ) {\n        'use strict';\n\n        var config = window.checkoutConfig.payment,\n            braintreeType = 'braintree',\n            payPalType = 'braintree_paypal';\n\n        if (config[braintreeType].isActive) {\n            rendererList.push(\n                {\n                    type: braintreeType,\n                    component: 'Magento_Braintree/js/view/payment/method-renderer/cc-form'\n                }\n            );\n        }\n\n        if (config[payPalType].isActive) {\n            rendererList.push(\n                {\n                    type: payPalType,\n                    component: 'Magento_Braintree/js/view/payment/method-renderer/paypal'\n                }\n            );\n        }\n\n        /** Add view logic here if needed */\n        return Component.extend({});\n    }\n);\n","Magento_Braintree/js/view/payment/method-renderer/paypal-vault.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\ndefine([\n    'jquery',\n    'underscore',\n    'Magento_Vault/js/view/payment/method-renderer/vault',\n    'Magento_Ui/js/model/messageList',\n    'Magento_Checkout/js/model/full-screen-loader'\n], function ($, _, VaultComponent, globalMessageList, fullScreenLoader) {\n    'use strict';\n\n    return VaultComponent.extend({\n        defaults: {\n            template: 'Magento_Braintree/payment/paypal/vault',\n            additionalData: {}\n        },\n\n        /**\n         * Get PayPal payer email\n         * @returns {String}\n         */\n        getPayerEmail: function () {\n            return this.details.payerEmail;\n        },\n\n        /**\n         * Get type of payment\n         * @returns {String}\n         */\n        getPaymentIcon: function () {\n            return window.checkoutConfig.payment['braintree_paypal'].paymentIcon;\n        },\n\n        /**\n         * Place order\n         */\n        beforePlaceOrder: function () {\n            this.getPaymentMethodNonce();\n        },\n\n        /**\n         * Send request to get payment method nonce\n         */\n        getPaymentMethodNonce: function () {\n            var self = this;\n\n            fullScreenLoader.startLoader();\n            $.getJSON(self.nonceUrl, {\n                'public_hash': self.publicHash\n            })\n                .done(function (response) {\n                    fullScreenLoader.stopLoader();\n                    self.additionalData['payment_method_nonce'] = response.paymentMethodNonce;\n                    self.placeOrder();\n                })\n                .fail(function (response) {\n                    var error = JSON.parse(response.responseText);\n\n                    fullScreenLoader.stopLoader();\n                    globalMessageList.addErrorMessage({\n                        message: error.message\n                    });\n                });\n        },\n\n        /**\n         * Get payment method data\n         * @returns {Object}\n         */\n        getData: function () {\n            var data = {\n                'method': this.code,\n                'additional_data': {\n                    'public_hash': this.publicHash\n                }\n            };\n\n            data['additional_data'] = _.extend(data['additional_data'], this.additionalData);\n\n            return data;\n        }\n    });\n});\n","Magento_Braintree/js/view/payment/method-renderer/paypal.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\ndefine([\n    'jquery',\n    'underscore',\n    'Magento_Checkout/js/view/payment/default',\n    'Magento_Braintree/js/view/payment/adapter',\n    'braintreePayPal',\n    'braintreePayPalCheckout',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/full-screen-loader',\n    'Magento_Checkout/js/model/payment/additional-validators',\n    'Magento_Vault/js/view/payment/vault-enabler',\n    'Magento_Checkout/js/action/create-billing-address',\n    'Magento_Braintree/js/view/payment/kount',\n    'mage/translate'\n], function (\n    $,\n    _,\n    Component,\n    BraintreeAdapter,\n    BraintreePayPal,\n    BraintreePayPalCheckout,\n    quote,\n    fullScreenLoader,\n    additionalValidators,\n    VaultEnabler,\n    createBillingAddress,\n    kount,\n    $t\n) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Braintree/payment/paypal',\n            code: 'braintree_paypal',\n            active: false,\n            grandTotalAmount: null,\n            isReviewRequired: false,\n            paypalCheckoutInstance: null,\n            customerEmail: null,\n            vaultEnabler: null,\n            paymentPayload: {\n                nonce: null\n            },\n            paypalButtonSelector: '[data-container=\"paypal-button\"]',\n\n            /**\n             * Additional payment data\n             *\n             * {Object}\n             */\n            additionalData: {},\n\n            imports: {\n                onActiveChange: 'active'\n            }\n        },\n\n        /**\n         * Initialize view.\n         *\n         * @return {exports}\n         */\n        initialize: function () {\n            var self = this;\n\n            self._super();\n\n            BraintreeAdapter.getApiClient().then(function (clientInstance) {\n                return BraintreePayPal.create({\n                    client: clientInstance\n                });\n            }).then(function (paypalCheckoutInstance) {\n                self.paypalCheckoutInstance = paypalCheckoutInstance;\n\n                return self.paypalCheckoutInstance;\n            });\n\n            kount.getDeviceData()\n                .then(function (deviceData) {\n                    self.additionalData['device_data'] = deviceData;\n                });\n\n            // for each component initialization need update property\n            this.isReviewRequired(false);\n\n            return self;\n        },\n\n        /**\n         * Set list of observable attributes\n         * @returns {exports.initObservable}\n         */\n        initObservable: function () {\n            var self = this;\n\n            this._super()\n                .observe(['active', 'isReviewRequired', 'customerEmail']);\n\n            this.vaultEnabler = new VaultEnabler();\n            this.vaultEnabler.setPaymentCode(this.getVaultCode());\n            this.vaultEnabler.isActivePaymentTokenEnabler.subscribe(function () {\n                self.onVaultPaymentTokenEnablerChange();\n            });\n\n            this.grandTotalAmount = quote.totals()['base_grand_total'];\n\n            quote.totals.subscribe(function () {\n                if (self.grandTotalAmount !== quote.totals()['base_grand_total']) {\n                    self.grandTotalAmount = quote.totals()['base_grand_total'];\n                }\n            });\n\n            quote.shippingAddress.subscribe(function () {\n                if (self.isActive()) {\n                    self.reInitPayPal();\n                }\n            });\n\n            return this;\n        },\n\n        /**\n         * Get payment name\n         *\n         * @returns {String}\n         */\n        getCode: function () {\n            return this.code;\n        },\n\n        /**\n         * Get payment title\n         *\n         * @returns {String}\n         */\n        getTitle: function () {\n            return window.checkoutConfig.payment[this.getCode()].title;\n        },\n\n        /**\n         * Check if payment is active\n         *\n         * @returns {Boolean}\n         */\n        isActive: function () {\n            var active = this.getCode() === this.isChecked();\n\n            this.active(active);\n\n            return active;\n        },\n\n        /**\n         * Triggers when payment method change\n         * @param {Boolean} isActive\n         */\n        onActiveChange: function (isActive) {\n            if (!isActive) {\n                return;\n            }\n\n            // need always re-init Braintree with PayPal configuration\n            this.reInitPayPal();\n        },\n\n        /**\n         * Sets payment payload\n         *\n         * @param {Object} paymentPayload\n         * @private\n         */\n        setPaymentPayload: function (paymentPayload) {\n            this.paymentPayload = paymentPayload;\n        },\n\n        /**\n         * Update quote billing address\n         * @param {Object}customer\n         * @param {Object}address\n         */\n        setBillingAddress: function (customer, address) {\n            var billingAddress = {\n                street: [address.streetAddress],\n                city: address.locality,\n                postcode: address.postalCode,\n                countryId: address.countryCodeAlpha2,\n                email: customer.email,\n                firstname: customer.firstName,\n                lastname: customer.lastName,\n                telephone: customer.phone\n            };\n\n            billingAddress['region_code'] = address.region;\n            billingAddress = createBillingAddress(billingAddress);\n            quote.billingAddress(billingAddress);\n        },\n\n        /**\n         * Prepare data to place order\n         * @param {Object} payload\n         */\n        beforePlaceOrder: function (payload) {\n            this.setPaymentPayload(payload);\n\n            if ((this.isRequiredBillingAddress() || quote.billingAddress() === null) &&\n                typeof payload.details.billingAddress !== 'undefined'\n            ) {\n                this.setBillingAddress(payload.details, payload.details.billingAddress);\n            }\n\n            if (this.isSkipOrderReview()) {\n                this.placeOrder();\n            } else {\n                this.customerEmail(payload.details.email);\n                this.isReviewRequired(true);\n            }\n        },\n\n        /**\n         * Re-init PayPal Auth Flow\n         */\n        reInitPayPal: function () {\n            var self = this;\n\n            $(self.paypalButtonSelector).html('');\n\n            return BraintreePayPalCheckout.Button.render({\n                env: this.getEnvironment(),\n                style: {\n                    color: 'blue',\n                    shape: 'rect',\n                    size: 'medium',\n                    label: 'pay',\n                    tagline: false\n                },\n\n                /**\n                 * Creates a PayPal payment\n                 */\n                payment: function () {\n                    return self.paypalCheckoutInstance.createPayment(\n                        self.getPayPalConfig()\n                    );\n                },\n\n                /**\n                 * Tokenizes the authorize data\n                 */\n                onAuthorize: function (data) {\n                    return self.paypalCheckoutInstance.tokenizePayment(data)\n                        .then(function (payload) {\n                            self.beforePlaceOrder(payload);\n                        });\n                },\n\n                /**\n                 * Triggers on error\n                 */\n                onError: function () {\n                    self.showError($t('Payment ' + self.getTitle() + ' can\\'t be initialized'));\n                }\n            }, self.paypalButtonSelector);\n        },\n\n        /**\n         * Get locale\n         * @returns {String}\n         */\n        getLocale: function () {\n            return window.checkoutConfig.payment[this.getCode()].locale;\n        },\n\n        /**\n         * Is shipping address can be editable on PayPal side\n         * @returns {Boolean}\n         */\n        isAllowOverrideShippingAddress: function () {\n            return window.checkoutConfig.payment[this.getCode()].isAllowShippingAddressOverride;\n        },\n\n        /**\n         * Is billing address required from PayPal side\n         * @returns {Boolean}\n         */\n        isRequiredBillingAddress: function () {\n            return window.checkoutConfig.payment[this.getCode()].isRequiredBillingAddress;\n        },\n\n        /**\n         * Get configuration for PayPal\n         * @returns {Object}\n         */\n        getPayPalConfig: function () {\n            var totals = quote.totals(),\n                config,\n                isActiveVaultEnabler = this.isActiveVault();\n\n            config = {\n                flow: !isActiveVaultEnabler ? 'checkout' : 'vault',\n                amount: this.grandTotalAmount,\n                currency: totals['base_currency_code'],\n                locale: this.getLocale(),\n                enableShippingAddress: true,\n                shippingAddressEditable: this.isAllowOverrideShippingAddress()\n            };\n\n            config.shippingAddressOverride = this.getShippingAddress();\n\n            if (this.getMerchantName()) {\n                config.displayName = this.getMerchantName();\n            }\n\n            return config;\n        },\n\n        /**\n         * Get shipping address\n         * @returns {Object}\n         */\n        getShippingAddress: function () {\n            var address = quote.shippingAddress();\n\n            if (_.isNull(address.postcode) || _.isUndefined(address.postcode)) {\n                return {};\n            }\n\n            return {\n                line1: address.street[0],\n                city: address.city,\n                state: address.regionCode,\n                postalCode: address.postcode,\n                countryCode: address.countryId,\n                phone: address.telephone,\n                recipientName: address.firstname + ' ' + address.lastname\n            };\n        },\n\n        /**\n         * Get merchant name\n         * @returns {String}\n         */\n        getMerchantName: function () {\n            return window.checkoutConfig.payment[this.getCode()].merchantName;\n        },\n\n        /**\n         * Get data\n         * @returns {Object}\n         */\n        getData: function () {\n            var data = {\n                'method': this.getCode(),\n                'additional_data': {\n                    'payment_method_nonce': this.paymentPayload.nonce\n                }\n            };\n\n            data['additional_data'] = _.extend(data['additional_data'], this.additionalData);\n\n            this.vaultEnabler.visitAdditionalData(data);\n\n            return data;\n        },\n\n        /**\n         * Returns payment acceptance mark image path\n         * @returns {String}\n         */\n        getPaymentAcceptanceMarkSrc: function () {\n\n            return window.checkoutConfig.payment[this.getCode()].paymentAcceptanceMarkSrc;\n        },\n\n        /**\n         * @returns {String}\n         */\n        getVaultCode: function () {\n            return window.checkoutConfig.payment[this.getCode()].vaultCode;\n        },\n\n        /**\n         * @returns {String}\n         */\n        getEnvironment: function () {\n            return window.checkoutConfig.payment[BraintreeAdapter.getCode()].environment;\n        },\n\n        /**\n         * Check if need to skip order review\n         * @returns {Boolean}\n         */\n        isSkipOrderReview: function () {\n            return window.checkoutConfig.payment[this.getCode()].skipOrderReview;\n        },\n\n        /**\n         * Checks if vault is active\n         * @returns {Boolean}\n         */\n        isActiveVault: function () {\n            return this.vaultEnabler.isVaultEnabled() && this.vaultEnabler.isActivePaymentTokenEnabler();\n        },\n\n        /**\n         * Re-init PayPal Auth flow to use Vault\n         */\n        onVaultPaymentTokenEnablerChange: function () {\n            this.reInitPayPal();\n        }\n    });\n});\n","Magento_Braintree/js/view/payment/method-renderer/vault.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\ndefine([\n    'jquery',\n    'Magento_Vault/js/view/payment/method-renderer/vault',\n    'Magento_Braintree/js/view/payment/adapter',\n    'Magento_Ui/js/model/messageList',\n    'Magento_Checkout/js/model/full-screen-loader'\n], function ($, VaultComponent, Braintree, globalMessageList, fullScreenLoader) {\n    'use strict';\n\n    return VaultComponent.extend({\n        defaults: {\n            template: 'Magento_Vault/payment/form',\n            modules: {\n                hostedFields: '${ $.parentName }.braintree'\n            }\n        },\n\n        /**\n         * Get last 4 digits of card\n         * @returns {String}\n         */\n        getMaskedCard: function () {\n            return this.details.maskedCC;\n        },\n\n        /**\n         * Get expiration date\n         * @returns {String}\n         */\n        getExpirationDate: function () {\n            return this.details.expirationDate;\n        },\n\n        /**\n         * Get card type\n         * @returns {String}\n         */\n        getCardType: function () {\n            return this.details.type;\n        },\n\n        /**\n         * Place order\n         */\n        placeOrder: function () {\n            var self = this;\n\n            self.getPaymentMethodNonce();\n        },\n\n        /**\n         * Send request to get payment method nonce\n         */\n        getPaymentMethodNonce: function () {\n            var self = this;\n\n            fullScreenLoader.startLoader();\n            $.getJSON(self.nonceUrl, {\n                'public_hash': self.publicHash\n            })\n                .done(function (response) {\n                    fullScreenLoader.stopLoader();\n                    self.hostedFields(function (formComponent) {\n                        formComponent.paymentPayload.nonce = response.paymentMethodNonce;\n                        formComponent.additionalData['public_hash'] = self.publicHash;\n                        formComponent.code = self.code;\n                        formComponent.messageContainer = self.messageContainer;\n                        formComponent.placeOrder();\n                    });\n                })\n                .fail(function (response) {\n                    var error = JSON.parse(response.responseText);\n\n                    fullScreenLoader.stopLoader();\n                    globalMessageList.addErrorMessage({\n                        message: error.message\n                    });\n                });\n        }\n    });\n});\n","Magento_Braintree/js/view/payment/method-renderer/cc-form.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\ndefine(\n    [\n        'underscore',\n        'jquery',\n        'Magento_Payment/js/view/payment/cc-form',\n        'Magento_Braintree/js/view/payment/adapter',\n        'braintreeHostedFields',\n        'Magento_Checkout/js/model/quote',\n        'Magento_Braintree/js/validator',\n        'Magento_Ui/js/model/messageList',\n        'Magento_Braintree/js/view/payment/validator-handler',\n        'Magento_Vault/js/view/payment/vault-enabler',\n        'Magento_Braintree/js/view/payment/kount',\n        'mage/translate',\n        'prototype',\n        'domReady!'\n    ],\n    function (\n        _,\n        $,\n        Component,\n        braintreeAdapter,\n        hostedFields,\n        quote,\n        validator,\n        globalMessageList,\n        validatorManager,\n        VaultEnabler,\n        kount,\n        $t\n    ) {\n        'use strict';\n\n        return Component.extend({\n            defaults: {\n                template: 'Magento_Braintree/payment/form',\n                active: false,\n                code: 'braintree',\n                lastBillingAddress: null,\n                hostedFieldsInstance: null,\n                selectorsMapper: {\n                    'expirationMonth': 'expirationMonth',\n                    'expirationYear': 'expirationYear',\n                    'number': 'cc_number',\n                    'cvv': 'cc_cid'\n                },\n                paymentPayload: {\n                    nonce: null\n                },\n                additionalData: {}\n            },\n\n            /**\n             * @returns {exports.initialize}\n             */\n            initialize: function () {\n                var self = this;\n\n                self._super();\n                self.vaultEnabler = new VaultEnabler();\n                self.vaultEnabler.setPaymentCode(self.getVaultCode());\n\n                kount.getDeviceData()\n                    .then(function (deviceData) {\n                        self.additionalData['device_data'] = deviceData;\n                    });\n\n                return self;\n            },\n\n            /**\n             * Init hosted fields.\n             *\n             * Is called after knockout finishes input fields bindings.\n             */\n            initHostedFields: function () {\n                var self = this;\n\n                braintreeAdapter.getApiClient()\n                    .then(function (clientInstance) {\n\n                        return hostedFields.create({\n                            client: clientInstance,\n                            fields: self.getFieldsConfiguration()\n                        });\n                    })\n                    .then(function (hostedFieldsInstance) {\n                        self.hostedFieldsInstance = hostedFieldsInstance;\n                        self.isPlaceOrderActionAllowed(true);\n                        self.initFormValidationEvents(hostedFieldsInstance);\n\n                        return self.hostedFieldsInstance;\n                    })\n                    .catch(function () {\n                        self.showError($t('Payment ' + self.getTitle() + ' can\\'t be initialized'));\n                    });\n            },\n\n            /**\n             * Set list of observable attributes\n             *\n             * @returns {exports.initObservable}\n             */\n            initObservable: function () {\n                validator.setConfig(window.checkoutConfig.payment[this.getCode()]);\n                this._super()\n                    .observe(['active']);\n\n                return this;\n            },\n\n            /**\n             * Get payment name\n             *\n             * @returns {String}\n             */\n            getCode: function () {\n                return this.code;\n            },\n\n            /**\n             * Check if payment is active\n             *\n             * @returns {Boolean}\n             */\n            isActive: function () {\n                var active = this.getCode() === this.isChecked();\n\n                this.active(active);\n\n                return active;\n            },\n\n            /**\n             * Get data\n             *\n             * @returns {Object}\n             */\n            getData: function () {\n                var data = {\n                    'method': this.getCode(),\n                    'additional_data': {\n                        'payment_method_nonce': this.paymentPayload.nonce\n                    }\n                };\n\n                data['additional_data'] = _.extend(data['additional_data'], this.additionalData);\n                this.vaultEnabler.visitAdditionalData(data);\n\n                return data;\n            },\n\n            /**\n             * Get list of available CC types\n             *\n             * @returns {Object}\n             */\n            getCcAvailableTypes: function () {\n                var availableTypes = validator.getAvailableCardTypes(),\n                    billingAddress = quote.billingAddress(),\n                    billingCountryId;\n\n                this.lastBillingAddress = quote.shippingAddress();\n\n                if (!billingAddress) {\n                    billingAddress = this.lastBillingAddress;\n                }\n\n                billingCountryId = billingAddress.countryId;\n\n                if (billingCountryId && validator.getCountrySpecificCardTypes(billingCountryId)) {\n                    return validator.collectTypes(\n                        availableTypes,\n                        validator.getCountrySpecificCardTypes(billingCountryId)\n                    );\n                }\n\n                return availableTypes;\n            },\n\n            /**\n             * @returns {Boolean}\n             */\n            isVaultEnabled: function () {\n                return this.vaultEnabler.isVaultEnabled();\n            },\n\n            /**\n             * Returns vault code.\n             *\n             * @returns {String}\n             */\n            getVaultCode: function () {\n                return window.checkoutConfig.payment[this.getCode()].ccVaultCode;\n            },\n\n            /**\n             * Action to place order\n             * @param {String} key\n             */\n            placeOrder: function (key) {\n                var self = this;\n\n                if (key) {\n                    return self._super();\n                }\n                // place order on success validation\n                validatorManager.validate(self, function () {\n                    return self.placeOrder('parent');\n                }, function (err) {\n\n                    if (err) {\n                        self.showError(err);\n                    }\n                });\n\n                return false;\n            },\n\n            /**\n             * Returns state of place order button\n             *\n             * @returns {Boolean}\n             */\n            isButtonActive: function () {\n                return this.isActive() && this.isPlaceOrderActionAllowed();\n            },\n\n            /**\n             * Trigger order placing\n             */\n            placeOrderClick: function () {\n                var self = this;\n\n                if (this.isFormValid(this.hostedFieldsInstance)) {\n                    self.hostedFieldsInstance.tokenize(function (err, payload) {\n                        if (err) {\n                            self.showError($t('Some payment input fields are invalid.'));\n\n                            return;\n                        }\n\n                        if (self.validateCardType()) {\n                            self.setPaymentPayload(payload);\n                            self.placeOrder();\n                        }\n                    });\n                }\n            },\n\n            /**\n             * Validates credit card form.\n             *\n             * @param {Object} hostedFieldsInstance\n             * @returns {Boolean}\n             * @private\n             */\n            isFormValid: function (hostedFieldsInstance) {\n                var self = this,\n                    state = hostedFieldsInstance.getState();\n\n                return Object.keys(state.fields).every(function (fieldKey) {\n                    if (fieldKey in self.selectorsMapper && state.fields[fieldKey].isValid === false) {\n                        self.addInvalidClass(self.selectorsMapper[fieldKey]);\n                    }\n\n                    return state.fields[fieldKey].isValid;\n                });\n            },\n\n            /**\n             * Init form validation events.\n             *\n             * @param {Object} hostedFieldsInstance\n             * @private\n             */\n            initFormValidationEvents: function (hostedFieldsInstance) {\n                var self = this;\n\n                hostedFieldsInstance.on('empty', function (event) {\n                    if (event.emittedBy === 'number') {\n                        self.selectedCardType(null);\n                    }\n\n                });\n\n                hostedFieldsInstance.on('blur', function (event) {\n                    if (event.emittedBy === 'number') {\n                        self.validateCardType();\n                    }\n                });\n\n                hostedFieldsInstance.on('validityChange', function (event) {\n                    var field = event.fields[event.emittedBy],\n                        fieldKey = event.emittedBy;\n\n                    if (fieldKey === 'number') {\n                        self.isValidCardNumber = field.isValid;\n                    }\n\n                    if (fieldKey in self.selectorsMapper && field.isValid === false) {\n                        self.addInvalidClass(self.selectorsMapper[fieldKey]);\n                    }\n                });\n\n                hostedFieldsInstance.on('cardTypeChange', function (event) {\n                    if (event.cards.length === 1) {\n                        self.selectedCardType(\n                            validator.getMageCardType(event.cards[0].type, self.getCcAvailableTypes())\n                        );\n                    }\n                });\n            },\n\n            /**\n             * Get full selector name\n             *\n             * @param {String} field\n             * @returns {String}\n             * @private\n             */\n            getSelector: function (field) {\n                return '#' + this.getCode() + '_' + field;\n            },\n\n            /**\n             * Add invalid class to field.\n             *\n             * @param {String} field\n             * @returns void\n             * @private\n             */\n            addInvalidClass: function (field) {\n                $(this.getSelector(field)).addClass('braintree-hosted-fields-invalid');\n            },\n\n            /**\n             * Remove invalid class from field.\n             *\n             * @param {String} field\n             * @returns void\n             * @private\n             */\n            removeInvalidClass: function (field) {\n                $(this.getSelector(field)).removeClass('braintree-hosted-fields-invalid');\n            },\n\n            /**\n             * Get Braintree Hosted Fields\n             *\n             * @returns {Object}\n             * @private\n             */\n            getFieldsConfiguration: function () {\n                var self = this,\n                    fields = {\n                        number: {\n                            selector: self.getSelector('cc_number')\n                        },\n                        expirationMonth: {\n                            selector: self.getSelector('expirationMonth'),\n                            placeholder: $t('MM')\n                        },\n                        expirationYear: {\n                            selector: self.getSelector('expirationYear'),\n                            placeholder: $t('YY')\n                        }\n                    };\n\n                if (self.hasVerification()) {\n                    fields.cvv = {\n                        selector: self.getSelector('cc_cid')\n                    };\n                }\n\n                return fields;\n            },\n\n            /**\n             * Validate current credit card type.\n             *\n             * @returns {Boolean}\n             * @private\n             */\n            validateCardType: function () {\n                var cardFieldName = 'cc_number';\n\n                this.removeInvalidClass(cardFieldName);\n\n                if (this.selectedCardType() === null || !this.isValidCardNumber) {\n                    this.addInvalidClass(cardFieldName);\n\n                    return false;\n                }\n\n                return true;\n            },\n\n            /**\n             * Sets payment payload\n             *\n             * @param {Object} paymentPayload\n             * @private\n             */\n            setPaymentPayload: function (paymentPayload) {\n                this.paymentPayload = paymentPayload;\n            },\n\n            /**\n             * Show error message\n             *\n             * @param {String} errorMessage\n             * @private\n             */\n            showError: function (errorMessage) {\n                globalMessageList.addErrorMessage({\n                    message: errorMessage\n                });\n            }\n        });\n    }\n);\n","Magento_Braintree/js/view/payment/method-renderer/multishipping/paypal.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\ndefine([\n    'jquery',\n    'underscore',\n    'Magento_Braintree/js/view/payment/method-renderer/paypal',\n    'Magento_Checkout/js/action/set-payment-information-extended',\n    'Magento_Checkout/js/model/payment/additional-validators',\n    'Magento_Checkout/js/model/full-screen-loader'\n], function (\n    $,\n    _,\n    Component,\n    setPaymentInformationExtended,\n    additionalValidators,\n    fullScreenLoader\n) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Braintree/payment/multishipping/paypal',\n            submitButtonSelector: '#payment-continue span',\n            paypalButtonSelector: '[id=\"parent-payment-continue\"]',\n            reviewButtonHtml: ''\n        },\n\n        /**\n         * @override\n         */\n        initObservable: function () {\n            this.reviewButtonHtml = $(this.paypalButtonSelector).html();\n\n            return this._super();\n        },\n\n        /**\n         * Get configuration for PayPal.\n         *\n         * @returns {Object}\n         */\n        getPayPalConfig: function () {\n            var config;\n\n            config = this._super();\n            config.flow = 'vault';\n            config.enableShippingAddress = false;\n            config.shippingAddressEditable = false;\n\n            return config;\n        },\n\n        /**\n         * @override\n         */\n        onActiveChange: function (isActive) {\n            this._super(isActive);\n            this.updateSubmitButton(isActive);\n        },\n\n        /**\n         * @override\n         */\n        beforePlaceOrder: function (data) {\n            this._super(data);\n\n            this.updateSubmitButton(true);\n        },\n\n        /**\n         * @override\n         */\n        getShippingAddress: function () {\n            return {};\n        },\n\n        /**\n         * @override\n         */\n        getData: function () {\n            var data = this._super();\n\n            data['additional_data']['is_active_payment_token_enabler'] = true;\n\n            return data;\n        },\n\n        /**\n         * @override\n         */\n        isActiveVault: function () {\n            return true;\n        },\n\n        /**\n         * Skipping order review step on checkout with multiple addresses is not allowed.\n         *\n         * @returns {Boolean}\n         */\n        isSkipOrderReview: function () {\n            return false;\n        },\n\n        /**\n         * Checks if payment method nonce is already received.\n         *\n         * @returns {Boolean}\n         */\n        isPaymentMethodNonceReceived: function () {\n            return this.paymentPayload.nonce !== null;\n        },\n\n        /**\n         * Updates submit button on multi-addresses checkout billing form.\n         *\n         * @param {Boolean} isActive\n         */\n        updateSubmitButton: function (isActive) {\n            if (this.isPaymentMethodNonceReceived() || !isActive) {\n                $(this.paypalButtonSelector).html(this.reviewButtonHtml);\n            }\n        },\n\n        /**\n         * @override\n         */\n        placeOrder: function () {\n            fullScreenLoader.startLoader();\n            $.when(\n                setPaymentInformationExtended(\n                    this.messageContainer,\n                    this.getData(),\n                    true\n                )\n            ).done(this.done.bind(this))\n                .fail(this.fail.bind(this));\n        },\n\n        /**\n         * {Function}\n         */\n        fail: function () {\n            fullScreenLoader.stopLoader();\n\n            return this;\n        },\n\n        /**\n         * {Function}\n         */\n        done: function () {\n            fullScreenLoader.stopLoader();\n            $('#multishipping-billing-form').submit();\n\n            return this;\n        }\n    });\n});\n","Magento_Braintree/js/view/payment/method-renderer/multishipping/cc-form.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n/*global define*/\n\ndefine([\n    'jquery',\n    'Magento_Braintree/js/view/payment/method-renderer/cc-form',\n    'Magento_Braintree/js/validator',\n    'Magento_Ui/js/model/messageList',\n    'mage/translate',\n    'Magento_Checkout/js/model/full-screen-loader',\n    'Magento_Checkout/js/action/set-payment-information-extended',\n    'Magento_Checkout/js/model/payment/additional-validators',\n    'Magento_Braintree/js/view/payment/validator-handler'\n], function (\n    $,\n    Component,\n    validator,\n    messageList,\n    $t,\n    fullScreenLoader,\n    setPaymentInformationExtended,\n    additionalValidators,\n    validatorManager\n) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Braintree/payment/multishipping/form'\n        },\n\n        /**\n         * @override\n         */\n        placeOrder: function () {\n            var self = this;\n\n            validatorManager.validate(self, function () {\n                return self.setPaymentInformation();\n            });\n        },\n\n        /**\n         * @override\n         */\n        setPaymentInformation: function () {\n            if (additionalValidators.validate()) {\n                fullScreenLoader.startLoader();\n                $.when(\n                    setPaymentInformationExtended(\n                        this.messageContainer,\n                        this.getData(),\n                        true\n                    )\n                ).done(this.done.bind(this))\n                    .fail(this.fail.bind(this));\n            }\n        },\n\n        /**\n         * {Function}\n         */\n        fail: function () {\n            fullScreenLoader.stopLoader();\n\n            return this;\n        },\n\n        /**\n         * {Function}\n         */\n        done: function () {\n            fullScreenLoader.stopLoader();\n            $('#multishipping-billing-form').submit();\n\n            return this;\n        }\n    });\n});\n","Magento_CardinalCommerce/js/cardinal-client.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'uiClass',\n    'Magento_CardinalCommerce/js/cardinal-factory',\n    'Magento_Checkout/js/model/quote',\n    'mage/translate'\n], function ($, Class, cardinalFactory, quote, $t) {\n    'use strict';\n\n    return {\n        /**\n         * Starts Cardinal Consumer Authentication\n         *\n         * @param {Object} cardData\n         * @return {jQuery.Deferred}\n         */\n        startAuthentication: function (cardData) {\n            var deferred = $.Deferred();\n\n            if (this.cardinalClient) {\n                this._startAuthentication(deferred, cardData);\n            } else {\n                cardinalFactory(this.getEnvironment())\n                    .done(function (client) {\n                        this.cardinalClient = client;\n                        this._startAuthentication(deferred, cardData);\n                    }.bind(this));\n            }\n\n            return deferred.promise();\n        },\n\n        /**\n         * Cardinal Consumer Authentication\n         *\n         * @param {jQuery.Deferred} deferred\n         * @param {Object} cardData\n         */\n        _startAuthentication: function (deferred, cardData) {\n            //this.cardinalClient.configure({ logging: { level: 'verbose' } });\n            this.cardinalClient.on('payments.validated', function (data, jwt) {\n                if (data.ErrorNumber !== 0) {\n                    deferred.reject(data.ErrorDescription);\n                } else if ($.inArray(data.ActionCode, ['FAILURE', 'ERROR']) !== -1) {\n                    deferred.reject($t('Authentication Failed. Please try again with another form of payment.'));\n                } else {\n                    deferred.resolve(jwt);\n                }\n                this.cardinalClient.off('payments.validated');\n            }.bind(this));\n\n            this.cardinalClient.on('payments.setupComplete', function () {\n                this.cardinalClient.start('cca', this.getRequestOrderObject(cardData));\n                this.cardinalClient.off('payments.setupComplete');\n            }.bind(this));\n\n            this.cardinalClient.setup('init', {\n                jwt: this.getRequestJWT()\n            });\n        },\n\n        /**\n         * Returns request order object.\n         *\n         * The request order object is structured object that is used to pass data\n         * to Cardinal that describes an order you'd like to process.\n         *\n         * If you pass a request object in both the JWT and the browser,\n         * Cardinal will merge the objects together where the browser overwrites\n         * the JWT object as it is considered the most recently captured data.\n         *\n         * @param {Object} cardData\n         * @returns {Object}\n         */\n        getRequestOrderObject: function (cardData) {\n            var totalAmount = quote.totals()['base_grand_total'],\n                currencyCode = quote.totals()['base_currency_code'],\n                billingAddress = quote.billingAddress(),\n                requestObject;\n\n            requestObject = {\n                OrderDetails: {\n                    Amount: totalAmount * 100,\n                    CurrencyCode: currencyCode\n                },\n                Consumer: {\n                    Account: {\n                        AccountNumber: cardData.accountNumber,\n                        ExpirationMonth: cardData.expMonth,\n                        ExpirationYear: cardData.expYear,\n                        CardCode: cardData.cardCode\n                    },\n                    BillingAddress: {\n                        FirstName: billingAddress.firstname,\n                        LastName: billingAddress.lastname,\n                        Address1: billingAddress.street[0],\n                        Address2: billingAddress.street[1],\n                        City: billingAddress.city,\n                        State: billingAddress.region,\n                        PostalCode: billingAddress.postcode,\n                        CountryCode: billingAddress.countryId,\n                        Phone1: billingAddress.telephone\n                    }\n                }\n            };\n\n            return requestObject;\n        },\n\n        /**\n         * Returns request JWT\n         * @returns {String}\n         */\n        getRequestJWT: function () {\n            return window.checkoutConfig.cardinal.requestJWT;\n        },\n\n        /**\n         * Returns type of environment\n         * @returns {String}\n         */\n        getEnvironment: function () {\n            return window.checkoutConfig.cardinal.environment;\n        }\n    };\n});\n","Magento_CardinalCommerce/js/cardinal-factory.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    return function (environment) {\n        var deferred = $.Deferred(),\n            dependency = 'cardinaljs';\n\n        if (environment === 'sandbox') {\n            dependency = 'cardinaljsSandbox';\n        }\n\n        require(\n            [dependency],\n            function (Cardinal) {\n                deferred.resolve(Cardinal);\n            },\n            deferred.reject\n        );\n\n        return deferred.promise();\n    };\n});\n","Magento_GoogleShoppingAds/js/gtag-cart.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/*global gtag*/\ndefine([\n    'jquery',\n    'Magento_Customer/js/customer-data',\n    'underscore',\n    'jquery/ui'\n], function ($, customerData, _) {\n    'use strict';\n\n    /**\n     * Delete cookie\n     *\n     * @param {String} name\n     */\n    function delCookie(name) {\n        var date = new Date(0);\n\n        document.cookie = name + '=' + '; path=/; expires=' + date.toUTCString();\n    }\n\n    $.widget('mage.sccGtagCart', {\n        options: {\n            cookieAddToCart: '',\n            cookieRemoveFromCart: ''\n        },\n\n        addedProducts: [],\n        removedProducts: [],\n\n        /**\n         * Format products array\n         *\n         * @param {Object} productsIn\n         *\n         * @private\n         */\n        _formatProductsArray: function (productsIn) {\n            var productsOut = [],\n                itemId,\n                i;\n\n            /* eslint-disable max-depth */\n            /* eslint-disable eqeqeq */\n            for (i in productsIn) {\n\n                if (i != 'length' && productsIn.hasOwnProperty(i)) {\n\n                    if (!_.isUndefined(productsIn[i].sku)) {\n                        itemId = productsIn[i].sku;\n                    } else {\n                        itemId = productsIn[i].id;\n                    }\n\n                    productsOut.push({\n                        'id': itemId,\n                        'name': productsIn[i].name,\n                        'price': productsIn[i].price,\n                        'quantity': parseInt(productsIn[i].qty, 10)\n                    });\n                }\n            }\n\n            /* eslint-enable max-depth */\n            /* eslint-enable eqeqeq */\n\n            return productsOut;\n        },\n\n        /**\n         * Cart item add action\n         *\n         * @private\n         */\n        _cartItemAdded: function () {\n            var products,\n                params,\n                i;\n\n            if (!this.addedProducts.length) {\n                return;\n            }\n            products = this._formatProductsArray(this.addedProducts);\n\n            for (i = 0; i < products.length; i++) {\n                params = {\n                    'ecomm_prodid': products[i].id,\n                    'ecomm_pagetype': 'cart'\n                };\n                gtag('event', 'add_to_cart', params);\n            }\n\n            this.addedProducts = [];\n        },\n\n        /**\n         * Cart item remove action\n         *\n         * @private\n         */\n        _cartItemRemoved: function () {\n            var products,\n                params,\n                i;\n\n            if (!this.removedProducts.length) {\n                return;\n            }\n            products = this._formatProductsArray(this.removedProducts);\n\n            for (i = 0; i < products.length; i++) {\n                params = {\n                    'ecomm_prodid': products[i].id,\n                    'ecomm_pagetype': 'cart'\n                };\n                gtag('event', 'remove_from_cart', params);\n            }\n\n            this.removedProducts = [];\n        },\n\n        /**\n         * Parse add from cart cookies.\n         *\n         * @private\n         */\n        _parseAddToCartCookies: function () {\n            var addProductsList;\n\n            if ($.mage.cookies.get(this.options.cookieAddToCart)) {\n                this.addedProducts = [];\n                addProductsList = decodeURIComponent($.mage.cookies.get(this.options.cookieAddToCart));\n                this.addedProducts = JSON.parse(addProductsList);\n                delCookie(this.options.cookieAddToCart);\n                this._cartItemAdded();\n            }\n        },\n\n        /**\n         * Parse remove from cart cookies.\n         *\n         * @private\n         */\n        _parseRemoveFromCartCookies: function () {\n            var removeProductsList;\n\n            if ($.mage.cookies.get(this.options.cookieRemoveFromCart)) {\n                this.removedProducts = [];\n                removeProductsList = decodeURIComponent($.mage.cookies.get(this.options.cookieRemoveFromCart));\n                this.removedProducts = JSON.parse(removeProductsList);\n                delCookie(this.options.cookieRemoveFromCart);\n                this._cartItemRemoved();\n            }\n        },\n\n        /**\n         * @inheritdoc\n         *\n         * @private\n         */\n        _create: function () {\n            var context = this;\n\n            setInterval(function () {\n                context._parseAddToCartCookies();\n                context._parseRemoveFromCartCookies();\n            }, 1000);\n        }\n    });\n\n    return $.mage.sccGtagCart;\n});\n","Magento_GoogleShoppingAds/js/gtag-configurable.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/*global gtag*/\ndefine([\n    'jquery',\n    'underscore'\n], function ($, _) {\n\n    $.widget('mage.sccGtagConfigurable', {\n        options: {\n            skuToOptions: {},\n            category: \"\"\n        },\n\n        /**\n         * Get currently selected options\n         *\n         * @private\n         */\n        _getSelectedOptionsArray: function () {\n            var optionsArray = {};\n            $('.super-attribute-select').each(function(key, element){\n                var optionVal = element.value;\n                var optionKey = element.name.replace(/\\D/g, '');\n                optionsArray[optionKey] = optionVal;\n            });\n            return optionsArray;\n         },\n\n        /**\n         * @inheritdoc\n         *\n         * @private\n         */\n        _create: function () {\n            var context = this;\n            $('body').on('change', 'input.super-attribute-select', function () {\n                var selectedOptions = context._getSelectedOptionsArray();\n                _.each(context.options.skuToOptions, function (options, sku) {\n                    if (_.isMatch(options, selectedOptions)) {\n                        gtag('event', 'view_item', {\n                            'ecomm_prodid': sku,\n                            'ecomm_pagetype': 'product',\n                            'ecomm_category': context.options.category\n                        });\n                    }\n                });\n            });\n        }\n    });\n\n    return $.mage.sccGtagConfigurable;\n});\n","Magento_GoogleAnalytics/js/google-analytics.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/* jscs:disable */\n/* eslint-disable */\ndefine([\n    'jquery',\n    'mage/cookies'\n], function ($) {\n    'use strict';\n\n    /**\n     * @param {Object} config\n     */\n    return function (config) {\n        var allowServices = false,\n            allowedCookies,\n            allowedWebsites;\n\n        if (config.isCookieRestrictionModeEnabled) {\n            allowedCookies = $.mage.cookies.get(config.cookieName);\n\n            if (allowedCookies !== null) {\n                allowedWebsites = JSON.parse(allowedCookies);\n\n                if (allowedWebsites[config.currentWebsite] === 1) {\n                    allowServices = true;\n                }\n            }\n        } else {\n            allowServices = true;\n        }\n\n        if (allowServices) {\n            (function (i, s, o, g, r, a, m) {\n                i.GoogleAnalyticsObject = r;\n                i[r] = i[r] || function () {\n                        (i[r].q = i[r].q || []).push(arguments)\n                    }, i[r].l = 1 * new Date();\n                a = s.createElement(o),\n                    m = s.getElementsByTagName(o)[0];\n                a.async = 1;\n                a.src = g;\n                m.parentNode.insertBefore(a, m)\n            })(window, document, 'script', '//www.google-analytics.com/analytics.js', 'ga');\n\n            // Process page info\n            ga('create', config.pageTrackingData.accountId, 'auto');\n\n            if (config.pageTrackingData.isAnonymizedIpActive) {\n                ga('set', 'anonymizeIp', true);\n            }\n\n            // Process orders data\n            if (config.ordersTrackingData.hasOwnProperty('currency')) {\n                ga('require', 'ec', 'ec.js');\n\n                ga('set', 'currencyCode', config.ordersTrackingData.currency);\n\n                // Collect product data for GA\n                if (config.ordersTrackingData.products) {\n                    $.each(config.ordersTrackingData.products, function (index, value) {\n                        ga('ec:addProduct', value);\n                    });\n                }\n\n                // Collect orders data for GA\n                if (config.ordersTrackingData.orders) {\n                    $.each(config.ordersTrackingData.orders, function (index, value) {\n                        ga('ec:setAction', 'purchase', value);\n                    });\n                }\n\n                ga('send', 'pageview');\n            } else {\n                // Process Data if not orders\n                ga('send', 'pageview' + config.pageTrackingData.optPageUrl);\n            }\n        }\n    }\n});\n","Magento_Swatches/js/catalog-add-to-cart.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\nrequire([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    /**\n     * Add selected swatch attributes to redirect url\n     *\n     * @see Magento_Catalog/js/catalog-add-to-cart\n     */\n    $('body').on('catalogCategoryAddToCartRedirect', function (event, data) {\n        $(data.form).find('[name*=\"super\"]').each(function (index, item) {\n            var $item = $(item),\n                attr;\n\n            if ($item.attr('data-attr-name')) {\n                attr = $item.attr('data-attr-name');\n            } else {\n                attr = $item.parent().attr('attribute-code');\n            }\n            data.redirectParameters.push(attr + '=' + $item.val());\n\n        });\n    });\n});\n","Magento_Swatches/js/swatch-renderer.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'mage/template',\n    'mage/smart-keyboard-handler',\n    'mage/translate',\n    'priceUtils',\n    'jquery-ui-modules/widget',\n    'jquery/jquery.parsequery',\n    'mage/validation/validation'\n], function ($, _, mageTemplate, keyboardHandler, $t, priceUtils) {\n    'use strict';\n\n    /**\n     * Extend form validation to support swatch accessibility\n     */\n    $.widget('mage.validation', $.mage.validation, {\n        /**\n         * Handle form with swatches validation. Focus on first invalid swatch block.\n         *\n         * @param {jQuery.Event} event\n         * @param {Object} validation\n         */\n        listenFormValidateHandler: function (event, validation) {\n            var swatchWrapper, firstActive, swatches, swatch, successList, errorList, firstSwatch;\n\n            this._superApply(arguments);\n\n            swatchWrapper = '.swatch-attribute-options';\n            swatches = $(event.target).find(swatchWrapper);\n\n            if (!swatches.length) {\n                return;\n            }\n\n            swatch = '.swatch-attribute';\n            firstActive = $(validation.errorList[0].element || []);\n            successList = validation.successList;\n            errorList = validation.errorList;\n            firstSwatch = $(firstActive).parent(swatch).find(swatchWrapper);\n\n            keyboardHandler.focus(swatches);\n\n            $.each(successList, function (index, item) {\n                $(item).parent(swatch).find(swatchWrapper).attr('aria-invalid', false);\n            });\n\n            $.each(errorList, function (index, item) {\n                $(item.element).parent(swatch).find(swatchWrapper).attr('aria-invalid', true);\n            });\n\n            if (firstSwatch.length) {\n                $(firstSwatch).focus();\n            }\n        }\n    });\n\n    /**\n     * Render tooltips by attributes (only to up).\n     * Required element attributes:\n     *  - option-type (integer, 0-3)\n     *  - option-label (string)\n     *  - option-tooltip-thumb\n     *  - option-tooltip-value\n     *  - thumb-width\n     *  - thumb-height\n     */\n    $.widget('mage.SwatchRendererTooltip', {\n        options: {\n            delay: 200,                             //how much ms before tooltip to show\n            tooltipClass: 'swatch-option-tooltip'  //configurable, but remember about css\n        },\n\n        /**\n         * @private\n         */\n        _init: function () {\n            var $widget = this,\n                $this = this.element,\n                $element = $('.' + $widget.options.tooltipClass),\n                timer,\n                type = parseInt($this.attr('option-type'), 10),\n                label = $this.attr('option-label'),\n                thumb = $this.attr('option-tooltip-thumb'),\n                value = $this.attr('option-tooltip-value'),\n                width = $this.attr('thumb-width'),\n                height = $this.attr('thumb-height'),\n                $image,\n                $title,\n                $corner;\n\n            if (!$element.length) {\n                $element = $('<div class=\"' +\n                    $widget.options.tooltipClass +\n                    '\"><div class=\"image\"></div><div class=\"title\"></div><div class=\"corner\"></div></div>'\n                );\n                $('body').append($element);\n            }\n\n            $image = $element.find('.image');\n            $title = $element.find('.title');\n            $corner = $element.find('.corner');\n\n            $this.hover(function () {\n                if (!$this.hasClass('disabled')) {\n                    timer = setTimeout(\n                        function () {\n                            var leftOpt = null,\n                                leftCorner = 0,\n                                left,\n                                $window;\n\n                            if (type === 2) {\n                                // Image\n                                $image.css({\n                                    'background': 'url(\"' + thumb + '\") no-repeat center', //Background case\n                                    'background-size': 'initial',\n                                    'width': width + 'px',\n                                    'height': height + 'px'\n                                });\n                                $image.show();\n                            } else if (type === 1) {\n                                // Color\n                                $image.css({\n                                    background: value\n                                });\n                                $image.show();\n                            } else if (type === 0 || type === 3) {\n                                // Default\n                                $image.hide();\n                            }\n\n                            $title.text(label);\n\n                            leftOpt = $this.offset().left;\n                            left = leftOpt + $this.width() / 2 - $element.width() / 2;\n                            $window = $(window);\n\n                            // the numbers (5 and 5) is magick constants for offset from left or right page\n                            if (left < 0) {\n                                left = 5;\n                            } else if (left + $element.width() > $window.width()) {\n                                left = $window.width() - $element.width() - 5;\n                            }\n\n                            // the numbers (6,  3 and 18) is magick constants for offset tooltip\n                            leftCorner = 0;\n\n                            if ($element.width() < $this.width()) {\n                                leftCorner = $element.width() / 2 - 3;\n                            } else {\n                                leftCorner = (leftOpt > left ? leftOpt - left : left - leftOpt) + $this.width() / 2 - 6;\n                            }\n\n                            $corner.css({\n                                left: leftCorner\n                            });\n                            $element.css({\n                                left: left,\n                                top: $this.offset().top - $element.height() - $corner.height() - 18\n                            }).show();\n                        },\n                        $widget.options.delay\n                    );\n                }\n            }, function () {\n                $element.hide();\n                clearTimeout(timer);\n            });\n\n            $(document).on('tap', function () {\n                $element.hide();\n                clearTimeout(timer);\n            });\n\n            $this.on('tap', function (event) {\n                event.stopPropagation();\n            });\n        }\n    });\n\n    /**\n     * Render swatch controls with options and use tooltips.\n     * Required two json:\n     *  - jsonConfig (magento's option config)\n     *  - jsonSwatchConfig (swatch's option config)\n     *\n     *  Tuning:\n     *  - numberToShow (show \"more\" button if options are more)\n     *  - onlySwatches (hide selectboxes)\n     *  - moreButtonText (text for \"more\" button)\n     *  - selectorProduct (selector for product container)\n     *  - selectorProductPrice (selector for change price)\n     */\n    $.widget('mage.SwatchRenderer', {\n        options: {\n            classes: {\n                attributeClass: 'swatch-attribute',\n                attributeLabelClass: 'swatch-attribute-label',\n                attributeSelectedOptionLabelClass: 'swatch-attribute-selected-option',\n                attributeOptionsWrapper: 'swatch-attribute-options',\n                attributeInput: 'swatch-input',\n                optionClass: 'swatch-option',\n                selectClass: 'swatch-select',\n                moreButton: 'swatch-more',\n                loader: 'swatch-option-loading'\n            },\n            // option's json config\n            jsonConfig: {},\n\n            // swatch's json config\n            jsonSwatchConfig: {},\n\n            // selector of parental block of prices and swatches (need to know where to seek for price block)\n            selectorProduct: '.product-info-main',\n\n            // selector of price wrapper (need to know where set price)\n            selectorProductPrice: '[data-role=priceBox]',\n\n            //selector of product images gallery wrapper\n            mediaGallerySelector: '[data-gallery-role=gallery-placeholder]',\n\n            // selector of category product tile wrapper\n            selectorProductTile: '.product-item',\n\n            // number of controls to show (false or zero = show all)\n            numberToShow: false,\n\n            // show only swatch controls\n            onlySwatches: false,\n\n            // enable label for control\n            enableControlLabel: true,\n\n            // control label id\n            controlLabelId: '',\n\n            // text for more button\n            moreButtonText: $t('More'),\n\n            // Callback url for media\n            mediaCallback: '',\n\n            // Local media cache\n            mediaCache: {},\n\n            // Cache for BaseProduct images. Needed when option unset\n            mediaGalleryInitial: [{}],\n\n            // Use ajax to get image data\n            useAjax: false,\n\n            /**\n             * Defines the mechanism of how images of a gallery should be\n             * updated when user switches between configurations of a product.\n             *\n             * As for now value of this option can be either 'replace' or 'prepend'.\n             *\n             * @type {String}\n             */\n            gallerySwitchStrategy: 'replace',\n\n            // whether swatches are rendered in product list or on product page\n            inProductList: false,\n\n            // sly-old-price block selector\n            slyOldPriceSelector: '.sly-old-price',\n\n            // tier prise selectors start\n            tierPriceTemplateSelector: '#tier-prices-template',\n            tierPriceBlockSelector: '[data-role=\"tier-price-block\"]',\n            tierPriceTemplate: '',\n            // tier prise selectors end\n\n            // A price label selector\n            normalPriceLabelSelector: '.normal-price .price-label'\n        },\n\n        /**\n         * Get chosen product\n         *\n         * @returns int|null\n         */\n        getProduct: function () {\n            var products = this._CalcProducts();\n\n            return _.isArray(products) ? products[0] : null;\n        },\n\n        /**\n         * @private\n         */\n        _init: function () {\n            if (_.isEmpty(this.options.jsonConfig.images)) {\n                this.options.useAjax = true;\n                // creates debounced variant of _LoadProductMedia()\n                // to use it in events handlers instead of _LoadProductMedia()\n                this._debouncedLoadProductMedia = _.debounce(this._LoadProductMedia.bind(this), 500);\n            }\n\n            if (this.options.jsonConfig !== '' && this.options.jsonSwatchConfig !== '') {\n                // store unsorted attributes\n                this.options.jsonConfig.mappedAttributes = _.clone(this.options.jsonConfig.attributes);\n                this._sortAttributes();\n                this._RenderControls();\n                this._setPreSelectedGallery();\n                $(this.element).trigger('swatch.initialized');\n            } else {\n                console.log('SwatchRenderer: No input data received');\n            }\n            this.options.tierPriceTemplate = $(this.options.tierPriceTemplateSelector).html();\n        },\n\n        /**\n         * @private\n         */\n        _sortAttributes: function () {\n            this.options.jsonConfig.attributes = _.sortBy(this.options.jsonConfig.attributes, function (attribute) {\n                return parseInt(attribute.position, 10);\n            });\n        },\n\n        /**\n         * @private\n         */\n        _create: function () {\n            var options = this.options,\n                gallery = $('[data-gallery-role=gallery-placeholder]', '.column.main'),\n                productData = this._determineProductData(),\n                $main = productData.isInProductView ?\n                    this.element.parents('.column.main') :\n                    this.element.parents('.product-item-info');\n\n            if (productData.isInProductView) {\n                gallery.data('gallery') ?\n                    this._onGalleryLoaded(gallery) :\n                    gallery.on('gallery:loaded', this._onGalleryLoaded.bind(this, gallery));\n            } else {\n                options.mediaGalleryInitial = [{\n                    'img': $main.find('.product-image-photo').attr('src')\n                }];\n            }\n\n            this.productForm = this.element.parents(this.options.selectorProductTile).find('form:first');\n            this.inProductList = this.productForm.length > 0;\n        },\n\n        /**\n         * Determine product id and related data\n         *\n         * @returns {{productId: *, isInProductView: bool}}\n         * @private\n         */\n        _determineProductData: function () {\n            // Check if product is in a list of products.\n            var productId,\n                isInProductView = false;\n\n            productId = this.element.parents('.product-item-details')\n                    .find('.price-box.price-final_price').attr('data-product-id');\n\n            if (!productId) {\n                // Check individual product.\n                productId = $('[name=product]').val();\n                isInProductView = productId > 0;\n            }\n\n            return {\n                productId: productId,\n                isInProductView: isInProductView\n            };\n        },\n\n        /**\n         * Render controls\n         *\n         * @private\n         */\n        _RenderControls: function () {\n            var $widget = this,\n                container = this.element,\n                classes = this.options.classes,\n                chooseText = this.options.jsonConfig.chooseText;\n\n            $widget.optionsMap = {};\n\n            $.each(this.options.jsonConfig.attributes, function () {\n                var item = this,\n                    controlLabelId = 'option-label-' + item.code + '-' + item.id,\n                    options = $widget._RenderSwatchOptions(item, controlLabelId),\n                    select = $widget._RenderSwatchSelect(item, chooseText),\n                    input = $widget._RenderFormInput(item),\n                    listLabel = '',\n                    label = '';\n\n                // Show only swatch controls\n                if ($widget.options.onlySwatches && !$widget.options.jsonSwatchConfig.hasOwnProperty(item.id)) {\n                    return;\n                }\n\n                if ($widget.options.enableControlLabel) {\n                    label +=\n                        '<span id=\"' + controlLabelId + '\" class=\"' + classes.attributeLabelClass + '\">' +\n                        $('<i></i>').text(item.label).html() +\n                        '</span>' +\n                        '<span class=\"' + classes.attributeSelectedOptionLabelClass + '\"></span>';\n                }\n\n                if ($widget.inProductList) {\n                    $widget.productForm.append(input);\n                    input = '';\n                    listLabel = 'aria-label=\"' + $('<i></i>').text(item.label).html() + '\"';\n                } else {\n                    listLabel = 'aria-labelledby=\"' + controlLabelId + '\"';\n                }\n\n                // Create new control\n                container.append(\n                    '<div class=\"' + classes.attributeClass + ' ' + item.code + '\" ' +\n                         'attribute-code=\"' + item.code + '\" ' +\n                         'attribute-id=\"' + item.id + '\">' +\n                        label +\n                        '<div aria-activedescendant=\"\" ' +\n                             'tabindex=\"0\" ' +\n                             'aria-invalid=\"false\" ' +\n                             'aria-required=\"true\" ' +\n                             'role=\"listbox\" ' + listLabel +\n                             'class=\"' + classes.attributeOptionsWrapper + ' clearfix\">' +\n                            options + select +\n                        '</div>' + input +\n                    '</div>'\n                );\n\n                $widget.optionsMap[item.id] = {};\n\n                // Aggregate options array to hash (key => value)\n                $.each(item.options, function () {\n                    if (this.products.length > 0) {\n                        $widget.optionsMap[item.id][this.id] = {\n                            price: parseInt(\n                                $widget.options.jsonConfig.optionPrices[this.products[0]].finalPrice.amount,\n                                10\n                            ),\n                            products: this.products\n                        };\n                    }\n                });\n            });\n\n            // Connect Tooltip\n            container\n                .find('[option-type=\"1\"], [option-type=\"2\"], [option-type=\"0\"], [option-type=\"3\"]')\n                .SwatchRendererTooltip();\n\n            // Hide all elements below more button\n            $('.' + classes.moreButton).nextAll().hide();\n\n            // Handle events like click or change\n            $widget._EventListener();\n\n            // Rewind options\n            $widget._Rewind(container);\n\n            //Emulate click on all swatches from Request\n            $widget._EmulateSelected($.parseQuery());\n            $widget._EmulateSelected($widget._getSelectedAttributes());\n        },\n\n        /**\n         * Render swatch options by part of config\n         *\n         * @param {Object} config\n         * @param {String} controlId\n         * @returns {String}\n         * @private\n         */\n        _RenderSwatchOptions: function (config, controlId) {\n            var optionConfig = this.options.jsonSwatchConfig[config.id],\n                optionClass = this.options.classes.optionClass,\n                sizeConfig = this.options.jsonSwatchImageSizeConfig,\n                moreLimit = parseInt(this.options.numberToShow, 10),\n                moreClass = this.options.classes.moreButton,\n                moreText = this.options.moreButtonText,\n                countAttributes = 0,\n                html = '';\n\n            if (!this.options.jsonSwatchConfig.hasOwnProperty(config.id)) {\n                return '';\n            }\n\n            $.each(config.options, function (index) {\n                var id,\n                    type,\n                    value,\n                    thumb,\n                    label,\n                    width,\n                    height,\n                    attr,\n                    swatchImageWidth,\n                    swatchImageHeight;\n\n                if (!optionConfig.hasOwnProperty(this.id)) {\n                    return '';\n                }\n\n                // Add more button\n                if (moreLimit === countAttributes++) {\n                    html += '<a href=\"#\" class=\"' + moreClass + '\"><span>' + moreText + '</span></a>';\n                }\n\n                id = this.id;\n                type = parseInt(optionConfig[id].type, 10);\n                value = optionConfig[id].hasOwnProperty('value') ?\n                    $('<i></i>').text(optionConfig[id].value).html() : '';\n                thumb = optionConfig[id].hasOwnProperty('thumb') ? optionConfig[id].thumb : '';\n                width = _.has(sizeConfig, 'swatchThumb') ? sizeConfig.swatchThumb.width : 110;\n                height = _.has(sizeConfig, 'swatchThumb') ? sizeConfig.swatchThumb.height : 90;\n                label = this.label ? $('<i></i>').text(this.label).html() : '';\n                attr =\n                    ' id=\"' + controlId + '-item-' + id + '\"' +\n                    ' index=\"' + index + '\"' +\n                    ' aria-checked=\"false\"' +\n                    ' aria-describedby=\"' + controlId + '\"' +\n                    ' tabindex=\"0\"' +\n                    ' option-type=\"' + type + '\"' +\n                    ' option-id=\"' + id + '\"' +\n                    ' option-label=\"' + label + '\"' +\n                    ' aria-label=\"' + label + '\"' +\n                    ' option-tooltip-thumb=\"' + thumb + '\"' +\n                    ' option-tooltip-value=\"' + value + '\"' +\n                    ' role=\"option\"' +\n                    ' thumb-width=\"' + width + '\"' +\n                    ' thumb-height=\"' + height + '\"';\n\n                swatchImageWidth = _.has(sizeConfig, 'swatchImage') ? sizeConfig.swatchImage.width : 30;\n                swatchImageHeight = _.has(sizeConfig, 'swatchImage') ? sizeConfig.swatchImage.height : 20;\n\n                if (!this.hasOwnProperty('products') || this.products.length <= 0) {\n                    attr += ' option-empty=\"true\"';\n                }\n\n                if (type === 0) {\n                    // Text\n                    html += '<div class=\"' + optionClass + ' text\" ' + attr + '>' + (value ? value : label) +\n                        '</div>';\n                } else if (type === 1) {\n                    // Color\n                    html += '<div class=\"' + optionClass + ' color\" ' + attr +\n                        ' style=\"background: ' + value +\n                        ' no-repeat center; background-size: initial;\">' + '' +\n                        '</div>';\n                } else if (type === 2) {\n                    // Image\n                    html += '<div class=\"' + optionClass + ' image\" ' + attr +\n                        ' style=\"background: url(' + value + ') no-repeat center; background-size: initial;width:' +\n                        swatchImageWidth + 'px; height:' + swatchImageHeight + 'px\">' + '' +\n                        '</div>';\n                } else if (type === 3) {\n                    // Clear\n                    html += '<div class=\"' + optionClass + '\" ' + attr + '></div>';\n                } else {\n                    // Default\n                    html += '<div class=\"' + optionClass + '\" ' + attr + '>' + label + '</div>';\n                }\n            });\n\n            return html;\n        },\n\n        /**\n         * Render select by part of config\n         *\n         * @param {Object} config\n         * @param {String} chooseText\n         * @returns {String}\n         * @private\n         */\n        _RenderSwatchSelect: function (config, chooseText) {\n            var html;\n\n            if (this.options.jsonSwatchConfig.hasOwnProperty(config.id)) {\n                return '';\n            }\n\n            html =\n                '<select class=\"' + this.options.classes.selectClass + ' ' + config.code + '\">' +\n                '<option value=\"0\" option-id=\"0\">' + chooseText + '</option>';\n\n            $.each(config.options, function () {\n                var label = this.label,\n                    attr = ' value=\"' + this.id + '\" option-id=\"' + this.id + '\"';\n\n                if (!this.hasOwnProperty('products') || this.products.length <= 0) {\n                    attr += ' option-empty=\"true\"';\n                }\n\n                html += '<option ' + attr + '>' + label + '</option>';\n            });\n\n            html += '</select>';\n\n            return html;\n        },\n\n        /**\n         * Input for submit form.\n         * This control shouldn't have \"type=hidden\", \"display: none\" for validation work :(\n         *\n         * @param {Object} config\n         * @private\n         */\n        _RenderFormInput: function (config) {\n            return '<input class=\"' + this.options.classes.attributeInput + ' super-attribute-select\" ' +\n                'name=\"super_attribute[' + config.id + ']\" ' +\n                'type=\"text\" ' +\n                'value=\"\" ' +\n                'data-selector=\"super_attribute[' + config.id + ']\" ' +\n                'data-validate=\"{required: true}\" ' +\n                'aria-required=\"true\" ' +\n                'aria-invalid=\"false\">';\n        },\n\n        /**\n         * Event listener\n         *\n         * @private\n         */\n        _EventListener: function () {\n            var $widget = this,\n                options = this.options.classes,\n                target;\n\n            $widget.element.on('click', '.' + options.optionClass, function () {\n                return $widget._OnClick($(this), $widget);\n            });\n\n            $widget.element.on('change', '.' + options.selectClass, function () {\n                return $widget._OnChange($(this), $widget);\n            });\n\n            $widget.element.on('click', '.' + options.moreButton, function (e) {\n                e.preventDefault();\n\n                return $widget._OnMoreClick($(this));\n            });\n\n            $widget.element.on('keydown', function (e) {\n                if (e.which === 13) {\n                    target = $(e.target);\n\n                    if (target.is('.' + options.optionClass)) {\n                        return $widget._OnClick(target, $widget);\n                    } else if (target.is('.' + options.selectClass)) {\n                        return $widget._OnChange(target, $widget);\n                    } else if (target.is('.' + options.moreButton)) {\n                        e.preventDefault();\n\n                        return $widget._OnMoreClick(target);\n                    }\n                }\n            });\n        },\n\n        /**\n         * Load media gallery using ajax or json config.\n         *\n         * @private\n         */\n        _loadMedia: function () {\n            var $main = this.inProductList ?\n                    this.element.parents('.product-item-info') :\n                    this.element.parents('.column.main'),\n                images;\n\n            if (this.options.useAjax) {\n                this._debouncedLoadProductMedia();\n            }  else {\n                images = this.options.jsonConfig.images[this.getProduct()];\n\n                if (!images) {\n                    images = this.options.mediaGalleryInitial;\n                }\n                this.updateBaseImage(this._sortImages(images), $main, !this.inProductList);\n            }\n        },\n\n        /**\n         * Sorting images array\n         *\n         * @private\n         */\n        _sortImages: function (images) {\n            return _.sortBy(images, function (image) {\n                return parseInt(image.position, 10);\n            });\n        },\n\n        /**\n         * Event for swatch options\n         *\n         * @param {Object} $this\n         * @param {Object} $widget\n         * @private\n         */\n        _OnClick: function ($this, $widget) {\n            var $parent = $this.parents('.' + $widget.options.classes.attributeClass),\n                $wrapper = $this.parents('.' + $widget.options.classes.attributeOptionsWrapper),\n                $label = $parent.find('.' + $widget.options.classes.attributeSelectedOptionLabelClass),\n                attributeId = $parent.attr('attribute-id'),\n                $input = $parent.find('.' + $widget.options.classes.attributeInput),\n                checkAdditionalData = JSON.parse(this.options.jsonSwatchConfig[attributeId]['additional_data']);\n\n            if ($widget.inProductList) {\n                $input = $widget.productForm.find(\n                    '.' + $widget.options.classes.attributeInput + '[name=\"super_attribute[' + attributeId + ']\"]'\n                );\n            }\n\n            if ($this.hasClass('disabled')) {\n                return;\n            }\n\n            if ($this.hasClass('selected')) {\n                $parent.removeAttr('option-selected').find('.selected').removeClass('selected');\n                $input.val('');\n                $label.text('');\n                $this.attr('aria-checked', false);\n            } else {\n                $parent.attr('option-selected', $this.attr('option-id')).find('.selected').removeClass('selected');\n                $label.text($this.attr('option-label'));\n                $input.val($this.attr('option-id'));\n                $input.attr('data-attr-name', this._getAttributeCodeById(attributeId));\n                $this.addClass('selected');\n                $widget._toggleCheckedAttributes($this, $wrapper);\n            }\n\n            $widget._Rebuild();\n\n            if ($widget.element.parents($widget.options.selectorProduct)\n                    .find(this.options.selectorProductPrice).is(':data(mage-priceBox)')\n            ) {\n                $widget._UpdatePrice();\n            }\n\n            $(document).trigger('updateMsrpPriceBlock',\n                [\n                    parseInt($this.attr('index'), 10) + 1,\n                    $widget.options.jsonConfig.optionPrices\n                ]);\n\n            if (checkAdditionalData['update_product_preview_image'] === '1') {\n                $widget._loadMedia();\n            }\n\n            $input.trigger('change');\n        },\n\n        /**\n         * Get human readable attribute code (eg. size, color) by it ID from configuration\n         *\n         * @param {Number} attributeId\n         * @returns {*}\n         * @private\n         */\n        _getAttributeCodeById: function (attributeId) {\n            var attribute = this.options.jsonConfig.mappedAttributes[attributeId];\n\n            return attribute ? attribute.code : attributeId;\n        },\n\n        /**\n         * Toggle accessibility attributes\n         *\n         * @param {Object} $this\n         * @param {Object} $wrapper\n         * @private\n         */\n        _toggleCheckedAttributes: function ($this, $wrapper) {\n            $wrapper.attr('aria-activedescendant', $this.attr('id'))\n                    .find('.' + this.options.classes.optionClass).attr('aria-checked', false);\n            $this.attr('aria-checked', true);\n        },\n\n        /**\n         * Event for select\n         *\n         * @param {Object} $this\n         * @param {Object} $widget\n         * @private\n         */\n        _OnChange: function ($this, $widget) {\n            var $parent = $this.parents('.' + $widget.options.classes.attributeClass),\n                attributeId = $parent.attr('attribute-id'),\n                $input = $parent.find('.' + $widget.options.classes.attributeInput);\n\n            if ($widget.productForm.length > 0) {\n                $input = $widget.productForm.find(\n                    '.' + $widget.options.classes.attributeInput + '[name=\"super_attribute[' + attributeId + ']\"]'\n                );\n            }\n\n            if ($this.val() > 0) {\n                $parent.attr('option-selected', $this.val());\n                $input.val($this.val());\n            } else {\n                $parent.removeAttr('option-selected');\n                $input.val('');\n            }\n\n            $widget._Rebuild();\n            $widget._UpdatePrice();\n            $widget._loadMedia();\n            $input.trigger('change');\n        },\n\n        /**\n         * Event for more switcher\n         *\n         * @param {Object} $this\n         * @private\n         */\n        _OnMoreClick: function ($this) {\n            $this.nextAll().show();\n            $this.blur().remove();\n        },\n\n        /**\n         * Rewind options for controls\n         *\n         * @private\n         */\n        _Rewind: function (controls) {\n            controls.find('div[option-id], option[option-id]').removeClass('disabled').removeAttr('disabled');\n            controls.find('div[option-empty], option[option-empty]').attr('disabled', true).addClass('disabled');\n        },\n\n        /**\n         * Rebuild container\n         *\n         * @private\n         */\n        _Rebuild: function () {\n            var $widget = this,\n                controls = $widget.element.find('.' + $widget.options.classes.attributeClass + '[attribute-id]'),\n                selected = controls.filter('[option-selected]');\n\n            // Enable all options\n            $widget._Rewind(controls);\n\n            // done if nothing selected\n            if (selected.length <= 0) {\n                return;\n            }\n\n            // Disable not available options\n            controls.each(function () {\n                var $this = $(this),\n                    id = $this.attr('attribute-id'),\n                    products = $widget._CalcProducts(id);\n\n                if (selected.length === 1 && selected.first().attr('attribute-id') === id) {\n                    return;\n                }\n\n                $this.find('[option-id]').each(function () {\n                    var $element = $(this),\n                        option = $element.attr('option-id');\n\n                    if (!$widget.optionsMap.hasOwnProperty(id) || !$widget.optionsMap[id].hasOwnProperty(option) ||\n                        $element.hasClass('selected') ||\n                        $element.is(':selected')) {\n                        return;\n                    }\n\n                    if (_.intersection(products, $widget.optionsMap[id][option].products).length <= 0) {\n                        $element.attr('disabled', true).addClass('disabled');\n                    }\n                });\n            });\n        },\n\n        /**\n         * Get selected product list\n         *\n         * @returns {Array}\n         * @private\n         */\n        _CalcProducts: function ($skipAttributeId) {\n            var $widget = this,\n                products = [];\n\n            // Generate intersection of products\n            $widget.element.find('.' + $widget.options.classes.attributeClass + '[option-selected]').each(function () {\n                var id = $(this).attr('attribute-id'),\n                    option = $(this).attr('option-selected');\n\n                if ($skipAttributeId !== undefined && $skipAttributeId === id) {\n                    return;\n                }\n\n                if (!$widget.optionsMap.hasOwnProperty(id) || !$widget.optionsMap[id].hasOwnProperty(option)) {\n                    return;\n                }\n\n                if (products.length === 0) {\n                    products = $widget.optionsMap[id][option].products;\n                } else {\n                    products = _.intersection(products, $widget.optionsMap[id][option].products);\n                }\n            });\n\n            return products;\n        },\n\n        /**\n         * Update total price\n         *\n         * @private\n         */\n        _UpdatePrice: function () {\n            var $widget = this,\n                $product = $widget.element.parents($widget.options.selectorProduct),\n                $productPrice = $product.find(this.options.selectorProductPrice),\n                options = _.object(_.keys($widget.optionsMap), {}),\n                result,\n                tierPriceHtml,\n                isShow;\n\n            $widget.element.find('.' + $widget.options.classes.attributeClass + '[option-selected]').each(function () {\n                var attributeId = $(this).attr('attribute-id');\n\n                options[attributeId] = $(this).attr('option-selected');\n            });\n\n            result = $widget.options.jsonConfig.optionPrices[_.findKey($widget.options.jsonConfig.index, options)];\n\n            $productPrice.trigger(\n                'updatePrice',\n                {\n                    'prices': $widget._getPrices(result, $productPrice.priceBox('option').prices)\n                }\n            );\n\n            isShow = typeof result != 'undefined' && result.oldPrice.amount !== result.finalPrice.amount;\n\n            $product.find(this.options.slyOldPriceSelector)[isShow ? 'show' : 'hide']();\n\n            if (typeof result != 'undefined' && result.tierPrices.length) {\n                if (this.options.tierPriceTemplate) {\n                    tierPriceHtml = mageTemplate(\n                        this.options.tierPriceTemplate,\n                        {\n                            'tierPrices': result.tierPrices,\n                            '$t': $t,\n                            'currencyFormat': this.options.jsonConfig.currencyFormat,\n                            'priceUtils': priceUtils\n                        }\n                    );\n                    $(this.options.tierPriceBlockSelector).html(tierPriceHtml).show();\n                }\n            } else {\n                $(this.options.tierPriceBlockSelector).hide();\n            }\n\n            $(this.options.normalPriceLabelSelector).hide();\n\n            _.each($('.' + this.options.classes.attributeOptionsWrapper), function (attribute) {\n                if ($(attribute).find('.' + this.options.classes.optionClass + '.selected').length === 0) {\n                    if ($(attribute).find('.' + this.options.classes.selectClass).length > 0) {\n                        _.each($(attribute).find('.' + this.options.classes.selectClass), function (dropdown) {\n                            if ($(dropdown).val() === '0') {\n                                $(this.options.normalPriceLabelSelector).show();\n                            }\n                        }.bind(this));\n                    } else {\n                        $(this.options.normalPriceLabelSelector).show();\n                    }\n                }\n            }.bind(this));\n        },\n\n        /**\n         * Get prices\n         *\n         * @param {Object} newPrices\n         * @param {Object} displayPrices\n         * @returns {*}\n         * @private\n         */\n        _getPrices: function (newPrices, displayPrices) {\n            var $widget = this,\n                optionPriceDiff = 0,\n                allowedProduct, optionPrices, basePrice, optionFinalPrice;\n\n            if (_.isEmpty(newPrices)) {\n                allowedProduct = this._getAllowedProductWithMinPrice(this._CalcProducts());\n                optionPrices = this.options.jsonConfig.optionPrices;\n                basePrice = parseFloat(this.options.jsonConfig.prices.basePrice.amount);\n\n                if (!_.isEmpty(allowedProduct)) {\n                    optionFinalPrice = parseFloat(optionPrices[allowedProduct].finalPrice.amount);\n                    optionPriceDiff = optionFinalPrice - basePrice;\n                }\n\n                if (optionPriceDiff !== 0) {\n                    newPrices  = this.options.jsonConfig.optionPrices[allowedProduct];\n                } else {\n                    newPrices = $widget.options.jsonConfig.prices;\n                }\n            }\n\n            _.each(displayPrices, function (price, code) {\n\n                if (newPrices[code]) {\n                    displayPrices[code].amount = newPrices[code].amount - displayPrices[code].amount;\n                }\n            });\n\n            return displayPrices;\n        },\n\n        /**\n         * Get product with minimum price from selected options.\n         *\n         * @param {Array} allowedProducts\n         * @returns {String}\n         * @private\n         */\n        _getAllowedProductWithMinPrice: function (allowedProducts) {\n            var optionPrices = this.options.jsonConfig.optionPrices,\n                product = {},\n                optionFinalPrice, optionMinPrice;\n\n            _.each(allowedProducts, function (allowedProduct) {\n                optionFinalPrice = parseFloat(optionPrices[allowedProduct].finalPrice.amount);\n\n                if (_.isEmpty(product) || optionFinalPrice < optionMinPrice) {\n                    optionMinPrice = optionFinalPrice;\n                    product = allowedProduct;\n                }\n            }, this);\n\n            return product;\n        },\n\n        /**\n         * Gets all product media and change current to the needed one\n         *\n         * @private\n         */\n        _LoadProductMedia: function () {\n            var $widget = this,\n                $this = $widget.element,\n                productData = this._determineProductData(),\n                mediaCallData,\n                mediaCacheKey,\n\n                /**\n                 * Processes product media data\n                 *\n                 * @param {Object} data\n                 * @returns void\n                 */\n                mediaSuccessCallback = function (data) {\n                    if (!(mediaCacheKey in $widget.options.mediaCache)) {\n                        $widget.options.mediaCache[mediaCacheKey] = data;\n                    }\n                    $widget._ProductMediaCallback($this, data, productData.isInProductView);\n                    setTimeout(function () {\n                        $widget._DisableProductMediaLoader($this);\n                    }, 300);\n                };\n\n            if (!$widget.options.mediaCallback) {\n                return;\n            }\n\n            mediaCallData = {\n                'product_id': this.getProduct()\n            };\n\n            mediaCacheKey = JSON.stringify(mediaCallData);\n\n            if (mediaCacheKey in $widget.options.mediaCache) {\n                $widget._XhrKiller();\n                $widget._EnableProductMediaLoader($this);\n                mediaSuccessCallback($widget.options.mediaCache[mediaCacheKey]);\n            } else {\n                mediaCallData.isAjax = true;\n                $widget._XhrKiller();\n                $widget._EnableProductMediaLoader($this);\n                $widget.xhr = $.ajax({\n                    url: $widget.options.mediaCallback,\n                    cache: true,\n                    type: 'GET',\n                    dataType: 'json',\n                    data: mediaCallData,\n                    success: mediaSuccessCallback\n                }).done(function () {\n                    $widget._XhrKiller();\n                });\n            }\n        },\n\n        /**\n         * Enable loader\n         *\n         * @param {Object} $this\n         * @private\n         */\n        _EnableProductMediaLoader: function ($this) {\n            var $widget = this;\n\n            if ($('body.catalog-product-view').length > 0) {\n                $this.parents('.column.main').find('.photo.image')\n                    .addClass($widget.options.classes.loader);\n            } else {\n                //Category View\n                $this.parents('.product-item-info').find('.product-image-photo')\n                    .addClass($widget.options.classes.loader);\n            }\n        },\n\n        /**\n         * Disable loader\n         *\n         * @param {Object} $this\n         * @private\n         */\n        _DisableProductMediaLoader: function ($this) {\n            var $widget = this;\n\n            if ($('body.catalog-product-view').length > 0) {\n                $this.parents('.column.main').find('.photo.image')\n                    .removeClass($widget.options.classes.loader);\n            } else {\n                //Category View\n                $this.parents('.product-item-info').find('.product-image-photo')\n                    .removeClass($widget.options.classes.loader);\n            }\n        },\n\n        /**\n         * Callback for product media\n         *\n         * @param {Object} $this\n         * @param {String} response\n         * @param {Boolean} isInProductView\n         * @private\n         */\n        _ProductMediaCallback: function ($this, response, isInProductView) {\n            var $main = isInProductView ? $this.parents('.column.main') : $this.parents('.product-item-info'),\n                $widget = this,\n                images = [],\n\n                /**\n                 * Check whether object supported or not\n                 *\n                 * @param {Object} e\n                 * @returns {*|Boolean}\n                 */\n                support = function (e) {\n                    return e.hasOwnProperty('large') && e.hasOwnProperty('medium') && e.hasOwnProperty('small');\n                };\n\n            if (_.size($widget) < 1 || !support(response)) {\n                this.updateBaseImage(this.options.mediaGalleryInitial, $main, isInProductView);\n\n                return;\n            }\n\n            images.push({\n                full: response.large,\n                img: response.medium,\n                thumb: response.small,\n                isMain: true\n            });\n\n            if (response.hasOwnProperty('gallery')) {\n                $.each(response.gallery, function () {\n                    if (!support(this) || response.large === this.large) {\n                        return;\n                    }\n                    images.push({\n                        full: this.large,\n                        img: this.medium,\n                        thumb: this.small\n                    });\n                });\n            }\n\n            this.updateBaseImage(images, $main, isInProductView);\n        },\n\n        /**\n         * Check if images to update are initial and set their type\n         * @param {Array} images\n         */\n        _setImageType: function (images) {\n            var initial = this.options.mediaGalleryInitial[0].img;\n\n            if (images[0].img === initial) {\n                images = $.extend(true, [], this.options.mediaGalleryInitial);\n            } else {\n                images.map(function (img) {\n                    if (!img.type) {\n                        img.type = 'image';\n                    }\n                });\n            }\n\n            return images;\n        },\n\n        /**\n         * Update [gallery-placeholder] or [product-image-photo]\n         * @param {Array} images\n         * @param {jQuery} context\n         * @param {Boolean} isInProductView\n         */\n        updateBaseImage: function (images, context, isInProductView) {\n            var justAnImage = images[0],\n                initialImages = this.options.mediaGalleryInitial,\n                imagesToUpdate,\n                gallery = context.find(this.options.mediaGallerySelector).data('gallery'),\n                isInitial;\n\n            if (isInProductView) {\n                imagesToUpdate = images.length ? this._setImageType($.extend(true, [], images)) : [];\n                isInitial = _.isEqual(imagesToUpdate, initialImages);\n\n                if (this.options.gallerySwitchStrategy === 'prepend' && !isInitial) {\n                    imagesToUpdate = imagesToUpdate.concat(initialImages);\n                }\n\n                imagesToUpdate = this._setImageIndex(imagesToUpdate);\n\n                if (!_.isUndefined(gallery)) {\n                    gallery.updateData(imagesToUpdate);\n                } else {\n                    context.find(this.options.mediaGallerySelector).on('gallery:loaded', function (loadedGallery) {\n                        loadedGallery = context.find(this.options.mediaGallerySelector).data('gallery');\n                        loadedGallery.updateData(imagesToUpdate);\n                    }.bind(this));\n                }\n\n                if (isInitial) {\n                    $(this.options.mediaGallerySelector).AddFotoramaVideoEvents();\n                } else {\n                    $(this.options.mediaGallerySelector).AddFotoramaVideoEvents({\n                        selectedOption: this.getProduct(),\n                        dataMergeStrategy: this.options.gallerySwitchStrategy\n                    });\n                }\n                gallery.first();\n            } else if (justAnImage && justAnImage.img) {\n                context.find('.product-image-photo').attr('src', justAnImage.img);\n            }\n        },\n\n        /**\n         * Set correct indexes for image set.\n         *\n         * @param {Array} images\n         * @private\n         */\n        _setImageIndex: function (images) {\n            var length = images.length,\n                i;\n\n            for (i = 0; length > i; i++) {\n                images[i].i = i + 1;\n            }\n\n            return images;\n        },\n\n        /**\n         * Kill doubled AJAX requests\n         *\n         * @private\n         */\n        _XhrKiller: function () {\n            var $widget = this;\n\n            if ($widget.xhr !== undefined && $widget.xhr !== null) {\n                $widget.xhr.abort();\n                $widget.xhr = null;\n            }\n        },\n\n        /**\n         * Emulate mouse click on all swatches that should be selected\n         * @param {Object} [selectedAttributes]\n         * @private\n         */\n        _EmulateSelected: function (selectedAttributes) {\n            $.each(selectedAttributes, $.proxy(function (attributeCode, optionId) {\n                var elem = this.element.find('.' + this.options.classes.attributeClass +\n                    '[attribute-code=\"' + attributeCode + '\"] [option-id=\"' + optionId + '\"]'),\n                    parentInput = elem.parent();\n\n                if (elem.hasClass('selected')) {\n                    return;\n                }\n\n                if (parentInput.hasClass(this.options.classes.selectClass)) {\n                    parentInput.val(optionId);\n                    parentInput.trigger('change');\n                } else {\n                    elem.trigger('click');\n                }\n            }, this));\n        },\n\n        /**\n         * Emulate mouse click or selection change on all swatches that should be selected\n         * @param {Object} [selectedAttributes]\n         * @private\n         */\n        _EmulateSelectedByAttributeId: function (selectedAttributes) {\n            $.each(selectedAttributes, $.proxy(function (attributeId, optionId) {\n                var elem = this.element.find('.' + this.options.classes.attributeClass +\n                    '[attribute-id=\"' + attributeId + '\"] [option-id=\"' + optionId + '\"]'),\n                    parentInput = elem.parent();\n\n                if (elem.hasClass('selected')) {\n                    return;\n                }\n\n                if (parentInput.hasClass(this.options.classes.selectClass)) {\n                    parentInput.val(optionId);\n                    parentInput.trigger('change');\n                } else {\n                    elem.trigger('click');\n                }\n            }, this));\n        },\n\n        /**\n         * Get default options values settings with either URL query parameters\n         * @private\n         */\n        _getSelectedAttributes: function () {\n            var hashIndex = window.location.href.indexOf('#'),\n                selectedAttributes = {},\n                params;\n\n            if (hashIndex !== -1) {\n                params = $.parseQuery(window.location.href.substr(hashIndex + 1));\n\n                selectedAttributes = _.invert(_.mapObject(_.invert(params), function (attributeId) {\n                    var attribute = this.options.jsonConfig.mappedAttributes[attributeId];\n\n                    return attribute ? attribute.code : attributeId;\n                }.bind(this)));\n            }\n\n            return selectedAttributes;\n        },\n\n        /**\n         * Callback which fired after gallery gets initialized.\n         *\n         * @param {HTMLElement} element - DOM element associated with a gallery.\n         */\n        _onGalleryLoaded: function (element) {\n            var galleryObject = element.data('gallery');\n\n            this.options.mediaGalleryInitial = galleryObject.returnCurrentImages();\n        },\n\n        /**\n         * Sets mediaCache for cases when jsonConfig contains preSelectedGallery on layered navigation result pages\n         *\n         * @private\n         */\n        _setPreSelectedGallery: function () {\n            var mediaCallData;\n\n            if (this.options.jsonConfig.preSelectedGallery) {\n                mediaCallData = {\n                    'product_id': this.getProduct()\n                };\n\n                this.options.mediaCache[JSON.stringify(mediaCallData)] = this.options.jsonConfig.preSelectedGallery;\n            }\n        }\n    });\n\n    return $.mage.SwatchRenderer;\n});\n","Magento_Swatches/js/configurable-customer-data.js":"require([\n    'jquery',\n    'Magento_ConfigurableProduct/js/options-updater'\n], function ($, Updater) {\n    'use strict';\n\n    var selectors = {\n            formSelector: '#product_addtocart_form',\n            swatchSelector: '.swatch-opt'\n        },\n        swatchWidgetName = 'mageSwatchRenderer',\n        widgetInitEvent = 'swatch.initialized',\n\n    /**\n    * Sets all configurable swatch attribute's selected values\n    */\n    updateSwatchOptions = function () {\n        var swatchWidget = $(selectors.swatchSelector).data(swatchWidgetName);\n\n        if (!swatchWidget || !swatchWidget._EmulateSelectedByAttributeId) {\n            return;\n        }\n        swatchWidget._EmulateSelectedByAttributeId(this.productOptions);\n    },\n    updater = new Updater(widgetInitEvent, updateSwatchOptions);\n\n    updater.listen();\n});\n","Magento_SendFriend/requirejs-config.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\nvar config = {\n    map: {\n        '*': {\n            'Magento_SendFriend/back-event': 'Magento_SendFriend/js/back-event'\n        }\n    }\n};\n","Magento_SendFriend/js/back-event.js":"/**\n* Copyright \u00a9 Magento, Inc. All rights reserved.\n* See COPYING.txt for license details.\n*/\n\ndefine([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    return function (config, element) {\n        $(element).on('click', function () {\n            history.back();\n\n            return false;\n        });\n    };\n});\n","Vertex_Tax/js/form/element/custom-option-flex-field-select.js":"/*\n * @copyright  Vertex. All rights reserved.  https://www.vertexinc.com/\n * @author     Mediotype                     https://www.mediotype.com/\n */\n\ndefine(['underscore', 'Magento_Ui/js/form/element/select'], function (_, Select) {\n    'use strict';\n\n    return Select.extend({\n        /**\n         * Overwrites the parent's filter to allow for checking if a value is\n         * in an array and for allowing the value of \"unmapped\" through all\n         * filters\n         *\n         * @param {String} value\n         * @param {String} field\n         */\n        filter: function (value, field) {\n            var source = this.initialOptions,\n                result;\n\n            field = field || this.filterBy.field;\n\n            result = _.filter(source, function (item) {\n                return Array.isArray(item[field]) && item[field].includes(value) ||\n                    item[field] === value ||\n                    item.value === '' ||\n                    item.value === 'unmapped';\n            });\n\n            this.setOptions(result);\n        }\n    });\n});\n","Vertex_Tax/js/view/checkout/summary/tax-messages.js":"/**\n * @copyright  Vertex. All rights reserved.  https://www.vertexinc.com/\n * @author     Mediotype                     https://www.mediotype.com/\n */\n\ndefine([\n    'underscore',\n    'ko',\n    'uiComponent',\n    'Magento_Checkout/js/model/totals'\n], function (_, ko, Component, totals) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            messages: []\n        },\n\n        /** @inheritdoc */\n        initialize: function () {\n            this._super();\n            this.subscribeTotals();\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super().observe('messages');\n            this.getMessages();\n\n            return this;\n        },\n\n        /**\n         * Retrieve messages\n         */\n        getMessages: function () {\n            var taxSegment = totals.getSegment('tax');\n\n            this.messages([]);\n\n            if (taxSegment && taxSegment['extension_attributes']) {\n                this.messages(taxSegment['extension_attributes']['vertex_tax_calculation_messages']);\n            }\n        },\n\n        /**\n         * Subscribe totals observer\n         */\n        subscribeTotals: function () {\n            var self = this;\n\n            totals.totals.subscribe(\n                function () {\n                    self.getMessages();\n                },\n                null,\n                'change'\n            );\n        }\n    });\n});\n","Vertex_Tax/js/model/set-checkout-messages.js":"/**\n * @copyright  Vertex. All rights reserved.  https://www.vertexinc.com/\n * @author     Mediotype                     https://www.mediotype.com/\n */\n\ndefine(\n    [\n        'underscore',\n        'Magento_Customer/js/customer-data',\n        'Magento_Ui/js/model/messageList'\n    ],\n    function (_, customerData, messageContainer) {\n        'use strict';\n\n        /**\n         * A utility for observing message updates in session storage. It is designed to subscribe to\n         * customer data updates and forward messages to the appropriate messageList model.\n         */\n        return function () {\n            var typeMap = {\n                    'success': 'addSuccessMessage',\n                    'warning': 'addErrorMessage',\n                    'error': 'addErrorMessage'\n                },\n\n                /**\n                 * Observe message section data changes and forward to the error processor.\n                 * @param {Object} data - The observable payload.\n                 * @return void\n                 */\n                messageSubscriptionCallback = function (data) {\n                    if ('messages' in data) {\n                        _.each(data.messages, function (message) {\n                            if (message.type in typeMap) {\n                                messageContainer[typeMap[message.type]]({\n                                    'message': message.text\n                                });\n                            }\n                        });\n                    }\n                };\n\n            customerData.get('messages').subscribe(messageSubscriptionCallback);\n        };\n    }\n);\n","Magento_Multishipping/js/multi-shipping.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.multiShipping', {\n        options: {\n            addNewAddressBtn: 'button[data-role=\"add-new-address\"]', // Add a new multishipping address.\n            addNewAddressFlag: '#add_new_address_flag', // Hidden input field with value 0 or 1.\n            canContinueBtn: 'button[data-role=\"can-continue\"]', // Continue (update quantity or go to shipping).\n            canContinueFlag: '#can_continue_flag' // Hidden input field with value 0 or 1.\n        },\n\n        /**\n         * Bind event handlers to click events for corresponding buttons.\n         * @private\n         */\n        _create: function () {\n            $(this.options.addNewAddressBtn).on('click', $.proxy(this._addNewAddress, this));\n            $(this.options.canContinueBtn).on('click', $.proxy(this._canContinue, this));\n        },\n\n        /**\n         * Add a new address. Set the hidden input field and submit the form. Then enter a new shipping address.\n         * @private\n         */\n        _addNewAddress: function () {\n            $(this.options.addNewAddressFlag).val(1);\n            this.element.submit();\n        },\n\n        /**\n         * Can the user continue to the next step? The data-flag attribute holds either 0 (no) or 1 (yes).\n         * @private\n         * @param {Event} event - Click event on the corresponding button.\n         */\n        _canContinue: function (event) {\n            $(this.options.canContinueFlag).val(parseInt($(event.currentTarget).data('flag'), 10));\n        }\n    });\n\n    return $.mage.multiShipping;\n});\n","Magento_Multishipping/js/overview.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget',\n    'mage/translate'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.orderOverview', {\n        options: {\n            opacity: 0.5, // CSS opacity for the 'Place Order' button when it's clicked and then disabled.\n            pleaseWaitLoader: 'span.please-wait', // 'Submitting order information...' Ajax loader.\n            placeOrderSubmit: 'button[type=\"submit\"]', // The 'Place Order' button.\n            agreements: '.checkout-agreements' // Container for all of the checkout agreements and terms/conditions\n        },\n\n        /**\n         * Bind a submit handler to the form.\n         * @private\n         */\n        _create: function () {\n            this.element.on('submit', $.proxy(this._showLoader, this));\n        },\n\n        /**\n         * Verify that all agreements and terms/conditions are checked. Show the Ajax loader. Disable\n         * the submit button (i.e. Place Order).\n         * @return {Boolean}\n         * @private\n         */\n        _showLoader: function () {\n            if ($(this.options.agreements).find('input[type=\"checkbox\"]:not(:checked)').length > 0) {\n                return false;\n            }\n            this.element.find(this.options.pleaseWaitLoader).show().end()\n                .find(this.options.placeOrderSubmit).prop('disabled', true).css('opacity', this.options.opacity);\n\n            return true;\n        }\n    });\n\n    return $.mage.orderOverview;\n});\n","Magento_Multishipping/js/payment.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/template',\n    'Magento_Ui/js/modal/alert',\n    'jquery-ui-modules/widget',\n    'mage/translate'\n], function ($, mageTemplate, alert) {\n    'use strict';\n\n    $.widget('mage.payment', {\n        options: {\n            continueSelector: '#payment-continue',\n            methodsContainer: '#payment-methods',\n            minBalance: 0,\n            tmpl: '<input id=\"hidden-free\" type=\"hidden\" name=\"payment[method]\" value=\"free\">'\n        },\n\n        /** @inheritdoc */\n        _create: function () {\n            this.element.find('dd [name^=\"payment[\"]').prop('disabled', true).end()\n                .on('click', this.options.continueSelector, $.proxy(this._submitHandler, this))\n                .on('updateCheckoutPrice', $.proxy(function (event, data) {\n                    //updating the checkoutPrice\n                    if (data.price) {\n                        this.options.checkoutPrice += data.price;\n                    }\n\n                    //updating total price\n                    if (data.totalPrice) {\n                        data.totalPrice = this.options.checkoutPrice;\n                    }\n\n                    if (this.options.checkoutPrice <= this.options.minBalance) {\n                        // Add free input field, hide and disable unchecked\n                        // checkbox payment method and all radio button payment methods\n                        this._disablePaymentMethods();\n                    } else {\n                        // Remove free input field, show all payment method\n                        this._enablePaymentMethods();\n                    }\n                }, this))\n                .on('click', 'dt input:radio', $.proxy(this._paymentMethodHandler, this));\n\n            if (this.options.checkoutPrice < this.options.minBalance) {\n                this._disablePaymentMethods();\n            } else {\n                this._enablePaymentMethods();\n            }\n        },\n\n        /**\n         * Display payment details when payment method radio button is checked\n         * @private\n         * @param {EventObject} e\n         */\n        _paymentMethodHandler: function (e) {\n            var element = $(e.target),\n                parentsDl = element.closest('dl');\n\n            parentsDl.find('dt input:radio').prop('checked', false);\n            parentsDl.find('dd').addClass('no-display').end()\n                .find('.items').hide()\n                .find('[name^=\"payment[\"]').prop('disabled', true);\n            element.prop('checked', true).parent()\n                .next('dd').removeClass('no-display')\n                .find('.items').show().find('[name^=\"payment[\"]').prop('disabled', false);\n        },\n\n        /**\n         * make sure one payment method is selected\n         * @private\n         * @return {Boolean}\n         */\n        _validatePaymentMethod: function () {\n            var methods = this.element.find('[name^=\"payment[\"]'),\n                isValid = false;\n\n            if (methods.length === 0) {\n                alert({\n                    content: $.mage.__('We can\\'t complete your order because you don\\'t have a payment method set up.')\n                });\n            } else if (this.options.checkoutPrice <= this.options.minBalance) {\n                isValid = true;\n            } else if (methods.filter('input:radio:checked').length) {\n                isValid = true;\n            } else {\n                alert({\n                    content: $.mage.__('Please choose a payment method.')\n                });\n            }\n\n            return isValid;\n        },\n\n        /**\n         * Disable and enable payment methods\n         * @private\n         */\n        _disablePaymentMethods: function () {\n            var tmpl = mageTemplate(this.options.tmpl, {\n                data: {}\n            });\n\n            this.element.find('input[name=\"payment[method]\"]').prop('disabled', true).end()\n                .find('input[id^=\"use\"][name^=\"payment[use\"]:not(:checked)').prop('disabled', true).parent().hide();\n            this.element.find('[name=\"payment[method]\"][value=\"free\"]').parent('dt').remove();\n            this.element.find(this.options.methodsContainer).hide().find('[name^=\"payment[\"]').prop('disabled', true);\n\n            $(tmpl).appendTo(this.element);\n        },\n\n        /**\n         * Enable and enable payment methods\n         * @private\n         */\n        _enablePaymentMethods: function () {\n            this.element.find('input[name=\"payment[method]\"]').prop('disabled', false).end()\n                .find('dt input:radio:checked').trigger('click').end()\n                .find('input[id^=\"use\"][name^=\"payment[use\"]:not(:checked)').prop('disabled', false).parent().show();\n            this.element.find(this.options.methodsContainer).show();\n        },\n\n        /**\n         * Returns checked payment method.\n         *\n         * @private\n         */\n        _getSelectedPaymentMethod: function () {\n            return this.element.find('input[name=\\'payment[method]\\']:checked');\n        },\n\n        /**\n         * Validate  before form submit\n         * @private\n         * @param {EventObject} e\n         */\n        _submitHandler: function (e) {\n            var currentMethod,\n                submitButton;\n\n            e.preventDefault();\n\n            if (this._validatePaymentMethod()) {\n                currentMethod = this._getSelectedPaymentMethod();\n                submitButton = currentMethod.parent().next('dd').find('button[type=submit]');\n\n                if (submitButton.length) {\n                    submitButton.first().trigger('click');\n                } else {\n                    this.element.submit();\n                }\n            }\n        }\n    });\n\n    return $.mage.payment;\n});\n","Magento_Reports/js/recently-viewed.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.recentlyViewedProducts', {\n        options: {\n            localStorageKey: 'recently-viewed-products',\n            productBlock: '#widget_viewed_item',\n            viewedContainer: 'ol'\n        },\n\n        /**\n         * Bind events to the appropriate handlers.\n         * @private\n         */\n        _create: function () {\n            var productHtml = $(this.options.productBlock).html(),\n                productSku = $(this.options.productBlock).data('sku'),\n                products = JSON.parse(window.localStorage.getItem(this.options.localStorageKey)),\n                productsLength, maximum, showed, index;\n\n            if (products) {\n                productsLength = products.sku.length;\n                maximum = $(this.element).data('count');\n                showed = 0;\n\n                for (index = 0; index <= productsLength; index++) {\n                    if (products.sku[index] == productSku || showed >= maximum) { //eslint-disable-line\n                        products.sku.splice(index, 1);\n                        products.html.splice(index, 1);\n                    } else {\n                        $(this.element).find(this.options.viewedContainer).append(products.html[index]);\n                        $(this.element).show();\n                        showed++;\n                    }\n                }\n                $(this.element).find(this.options.productBlock).show();\n            } else {\n                products = {};\n                products.sku = [];\n                products.html = [];\n            }\n            products.sku.unshift(productSku);\n            products.html.unshift(productHtml);\n            window.localStorage.setItem(this.options.localStorageKey, JSON.stringify(products));\n        }\n    });\n\n    return $.mage.recentlyViewedProducts;\n});\n","jquery/jquery.tabs.js":"/* ========================================================\n * bootstrap-tab.js v2.0.4\n * http://twitter.github.com/bootstrap/javascript.html#tabs\n * ========================================================\n * Copyright 2012 Twitter, Inc.\n *\n * Licensed under the Apache License, Version 2.0 (the \"License\");\n * you may not use this file except in compliance with the License.\n * You may obtain a copy of the License at\n *\n * http://www.apache.org/licenses/LICENSE-2.0\n *\n * Unless required by applicable law or agreed to in writing, software\n * distributed under the License is distributed on an \"AS IS\" BASIS,\n * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\n * See the License for the specific language governing permissions and\n * limitations under the License.\n * ======================================================== */\n\ndefine([\n  \"jquery\"\n], function(jQuery){\n\n\n!function ($) {\n\n  \"use strict\"; // jshint ;_;\n\n\n /* TAB CLASS DEFINITION\n  * ==================== */\n\n  var Tab = function ( element ) {\n    this.element = $(element)\n  };\n\n  Tab.prototype = {\n\n    constructor: Tab\n\n  , show: function () {\n      var $this = this.element\n        , $ul = $this.closest('ul:not(.dropdown-menu)')\n        , selector = $this.attr('data-target')\n        , previous\n        , $target\n        , e;\n\n      if (!selector) {\n        selector = $this.attr('href');\n        selector = selector && selector.replace(/.*(?=#[^\\s]*$)/, ''); //strip for ie7\n      }\n\n      if ( $this.parent('li').hasClass('active') ) return;\n\n      previous = $ul.find('.active a').last()[0];\n\n      e = $.Event('show', {\n        relatedTarget: previous\n      });\n\n      $this.trigger(e);\n\n      if (e.isDefaultPrevented()) return;\n\n      $target = $(selector);\n\n      this.activate($this.parent('li'), $ul);\n      this.activate($target, $target.parent(), function () {\n        $this.trigger({\n          type: 'shown'\n        , relatedTarget: previous\n        })\n      })\n    }\n\n  , activate: function ( element, container, callback) {\n      var $active = container.find('> .active')\n        , transition = callback\n            && $.support.transition\n            && $active.hasClass('fade');\n\n      function next() {\n        $active\n          .removeClass('active')\n          .find('> .dropdown-menu > .active')\n          .removeClass('active');\n\n        element.addClass('active');\n\n        if (transition) {\n          element[0].offsetWidth; // reflow for transition\n          element.addClass('in');\n        } else {\n          element.removeClass('fade')\n        }\n\n        if ( element.parent('.dropdown-menu') ) {\n          element.closest('li.dropdown').addClass('active')\n        }\n\n        callback && callback()\n      }\n\n      transition ?\n        $active.one($.support.transition.end, next) :\n        next();\n\n      $active.removeClass('in')\n    }\n  };\n\n\n /* TAB PLUGIN DEFINITION\n  * ===================== */\n\n  $.fn.tab = function ( option ) {\n    return this.each(function () {\n      var $this = $(this)\n        , data = $this.data('tab');\n      if (!data) $this.data('tab', (data = new Tab(this)));\n      if (typeof option == 'string') data[option]()\n    })\n  };\n\n  $.fn.tab.Constructor = Tab;\n\n\n /* TAB DATA-API\n  * ============ */\n\n  $(function () {\n    $('body').on('click.tab.data-api', '[data-toggle=\"tab\"], [data-toggle=\"pill\"]', function (e) {\n      e.preventDefault();\n      $(this).tab('show')\n    })\n  })\n\n}(jQuery);\n\n/* =============================================================\n * bootstrap-collapse.js v2.0.4\n * http://twitter.github.com/bootstrap/javascript.html#collapse\n * =============================================================\n * Copyright 2012 Twitter, Inc.\n *\n * Licensed under the Apache License, Version 2.0 (the \"License\");\n * you may not use this file except in compliance with the License.\n * You may obtain a copy of the License at\n *\n * http://www.apache.org/licenses/LICENSE-2.0\n *\n * Unless required by applicable law or agreed to in writing, software\n * distributed under the License is distributed on an \"AS IS\" BASIS,\n * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\n * See the License for the specific language governing permissions and\n * limitations under the License.\n * ============================================================ */\n\n\n!function ($) {\n\n  \"use strict\"; // jshint ;_;\n\n\n /* COLLAPSE PUBLIC CLASS DEFINITION\n  * ================================ */\n\n  var Collapse = function (element, options) {\n    this.$element = $(element);\n    this.options = $.extend({}, $.fn.collapse.defaults, options);\n\n    if (this.options.parent) {\n      this.$parent = $(this.options.parent)\n    }\n\n    this.options.toggle && this.toggle()\n  };\n\n  Collapse.prototype = {\n\n    constructor: Collapse\n\n  , dimension: function () {\n      var hasWidth = this.$element.hasClass('width');\n      return hasWidth ? 'width' : 'height'\n    }\n\n  , show: function () {\n      var dimension\n        , scroll\n        , actives\n        , hasData;\n\n      if (this.transitioning) return;\n\n      dimension = this.dimension();\n      scroll = $.camelCase(['scroll', dimension].join('-'));\n      actives = this.$parent && this.$parent.find('> .accordion-group > .in');\n\n      if (actives && actives.length) {\n        hasData = actives.data('collapse');\n        if (hasData && hasData.transitioning) return;\n        actives.collapse('hide');\n        hasData || actives.data('collapse', null)\n      }\n\n      this.$element[dimension](0);\n      this.transition('addClass', $.Event('show'), 'shown');\n      this.$element[dimension](this.$element[0][scroll]);\n    }\n\n  , hide: function () {\n      var dimension;\n      if (this.transitioning) return;\n      dimension = this.dimension();\n      this.reset(this.$element[dimension]());\n      this.transition('removeClass', $.Event('hide'), 'hidden');\n      this.$element[dimension](0)\n    }\n\n  , reset: function (size) {\n      var dimension = this.dimension();\n\n      this.$element\n        .removeClass('collapse')\n        [dimension](size || 'auto')\n        [0].offsetWidth;\n\n      this.$element[size !== null ? 'addClass' : 'removeClass']('collapse');\n\n      return this\n    }\n\n  , transition: function (method, startEvent, completeEvent) {\n      var that = this\n        , complete = function () {\n            if (startEvent.type == 'show') that.reset();\n            that.transitioning = 0;\n            that.$element.trigger(completeEvent)\n          };\n\n      this.$element.trigger(startEvent);\n\n      if (startEvent.isDefaultPrevented()) return;\n\n      this.transitioning = 1;\n\n      this.$element[method]('in');\n\n      $.support.transition && this.$element.hasClass('collapse') ?\n        this.$element.one($.support.transition.end, complete) :\n        complete()\n    }\n\n  , toggle: function () {\n      this[this.$element.hasClass('in') ? 'hide' : 'show']();\n    }\n\n  };\n\n\n /* COLLAPSIBLE PLUGIN DEFINITION\n  * ============================== */\n\n  $.fn.collapse = function (option) {\n    return this.each(function () {\n      var $this = $(this)\n        , data = $this.data('collapse')\n        , options = typeof option == 'object' && option;\n      if (!data) $this.data('collapse', (data = new Collapse(this, options)));\n      if (typeof option == 'string') data[option]()\n    })\n  };\n\n  $.fn.collapse.defaults = {\n    toggle: true\n  };\n\n  $.fn.collapse.Constructor = Collapse;\n\n\n /* COLLAPSIBLE DATA-API\n  * ==================== */\n\n  $(function () {\n    $('body').on('click.collapse.data-api', '[data-toggle=collapse]', function ( e ) {\n      var $this = $(this), href\n        , target = $this.attr('data-target')\n          || e.preventDefault()\n          || (href = $this.attr('href')) && href.replace(/.*(?=#[^\\s]+$)/, '') //strip for ie7\n        , option = $(target).data('collapse') ? 'toggle' : $this.data();\n      $(target).collapse(option);\n        $(this).toggleClass('active');\n    })\n  })\n\n}(jQuery);\n\n});","jquery/jquery.parsequery.js":"/**\n * Copyright (c) 2010 Conrad Irwin <conrad@rapportive.com> MIT license.\n * Based loosely on original: Copyright (c) 2008 mkmanning MIT license.\n *\n * Parses CGI query strings into javascript objects.\n *\n * See the README for details.\n **/\n/*jshint jquery:true */\n/*global window:true */\ndefine([\n    \"jquery\"\n], function($){\n\n    $.parseQuery = function(options) {\n        var config = {query: window.location.search || \"\"},\n            params = {};\n\n        if (typeof options === 'string') {\n            options = {query: options};\n        }\n        $.extend(config, $.parseQuery, options);\n        config.query = config.query.replace(/^\\?/, '');\n\n        $.each(config.query.split(config.separator), function(i, param) {\n            var pair = param.split('='),\n                key = config.decode(pair.shift(), null).toString(),\n                value = config.decode(pair.length ? pair.join('=') : null, key);\n\n            if (config.array_keys(key)) {\n                params[key] = params[key] || [];\n                params[key].push(value);\n            } else {\n                params[key] = value;\n            }\n        });\n\n        return params;\n    };\n\n    $.parseQuery.decode = $.parseQuery.default_decode = function(string) {\n        return decodeURIComponent((string || \"\").replace('+', ' '));\n    };\n\n    $.parseQuery.array_keys = function() {\n        return false;\n    };\n\n    $.parseQuery.separator = \"&\";\n\n});\n","jquery/jquery-migrate.js":"/*!\n * jQuery Migrate - v1.4.1 - 2016-05-19\n * Copyright jQuery Foundation and other contributors\n */\n(function( jQuery, window, undefined ) {\n// See http://bugs.jquery.com/ticket/13335\n// \"use strict\";\n\n\njQuery.migrateVersion = \"1.4.1\";\n\n\nvar warnedAbout = {};\n\n// List of warnings already given; public read only\njQuery.migrateWarnings = [];\n\n// Set to true to prevent console output; migrateWarnings still maintained\n// jQuery.migrateMute = false;\n\n// Show a message on the console so devs know we're active\nif ( window.console && window.console.log ) {\n\twindow.console.log( \"JQMIGRATE: Migrate is installed\" +\n\t\t( jQuery.migrateMute ? \"\" : \" with logging active\" ) +\n\t\t\", version \" + jQuery.migrateVersion );\n}\n\n// Set to false to disable traces that appear with warnings\nif ( jQuery.migrateTrace === undefined ) {\n\tjQuery.migrateTrace = true;\n}\n\n// Forget any warnings we've already given; public\njQuery.migrateReset = function() {\n\twarnedAbout = {};\n\tjQuery.migrateWarnings.length = 0;\n};\n\nfunction migrateWarn( msg) {\n\tvar console = window.console;\n\tif ( !warnedAbout[ msg ] ) {\n\t\twarnedAbout[ msg ] = true;\n\t\tjQuery.migrateWarnings.push( msg );\n\t\tif ( console && console.warn && !jQuery.migrateMute ) {\n\t\t\tconsole.warn( \"JQMIGRATE: \" + msg );\n\t\t\tif ( jQuery.migrateTrace && console.trace ) {\n\t\t\t\tconsole.trace();\n\t\t\t}\n\t\t}\n\t}\n}\n\nfunction migrateWarnProp( obj, prop, value, msg ) {\n\tif ( Object.defineProperty ) {\n\t\t// On ES5 browsers (non-oldIE), warn if the code tries to get prop;\n\t\t// allow property to be overwritten in case some other plugin wants it\n\t\ttry {\n\t\t\tObject.defineProperty( obj, prop, {\n\t\t\t\tconfigurable: true,\n\t\t\t\tenumerable: true,\n\t\t\t\tget: function() {\n\t\t\t\t\tmigrateWarn( msg );\n\t\t\t\t\treturn value;\n\t\t\t\t},\n\t\t\t\tset: function( newValue ) {\n\t\t\t\t\tmigrateWarn( msg );\n\t\t\t\t\tvalue = newValue;\n\t\t\t\t}\n\t\t\t});\n\t\t\treturn;\n\t\t} catch( err ) {\n\t\t\t// IE8 is a dope about Object.defineProperty, can't warn there\n\t\t}\n\t}\n\n\t// Non-ES5 (or broken) browser; just set the property\n\tjQuery._definePropertyBroken = true;\n\tobj[ prop ] = value;\n}\n\nif ( document.compatMode === \"BackCompat\" ) {\n\t// jQuery has never supported or tested Quirks Mode\n\tmigrateWarn( \"jQuery is not compatible with Quirks Mode\" );\n}\n\n\nvar attrFn = jQuery( \"<input/>\", { size: 1 } ).attr(\"size\") && jQuery.attrFn,\n\toldAttr = jQuery.attr,\n\tvalueAttrGet = jQuery.attrHooks.value && jQuery.attrHooks.value.get ||\n\t\tfunction() { return null; },\n\tvalueAttrSet = jQuery.attrHooks.value && jQuery.attrHooks.value.set ||\n\t\tfunction() { return undefined; },\n\trnoType = /^(?:input|button)$/i,\n\trnoAttrNodeType = /^[238]$/,\n\trboolean = /^(?:autofocus|autoplay|async|checked|controls|defer|disabled|hidden|loop|multiple|open|readonly|required|scoped|selected)$/i,\n\truseDefault = /^(?:checked|selected)$/i;\n\n// jQuery.attrFn\nmigrateWarnProp( jQuery, \"attrFn\", attrFn || {}, \"jQuery.attrFn is deprecated\" );\n\njQuery.attr = function( elem, name, value, pass ) {\n\tvar lowerName = name.toLowerCase(),\n\t\tnType = elem && elem.nodeType;\n\n\tif ( pass ) {\n\t\t// Since pass is used internally, we only warn for new jQuery\n\t\t// versions where there isn't a pass arg in the formal params\n\t\tif ( oldAttr.length < 4 ) {\n\t\t\tmigrateWarn(\"jQuery.fn.attr( props, pass ) is deprecated\");\n\t\t}\n\t\tif ( elem && !rnoAttrNodeType.test( nType ) &&\n\t\t\t(attrFn ? name in attrFn : jQuery.isFunction(jQuery.fn[name])) ) {\n\t\t\treturn jQuery( elem )[ name ]( value );\n\t\t}\n\t}\n\n\t// Warn if user tries to set `type`, since it breaks on IE 6/7/8; by checking\n\t// for disconnected elements we don't warn on $( \"<button>\", { type: \"button\" } ).\n\tif ( name === \"type\" && value !== undefined && rnoType.test( elem.nodeName ) && elem.parentNode ) {\n\t\tmigrateWarn(\"Can't change the 'type' of an input or button in IE 6/7/8\");\n\t}\n\n\t// Restore boolHook for boolean property/attribute synchronization\n\tif ( !jQuery.attrHooks[ lowerName ] && rboolean.test( lowerName ) ) {\n\t\tjQuery.attrHooks[ lowerName ] = {\n\t\t\tget: function( elem, name ) {\n\t\t\t\t// Align boolean attributes with corresponding properties\n\t\t\t\t// Fall back to attribute presence where some booleans are not supported\n\t\t\t\tvar attrNode,\n\t\t\t\t\tproperty = jQuery.prop( elem, name );\n\t\t\t\treturn property === true || typeof property !== \"boolean\" &&\n\t\t\t\t\t( attrNode = elem.getAttributeNode(name) ) && attrNode.nodeValue !== false ?\n\n\t\t\t\t\tname.toLowerCase() :\n\t\t\t\t\tundefined;\n\t\t\t},\n\t\t\tset: function( elem, value, name ) {\n\t\t\t\tvar propName;\n\t\t\t\tif ( value === false ) {\n\t\t\t\t\t// Remove boolean attributes when set to false\n\t\t\t\t\tjQuery.removeAttr( elem, name );\n\t\t\t\t} else {\n\t\t\t\t\t// value is true since we know at this point it's type boolean and not false\n\t\t\t\t\t// Set boolean attributes to the same name and set the DOM property\n\t\t\t\t\tpropName = jQuery.propFix[ name ] || name;\n\t\t\t\t\tif ( propName in elem ) {\n\t\t\t\t\t\t// Only set the IDL specifically if it already exists on the element\n\t\t\t\t\t\telem[ propName ] = true;\n\t\t\t\t\t}\n\n\t\t\t\t\telem.setAttribute( name, name.toLowerCase() );\n\t\t\t\t}\n\t\t\t\treturn name;\n\t\t\t}\n\t\t};\n\n\t\t// Warn only for attributes that can remain distinct from their properties post-1.9\n\t\tif ( ruseDefault.test( lowerName ) ) {\n\t\t\tmigrateWarn( \"jQuery.fn.attr('\" + lowerName + \"') might use property instead of attribute\" );\n\t\t}\n\t}\n\n\treturn oldAttr.call( jQuery, elem, name, value );\n};\n\n// attrHooks: value\njQuery.attrHooks.value = {\n\tget: function( elem, name ) {\n\t\tvar nodeName = ( elem.nodeName || \"\" ).toLowerCase();\n\t\tif ( nodeName === \"button\" ) {\n\t\t\treturn valueAttrGet.apply( this, arguments );\n\t\t}\n\t\tif ( nodeName !== \"input\" && nodeName !== \"option\" ) {\n\t\t\tmigrateWarn(\"jQuery.fn.attr('value') no longer gets properties\");\n\t\t}\n\t\treturn name in elem ?\n\t\t\telem.value :\n\t\t\tnull;\n\t},\n\tset: function( elem, value ) {\n\t\tvar nodeName = ( elem.nodeName || \"\" ).toLowerCase();\n\t\tif ( nodeName === \"button\" ) {\n\t\t\treturn valueAttrSet.apply( this, arguments );\n\t\t}\n\t\tif ( nodeName !== \"input\" && nodeName !== \"option\" ) {\n\t\t\tmigrateWarn(\"jQuery.fn.attr('value', val) no longer sets properties\");\n\t\t}\n\t\t// Does not return so that setAttribute is also used\n\t\telem.value = value;\n\t}\n};\n\n\nvar matched, browser,\n\toldInit = jQuery.fn.init,\n\toldFind = jQuery.find,\n\toldParseJSON = jQuery.parseJSON,\n\trspaceAngle = /^\\s*</,\n\trattrHashTest = /\\[(\\s*[-\\w]+\\s*)([~|^$*]?=)\\s*([-\\w#]*?#[-\\w#]*)\\s*\\]/,\n\trattrHashGlob = /\\[(\\s*[-\\w]+\\s*)([~|^$*]?=)\\s*([-\\w#]*?#[-\\w#]*)\\s*\\]/g,\n\t// Note: XSS check is done below after string is trimmed\n\trquickExpr = /^([^<]*)(<[\\w\\W]+>)([^>]*)$/;\n\n// $(html) \"looks like html\" rule change\njQuery.fn.init = function( selector, context, rootjQuery ) {\n\tvar match, ret;\n\n\tif ( selector && typeof selector === \"string\" ) {\n\t\tif ( !jQuery.isPlainObject( context ) &&\n\t\t\t\t(match = rquickExpr.exec( jQuery.trim( selector ) )) && match[ 0 ] ) {\n\n\t\t\t// This is an HTML string according to the \"old\" rules; is it still?\n\t\t\tif ( !rspaceAngle.test( selector ) ) {\n\t\t\t\tmigrateWarn(\"$(html) HTML strings must start with '<' character\");\n\t\t\t}\n\t\t\tif ( match[ 3 ] ) {\n\t\t\t\tmigrateWarn(\"$(html) HTML text after last tag is ignored\");\n\t\t\t}\n\n\t\t\t// Consistently reject any HTML-like string starting with a hash (gh-9521)\n\t\t\t// Note that this may break jQuery 1.6.x code that otherwise would work.\n\t\t\tif ( match[ 0 ].charAt( 0 ) === \"#\" ) {\n\t\t\t\tmigrateWarn(\"HTML string cannot start with a '#' character\");\n\t\t\t\tjQuery.error(\"JQMIGRATE: Invalid selector string (XSS)\");\n\t\t\t}\n\n\t\t\t// Now process using loose rules; let pre-1.8 play too\n\t\t\t// Is this a jQuery context? parseHTML expects a DOM element (#178)\n\t\t\tif ( context && context.context && context.context.nodeType ) {\n\t\t\t\tcontext = context.context;\n\t\t\t}\n\n\t\t\tif ( jQuery.parseHTML ) {\n\t\t\t\treturn oldInit.call( this,\n\t\t\t\t\t\tjQuery.parseHTML( match[ 2 ], context && context.ownerDocument ||\n\t\t\t\t\t\t\tcontext || document, true ), context, rootjQuery );\n\t\t\t}\n\t\t}\n\t}\n\n\tret = oldInit.apply( this, arguments );\n\n\t// Fill in selector and context properties so .live() works\n\tif ( selector && selector.selector !== undefined ) {\n\t\t// A jQuery object, copy its properties\n\t\tret.selector = selector.selector;\n\t\tret.context = selector.context;\n\n\t} else {\n\t\tret.selector = typeof selector === \"string\" ? selector : \"\";\n\t\tif ( selector ) {\n\t\t\tret.context = selector.nodeType? selector : context || document;\n\t\t}\n\t}\n\n\treturn ret;\n};\njQuery.fn.init.prototype = jQuery.fn;\n\njQuery.find = function( selector ) {\n\tvar args = Array.prototype.slice.call( arguments );\n\n\t// Support: PhantomJS 1.x\n\t// String#match fails to match when used with a //g RegExp, only on some strings\n\tif ( typeof selector === \"string\" && rattrHashTest.test( selector ) ) {\n\n\t\t// The nonstandard and undocumented unquoted-hash was removed in jQuery 1.12.0\n\t\t// First see if qS thinks it's a valid selector, if so avoid a false positive\n\t\ttry {\n\t\t\tdocument.querySelector( selector );\n\t\t} catch ( err1 ) {\n\n\t\t\t// Didn't *look* valid to qSA, warn and try quoting what we think is the value\n\t\t\tselector = selector.replace( rattrHashGlob, function( _, attr, op, value ) {\n\t\t\t\treturn \"[\" + attr + op + \"\\\"\" + value + \"\\\"]\";\n\t\t\t} );\n\n\t\t\t// If the regexp *may* have created an invalid selector, don't update it\n\t\t\t// Note that there may be false alarms if selector uses jQuery extensions\n\t\t\ttry {\n\t\t\t\tdocument.querySelector( selector );\n\t\t\t\tmigrateWarn( \"Attribute selector with '#' must be quoted: \" + args[ 0 ] );\n\t\t\t\targs[ 0 ] = selector;\n\t\t\t} catch ( err2 ) {\n\t\t\t\tmigrateWarn( \"Attribute selector with '#' was not fixed: \" + args[ 0 ] );\n\t\t\t}\n\t\t}\n\t}\n\n\treturn oldFind.apply( this, args );\n};\n\n// Copy properties attached to original jQuery.find method (e.g. .attr, .isXML)\nvar findProp;\nfor ( findProp in oldFind ) {\n\tif ( Object.prototype.hasOwnProperty.call( oldFind, findProp ) ) {\n\t\tjQuery.find[ findProp ] = oldFind[ findProp ];\n\t}\n}\n\n// Let $.parseJSON(falsy_value) return null\njQuery.parseJSON = function( json ) {\n\tif ( !json ) {\n\t\tmigrateWarn(\"jQuery.parseJSON requires a valid JSON string\");\n\t\treturn null;\n\t}\n\treturn oldParseJSON.apply( this, arguments );\n};\n\njQuery.uaMatch = function( ua ) {\n\tua = ua.toLowerCase();\n\n\tvar match = /(chrome)[ \\/]([\\w.]+)/.exec( ua ) ||\n\t\t/(webkit)[ \\/]([\\w.]+)/.exec( ua ) ||\n\t\t/(opera)(?:.*version|)[ \\/]([\\w.]+)/.exec( ua ) ||\n\t\t/(msie) ([\\w.]+)/.exec( ua ) ||\n\t\tua.indexOf(\"compatible\") < 0 && /(mozilla)(?:.*? rv:([\\w.]+)|)/.exec( ua ) ||\n\t\t[];\n\n\treturn {\n\t\tbrowser: match[ 1 ] || \"\",\n\t\tversion: match[ 2 ] || \"0\"\n\t};\n};\n\n// Don't clobber any existing jQuery.browser in case it's different\nif ( !jQuery.browser ) {\n\tmatched = jQuery.uaMatch( navigator.userAgent );\n\tbrowser = {};\n\n\tif ( matched.browser ) {\n\t\tbrowser[ matched.browser ] = true;\n\t\tbrowser.version = matched.version;\n\t}\n\n\t// Chrome is Webkit, but Webkit is also Safari.\n\tif ( browser.chrome ) {\n\t\tbrowser.webkit = true;\n\t} else if ( browser.webkit ) {\n\t\tbrowser.safari = true;\n\t}\n\n\tjQuery.browser = browser;\n}\n\n// Warn if the code tries to get jQuery.browser\nmigrateWarnProp( jQuery, \"browser\", jQuery.browser, \"jQuery.browser is deprecated\" );\n\n// jQuery.boxModel deprecated in 1.3, jQuery.support.boxModel deprecated in 1.7\njQuery.boxModel = jQuery.support.boxModel = (document.compatMode === \"CSS1Compat\");\nmigrateWarnProp( jQuery, \"boxModel\", jQuery.boxModel, \"jQuery.boxModel is deprecated\" );\nmigrateWarnProp( jQuery.support, \"boxModel\", jQuery.support.boxModel, \"jQuery.support.boxModel is deprecated\" );\n\njQuery.sub = function() {\n\tfunction jQuerySub( selector, context ) {\n\t\treturn new jQuerySub.fn.init( selector, context );\n\t}\n\tjQuery.extend( true, jQuerySub, this );\n\tjQuerySub.superclass = this;\n\tjQuerySub.fn = jQuerySub.prototype = this();\n\tjQuerySub.fn.constructor = jQuerySub;\n\tjQuerySub.sub = this.sub;\n\tjQuerySub.fn.init = function init( selector, context ) {\n\t\tvar instance = jQuery.fn.init.call( this, selector, context, rootjQuerySub );\n\t\treturn instance instanceof jQuerySub ?\n\t\t\tinstance :\n\t\t\tjQuerySub( instance );\n\t};\n\tjQuerySub.fn.init.prototype = jQuerySub.fn;\n\tvar rootjQuerySub = jQuerySub(document);\n\tmigrateWarn( \"jQuery.sub() is deprecated\" );\n\treturn jQuerySub;\n};\n\n// The number of elements contained in the matched element set\njQuery.fn.size = function() {\n\tmigrateWarn( \"jQuery.fn.size() is deprecated; use the .length property\" );\n\treturn this.length;\n};\n\n\nvar internalSwapCall = false;\n\n// If this version of jQuery has .swap(), don't false-alarm on internal uses\nif ( jQuery.swap ) {\n\tjQuery.each( [ \"height\", \"width\", \"reliableMarginRight\" ], function( _, name ) {\n\t\tvar oldHook = jQuery.cssHooks[ name ] && jQuery.cssHooks[ name ].get;\n\n\t\tif ( oldHook ) {\n\t\t\tjQuery.cssHooks[ name ].get = function() {\n\t\t\t\tvar ret;\n\n\t\t\t\tinternalSwapCall = true;\n\t\t\t\tret = oldHook.apply( this, arguments );\n\t\t\t\tinternalSwapCall = false;\n\t\t\t\treturn ret;\n\t\t\t};\n\t\t}\n\t});\n}\n\njQuery.swap = function( elem, options, callback, args ) {\n\tvar ret, name,\n\t\told = {};\n\n\tif ( !internalSwapCall ) {\n\t\tmigrateWarn( \"jQuery.swap() is undocumented and deprecated\" );\n\t}\n\n\t// Remember the old values, and insert the new ones\n\tfor ( name in options ) {\n\t\told[ name ] = elem.style[ name ];\n\t\telem.style[ name ] = options[ name ];\n\t}\n\n\tret = callback.apply( elem, args || [] );\n\n\t// Revert the old values\n\tfor ( name in options ) {\n\t\telem.style[ name ] = old[ name ];\n\t}\n\n\treturn ret;\n};\n\n\n// Ensure that $.ajax gets the new parseJSON defined in core.js\njQuery.ajaxSetup({\n\tconverters: {\n\t\t\"text json\": jQuery.parseJSON\n\t}\n});\n\n\nvar oldFnData = jQuery.fn.data;\n\njQuery.fn.data = function( name ) {\n\tvar ret, evt,\n\t\telem = this[0];\n\n\t// Handles 1.7 which has this behavior and 1.8 which doesn't\n\tif ( elem && name === \"events\" && arguments.length === 1 ) {\n\t\tret = jQuery.data( elem, name );\n\t\tevt = jQuery._data( elem, name );\n\t\tif ( ( ret === undefined || ret === evt ) && evt !== undefined ) {\n\t\t\tmigrateWarn(\"Use of jQuery.fn.data('events') is deprecated\");\n\t\t\treturn evt;\n\t\t}\n\t}\n\treturn oldFnData.apply( this, arguments );\n};\n\n\nvar rscriptType = /\\/(java|ecma)script/i;\n\n// Since jQuery.clean is used internally on older versions, we only shim if it's missing\nif ( !jQuery.clean ) {\n\tjQuery.clean = function( elems, context, fragment, scripts ) {\n\t\t// Set context per 1.8 logic\n\t\tcontext = context || document;\n\t\tcontext = !context.nodeType && context[0] || context;\n\t\tcontext = context.ownerDocument || context;\n\n\t\tmigrateWarn(\"jQuery.clean() is deprecated\");\n\n\t\tvar i, elem, handleScript, jsTags,\n\t\t\tret = [];\n\n\t\tjQuery.merge( ret, jQuery.buildFragment( elems, context ).childNodes );\n\n\t\t// Complex logic lifted directly from jQuery 1.8\n\t\tif ( fragment ) {\n\t\t\t// Special handling of each script element\n\t\t\thandleScript = function( elem ) {\n\t\t\t\t// Check if we consider it executable\n\t\t\t\tif ( !elem.type || rscriptType.test( elem.type ) ) {\n\t\t\t\t\t// Detach the script and store it in the scripts array (if provided) or the fragment\n\t\t\t\t\t// Return truthy to indicate that it has been handled\n\t\t\t\t\treturn scripts ?\n\t\t\t\t\t\tscripts.push( elem.parentNode ? elem.parentNode.removeChild( elem ) : elem ) :\n\t\t\t\t\t\tfragment.appendChild( elem );\n\t\t\t\t}\n\t\t\t};\n\n\t\t\tfor ( i = 0; (elem = ret[i]) != null; i++ ) {\n\t\t\t\t// Check if we're done after handling an executable script\n\t\t\t\tif ( !( jQuery.nodeName( elem, \"script\" ) && handleScript( elem ) ) ) {\n\t\t\t\t\t// Append to fragment and handle embedded scripts\n\t\t\t\t\tfragment.appendChild( elem );\n\t\t\t\t\tif ( typeof elem.getElementsByTagName !== \"undefined\" ) {\n\t\t\t\t\t\t// handleScript alters the DOM, so use jQuery.merge to ensure snapshot iteration\n\t\t\t\t\t\tjsTags = jQuery.grep( jQuery.merge( [], elem.getElementsByTagName(\"script\") ), handleScript );\n\n\t\t\t\t\t\t// Splice the scripts into ret after their former ancestor and advance our index beyond them\n\t\t\t\t\t\tret.splice.apply( ret, [i + 1, 0].concat( jsTags ) );\n\t\t\t\t\t\ti += jsTags.length;\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t}\n\t\t}\n\n\t\treturn ret;\n\t};\n}\n\nvar eventAdd = jQuery.event.add,\n\teventRemove = jQuery.event.remove,\n\teventTrigger = jQuery.event.trigger,\n\toldToggle = jQuery.fn.toggle,\n\toldLive = jQuery.fn.live,\n\toldDie = jQuery.fn.die,\n\toldLoad = jQuery.fn.load,\n\tajaxEvents = \"ajaxStart|ajaxStop|ajaxSend|ajaxComplete|ajaxError|ajaxSuccess\",\n\trajaxEvent = new RegExp( \"\\\\b(?:\" + ajaxEvents + \")\\\\b\" ),\n\trhoverHack = /(?:^|\\s)hover(\\.\\S+|)\\b/,\n\thoverHack = function( events ) {\n\t\tif ( typeof( events ) !== \"string\" || jQuery.event.special.hover ) {\n\t\t\treturn events;\n\t\t}\n\t\tif ( rhoverHack.test( events ) ) {\n\t\t\tmigrateWarn(\"'hover' pseudo-event is deprecated, use 'mouseenter mouseleave'\");\n\t\t}\n\t\treturn events && events.replace( rhoverHack, \"mouseenter$1 mouseleave$1\" );\n\t};\n\n// Event props removed in 1.9, put them back if needed; no practical way to warn them\nif ( jQuery.event.props && jQuery.event.props[ 0 ] !== \"attrChange\" ) {\n\tjQuery.event.props.unshift( \"attrChange\", \"attrName\", \"relatedNode\", \"srcElement\" );\n}\n\n// Undocumented jQuery.event.handle was \"deprecated\" in jQuery 1.7\nif ( jQuery.event.dispatch ) {\n\tmigrateWarnProp( jQuery.event, \"handle\", jQuery.event.dispatch, \"jQuery.event.handle is undocumented and deprecated\" );\n}\n\n// Support for 'hover' pseudo-event and ajax event warnings\njQuery.event.add = function( elem, types, handler, data, selector ){\n\tif ( elem !== document && rajaxEvent.test( types ) ) {\n\t\tmigrateWarn( \"AJAX events should be attached to document: \" + types );\n\t}\n\teventAdd.call( this, elem, hoverHack( types || \"\" ), handler, data, selector );\n};\njQuery.event.remove = function( elem, types, handler, selector, mappedTypes ){\n\teventRemove.call( this, elem, hoverHack( types ) || \"\", handler, selector, mappedTypes );\n};\n\njQuery.each( [ \"load\", \"unload\", \"error\" ], function( _, name ) {\n\n\tjQuery.fn[ name ] = function() {\n\t\tvar args = Array.prototype.slice.call( arguments, 0 );\n\n\t\t// If this is an ajax load() the first arg should be the string URL;\n\t\t// technically this could also be the \"Anything\" arg of the event .load()\n\t\t// which just goes to show why this dumb signature has been deprecated!\n\t\t// jQuery custom builds that exclude the Ajax module justifiably die here.\n\t\tif ( name === \"load\" && typeof args[ 0 ] === \"string\" ) {\n\t\t\treturn oldLoad.apply( this, args );\n\t\t}\n\n\t\tmigrateWarn( \"jQuery.fn.\" + name + \"() is deprecated\" );\n\n\t\targs.splice( 0, 0, name );\n\t\tif ( arguments.length ) {\n\t\t\treturn this.bind.apply( this, args );\n\t\t}\n\n\t\t// Use .triggerHandler here because:\n\t\t// - load and unload events don't need to bubble, only applied to window or image\n\t\t// - error event should not bubble to window, although it does pre-1.7\n\t\t// See http://bugs.jquery.com/ticket/11820\n\t\tthis.triggerHandler.apply( this, args );\n\t\treturn this;\n\t};\n\n});\n\njQuery.fn.toggle = function( fn, fn2 ) {\n\n\t// Don't mess with animation or css toggles\n\tif ( !jQuery.isFunction( fn ) || !jQuery.isFunction( fn2 ) ) {\n\t\treturn oldToggle.apply( this, arguments );\n\t}\n\tmigrateWarn(\"jQuery.fn.toggle(handler, handler...) is deprecated\");\n\n\t// Save reference to arguments for access in closure\n\tvar args = arguments,\n\t\tguid = fn.guid || jQuery.guid++,\n\t\ti = 0,\n\t\ttoggler = function( event ) {\n\t\t\t// Figure out which function to execute\n\t\t\tvar lastToggle = ( jQuery._data( this, \"lastToggle\" + fn.guid ) || 0 ) % i;\n\t\t\tjQuery._data( this, \"lastToggle\" + fn.guid, lastToggle + 1 );\n\n\t\t\t// Make sure that clicks stop\n\t\t\tevent.preventDefault();\n\n\t\t\t// and execute the function\n\t\t\treturn args[ lastToggle ].apply( this, arguments ) || false;\n\t\t};\n\n\t// link all the functions, so any of them can unbind this click handler\n\ttoggler.guid = guid;\n\twhile ( i < args.length ) {\n\t\targs[ i++ ].guid = guid;\n\t}\n\n\treturn this.click( toggler );\n};\n\njQuery.fn.live = function( types, data, fn ) {\n\tmigrateWarn(\"jQuery.fn.live() is deprecated\");\n\tif ( oldLive ) {\n\t\treturn oldLive.apply( this, arguments );\n\t}\n\tjQuery( this.context ).on( types, this.selector, data, fn );\n\treturn this;\n};\n\njQuery.fn.die = function( types, fn ) {\n\tmigrateWarn(\"jQuery.fn.die() is deprecated\");\n\tif ( oldDie ) {\n\t\treturn oldDie.apply( this, arguments );\n\t}\n\tjQuery( this.context ).off( types, this.selector || \"**\", fn );\n\treturn this;\n};\n\n// Turn global events into document-triggered events\njQuery.event.trigger = function( event, data, elem, onlyHandlers  ){\n\tif ( !elem && !rajaxEvent.test( event ) ) {\n\t\tmigrateWarn( \"Global events are undocumented and deprecated\" );\n\t}\n\treturn eventTrigger.call( this,  event, data, elem || document, onlyHandlers  );\n};\njQuery.each( ajaxEvents.split(\"|\"),\n\tfunction( _, name ) {\n\t\tjQuery.event.special[ name ] = {\n\t\t\tsetup: function() {\n\t\t\t\tvar elem = this;\n\n\t\t\t\t// The document needs no shimming; must be !== for oldIE\n\t\t\t\tif ( elem !== document ) {\n\t\t\t\t\tjQuery.event.add( document, name + \".\" + jQuery.guid, function() {\n\t\t\t\t\t\tjQuery.event.trigger( name, Array.prototype.slice.call( arguments, 1 ), elem, true );\n\t\t\t\t\t});\n\t\t\t\t\tjQuery._data( this, name, jQuery.guid++ );\n\t\t\t\t}\n\t\t\t\treturn false;\n\t\t\t},\n\t\t\tteardown: function() {\n\t\t\t\tif ( this !== document ) {\n\t\t\t\t\tjQuery.event.remove( document, name + \".\" + jQuery._data( this, name ) );\n\t\t\t\t}\n\t\t\t\treturn false;\n\t\t\t}\n\t\t};\n\t}\n);\n\njQuery.event.special.ready = {\n\tsetup: function() {\n\t\tif ( this === document ) {\n\t\t\tmigrateWarn( \"'ready' event is deprecated\" );\n\t\t}\n\t}\n};\n\nvar oldSelf = jQuery.fn.andSelf || jQuery.fn.addBack,\n\toldFnFind = jQuery.fn.find;\n\njQuery.fn.andSelf = function() {\n\tmigrateWarn(\"jQuery.fn.andSelf() replaced by jQuery.fn.addBack()\");\n\treturn oldSelf.apply( this, arguments );\n};\n\njQuery.fn.find = function( selector ) {\n\tvar ret = oldFnFind.apply( this, arguments );\n\tret.context = this.context;\n\tret.selector = this.selector ? this.selector + \" \" + selector : selector;\n\treturn ret;\n};\n\n\n// jQuery 1.6 did not support Callbacks, do not warn there\nif ( jQuery.Callbacks ) {\n\n\tvar oldDeferred = jQuery.Deferred,\n\t\ttuples = [\n\t\t\t// action, add listener, callbacks, .then handlers, final state\n\t\t\t[ \"resolve\", \"done\", jQuery.Callbacks(\"once memory\"),\n\t\t\t\tjQuery.Callbacks(\"once memory\"), \"resolved\" ],\n\t\t\t[ \"reject\", \"fail\", jQuery.Callbacks(\"once memory\"),\n\t\t\t\tjQuery.Callbacks(\"once memory\"), \"rejected\" ],\n\t\t\t[ \"notify\", \"progress\", jQuery.Callbacks(\"memory\"),\n\t\t\t\tjQuery.Callbacks(\"memory\") ]\n\t\t];\n\n\tjQuery.Deferred = function( func ) {\n\t\tvar deferred = oldDeferred(),\n\t\t\tpromise = deferred.promise();\n\n\t\tdeferred.pipe = promise.pipe = function( /* fnDone, fnFail, fnProgress */ ) {\n\t\t\tvar fns = arguments;\n\n\t\t\tmigrateWarn( \"deferred.pipe() is deprecated\" );\n\n\t\t\treturn jQuery.Deferred(function( newDefer ) {\n\t\t\t\tjQuery.each( tuples, function( i, tuple ) {\n\t\t\t\t\tvar fn = jQuery.isFunction( fns[ i ] ) && fns[ i ];\n\t\t\t\t\t// deferred.done(function() { bind to newDefer or newDefer.resolve })\n\t\t\t\t\t// deferred.fail(function() { bind to newDefer or newDefer.reject })\n\t\t\t\t\t// deferred.progress(function() { bind to newDefer or newDefer.notify })\n\t\t\t\t\tdeferred[ tuple[1] ](function() {\n\t\t\t\t\t\tvar returned = fn && fn.apply( this, arguments );\n\t\t\t\t\t\tif ( returned && jQuery.isFunction( returned.promise ) ) {\n\t\t\t\t\t\t\treturned.promise()\n\t\t\t\t\t\t\t\t.done( newDefer.resolve )\n\t\t\t\t\t\t\t\t.fail( newDefer.reject )\n\t\t\t\t\t\t\t\t.progress( newDefer.notify );\n\t\t\t\t\t\t} else {\n\t\t\t\t\t\t\tnewDefer[ tuple[ 0 ] + \"With\" ](\n\t\t\t\t\t\t\t\tthis === promise ? newDefer.promise() : this,\n\t\t\t\t\t\t\t\tfn ? [ returned ] : arguments\n\t\t\t\t\t\t\t);\n\t\t\t\t\t\t}\n\t\t\t\t\t});\n\t\t\t\t});\n\t\t\t\tfns = null;\n\t\t\t}).promise();\n\n\t\t};\n\n\t\tdeferred.isResolved = function() {\n\t\t\tmigrateWarn( \"deferred.isResolved is deprecated\" );\n\t\t\treturn deferred.state() === \"resolved\";\n\t\t};\n\n\t\tdeferred.isRejected = function() {\n\t\t\tmigrateWarn( \"deferred.isRejected is deprecated\" );\n\t\t\treturn deferred.state() === \"rejected\";\n\t\t};\n\n\t\tif ( func ) {\n\t\t\tfunc.call( deferred, deferred );\n\t\t}\n\n\t\treturn deferred;\n\t};\n\n}\n\n})( jQuery, window );\n\n","jquery/jquery.validate.js":"/**\n * jQuery Validation Plugin @VERSION\n *\n * http://bassistance.de/jquery-plugins/jquery-plugin-validation/\n * http://docs.jquery.com/Plugins/Validation\n *\n * Copyright (c) 2012 J\u00f6rn Zaefferer\n *\n * Dual licensed under the MIT and GPL licenses:\n *   http://www.opensource.org/licenses/mit-license.php\n *   http://www.gnu.org/licenses/gpl.html\n */\n(function (factory) {\n    if (typeof define === 'function' && define.amd) {\n        define([\n            \"jquery\",\n            \"jquery/jquery.metadata\"\n        ], factory);\n    } else {\n        factory(jQuery);\n    }\n}(function (jQuery) {\n\n(function ($) {\n\n    $.extend($.fn, {\n        // http://docs.jquery.com/Plugins/Validation/validate\n        validate: function (options) {\n\n            // if nothing is selected, return nothing; can't chain anyway\n            if (!this.length) {\n                if (options && options.debug && window.console) {\n                    console.warn(\"nothing selected, can't validate, returning nothing\");\n                }\n                return;\n            }\n\n            // check if a validator for this form was already created\n            var validator = $.data(this[0], 'validator');\n            if (validator) {\n                return validator;\n            }\n\n            // Add novalidate tag if HTML5.\n            this.attr('novalidate', 'novalidate');\n\n            validator = new $.validator(options, this[0]);\n            $.data(this[0], 'validator', validator);\n\n            if (validator.settings.onsubmit) {\n\n                this.validateDelegate(\":submit\", \"click\", function (ev) {\n                    if (validator.settings.submitHandler) {\n                        validator.submitButton = ev.target;\n                    }\n                    // allow suppressing validation by adding a cancel class to the submit button\n                    if ($(ev.target).hasClass('cancel')) {\n                        validator.cancelSubmit = true;\n                    }\n                });\n\n                // validate the form on submit\n                this.submit(function (event) {\n                    if (validator.settings.debug) {\n                        // prevent form submit to be able to see console output\n                        event.preventDefault();\n                    }\n                    function handle() {\n                        var hidden;\n                        if (validator.settings.submitHandler) {\n                            if (validator.submitButton) {\n                                // insert a hidden input as a replacement for the missing submit button\n                                hidden = $(\"<input type='hidden'/>\").attr(\"name\", validator.submitButton.name).val(validator.submitButton.value).appendTo(validator.currentForm);\n                            }\n                            validator.settings.submitHandler.call(validator, validator.currentForm, event);\n                            if (validator.submitButton) {\n                                // and clean up afterwards; thanks to no-block-scope, hidden can be referenced\n                                hidden.remove();\n                            }\n                            return false;\n                        }\n                        return true;\n                    }\n\n                    // prevent submit for invalid forms or custom submit handlers\n                    if (validator.cancelSubmit) {\n                        validator.cancelSubmit = false;\n                        return handle();\n                    }\n                    if (validator.form()) {\n                        if (validator.pendingRequest) {\n                            validator.formSubmitted = true;\n                            return false;\n                        }\n                        return handle();\n                    } else {\n                        validator.focusInvalid();\n                        return false;\n                    }\n                });\n            }\n\n            return validator;\n        },\n        // http://docs.jquery.com/Plugins/Validation/valid\n        valid: function () {\n            if ($(this[0]).is('form')) {\n                return this.validate().form();\n            } else {\n                var valid = true;\n                var validator = $(this[0].form).validate();\n                this.each(function () {\n                    valid &= validator.element(this);\n                });\n                return valid;\n            }\n        },\n        // attributes: space seperated list of attributes to retrieve and remove\n        removeAttrs: function (attributes) {\n            var result = {},\n                $element = this;\n            $.each(attributes.split(/\\s/), function (index, value) {\n                result[value] = $element.attr(value);\n                $element.removeAttr(value);\n            });\n            return result;\n        },\n        // http://docs.jquery.com/Plugins/Validation/rules\n        rules: function (command, argument) {\n            var element = this[0];\n\n            if (command) {\n                var settings = $.data(element.form, 'validator').settings;\n                var staticRules = settings.rules;\n                var existingRules = $.validator.staticRules(element);\n                switch (command) {\n                    case \"add\":\n                        $.extend(existingRules, $.validator.normalizeRule(argument));\n                        staticRules[element.name] = existingRules;\n                        if (argument.messages) {\n                            settings.messages[element.name] = $.extend(settings.messages[element.name], argument.messages);\n                        }\n                        break;\n                    case \"remove\":\n                        if (!argument) {\n                            delete staticRules[element.name];\n                            return existingRules;\n                        }\n                        var filtered = {};\n                        $.each(argument.split(/\\s/), function (index, method) {\n                            filtered[method] = existingRules[method];\n                            delete existingRules[method];\n                        });\n                        return filtered;\n                }\n            }\n\n            var data = $.validator.normalizeRules(\n                $.extend(\n                    {},\n                    $.validator.metadataRules(element),\n                    $.validator.classRules(element),\n                    $.validator.attributeRules(element),\n                    $.validator.staticRules(element)\n                ), element);\n\n            // make sure required is at front\n            if (data.required) {\n                var param = data.required;\n                delete data.required;\n                data = $.extend({required: param}, data);\n            }\n\n            return data;\n        }\n    });\n\n// Custom selectors\n    $.extend($.expr[\":\"], {\n        // http://docs.jquery.com/Plugins/Validation/blank\n        blank: function (a) {\n            return !$.trim(\"\" + a.value);\n        },\n        // http://docs.jquery.com/Plugins/Validation/filled\n        filled: function (a) {\n            return !!$.trim(\"\" + a.value);\n        },\n        // http://docs.jquery.com/Plugins/Validation/unchecked\n        unchecked: function (a) {\n            return !a.checked;\n        }\n    });\n\n// constructor for validator\n    $.validator = function (options, form) {\n        this.settings = $.extend(true, {}, $.validator.defaults, options);\n        this.currentForm = form;\n        this.init();\n    };\n\n    $.validator.format = function (source, params) {\n        if (arguments.length === 1) {\n            return function () {\n                var args = $.makeArray(arguments);\n                args.unshift(source);\n                return $.validator.format.apply(this, args);\n            };\n        }\n        if (arguments.length > 2 && params.constructor !== Array) {\n            params = $.makeArray(arguments).slice(1);\n        }\n        if (params.constructor !== Array) {\n            params = [ params ];\n        }\n        $.each(params, function (i, n) {\n            source = source.replace(new RegExp(\"\\\\{\" + i + \"\\\\}\", \"g\"), n);\n        });\n        return source;\n    };\n\n    $.extend($.validator, {\n\n        defaults: {\n            messages: {},\n            groups: {},\n            rules: {},\n            errorClass: \"error\",\n            validClass: \"valid\",\n            errorElement: \"label\",\n            focusInvalid: true,\n            errorContainer: $([]),\n            errorLabelContainer: $([]),\n            onsubmit: true,\n            ignore: \":hidden\",\n            ignoreTitle: false,\n            onfocusin: function (element, event) {\n                this.lastActive = element;\n\n                // hide error label and remove error class on focus if enabled\n                if (this.settings.focusCleanup && !this.blockFocusCleanup) {\n                    if (this.settings.unhighlight) {\n                        this.settings.unhighlight.call(this, element, this.settings.errorClass, this.settings.validClass);\n                    }\n                    this.addWrapper(this.errorsFor(element)).hide();\n                }\n            },\n            onfocusout: function (element, event) {\n                if (!this.checkable(element) && (element.name in this.submitted || !this.optional(element))) {\n                    this.element(element);\n                }\n            },\n            onkeyup: function (element, event) {\n                if (event.which == 9 && this.elementValue(element) === '') {\n                    return;\n                } else if (element.name in this.submitted || element === this.lastActive) {\n                    this.element(element);\n                }\n            },\n            onclick: function (element, event) {\n                // click on selects, radiobuttons and checkboxes\n                if (element.name in this.submitted) {\n                    this.element(element);\n                }\n                // or option elements, check parent select in that case\n                else if (element.parentNode.name in this.submitted) {\n                    this.element(element.parentNode);\n                }\n            },\n            highlight: function (element, errorClass, validClass) {\n                if (element.type === 'radio') {\n                    this.findByName(element.name).addClass(errorClass).removeClass(validClass);\n                } else {\n                    $(element).addClass(errorClass).removeClass(validClass);\n                }\n            },\n            unhighlight: function (element, errorClass, validClass) {\n                if (element.type === 'radio') {\n                    this.findByName(element.name).removeClass(errorClass).addClass(validClass);\n                } else {\n                    $(element).removeClass(errorClass).addClass(validClass);\n                }\n            }\n        },\n\n        // http://docs.jquery.com/Plugins/Validation/Validator/setDefaults\n        setDefaults: function (settings) {\n            $.extend($.validator.defaults, settings);\n        },\n\n        messages: {\n            required: \"This field is required.\",\n            remote: \"Please fix this field.\",\n            email: \"Please enter a valid email address.\",\n            url: \"Please enter a valid URL.\",\n            date: \"Please enter a valid date.\",\n            dateISO: \"Please enter a valid date (ISO).\",\n            number: \"Please enter a valid number.\",\n            digits: \"Please enter only digits.\",\n            creditcard: \"Please enter a valid credit card number.\",\n            equalTo: \"Please enter the same value again.\",\n            maxlength: $.validator.format(\"Please enter no more than {0} characters.\"),\n            minlength: $.validator.format(\"Please enter at least {0} characters.\"),\n            rangelength: $.validator.format(\"Please enter a value between {0} and {1} characters long.\"),\n            range: $.validator.format(\"Please enter a value between {0} and {1}.\"),\n            max: $.validator.format(\"Please enter a value less than or equal to {0}.\"),\n            min: $.validator.format(\"Please enter a value greater than or equal to {0}.\")\n        },\n\n        autoCreateRanges: false,\n\n        prototype: {\n\n            init: function () {\n                this.labelContainer = $(this.settings.errorLabelContainer);\n                this.errorContext = this.labelContainer.length && this.labelContainer || $(this.currentForm);\n                this.containers = $(this.settings.errorContainer).add(this.settings.errorLabelContainer);\n                this.submitted = {};\n                this.valueCache = {};\n                this.pendingRequest = 0;\n                this.pending = {};\n                this.invalid = {};\n                this.reset();\n\n                var groups = (this.groups = {});\n                $.each(this.settings.groups, function (key, value) {\n                    $.each(value.split(/\\s/), function (index, name) {\n                        groups[name] = key;\n                    });\n                });\n                var rules = this.settings.rules;\n                $.each(rules, function (key, value) {\n                    rules[key] = $.validator.normalizeRule(value);\n                });\n\n                function delegate(event) {\n                    var validator = $.data(this[0].form, \"validator\"),\n                        eventType = \"on\" + event.type.replace(/^validate/, \"\");\n                    if (validator.settings[eventType]) {\n                        validator.settings[eventType].call(validator, this[0], event);\n                    }\n                }\n\n                $(this.currentForm)\n                    .validateDelegate(\":text, [type='password'], [type='file'], select, textarea, \" +\n                    \"[type='number'], [type='search'], input.search, [type='tel'], [type='url'], \" +\n                    \"[type='email'], [type='datetime'], [type='date'], [type='month'], \" +\n                    \"[type='week'], [type='time'], [type='datetime-local'], \" +\n                    \"[type='range'], [type='color'] \",\n                    \"focusin focusout keyup\", delegate)\n                    .validateDelegate(\"[type='radio'], [type='checkbox'], select, option\", \"click\", delegate);\n\n                if (this.settings.invalidHandler) {\n                    $(this.currentForm).bind(\"invalid-form.validate\", this.settings.invalidHandler);\n                }\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Validator/form\n            form: function () {\n                this.checkForm();\n                $.extend(this.submitted, this.errorMap);\n                this.invalid = $.extend({}, this.errorMap);\n                if (!this.valid()) {\n                    $(this.currentForm).triggerHandler(\"invalid-form\", [this]);\n                }\n                this.showErrors();\n                return this.valid();\n            },\n\n            checkForm: function () {\n                this.prepareForm();\n                for (var i = 0, elements = (this.currentElements = this.elements()); elements[i]; i++) {\n                    this.check(elements[i]);\n                }\n                return this.valid();\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Validator/element\n            element: function (element) {\n                element = this.validationTargetFor(this.clean(element));\n                this.lastElement = element;\n                this.prepareElement(element);\n                this.currentElements = $(element);\n                var result = this.check(element) !== false;\n                if (result) {\n                    delete this.invalid[element.name];\n                } else {\n                    this.invalid[element.name] = true;\n                }\n                if (!this.numberOfInvalids()) {\n                    // Hide error containers on last error\n                    this.toHide = this.toHide.add(this.containers);\n                }\n                this.showErrors();\n                return result;\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Validator/showErrors\n            showErrors: function (errors) {\n                if (errors) {\n                    // add items to error list and map\n                    $.extend(this.errorMap, errors);\n                    this.errorList = [];\n                    for (var name in errors) {\n                        this.errorList.push({\n                            message: errors[name],\n                            element: this.findByName(name)[0]\n                        });\n                    }\n                    // remove items from success list\n                    this.successList = $.grep(this.successList, function (element) {\n                        return !(element.name in errors);\n                    });\n                }\n                if (this.settings.showErrors) {\n                    this.settings.showErrors.call(this, this.errorMap, this.errorList);\n                } else {\n                    this.defaultShowErrors();\n                }\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Validator/resetForm\n            resetForm: function () {\n                if ($.fn.resetForm) {\n                    $(this.currentForm).resetForm();\n                }\n                this.submitted = {};\n                this.lastElement = null;\n                this.prepareForm();\n                this.hideErrors();\n                this.elements().removeClass(this.settings.errorClass).removeData(\"previousValue\");\n            },\n\n            numberOfInvalids: function () {\n                return this.objectLength(this.invalid);\n            },\n\n            objectLength: function (obj) {\n                var count = 0;\n                for (var i in obj) {\n                    count++;\n                }\n                return count;\n            },\n\n            hideErrors: function () {\n                this.addWrapper(this.toHide).hide();\n            },\n\n            valid: function () {\n                return this.size() === 0;\n            },\n\n            size: function () {\n                return this.errorList.length;\n            },\n\n            focusInvalid: function () {\n                if (this.settings.focusInvalid) {\n                    try {\n                        $(this.findLastActive() || this.errorList.length && this.errorList[0].element || [])\n                            .filter(\":visible\")\n                            .focus()\n                            // manually trigger focusin event; without it, focusin handler isn't called, findLastActive won't have anything to find\n                            .trigger(\"focusin\");\n                    } catch (e) {\n                        // ignore IE throwing errors when focusing hidden elements\n                    }\n                }\n            },\n\n            findLastActive: function () {\n                var lastActive = this.lastActive;\n                return lastActive && $.grep(this.errorList,function (n) {\n                    return n.element.name === lastActive.name;\n                }).length === 1 && lastActive;\n            },\n\n            elements: function () {\n                var validator = this,\n                    rulesCache = {};\n\n                // select all valid inputs inside the form (no submit or reset buttons)\n                return $(this.currentForm)\n                    .find(\"input, select, textarea\")\n                    .not(\":submit, :reset, :image, [disabled]\")\n                    .not(this.settings.ignore)\n                    .filter(function () {\n                        if (!this.name && validator.settings.debug && window.console) {\n                            console.error(\"%o has no name assigned\", this);\n                        }\n\n                        // select only the first element for each name, and only those with rules specified\n                        if (this.name in rulesCache || !validator.objectLength($(this).rules())) {\n                            return false;\n                        }\n\n                        rulesCache[this.name] = true;\n                        return true;\n                    });\n            },\n\n            clean: function (selector) {\n                return $(selector)[0];\n            },\n\n            errors: function () {\n                var errorClass = this.settings.errorClass.replace(' ', '.');\n                return $(this.settings.errorElement + \".\" + errorClass, this.errorContext);\n            },\n\n            reset: function () {\n                this.successList = [];\n                this.errorList = [];\n                this.errorMap = {};\n                this.toShow = $([]);\n                this.toHide = $([]);\n                this.currentElements = $([]);\n            },\n\n            prepareForm: function () {\n                this.reset();\n                this.toHide = this.errors().add(this.containers);\n            },\n\n            prepareElement: function (element) {\n                this.reset();\n                this.toHide = this.errorsFor(element);\n            },\n\n            elementValue: function (element) {\n                var type = $(element).attr('type'),\n                    val = $(element).val();\n\n                if (type === 'radio' || type === 'checkbox') {\n                    return $('input[name=\"' + $(element).attr('name') + '\"]:checked').val();\n                }\n\n                if (typeof val === 'string') {\n                    return val.replace(/\\r/g, \"\");\n                }\n                return val;\n            },\n\n            check: function (element) {\n                element = this.validationTargetFor(this.clean(element));\n\n                var rules = $(element).rules();\n                var dependencyMismatch = false;\n                var val = this.elementValue(element);\n                var result;\n\n                for (var method in rules) {\n                    var rule = { method: method, parameters: rules[method] };\n                    try {\n\n                        result = $.validator.methods[method].call(this, val, element, rule.parameters);\n\n                        // if a method indicates that the field is optional and therefore valid,\n                        // don't mark it as valid when there are no other rules\n                        if (result === \"dependency-mismatch\") {\n                            dependencyMismatch = true;\n                            continue;\n                        }\n                        dependencyMismatch = false;\n\n                        if (result === \"pending\") {\n                            this.toHide = this.toHide.not(this.errorsFor(element));\n                            return;\n                        }\n\n                        if (!result) {\n                            this.formatAndAdd(element, rule);\n                            return false;\n                        }\n                    } catch (e) {\n                        if (this.settings.debug && window.console) {\n                            console.log(\"exception occurred when checking element \" + element.id + \", check the '\" + rule.method + \"' method\", e);\n                        }\n                        throw e;\n                    }\n                }\n                if (dependencyMismatch) {\n                    return;\n                }\n                if (this.objectLength(rules)) {\n                    this.successList.push(element);\n                }\n                return true;\n            },\n\n            // return the custom message for the given element and validation method\n            // specified in the element's \"messages\" metadata\n            customMetaMessage: function (element, method) {\n                if (!$.metadata) {\n                    return;\n                }\n                var meta = this.settings.meta ? $(element).metadata()[this.settings.meta] : $(element).metadata();\n                return meta && meta.messages && meta.messages[method];\n            },\n\n            // return the custom message for the given element and validation method\n            // specified in the element's HTML5 data attribute\n            customDataMessage: function (element, method) {\n                return $(element).data('msg-' + method.toLowerCase()) || (element.attributes && $(element).attr('data-msg-' + method.toLowerCase()));\n            },\n\n            // return the custom message for the given element name and validation method\n            customMessage: function (name, method) {\n                var m = this.settings.messages[name];\n                return m && (m.constructor === String ? m : m[method]);\n            },\n\n            // return the first defined argument, allowing empty strings\n            findDefined: function () {\n                for (var i = 0; i < arguments.length; i++) {\n                    if (arguments[i] !== undefined) {\n                        return arguments[i];\n                    }\n                }\n                return undefined;\n            },\n\n            defaultMessage: function (element, method) {\n                var noMessage = $.mage.__(\"Warning: No message defined for %s\");\n                return this.findDefined(\n                    this.customMessage(element.name, method),\n                    this.customDataMessage(element, method),\n                    this.customMetaMessage(element, method),\n                    // title is never undefined, so handle empty string as undefined\n                    !this.settings.ignoreTitle && element.title || undefined,\n                    $.validator.messages[method],\n                    \"<strong>\" + noMessage.replace('%s', element.name) + \"</strong>\"\n                );\n            },\n\n            formatAndAdd: function (element, rule) {\n                var message = this.defaultMessage(element, rule.method),\n                    theregex = /\\$?\\{(\\d+)\\}/g;\n                if (typeof message === \"function\") {\n                    message = message.call(this, rule.parameters, element);\n                } else if (theregex.test(message)) {\n                    message = $.validator.format(message.replace(theregex, '{$1}'), rule.parameters);\n                }\n                this.errorList.push({\n                    message: message,\n                    element: element\n                });\n\n                this.errorMap[element.name] = message;\n                this.submitted[element.name] = message;\n            },\n\n            addWrapper: function (toToggle) {\n                if (this.settings.wrapper) {\n                    toToggle = toToggle.add(toToggle.parent(this.settings.wrapper));\n                }\n                return toToggle;\n            },\n\n            defaultShowErrors: function () {\n                var i, elements;\n                for (i = 0; this.errorList[i]; i++) {\n                    var error = this.errorList[i];\n                    if (this.settings.highlight) {\n                        this.settings.highlight.call(this, error.element, this.settings.errorClass, this.settings.validClass);\n                    }\n                    this.showLabel(error.element, error.message);\n                }\n                if (this.errorList.length) {\n                    this.toShow = this.toShow.add(this.containers);\n                }\n                if (this.settings.success) {\n                    for (i = 0; this.successList[i]; i++) {\n                        this.showLabel(this.successList[i]);\n                    }\n                }\n                if (this.settings.unhighlight) {\n                    for (i = 0, elements = this.validElements(); elements[i]; i++) {\n                        this.settings.unhighlight.call(this, elements[i], this.settings.errorClass, this.settings.validClass);\n                    }\n                }\n                this.toHide = this.toHide.not(this.toShow);\n                this.hideErrors();\n                this.addWrapper(this.toShow).show();\n            },\n\n            validElements: function () {\n                return this.currentElements.not(this.invalidElements());\n            },\n\n            invalidElements: function () {\n                return $(this.errorList).map(function () {\n                    return this.element;\n                });\n            },\n\n            showLabel: function (element, message) {\n                var label = this.errorsFor(element);\n                if (label.length) {\n                    // refresh error/success class\n                    label.removeClass(this.settings.validClass).addClass(this.settings.errorClass);\n\n                    // check if we have a generated label, replace the message then\n                    if (label.attr(\"generated\")) {\n                        label.hide().html(message);\n                    }\n                } else {\n                    // create label\n                    label = $(\"<\" + this.settings.errorElement + \"/>\")\n                        .attr({\"for\": this.idOrName(element), generated: true})\n                        .addClass(this.settings.errorClass)\n                        .html(message || \"\");\n                    if (this.settings.wrapper) {\n                        // make sure the element is visible, even in IE\n                        // actually showing the wrapped element is handled elsewhere\n                        label = label.hide().show().wrap(\"<\" + this.settings.wrapper + \"/>\").parent();\n                    }\n                    if (!this.labelContainer.append(label).length) {\n                        if (this.settings.errorPlacement) {\n                            this.settings.errorPlacement(label, $(element));\n                        } else {\n                            label.insertAfter(element);\n                        }\n                    }\n                }\n                if (!message && this.settings.success) {\n                    label.text(\"\");\n                    if (typeof this.settings.success === \"string\") {\n                        label.addClass(this.settings.success);\n                    } else {\n                        this.settings.success(label, element);\n                    }\n                }\n                this.toShow = this.toShow.add(label);\n            },\n\n            errorsFor: function (element) {\n                var name = this.idOrName(element);\n                return this.errors().filter(function () {\n                    return $(this).attr('for') === name;\n                });\n            },\n\n            idOrName: function (element) {\n                return this.groups[element.name] || (this.checkable(element) ? element.name : element.id || element.name);\n            },\n\n            validationTargetFor: function (element) {\n                // if radio/checkbox, validate first element in group instead\n                if (this.checkable(element)) {\n                    element = this.findByName(element.name).not(this.settings.ignore)[0];\n                }\n                return element;\n            },\n\n            checkable: function (element) {\n                return (/radio|checkbox/i).test(element.type);\n            },\n\n            findByName: function (name) {\n                return $(this.currentForm).find('[name=\"' + name + '\"]');\n            },\n\n            getLength: function (value, element) {\n                switch (element.nodeName.toLowerCase()) {\n                    case 'select':\n                        return $(\"option:selected\", element).length;\n                    case 'input':\n                        if (this.checkable(element)) {\n                            return this.findByName(element.name).filter(':checked').length;\n                        }\n                }\n                return value.length;\n            },\n\n            depend: function (param, element) {\n                return this.dependTypes[typeof param] ? this.dependTypes[typeof param](param, element) : true;\n            },\n\n            dependTypes: {\n                \"boolean\": function (param, element) {\n                    return param;\n                },\n                \"string\": function (param, element) {\n                    return !!$(param, element.form).length;\n                },\n                \"function\": function (param, element) {\n                    return param(element);\n                }\n            },\n\n            optional: function (element) {\n                var val = this.elementValue(element);\n                return !$.validator.methods.required.call(this, val, element) && \"dependency-mismatch\";\n            },\n\n            startRequest: function (element) {\n                if (!this.pending[element.name]) {\n                    this.pendingRequest++;\n                    this.pending[element.name] = true;\n                }\n            },\n\n            stopRequest: function (element, valid) {\n                this.pendingRequest--;\n                // sometimes synchronization fails, make sure pendingRequest is never < 0\n                if (this.pendingRequest < 0) {\n                    this.pendingRequest = 0;\n                }\n                delete this.pending[element.name];\n                if (valid && this.pendingRequest === 0 && this.formSubmitted && this.form()) {\n                    $(this.currentForm).submit();\n                    this.formSubmitted = false;\n                } else if (!valid && this.pendingRequest === 0 && this.formSubmitted) {\n                    $(this.currentForm).triggerHandler(\"invalid-form\", [this]);\n                    this.formSubmitted = false;\n                }\n            },\n\n            previousValue: function (element) {\n                return $.data(element, \"previousValue\") || $.data(element, \"previousValue\", {\n                    old: null,\n                    valid: true,\n                    message: this.defaultMessage(element, \"remote\")\n                });\n            }\n\n        },\n\n        classRuleSettings: {\n            required: {required: true},\n            email: {email: true},\n            url: {url: true},\n            date: {date: true},\n            dateISO: {dateISO: true},\n            number: {number: true},\n            digits: {digits: true},\n            creditcard: {creditcard: true}\n        },\n\n        addClassRules: function (className, rules) {\n            if (className.constructor === String) {\n                this.classRuleSettings[className] = rules;\n            } else {\n                $.extend(this.classRuleSettings, className);\n            }\n        },\n\n        classRules: function (element) {\n            var rules = {};\n            var classes = $(element).attr('class');\n            if (classes) {\n                $.each(classes.split(' '), function () {\n                    if (this in $.validator.classRuleSettings) {\n                        $.extend(rules, $.validator.classRuleSettings[this]);\n                    }\n                });\n            }\n            return rules;\n        },\n\n        attributeRules: function (element) {\n            var rules = {};\n            var $element = $(element);\n\n            for (var method in $.validator.methods) {\n                var value;\n\n                // support for <input required> in both html5 and older browsers\n                if (method === 'required') {\n                    value = $element.get(0).getAttribute(method);\n                    // Some browsers return an empty string for the required attribute\n                    // and non-HTML5 browsers might have required=\"\" markup\n                    if (value === \"\") {\n                        value = true;\n                    }\n                    // force non-HTML5 browsers to return bool\n                    value = !!value;\n                } else {\n                    value = $element.attr(method);\n                }\n\n                if (value) {\n                    rules[method] = value;\n                } else if ($element[0].getAttribute(\"type\") === method) {\n                    rules[method] = true;\n                }\n            }\n\n            // maxlength may be returned as -1, 2147483647 (IE) and 524288 (safari) for text inputs\n            if (rules.maxlength && /-1|2147483647|524288/.test(rules.maxlength)) {\n                delete rules.maxlength;\n            }\n\n            return rules;\n        },\n\n        metadataRules: function (element) {\n            if (!$.metadata) {\n                return {};\n            }\n\n            var meta = $.data(element.form, 'validator').settings.meta;\n            return meta ?\n                $(element).metadata()[meta] :\n                $(element).metadata();\n        },\n\n        staticRules: function (element) {\n            var rules = {};\n            var validator = $.data(element.form, 'validator');\n            if (validator.settings.rules) {\n                rules = $.validator.normalizeRule(validator.settings.rules[element.name]) || {};\n            }\n            return rules;\n        },\n\n        normalizeRules: function (rules, element) {\n            // handle dependency check\n            $.each(rules, function (prop, val) {\n                // ignore rule when param is explicitly false, eg. required:false\n                if (val === false) {\n                    delete rules[prop];\n                    return;\n                }\n                if (val.param || val.depends) {\n                    var keepRule = true;\n                    switch (typeof val.depends) {\n                        case \"string\":\n                            keepRule = !!$(val.depends, element.form).length;\n                            break;\n                        case \"function\":\n                            keepRule = val.depends.call(element, element);\n                            break;\n                    }\n                    if (keepRule) {\n                        rules[prop] = val.param !== undefined ? val.param : true;\n                    } else {\n                        delete rules[prop];\n                    }\n                }\n            });\n\n            // evaluate parameters\n            $.each(rules, function (rule, parameter) {\n                rules[rule] = $.isFunction(parameter) ? parameter(element) : parameter;\n            });\n\n            // clean number parameters\n            $.each(['minlength', 'maxlength', 'min', 'max'], function () {\n                if (rules[this]) {\n                    rules[this] = Number(rules[this]);\n                }\n            });\n            $.each(['rangelength', 'range'], function () {\n                if (rules[this]) {\n                    rules[this] = [Number(rules[this][0]), Number(rules[this][1])];\n                }\n            });\n\n            if ($.validator.autoCreateRanges) {\n                // auto-create ranges\n                if (rules.min && rules.max) {\n                    rules.range = [rules.min, rules.max];\n                    delete rules.min;\n                    delete rules.max;\n                }\n                if (rules.minlength && rules.maxlength) {\n                    rules.rangelength = [rules.minlength, rules.maxlength];\n                    delete rules.minlength;\n                    delete rules.maxlength;\n                }\n            }\n\n            // To support custom messages in metadata ignore rule methods titled \"messages\"\n            if (rules.messages) {\n                delete rules.messages;\n            }\n\n            return rules;\n        },\n\n        // Converts a simple string to a {string: true} rule, e.g., \"required\" to {required:true}\n        normalizeRule: function (data) {\n            if (typeof data === \"string\") {\n                var transformed = {};\n                $.each(data.split(/\\s/), function () {\n                    transformed[this] = true;\n                });\n                data = transformed;\n            }\n            return data;\n        },\n\n        // http://docs.jquery.com/Plugins/Validation/Validator/addMethod\n        addMethod: function (name, method, message) {\n            $.validator.methods[name] = method;\n            $.validator.messages[name] = message !== undefined ? message : $.validator.messages[name];\n            if (method.length < 3) {\n                $.validator.addClassRules(name, $.validator.normalizeRule(name));\n            }\n        },\n\n        methods: {\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/required\n            required: function (value, element, param) {\n                // check if dependency is met\n                if (!this.depend(param, element)) {\n                    return \"dependency-mismatch\";\n                }\n                if (element.nodeName.toLowerCase() === \"select\") {\n                    // could be an array for select-multiple or a string, both are fine this way\n                    var val = $(element).val();\n                    return val && val.length > 0;\n                }\n                if (this.checkable(element)) {\n                    return this.getLength(value, element) > 0;\n                }\n                return $.trim(value).length > 0;\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/remote\n            remote: function (value, element, param) {\n                if (this.optional(element)) {\n                    return \"dependency-mismatch\";\n                }\n\n                var previous = this.previousValue(element);\n                if (!this.settings.messages[element.name]) {\n                    this.settings.messages[element.name] = {};\n                }\n                previous.originalMessage = this.settings.messages[element.name].remote;\n                this.settings.messages[element.name].remote = previous.message;\n\n                param = typeof param === \"string\" && {url: param} || param;\n\n                if (this.pending[element.name]) {\n                    return \"pending\";\n                }\n                if (previous.old === value) {\n                    return previous.valid;\n                }\n\n                previous.old = value;\n                var validator = this;\n                this.startRequest(element);\n                var data = {};\n                data[element.name] = value;\n                $.ajax($.extend(true, {\n                    url: param,\n                    mode: \"abort\",\n                    port: \"validate\" + element.name,\n                    dataType: \"json\",\n                    data: data,\n                    success: function (response) {\n                        validator.settings.messages[element.name].remote = previous.originalMessage;\n                        var valid = response === true || response === \"true\";\n                        if (valid) {\n                            var submitted = validator.formSubmitted;\n                            validator.prepareElement(element);\n                            validator.formSubmitted = submitted;\n                            validator.successList.push(element);\n                            delete validator.invalid[element.name];\n                            validator.showErrors();\n                        } else {\n                            var errors = {};\n                            var message = response || validator.defaultMessage(element, \"remote\");\n                            errors[element.name] = previous.message = $.isFunction(message) ? message(value) : message;\n                            validator.invalid[element.name] = true;\n                            validator.showErrors(errors);\n                        }\n                        previous.valid = valid;\n                        validator.stopRequest(element, valid);\n                    }\n                }, param));\n                return \"pending\";\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/minlength\n            minlength: function (value, element, param) {\n                var length = $.isArray(value) ? value.length : this.getLength($.trim(value), element);\n                return this.optional(element) || length >= param;\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/maxlength\n            maxlength: function (value, element, param) {\n                var length = $.isArray(value) ? value.length : this.getLength($.trim(value), element);\n                return this.optional(element) || length <= param;\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/rangelength\n            rangelength: function (value, element, param) {\n                var length = $.isArray(value) ? value.length : this.getLength($.trim(value), element);\n                return this.optional(element) || ( length >= param[0] && length <= param[1] );\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/min\n            min: function (value, element, param) {\n                return this.optional(element) || value >= param;\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/max\n            max: function (value, element, param) {\n                return this.optional(element) || value <= param;\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/range\n            range: function (value, element, param) {\n                return this.optional(element) || ( value >= param[0] && value <= param[1] );\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/email\n            email: function (value, element) {\n                // contributed by Scott Gonzalez: http://projects.scottsplayground.com/email_address_validation/\n                return this.optional(element) || /^((([a-z]|\\d|[!#\\$%&'\\*\\+\\-\\/=\\?\\^_`{\\|}~]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z]|\\d|[!#\\$%&'\\*\\+\\-\\/=\\?\\^_`{\\|}~]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*)|((\\x22)((((\\x20|\\x09)*(\\x0d\\x0a))?(\\x20|\\x09)+)?(([\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x7f]|\\x21|[\\x23-\\x5b]|[\\x5d-\\x7e]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(\\\\([\\x01-\\x09\\x0b\\x0c\\x0d-\\x7f]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]))))*(((\\x20|\\x09)*(\\x0d\\x0a))?(\\x20|\\x09)+)?(\\x22)))@((([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.)+(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))$/i.test(value);\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/url\n            url: function (value, element) {\n                // contributed by Scott Gonzalez: http://projects.scottsplayground.com/iri/\n                return this.optional(element) || /^(https?|ftp):\\/\\/(((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&'\\(\\)\\*\\+,;=]|:)*@)?(((\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5]))|((([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.)+(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.?)(:\\d*)?)(\\/((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&'\\(\\)\\*\\+,;=]|:|@)+(\\/(([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&'\\(\\)\\*\\+,;=]|:|@)*)*)?)?(\\?((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&'\\(\\)\\*\\+,;=]|:|@)|[\\uE000-\\uF8FF]|\\/|\\?)*)?(\\#((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&'\\(\\)\\*\\+,;=]|:|@)|\\/|\\?)*)?$/i.test(value);\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/date\n            date: function (value, element) {\n                return this.optional(element) || !/Invalid|NaN/.test(new Date(value));\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/dateISO\n            dateISO: function (value, element) {\n                return this.optional(element) || /^\\d{4}[\\/\\-]\\d{1,2}[\\/\\-]\\d{1,2}$/.test(value);\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/number\n            number: function (value, element) {\n                return this.optional(element) || /^-?(?:\\d+|\\d{1,3}(?:,\\d{3})+)?(?:\\.\\d+)?$/.test(value);\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/digits\n            digits: function (value, element) {\n                return this.optional(element) || /^\\d+$/.test(value);\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/creditcard\n            // based on http://en.wikipedia.org/wiki/Luhn\n            creditcard: function (value, element) {\n                if (this.optional(element)) {\n                    return \"dependency-mismatch\";\n                }\n                // accept only spaces, digits and dashes\n                if (/[^0-9 \\-]+/.test(value)) {\n                    return false;\n                }\n                var nCheck = 0,\n                    nDigit = 0,\n                    bEven = false;\n\n                value = value.replace(/\\D/g, \"\");\n\n                for (var n = value.length - 1; n >= 0; n--) {\n                    var cDigit = value.charAt(n);\n                    nDigit = parseInt(cDigit, 10);\n                    if (bEven) {\n                        if ((nDigit *= 2) > 9) {\n                            nDigit -= 9;\n                        }\n                    }\n                    nCheck += nDigit;\n                    bEven = !bEven;\n                }\n\n                return (nCheck % 10) === 0;\n            },\n\n            // http://docs.jquery.com/Plugins/Validation/Methods/equalTo\n            equalTo: function (value, element, param) {\n                // bind to the blur event of the target in order to revalidate whenever the target field is updated\n                // TODO find a way to bind the event just once, avoiding the unbind-rebind overhead\n                var target = $(param);\n                if (this.settings.onfocusout) {\n                    target.unbind(\".validate-equalTo\").bind(\"blur.validate-equalTo\", function () {\n                        $(element).valid();\n                    });\n                }\n                return value === target.val();\n            }\n\n        }\n\n    });\n\n// deprecated, use $.validator.format instead\n    $.format = $.validator.format;\n\n}(jQuery));\n\n// ajax mode: abort\n// usage: $.ajax({ mode: \"abort\"[, port: \"uniqueport\"]});\n// if mode:\"abort\" is used, the previous request on that port (port can be undefined) is aborted via XMLHttpRequest.abort()\n(function ($) {\n    var pendingRequests = {};\n    // Use a prefilter if available (1.5+)\n    if ($.ajaxPrefilter) {\n        $.ajaxPrefilter(function (settings, _, xhr) {\n            var port = settings.port;\n            if (settings.mode === \"abort\") {\n                if (pendingRequests[port]) {\n                    pendingRequests[port].abort();\n                }\n                pendingRequests[port] = xhr;\n            }\n        });\n    } else {\n        // Proxy ajax\n        var ajax = $.ajax;\n        $.ajax = function (settings) {\n            var mode = ( \"mode\" in settings ? settings : $.ajaxSettings ).mode,\n                port = ( \"port\" in settings ? settings : $.ajaxSettings ).port;\n            if (mode === \"abort\") {\n                if (pendingRequests[port]) {\n                    pendingRequests[port].abort();\n                }\n                return (pendingRequests[port] = ajax.apply(this, arguments));\n            }\n            return ajax.apply(this, arguments);\n        };\n    }\n}(jQuery));\n\n// provides cross-browser focusin and focusout events\n// IE has native support, in other browsers, use event caputuring (neither bubbles)\n\n// provides delegate(type: String, delegate: Selector, handler: Callback) plugin for easier event delegation\n// handler is only called when $(event.target).is(delegate), in the scope of the jquery-object for event.target\n(function ($) {\n    // only implement if not provided by jQuery core (since 1.4)\n    // TODO verify if jQuery 1.4's implementation is compatible with older jQuery special-event APIs\n    if (!jQuery.event.special.focusin && !jQuery.event.special.focusout && document.addEventListener) {\n        $.each({\n            focus: 'focusin',\n            blur: 'focusout'\n        }, function (original, fix) {\n            $.event.special[fix] = {\n                setup: function () {\n                    this.addEventListener(original, handler, true);\n                },\n                teardown: function () {\n                    this.removeEventListener(original, handler, true);\n                },\n                handler: function (e) {\n                    var args = arguments;\n                    args[0] = $.event.fix(e);\n                    args[0].type = fix;\n                    return $.event.handle.apply(this, args);\n                }\n            };\n            function handler(e) {\n                e = $.event.fix(e);\n                e.type = fix;\n                return $.event.handle.call(this, e);\n            }\n        });\n    }\n    $.extend($.fn, {\n        validateDelegate: function (delegate, type, handler) {\n            return this.bind(type, function (event) {\n                var target = $(event.target);\n                if (target.is(delegate)) {\n                    return handler.apply(target, arguments);\n                }\n            });\n        }\n    });\n}(jQuery));\n\n}));","jquery/jquery.mobile.custom.js":"/*\n* jQuery Mobile v1.4.3\n* http://jquerymobile.com\n*\n* Copyright 2010, 2014 jQuery Foundation, Inc. and other contributors\n* Released under the MIT license.\n* http://jquery.org/license\n*\n*/\n\n(function ( root, doc, factory ) {\n\tif ( typeof define === \"function\" && define.amd ) {\n\t\t// AMD. Register as an anonymous module.\n\t\tdefine( [ \"jquery\" ], function ( $ ) {\n\t\t\tfactory( $, root, doc );\n\t\t\treturn $.mobile;\n\t\t});\n\t} else {\n\t\t// Browser globals\n\t\tfactory( root.jQuery, root, doc );\n\t}\n}( this, document, function ( jQuery, window, document, undefined ) {// This plugin is an experiment for abstracting away the touch and mouse\n// events so that developers don't have to worry about which method of input\n// the device their document is loaded on supports.\n//\n// The idea here is to allow the developer to register listeners for the\n// basic mouse events, such as mousedown, mousemove, mouseup, and click,\n// and the plugin will take care of registering the correct listeners\n// behind the scenes to invoke the listener at the fastest possible time\n// for that device, while still retaining the order of event firing in\n// the traditional mouse environment, should multiple handlers be registered\n// on the same element for different events.\n//\n// The current version exposes the following virtual events to jQuery bind methods:\n// \"vmouseover vmousedown vmousemove vmouseup vclick vmouseout vmousecancel\"\n\n(function( $, window, document, undefined ) {\n\nvar dataPropertyName = \"virtualMouseBindings\",\n\ttouchTargetPropertyName = \"virtualTouchID\",\n\tvirtualEventNames = \"vmouseover vmousedown vmousemove vmouseup vclick vmouseout vmousecancel\".split( \" \" ),\n\ttouchEventProps = \"clientX clientY pageX pageY screenX screenY\".split( \" \" ),\n\tmouseHookProps = $.event.mouseHooks ? $.event.mouseHooks.props : [],\n\tmouseEventProps = $.event.props.concat( mouseHookProps ),\n\tactiveDocHandlers = {},\n\tresetTimerID = 0,\n\tstartX = 0,\n\tstartY = 0,\n\tdidScroll = false,\n\tclickBlockList = [],\n\tblockMouseTriggers = false,\n\tblockTouchTriggers = false,\n\teventCaptureSupported = \"addEventListener\" in document,\n\t$document = $( document ),\n\tnextTouchID = 1,\n\tlastTouchID = 0, threshold,\n\ti;\n\n$.vmouse = {\n\tmoveDistanceThreshold: 10,\n\tclickDistanceThreshold: 10,\n\tresetTimerDuration: 1500\n};\n\nfunction getNativeEvent( event ) {\n\n\twhile ( event && typeof event.originalEvent !== \"undefined\" ) {\n\t\tevent = event.originalEvent;\n\t}\n\treturn event;\n}\n\nfunction createVirtualEvent( event, eventType ) {\n\n\tvar t = event.type,\n\t\toe, props, ne, prop, ct, touch, i, j, len;\n\n\tevent = $.Event( event );\n\tevent.type = eventType;\n\n\toe = event.originalEvent;\n\tprops = $.event.props;\n\n\t// addresses separation of $.event.props in to $.event.mouseHook.props and Issue 3280\n\t// https://github.com/jquery/jquery-mobile/issues/3280\n\tif ( t.search( /^(mouse|click)/ ) > -1 ) {\n\t\tprops = mouseEventProps;\n\t}\n\n\t// copy original event properties over to the new event\n\t// this would happen if we could call $.event.fix instead of $.Event\n\t// but we don't have a way to force an event to be fixed multiple times\n\tif ( oe ) {\n\t\tfor ( i = props.length, prop; i; ) {\n\t\t\tprop = props[ --i ];\n\t\t\tevent[ prop ] = oe[ prop ];\n\t\t}\n\t}\n\n\t// make sure that if the mouse and click virtual events are generated\n\t// without a .which one is defined\n\tif ( t.search(/mouse(down|up)|click/) > -1 && !event.which ) {\n\t\tevent.which = 1;\n\t}\n\n\tif ( t.search(/^touch/) !== -1 ) {\n\t\tne = getNativeEvent( oe );\n\t\tt = ne.touches;\n\t\tct = ne.changedTouches;\n\t\ttouch = ( t && t.length ) ? t[0] : ( ( ct && ct.length ) ? ct[ 0 ] : undefined );\n\n\t\tif ( touch ) {\n\t\t\tfor ( j = 0, len = touchEventProps.length; j < len; j++) {\n\t\t\t\tprop = touchEventProps[ j ];\n\t\t\t\tevent[ prop ] = touch[ prop ];\n\t\t\t}\n\t\t}\n\t}\n\n\treturn event;\n}\n\nfunction getVirtualBindingFlags( element ) {\n\n\tvar flags = {},\n\t\tb, k;\n\n\twhile ( element ) {\n\n\t\tb = $.data( element, dataPropertyName );\n\n\t\tfor (  k in b ) {\n\t\t\tif ( b[ k ] ) {\n\t\t\t\tflags[ k ] = flags.hasVirtualBinding = true;\n\t\t\t}\n\t\t}\n\t\telement = element.parentNode;\n\t}\n\treturn flags;\n}\n\nfunction getClosestElementWithVirtualBinding( element, eventType ) {\n\tvar b;\n\twhile ( element ) {\n\n\t\tb = $.data( element, dataPropertyName );\n\n\t\tif ( b && ( !eventType || b[ eventType ] ) ) {\n\t\t\treturn element;\n\t\t}\n\t\telement = element.parentNode;\n\t}\n\treturn null;\n}\n\nfunction enableTouchBindings() {\n\tblockTouchTriggers = false;\n}\n\nfunction disableTouchBindings() {\n\tblockTouchTriggers = true;\n}\n\nfunction enableMouseBindings() {\n\tlastTouchID = 0;\n\tclickBlockList.length = 0;\n\tblockMouseTriggers = false;\n\n\t// When mouse bindings are enabled, our\n\t// touch bindings are disabled.\n\tdisableTouchBindings();\n}\n\nfunction disableMouseBindings() {\n\t// When mouse bindings are disabled, our\n\t// touch bindings are enabled.\n\tenableTouchBindings();\n}\n\nfunction startResetTimer() {\n\tclearResetTimer();\n\tresetTimerID = setTimeout( function() {\n\t\tresetTimerID = 0;\n\t\tenableMouseBindings();\n\t}, $.vmouse.resetTimerDuration );\n}\n\nfunction clearResetTimer() {\n\tif ( resetTimerID ) {\n\t\tclearTimeout( resetTimerID );\n\t\tresetTimerID = 0;\n\t}\n}\n\nfunction triggerVirtualEvent( eventType, event, flags ) {\n\tvar ve;\n\n\tif ( ( flags && flags[ eventType ] ) ||\n\t\t\t\t( !flags && getClosestElementWithVirtualBinding( event.target, eventType ) ) ) {\n\n\t\tve = createVirtualEvent( event, eventType );\n\n\t\t$( event.target).trigger( ve );\n\t}\n\n\treturn ve;\n}\n\nfunction mouseEventCallback( event ) {\n\tvar touchID = $.data( event.target, touchTargetPropertyName ),\n\t\tve;\n\n\tif ( !blockMouseTriggers && ( !lastTouchID || lastTouchID !== touchID ) ) {\n\t\tve = triggerVirtualEvent( \"v\" + event.type, event );\n\t\tif ( ve ) {\n\t\t\tif ( ve.isDefaultPrevented() ) {\n\t\t\t\tevent.preventDefault();\n\t\t\t}\n\t\t\tif ( ve.isPropagationStopped() ) {\n\t\t\t\tevent.stopPropagation();\n\t\t\t}\n\t\t\tif ( ve.isImmediatePropagationStopped() ) {\n\t\t\t\tevent.stopImmediatePropagation();\n\t\t\t}\n\t\t}\n\t}\n}\n\nfunction handleTouchStart( event ) {\n\n\tvar touches = getNativeEvent( event ).touches,\n\t\ttarget, flags, t;\n\n\tif ( touches && touches.length === 1 ) {\n\n\t\ttarget = event.target;\n\t\tflags = getVirtualBindingFlags( target );\n\n\t\tif ( flags.hasVirtualBinding ) {\n\n\t\t\tlastTouchID = nextTouchID++;\n\t\t\t$.data( target, touchTargetPropertyName, lastTouchID );\n\n\t\t\tclearResetTimer();\n\n\t\t\tdisableMouseBindings();\n\t\t\tdidScroll = false;\n\n\t\t\tt = getNativeEvent( event ).touches[ 0 ];\n\t\t\tstartX = t.pageX;\n\t\t\tstartY = t.pageY;\n\n\t\t\ttriggerVirtualEvent( \"vmouseover\", event, flags );\n\t\t\ttriggerVirtualEvent( \"vmousedown\", event, flags );\n\t\t}\n\t}\n}\n\nfunction handleScroll( event ) {\n\tif ( blockTouchTriggers ) {\n\t\treturn;\n\t}\n\n\tif ( !didScroll ) {\n\t\ttriggerVirtualEvent( \"vmousecancel\", event, getVirtualBindingFlags( event.target ) );\n\t}\n\n\tdidScroll = true;\n\tstartResetTimer();\n}\n\nfunction handleTouchMove( event ) {\n\tif ( blockTouchTriggers ) {\n\t\treturn;\n\t}\n\n\tvar t = getNativeEvent( event ).touches[ 0 ],\n\t\tdidCancel = didScroll,\n\t\tmoveThreshold = $.vmouse.moveDistanceThreshold,\n\t\tflags = getVirtualBindingFlags( event.target );\n\n\t\tdidScroll = didScroll ||\n\t\t\t( Math.abs( t.pageX - startX ) > moveThreshold ||\n\t\t\t\tMath.abs( t.pageY - startY ) > moveThreshold );\n\n\tif ( didScroll && !didCancel ) {\n\t\ttriggerVirtualEvent( \"vmousecancel\", event, flags );\n\t}\n\n\ttriggerVirtualEvent( \"vmousemove\", event, flags );\n\tstartResetTimer();\n}\n\nfunction handleTouchEnd( event ) {\n\tif ( blockTouchTriggers ) {\n\t\treturn;\n\t}\n\n\tdisableTouchBindings();\n\n\tvar flags = getVirtualBindingFlags( event.target ),\n\t\tve, t;\n\ttriggerVirtualEvent( \"vmouseup\", event, flags );\n\n\tif ( !didScroll ) {\n\t\tve = triggerVirtualEvent( \"vclick\", event, flags );\n\t\tif ( ve && ve.isDefaultPrevented() ) {\n\t\t\t// The target of the mouse events that follow the touchend\n\t\t\t// event don't necessarily match the target used during the\n\t\t\t// touch. This means we need to rely on coordinates for blocking\n\t\t\t// any click that is generated.\n\t\t\tt = getNativeEvent( event ).changedTouches[ 0 ];\n\t\t\tclickBlockList.push({\n\t\t\t\ttouchID: lastTouchID,\n\t\t\t\tx: t.clientX,\n\t\t\t\ty: t.clientY\n\t\t\t});\n\n\t\t\t// Prevent any mouse events that follow from triggering\n\t\t\t// virtual event notifications.\n\t\t\tblockMouseTriggers = true;\n\t\t}\n\t}\n\ttriggerVirtualEvent( \"vmouseout\", event, flags);\n\tdidScroll = false;\n\n\tstartResetTimer();\n}\n\nfunction hasVirtualBindings( ele ) {\n\tvar bindings = $.data( ele, dataPropertyName ),\n\t\tk;\n\n\tif ( bindings ) {\n\t\tfor ( k in bindings ) {\n\t\t\tif ( bindings[ k ] ) {\n\t\t\t\treturn true;\n\t\t\t}\n\t\t}\n\t}\n\treturn false;\n}\n\nfunction dummyMouseHandler() {}\n\nfunction getSpecialEventObject( eventType ) {\n\tvar realType = eventType.substr( 1 );\n\n\treturn {\n\t\tsetup: function(/* data, namespace */) {\n\t\t\t// If this is the first virtual mouse binding for this element,\n\t\t\t// add a bindings object to its data.\n\n\t\t\tif ( !hasVirtualBindings( this ) ) {\n\t\t\t\t$.data( this, dataPropertyName, {} );\n\t\t\t}\n\n\t\t\t// If setup is called, we know it is the first binding for this\n\t\t\t// eventType, so initialize the count for the eventType to zero.\n\t\t\tvar bindings = $.data( this, dataPropertyName );\n\t\t\tbindings[ eventType ] = true;\n\n\t\t\t// If this is the first virtual mouse event for this type,\n\t\t\t// register a global handler on the document.\n\n\t\t\tactiveDocHandlers[ eventType ] = ( activeDocHandlers[ eventType ] || 0 ) + 1;\n\n\t\t\tif ( activeDocHandlers[ eventType ] === 1 ) {\n\t\t\t\t$document.bind( realType, mouseEventCallback );\n\t\t\t}\n\n\t\t\t// Some browsers, like Opera Mini, won't dispatch mouse/click events\n\t\t\t// for elements unless they actually have handlers registered on them.\n\t\t\t// To get around this, we register dummy handlers on the elements.\n\n\t\t\t$( this ).bind( realType, dummyMouseHandler );\n\n\t\t\t// For now, if event capture is not supported, we rely on mouse handlers.\n\t\t\tif ( eventCaptureSupported ) {\n\t\t\t\t// If this is the first virtual mouse binding for the document,\n\t\t\t\t// register our touchstart handler on the document.\n\n\t\t\t\tactiveDocHandlers[ \"touchstart\" ] = ( activeDocHandlers[ \"touchstart\" ] || 0) + 1;\n\n\t\t\t\tif ( activeDocHandlers[ \"touchstart\" ] === 1 ) {\n\t\t\t\t\t$document.bind( \"touchstart\", handleTouchStart )\n\t\t\t\t\t\t.bind( \"touchend\", handleTouchEnd )\n\n\t\t\t\t\t\t// On touch platforms, touching the screen and then dragging your finger\n\t\t\t\t\t\t// causes the window content to scroll after some distance threshold is\n\t\t\t\t\t\t// exceeded. On these platforms, a scroll prevents a click event from being\n\t\t\t\t\t\t// dispatched, and on some platforms, even the touchend is suppressed. To\n\t\t\t\t\t\t// mimic the suppression of the click event, we need to watch for a scroll\n\t\t\t\t\t\t// event. Unfortunately, some platforms like iOS don't dispatch scroll\n\t\t\t\t\t\t// events until *AFTER* the user lifts their finger (touchend). This means\n\t\t\t\t\t\t// we need to watch both scroll and touchmove events to figure out whether\n\t\t\t\t\t\t// or not a scroll happenens before the touchend event is fired.\n\n\t\t\t\t\t\t.bind( \"touchmove\", handleTouchMove )\n\t\t\t\t\t\t.bind( \"scroll\", handleScroll );\n\t\t\t\t}\n\t\t\t}\n\t\t},\n\n\t\tteardown: function(/* data, namespace */) {\n\t\t\t// If this is the last virtual binding for this eventType,\n\t\t\t// remove its global handler from the document.\n\n\t\t\t--activeDocHandlers[ eventType ];\n\n\t\t\tif ( !activeDocHandlers[ eventType ] ) {\n\t\t\t\t$document.unbind( realType, mouseEventCallback );\n\t\t\t}\n\n\t\t\tif ( eventCaptureSupported ) {\n\t\t\t\t// If this is the last virtual mouse binding in existence,\n\t\t\t\t// remove our document touchstart listener.\n\n\t\t\t\t--activeDocHandlers[ \"touchstart\" ];\n\n\t\t\t\tif ( !activeDocHandlers[ \"touchstart\" ] ) {\n\t\t\t\t\t$document.unbind( \"touchstart\", handleTouchStart )\n\t\t\t\t\t\t.unbind( \"touchmove\", handleTouchMove )\n\t\t\t\t\t\t.unbind( \"touchend\", handleTouchEnd )\n\t\t\t\t\t\t.unbind( \"scroll\", handleScroll );\n\t\t\t\t}\n\t\t\t}\n\n\t\t\tvar $this = $( this ),\n\t\t\t\tbindings = $.data( this, dataPropertyName );\n\n\t\t\t// teardown may be called when an element was\n\t\t\t// removed from the DOM. If this is the case,\n\t\t\t// jQuery core may have already stripped the element\n\t\t\t// of any data bindings so we need to check it before\n\t\t\t// using it.\n\t\t\tif ( bindings ) {\n\t\t\t\tbindings[ eventType ] = false;\n\t\t\t}\n\n\t\t\t// Unregister the dummy event handler.\n\n\t\t\t$this.unbind( realType, dummyMouseHandler );\n\n\t\t\t// If this is the last virtual mouse binding on the\n\t\t\t// element, remove the binding data from the element.\n\n\t\t\tif ( !hasVirtualBindings( this ) ) {\n\t\t\t\t$this.removeData( dataPropertyName );\n\t\t\t}\n\t\t}\n\t};\n}\n\n// Expose our custom events to the jQuery bind/unbind mechanism.\n\nfor ( i = 0; i < virtualEventNames.length; i++ ) {\n\t$.event.special[ virtualEventNames[ i ] ] = getSpecialEventObject( virtualEventNames[ i ] );\n}\n\n// Add a capture click handler to block clicks.\n// Note that we require event capture support for this so if the device\n// doesn't support it, we punt for now and rely solely on mouse events.\nif ( eventCaptureSupported ) {\n\tdocument.addEventListener( \"click\", function( e ) {\n\t\tvar cnt = clickBlockList.length,\n\t\t\ttarget = e.target,\n\t\t\tx, y, ele, i, o, touchID;\n\n\t\tif ( cnt ) {\n\t\t\tx = e.clientX;\n\t\t\ty = e.clientY;\n\t\t\tthreshold = $.vmouse.clickDistanceThreshold;\n\n\t\t\t// The idea here is to run through the clickBlockList to see if\n\t\t\t// the current click event is in the proximity of one of our\n\t\t\t// vclick events that had preventDefault() called on it. If we find\n\t\t\t// one, then we block the click.\n\t\t\t//\n\t\t\t// Why do we have to rely on proximity?\n\t\t\t//\n\t\t\t// Because the target of the touch event that triggered the vclick\n\t\t\t// can be different from the target of the click event synthesized\n\t\t\t// by the browser. The target of a mouse/click event that is synthesized\n\t\t\t// from a touch event seems to be implementation specific. For example,\n\t\t\t// some browsers will fire mouse/click events for a link that is near\n\t\t\t// a touch event, even though the target of the touchstart/touchend event\n\t\t\t// says the user touched outside the link. Also, it seems that with most\n\t\t\t// browsers, the target of the mouse/click event is not calculated until the\n\t\t\t// time it is dispatched, so if you replace an element that you touched\n\t\t\t// with another element, the target of the mouse/click will be the new\n\t\t\t// element underneath that point.\n\t\t\t//\n\t\t\t// Aside from proximity, we also check to see if the target and any\n\t\t\t// of its ancestors were the ones that blocked a click. This is necessary\n\t\t\t// because of the strange mouse/click target calculation done in the\n\t\t\t// Android 2.1 browser, where if you click on an element, and there is a\n\t\t\t// mouse/click handler on one of its ancestors, the target will be the\n\t\t\t// innermost child of the touched element, even if that child is no where\n\t\t\t// near the point of touch.\n\n\t\t\tele = target;\n\n\t\t\twhile ( ele ) {\n\t\t\t\tfor ( i = 0; i < cnt; i++ ) {\n\t\t\t\t\to = clickBlockList[ i ];\n\t\t\t\t\ttouchID = 0;\n\n\t\t\t\t\tif ( ( ele === target && Math.abs( o.x - x ) < threshold && Math.abs( o.y - y ) < threshold ) ||\n\t\t\t\t\t\t\t\t$.data( ele, touchTargetPropertyName ) === o.touchID ) {\n\t\t\t\t\t\t// XXX: We may want to consider removing matches from the block list\n\t\t\t\t\t\t//      instead of waiting for the reset timer to fire.\n\t\t\t\t\t\te.preventDefault();\n\t\t\t\t\t\te.stopPropagation();\n\t\t\t\t\t\treturn;\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t\tele = ele.parentNode;\n\t\t\t}\n\t\t}\n\t}, true);\n}\n})( jQuery, window, document );\n\n(function( $ ) {\n\t$.mobile = {};\n}( jQuery ));\n\n\t(function( $, undefined ) {\n\t\tvar support = {\n\t\t\ttouch: \"ontouchend\" in document\n\t\t};\n\n\t\t$.mobile.support = $.mobile.support || {};\n\t\t$.extend( $.support, support );\n\t\t$.extend( $.mobile.support, support );\n\t}( jQuery ));\n\n\n(function( $, window, undefined ) {\n\tvar $document = $( document ),\n\t\tsupportTouch = $.mobile.support.touch,\n\t\tscrollEvent = \"touchmove scroll\",\n\t\ttouchStartEvent = supportTouch ? \"touchstart\" : \"mousedown\",\n\t\ttouchStopEvent = supportTouch ? \"touchend\" : \"mouseup\",\n\t\ttouchMoveEvent = supportTouch ? \"touchmove\" : \"mousemove\";\n\n\t// setup new event shortcuts\n\t$.each( ( \"touchstart touchmove touchend \" +\n\t\t\"tap taphold \" +\n\t\t\"swipe swipeleft swiperight \" +\n\t\t\"scrollstart scrollstop\" ).split( \" \" ), function( i, name ) {\n\n\t\t$.fn[ name ] = function( fn ) {\n\t\t\treturn fn ? this.bind( name, fn ) : this.trigger( name );\n\t\t};\n\n\t\t// jQuery < 1.8\n\t\tif ( $.attrFn ) {\n\t\t\t$.attrFn[ name ] = true;\n\t\t}\n\t});\n\n\tfunction triggerCustomEvent( obj, eventType, event, bubble ) {\n\t\tvar originalType = event.type;\n\t\tevent.type = eventType;\n\t\tif ( bubble ) {\n\t\t\t$.event.trigger( event, undefined, obj );\n\t\t} else {\n\t\t\t$.event.dispatch.call( obj, event );\n\t\t}\n\t\tevent.type = originalType;\n\t}\n\n\t// also handles scrollstop\n\t$.event.special.scrollstart = {\n\n\t\tenabled: true,\n\t\tsetup: function() {\n\n\t\t\tvar thisObject = this,\n\t\t\t\t$this = $( thisObject ),\n\t\t\t\tscrolling,\n\t\t\t\ttimer;\n\n\t\t\tfunction trigger( event, state ) {\n\t\t\t\tscrolling = state;\n\t\t\t\ttriggerCustomEvent( thisObject, scrolling ? \"scrollstart\" : \"scrollstop\", event );\n\t\t\t}\n\n\t\t\t// iPhone triggers scroll after a small delay; use touchmove instead\n\t\t\t$this.bind( scrollEvent, function( event ) {\n\n\t\t\t\tif ( !$.event.special.scrollstart.enabled ) {\n\t\t\t\t\treturn;\n\t\t\t\t}\n\n\t\t\t\tif ( !scrolling ) {\n\t\t\t\t\ttrigger( event, true );\n\t\t\t\t}\n\n\t\t\t\tclearTimeout( timer );\n\t\t\t\ttimer = setTimeout( function() {\n\t\t\t\t\ttrigger( event, false );\n\t\t\t\t}, 50 );\n\t\t\t});\n\t\t},\n\t\tteardown: function() {\n\t\t\t$( this ).unbind( scrollEvent );\n\t\t}\n\t};\n\n\t// also handles taphold\n\t$.event.special.tap = {\n\t\ttapholdThreshold: 750,\n\t\temitTapOnTaphold: true,\n\t\tsetup: function() {\n\t\t\tvar thisObject = this,\n\t\t\t\t$this = $( thisObject ),\n\t\t\t\tisTaphold = false;\n\n\t\t\t$this.bind( \"vmousedown\", function( event ) {\n\t\t\t\tisTaphold = false;\n\t\t\t\tif ( event.which && event.which !== 1 ) {\n\t\t\t\t\treturn false;\n\t\t\t\t}\n\n\t\t\t\tvar origTarget = event.target,\n\t\t\t\t\ttimer;\n\n\t\t\t\tfunction clearTapTimer() {\n\t\t\t\t\tclearTimeout( timer );\n\t\t\t\t}\n\n\t\t\t\tfunction clearTapHandlers() {\n\t\t\t\t\tclearTapTimer();\n\n\t\t\t\t\t$this.unbind( \"vclick\", clickHandler )\n\t\t\t\t\t\t.unbind( \"vmouseup\", clearTapTimer );\n\t\t\t\t\t$document.unbind( \"vmousecancel\", clearTapHandlers );\n\t\t\t\t}\n\n\t\t\t\tfunction clickHandler( event ) {\n\t\t\t\t\tclearTapHandlers();\n\n\t\t\t\t\t// ONLY trigger a 'tap' event if the start target is\n\t\t\t\t\t// the same as the stop target.\n\t\t\t\t\tif ( !isTaphold && origTarget === event.target ) {\n\t\t\t\t\t\ttriggerCustomEvent( thisObject, \"tap\", event );\n\t\t\t\t\t} else if ( isTaphold ) {\n\t\t\t\t\t\tevent.preventDefault();\n\t\t\t\t\t}\n\t\t\t\t}\n\n\t\t\t\t$this.bind( \"vmouseup\", clearTapTimer )\n\t\t\t\t\t.bind( \"vclick\", clickHandler );\n\t\t\t\t$document.bind( \"vmousecancel\", clearTapHandlers );\n\n\t\t\t\ttimer = setTimeout( function() {\n\t\t\t\t\tif ( !$.event.special.tap.emitTapOnTaphold ) {\n\t\t\t\t\t\tisTaphold = true;\n\t\t\t\t\t}\n\t\t\t\t\ttriggerCustomEvent( thisObject, \"taphold\", $.Event( \"taphold\", { target: origTarget } ) );\n\t\t\t\t}, $.event.special.tap.tapholdThreshold );\n\t\t\t});\n\t\t},\n\t\tteardown: function() {\n\t\t\t$( this ).unbind( \"vmousedown\" ).unbind( \"vclick\" ).unbind( \"vmouseup\" );\n\t\t\t$document.unbind( \"vmousecancel\" );\n\t\t}\n\t};\n\n\t// Also handles swipeleft, swiperight\n\t$.event.special.swipe = {\n\n\t\t// More than this horizontal displacement, and we will suppress scrolling.\n\t\tscrollSupressionThreshold: 30,\n\n\t\t// More time than this, and it isn't a swipe.\n\t\tdurationThreshold: 1000,\n\n\t\t// Swipe horizontal displacement must be more than this.\n\t\thorizontalDistanceThreshold: 30,\n\n\t\t// Swipe vertical displacement must be less than this.\n\t\tverticalDistanceThreshold: 30,\n\n\t\tgetLocation: function ( event ) {\n\t\t\tvar winPageX = window.pageXOffset,\n\t\t\t\twinPageY = window.pageYOffset,\n\t\t\t\tx = event.clientX,\n\t\t\t\ty = event.clientY;\n\n\t\t\tif ( event.pageY === 0 && Math.floor( y ) > Math.floor( event.pageY ) ||\n\t\t\t\tevent.pageX === 0 && Math.floor( x ) > Math.floor( event.pageX ) ) {\n\n\t\t\t\t// iOS4 clientX/clientY have the value that should have been\n\t\t\t\t// in pageX/pageY. While pageX/page/ have the value 0\n\t\t\t\tx = x - winPageX;\n\t\t\t\ty = y - winPageY;\n\t\t\t} else if ( y < ( event.pageY - winPageY) || x < ( event.pageX - winPageX ) ) {\n\n\t\t\t\t// Some Android browsers have totally bogus values for clientX/Y\n\t\t\t\t// when scrolling/zooming a page. Detectable since clientX/clientY\n\t\t\t\t// should never be smaller than pageX/pageY minus page scroll\n\t\t\t\tx = event.pageX - winPageX;\n\t\t\t\ty = event.pageY - winPageY;\n\t\t\t}\n\n\t\t\treturn {\n\t\t\t\tx: x,\n\t\t\t\ty: y\n\t\t\t};\n\t\t},\n\n\t\tstart: function( event ) {\n\t\t\tvar data = event.originalEvent.touches ?\n\t\t\t\t\tevent.originalEvent.touches[ 0 ] : event,\n\t\t\t\tlocation = $.event.special.swipe.getLocation( data );\n\t\t\treturn {\n\t\t\t\t\t\ttime: ( new Date() ).getTime(),\n\t\t\t\t\t\tcoords: [ location.x, location.y ],\n\t\t\t\t\t\torigin: $( event.target )\n\t\t\t\t\t};\n\t\t},\n\n\t\tstop: function( event ) {\n\t\t\tvar data = event.originalEvent.touches ?\n\t\t\t\t\tevent.originalEvent.touches[ 0 ] : event,\n\t\t\t\tlocation = $.event.special.swipe.getLocation( data );\n\t\t\treturn {\n\t\t\t\t\t\ttime: ( new Date() ).getTime(),\n\t\t\t\t\t\tcoords: [ location.x, location.y ]\n\t\t\t\t\t};\n\t\t},\n\n\t\thandleSwipe: function( start, stop, thisObject, origTarget ) {\n\t\t\tif ( stop.time - start.time < $.event.special.swipe.durationThreshold &&\n\t\t\t\tMath.abs( start.coords[ 0 ] - stop.coords[ 0 ] ) > $.event.special.swipe.horizontalDistanceThreshold &&\n\t\t\t\tMath.abs( start.coords[ 1 ] - stop.coords[ 1 ] ) < $.event.special.swipe.verticalDistanceThreshold ) {\n\t\t\t\tvar direction = start.coords[0] > stop.coords[ 0 ] ? \"swipeleft\" : \"swiperight\";\n\n\t\t\t\ttriggerCustomEvent( thisObject, \"swipe\", $.Event( \"swipe\", { target: origTarget, swipestart: start, swipestop: stop }), true );\n\t\t\t\ttriggerCustomEvent( thisObject, direction,$.Event( direction, { target: origTarget, swipestart: start, swipestop: stop } ), true );\n\t\t\t\treturn true;\n\t\t\t}\n\t\t\treturn false;\n\n\t\t},\n\n\t\t// This serves as a flag to ensure that at most one swipe event event is\n\t\t// in work at any given time\n\t\teventInProgress: false,\n\n\t\tsetup: function() {\n\t\t\tvar events,\n\t\t\t\tthisObject = this,\n\t\t\t\t$this = $( thisObject ),\n\t\t\t\tcontext = {};\n\n\t\t\t// Retrieve the events data for this element and add the swipe context\n\t\t\tevents = $.data( this, \"mobile-events\" );\n\t\t\tif ( !events ) {\n\t\t\t\tevents = { length: 0 };\n\t\t\t\t$.data( this, \"mobile-events\", events );\n\t\t\t}\n\t\t\tevents.length++;\n\t\t\tevents.swipe = context;\n\n\t\t\tcontext.start = function( event ) {\n\n\t\t\t\t// Bail if we're already working on a swipe event\n\t\t\t\tif ( $.event.special.swipe.eventInProgress ) {\n\t\t\t\t\treturn;\n\t\t\t\t}\n\t\t\t\t$.event.special.swipe.eventInProgress = true;\n\n\t\t\t\tvar stop,\n\t\t\t\t\tstart = $.event.special.swipe.start( event ),\n\t\t\t\t\torigTarget = event.target,\n\t\t\t\t\temitted = false;\n\n\t\t\t\tcontext.move = function( event ) {\n\t\t\t\t\tif ( !start ) {\n\t\t\t\t\t\treturn;\n\t\t\t\t\t}\n\n\t\t\t\t\tstop = $.event.special.swipe.stop( event );\n\t\t\t\t\tif ( !emitted ) {\n\t\t\t\t\t\temitted = $.event.special.swipe.handleSwipe( start, stop, thisObject, origTarget );\n\t\t\t\t\t\tif ( emitted ) {\n\n\t\t\t\t\t\t\t// Reset the context to make way for the next swipe event\n\t\t\t\t\t\t\t$.event.special.swipe.eventInProgress = false;\n\t\t\t\t\t\t}\n\t\t\t\t\t}\n\t\t\t\t\t// prevent scrolling\n\t\t\t\t\tif ( Math.abs( start.coords[ 0 ] - stop.coords[ 0 ] ) > $.event.special.swipe.scrollSupressionThreshold ) {\n\t\t\t\t\t\tevent.preventDefault();\n\t\t\t\t\t}\n\t\t\t\t};\n\n\t\t\t\tcontext.stop = function() {\n\t\t\t\t\t\temitted = true;\n\n\t\t\t\t\t\t// Reset the context to make way for the next swipe event\n\t\t\t\t\t\t$.event.special.swipe.eventInProgress = false;\n\t\t\t\t\t\t$document.off( touchMoveEvent, context.move );\n\t\t\t\t\t\tcontext.move = null;\n\t\t\t\t};\n\n\t\t\t\t$document.on( touchMoveEvent, context.move )\n\t\t\t\t\t.one( touchStopEvent, context.stop );\n\t\t\t};\n\t\t\t$this.on( touchStartEvent, context.start );\n\t\t},\n\n\t\tteardown: function() {\n\t\t\tvar events, context;\n\n\t\t\tevents = $.data( this, \"mobile-events\" );\n\t\t\tif ( events ) {\n\t\t\t\tcontext = events.swipe;\n\t\t\t\tdelete events.swipe;\n\t\t\t\tevents.length--;\n\t\t\t\tif ( events.length === 0 ) {\n\t\t\t\t\t$.removeData( this, \"mobile-events\" );\n\t\t\t\t}\n\t\t\t}\n\n\t\t\tif ( context ) {\n\t\t\t\tif ( context.start ) {\n\t\t\t\t\t$( this ).off( touchStartEvent, context.start );\n\t\t\t\t}\n\t\t\t\tif ( context.move ) {\n\t\t\t\t\t$document.off( touchMoveEvent, context.move );\n\t\t\t\t}\n\t\t\t\tif ( context.stop ) {\n\t\t\t\t\t$document.off( touchStopEvent, context.stop );\n\t\t\t\t}\n\t\t\t}\n\t\t}\n\t};\n\t$.each({\n\t\tscrollstop: \"scrollstart\",\n\t\ttaphold: \"tap\",\n\t\tswipeleft: \"swipe.left\",\n\t\tswiperight: \"swipe.right\"\n\t}, function( event, sourceEvent ) {\n\n\t\t$.event.special[ event ] = {\n\t\t\tsetup: function() {\n\t\t\t\t$( this ).bind( sourceEvent, $.noop );\n\t\t\t},\n\t\t\tteardown: function() {\n\t\t\t\t$( this ).unbind( sourceEvent );\n\t\t\t}\n\t\t};\n\t});\n\n})( jQuery, this );\n\n\n}));\n","jquery/jquery.cookie.js":"/*jshint eqnull:true */\n/*!\n * jQuery Cookie Plugin v1.1\n * https://github.com/carhartl/jquery-cookie\n *\n * Copyright 2011, Klaus Hartl\n * Dual licensed under the MIT or GPL Version 2 licenses.\n * http://www.opensource.org/licenses/mit-license.php\n * http://www.opensource.org/licenses/GPL-2.0\n */\n(function (factory) {\n    if (typeof define === 'function' && define.amd) {\n        define([\"jquery\"], factory);\n    } else {\n        factory(jQuery);\n    }\n}(function ($) {\n\n    var pluses = /\\+/g;\n    function raw(s) {\n        return s;\n    }\n    function decoded(s) {\n        return decodeURIComponent(s.replace(pluses, ' '));\n    }\n\n    $.cookie = function(key, value, options) {\n\n        // key and at least value given, set cookie...\n        if (arguments.length > 1 && (!/Object/.test(Object.prototype.toString.call(value)) || value == null)) {\n            options = $.extend({}, $.cookie.defaults, options);\n\n            if (value == null) {\n                options.expires = -1;\n            }\n\n            if (typeof options.expires === 'number') {\n                var days = options.expires, t = options.expires = new Date();\n                t.setDate(t.getDate() + days);\n            }\n\n            value = String(value);\n\n            return (document.cookie = [\n                encodeURIComponent(key), '=', options.raw ? value : encodeURIComponent(value),\n                options.expires ? '; expires=' + options.expires.toUTCString() : '', // use expires attribute, max-age is not supported by IE\n                options.path    ? '; path=' + options.path : '',\n                options.domain  ? '; domain=' + options.domain : '',\n                options.secure  ? '; secure' : ''\n            ].join(''));\n        }\n\n        // key and possibly options given, get cookie...\n        options = value || $.cookie.defaults || {};\n        var decode = options.raw ? raw : decoded;\n        var cookies = document.cookie.split('; ');\n        for (var i = 0, parts; (parts = cookies[i] && cookies[i].split('=')); i++) {\n            if (decode(parts.shift()) === key) {\n                return decode(parts.join('='));\n            }\n        }\n        return null;\n    };\n\n    $.cookie.defaults = {};\n\n}));\n","jquery/jquery-ui-timepicker-addon.js":"/*! jQuery Timepicker Addon - v1.4.3 - 2013-11-30\n* http://trentrichardson.com/examples/timepicker\n* Copyright (c) 2013 Trent Richardson; Licensed MIT */\n(function (factory) {\n    if (typeof define === 'function' && define.amd) {\n        define([\n            \"jquery\",\n            \"jquery/ui\"\n        ], factory);\n    } else {\n        factory(jQuery);\n    }\n}(function ($) {\n\n\t/*\n\t* Lets not redefine timepicker, Prevent \"Uncaught RangeError: Maximum call stack size exceeded\"\n\t*/\n\t$.ui.timepicker = $.ui.timepicker || {};\n\tif ($.ui.timepicker.version) {\n\t\treturn;\n\t}\n\n\t/*\n\t* Extend jQueryUI, get it started with our version number\n\t*/\n\t$.extend($.ui, {\n\t\ttimepicker: {\n\t\t\tversion: \"1.4.3\"\n\t\t}\n\t});\n\n\t/* \n\t* Timepicker manager.\n\t* Use the singleton instance of this class, $.timepicker, to interact with the time picker.\n\t* Settings for (groups of) time pickers are maintained in an instance object,\n\t* allowing multiple different settings on the same page.\n\t*/\n\tvar Timepicker = function () {\n\t\tthis.regional = []; // Available regional settings, indexed by language code\n\t\tthis.regional[''] = { // Default regional settings\n\t\t\tcurrentText: 'Now',\n\t\t\tcloseText: 'Done',\n\t\t\tamNames: ['AM', 'A'],\n\t\t\tpmNames: ['PM', 'P'],\n\t\t\ttimeFormat: 'HH:mm',\n\t\t\ttimeSuffix: '',\n\t\t\ttimeOnlyTitle: 'Choose Time',\n\t\t\ttimeText: 'Time',\n\t\t\thourText: 'Hour',\n\t\t\tminuteText: 'Minute',\n\t\t\tsecondText: 'Second',\n\t\t\tmillisecText: 'Millisecond',\n\t\t\tmicrosecText: 'Microsecond',\n\t\t\ttimezoneText: 'Time Zone',\n\t\t\tisRTL: false\n\t\t};\n\t\tthis._defaults = { // Global defaults for all the datetime picker instances\n\t\t\tshowButtonPanel: true,\n\t\t\ttimeOnly: false,\n\t\t\tshowHour: null,\n\t\t\tshowMinute: null,\n\t\t\tshowSecond: null,\n\t\t\tshowMillisec: null,\n\t\t\tshowMicrosec: null,\n\t\t\tshowTimezone: null,\n\t\t\tshowTime: true,\n\t\t\tstepHour: 1,\n\t\t\tstepMinute: 1,\n\t\t\tstepSecond: 1,\n\t\t\tstepMillisec: 1,\n\t\t\tstepMicrosec: 1,\n\t\t\thour: 0,\n\t\t\tminute: 0,\n\t\t\tsecond: 0,\n\t\t\tmillisec: 0,\n\t\t\tmicrosec: 0,\n\t\t\ttimezone: null,\n\t\t\thourMin: 0,\n\t\t\tminuteMin: 0,\n\t\t\tsecondMin: 0,\n\t\t\tmillisecMin: 0,\n\t\t\tmicrosecMin: 0,\n\t\t\thourMax: 23,\n\t\t\tminuteMax: 59,\n\t\t\tsecondMax: 59,\n\t\t\tmillisecMax: 999,\n\t\t\tmicrosecMax: 999,\n\t\t\tminDateTime: null,\n\t\t\tmaxDateTime: null,\n\t\t\tonSelect: null,\n\t\t\thourGrid: 0,\n\t\t\tminuteGrid: 0,\n\t\t\tsecondGrid: 0,\n\t\t\tmillisecGrid: 0,\n\t\t\tmicrosecGrid: 0,\n\t\t\talwaysSetTime: true,\n\t\t\tseparator: ' ',\n\t\t\taltFieldTimeOnly: true,\n\t\t\taltTimeFormat: null,\n\t\t\taltSeparator: null,\n\t\t\taltTimeSuffix: null,\n\t\t\tpickerTimeFormat: null,\n\t\t\tpickerTimeSuffix: null,\n\t\t\tshowTimepicker: true,\n\t\t\ttimezoneList: null,\n\t\t\taddSliderAccess: false,\n\t\t\tsliderAccessArgs: null,\n\t\t\tcontrolType: 'slider',\n\t\t\tdefaultValue: null,\n\t\t\tparse: 'strict'\n\t\t};\n\t\t$.extend(this._defaults, this.regional['']);\n\t};\n\n\t$.extend(Timepicker.prototype, {\n\t\t$input: null,\n\t\t$altInput: null,\n\t\t$timeObj: null,\n\t\tinst: null,\n\t\thour_slider: null,\n\t\tminute_slider: null,\n\t\tsecond_slider: null,\n\t\tmillisec_slider: null,\n\t\tmicrosec_slider: null,\n\t\ttimezone_select: null,\n\t\thour: 0,\n\t\tminute: 0,\n\t\tsecond: 0,\n\t\tmillisec: 0,\n\t\tmicrosec: 0,\n\t\ttimezone: null,\n\t\thourMinOriginal: null,\n\t\tminuteMinOriginal: null,\n\t\tsecondMinOriginal: null,\n\t\tmillisecMinOriginal: null,\n\t\tmicrosecMinOriginal: null,\n\t\thourMaxOriginal: null,\n\t\tminuteMaxOriginal: null,\n\t\tsecondMaxOriginal: null,\n\t\tmillisecMaxOriginal: null,\n\t\tmicrosecMaxOriginal: null,\n\t\tampm: '',\n\t\tformattedDate: '',\n\t\tformattedTime: '',\n\t\tformattedDateTime: '',\n\t\ttimezoneList: null,\n\t\tunits: ['hour', 'minute', 'second', 'millisec', 'microsec'],\n\t\tsupport: {},\n\t\tcontrol: null,\n\n\t\t/* \n\t\t* Override the default settings for all instances of the time picker.\n\t\t* @param  {Object} settings  object - the new settings to use as defaults (anonymous object)\n\t\t* @return {Object} the manager object\n\t\t*/\n\t\tsetDefaults: function (settings) {\n\t\t\textendRemove(this._defaults, settings || {});\n\t\t\treturn this;\n\t\t},\n\n\t\t/*\n\t\t* Create a new Timepicker instance\n\t\t*/\n\t\t_newInst: function ($input, opts) {\n\t\t\tvar tp_inst = new Timepicker(),\n\t\t\t\tinlineSettings = {},\n\t\t\t\tfns = {},\n\t\t\t\toverrides, i;\n\n\t\t\tfor (var attrName in this._defaults) {\n\t\t\t\tif (this._defaults.hasOwnProperty(attrName)) {\n\t\t\t\t\tvar attrValue = $input.attr('time:' + attrName);\n\t\t\t\t\tif (attrValue) {\n\t\t\t\t\t\ttry {\n\t\t\t\t\t\t\tinlineSettings[attrName] = eval(attrValue);\n\t\t\t\t\t\t} catch (err) {\n\t\t\t\t\t\t\tinlineSettings[attrName] = attrValue;\n\t\t\t\t\t\t}\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t}\n\n\t\t\toverrides = {\n\t\t\t\tbeforeShow: function (input, dp_inst) {\n\t\t\t\t\tif ($.isFunction(tp_inst._defaults.evnts.beforeShow)) {\n\t\t\t\t\t\treturn tp_inst._defaults.evnts.beforeShow.call($input[0], input, dp_inst, tp_inst);\n\t\t\t\t\t}\n\t\t\t\t},\n\t\t\t\tonChangeMonthYear: function (year, month, dp_inst) {\n\t\t\t\t\t// Update the time as well : this prevents the time from disappearing from the $input field.\n\t\t\t\t\ttp_inst._updateDateTime(dp_inst);\n\t\t\t\t\tif ($.isFunction(tp_inst._defaults.evnts.onChangeMonthYear)) {\n\t\t\t\t\t\ttp_inst._defaults.evnts.onChangeMonthYear.call($input[0], year, month, dp_inst, tp_inst);\n\t\t\t\t\t}\n\t\t\t\t},\n\t\t\t\tonClose: function (dateText, dp_inst) {\n\t\t\t\t\tif (tp_inst.timeDefined === true && $input.val() !== '') {\n\t\t\t\t\t\ttp_inst._updateDateTime(dp_inst);\n\t\t\t\t\t}\n\t\t\t\t\tif ($.isFunction(tp_inst._defaults.evnts.onClose)) {\n\t\t\t\t\t\ttp_inst._defaults.evnts.onClose.call($input[0], dateText, dp_inst, tp_inst);\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t};\n\t\t\tfor (i in overrides) {\n\t\t\t\tif (overrides.hasOwnProperty(i)) {\n\t\t\t\t\tfns[i] = opts[i] || null;\n\t\t\t\t}\n\t\t\t}\n\n\t\t\ttp_inst._defaults = $.extend({}, this._defaults, inlineSettings, opts, overrides, {\n\t\t\t\tevnts: fns,\n\t\t\t\ttimepicker: tp_inst // add timepicker as a property of datepicker: $.datepicker._get(dp_inst, 'timepicker');\n\t\t\t});\n\t\t\ttp_inst.amNames = $.map(tp_inst._defaults.amNames, function (val) {\n\t\t\t\treturn val.toUpperCase();\n\t\t\t});\n\t\t\ttp_inst.pmNames = $.map(tp_inst._defaults.pmNames, function (val) {\n\t\t\t\treturn val.toUpperCase();\n\t\t\t});\n\n\t\t\t// detect which units are supported\n\t\t\ttp_inst.support = detectSupport(\n\t\t\t\t\ttp_inst._defaults.timeFormat + \n\t\t\t\t\t(tp_inst._defaults.pickerTimeFormat ? tp_inst._defaults.pickerTimeFormat : '') +\n\t\t\t\t\t(tp_inst._defaults.altTimeFormat ? tp_inst._defaults.altTimeFormat : ''));\n\n\t\t\t// controlType is string - key to our this._controls\n\t\t\tif (typeof(tp_inst._defaults.controlType) === 'string') {\n\t\t\t\tif (tp_inst._defaults.controlType === 'slider' && typeof($.ui.slider) === 'undefined') {\n\t\t\t\t\ttp_inst._defaults.controlType = 'select';\n\t\t\t\t}\n\t\t\t\ttp_inst.control = tp_inst._controls[tp_inst._defaults.controlType];\n\t\t\t}\n\t\t\t// controlType is an object and must implement create, options, value methods\n\t\t\telse {\n\t\t\t\ttp_inst.control = tp_inst._defaults.controlType;\n\t\t\t}\n\n\t\t\t// prep the timezone options\n\t\t\tvar timezoneList = [-720, -660, -600, -570, -540, -480, -420, -360, -300, -270, -240, -210, -180, -120, -60,\n\t\t\t\t\t0, 60, 120, 180, 210, 240, 270, 300, 330, 345, 360, 390, 420, 480, 525, 540, 570, 600, 630, 660, 690, 720, 765, 780, 840];\n\t\t\tif (tp_inst._defaults.timezoneList !== null) {\n\t\t\t\ttimezoneList = tp_inst._defaults.timezoneList;\n\t\t\t}\n\t\t\tvar tzl = timezoneList.length, tzi = 0, tzv = null;\n\t\t\tif (tzl > 0 && typeof timezoneList[0] !== 'object') {\n\t\t\t\tfor (; tzi < tzl; tzi++) {\n\t\t\t\t\ttzv = timezoneList[tzi];\n\t\t\t\t\ttimezoneList[tzi] = { value: tzv, label: $.timepicker.timezoneOffsetString(tzv, tp_inst.support.iso8601) };\n\t\t\t\t}\n\t\t\t}\n\t\t\ttp_inst._defaults.timezoneList = timezoneList;\n\n\t\t\t// set the default units\n\t\t\ttp_inst.timezone = tp_inst._defaults.timezone !== null ? $.timepicker.timezoneOffsetNumber(tp_inst._defaults.timezone) :\n\t\t\t\t\t\t\t((new Date()).getTimezoneOffset() * -1);\n\t\t\ttp_inst.hour = tp_inst._defaults.hour < tp_inst._defaults.hourMin ? tp_inst._defaults.hourMin :\n\t\t\t\t\t\t\ttp_inst._defaults.hour > tp_inst._defaults.hourMax ? tp_inst._defaults.hourMax : tp_inst._defaults.hour;\n\t\t\ttp_inst.minute = tp_inst._defaults.minute < tp_inst._defaults.minuteMin ? tp_inst._defaults.minuteMin :\n\t\t\t\t\t\t\ttp_inst._defaults.minute > tp_inst._defaults.minuteMax ? tp_inst._defaults.minuteMax : tp_inst._defaults.minute;\n\t\t\ttp_inst.second = tp_inst._defaults.second < tp_inst._defaults.secondMin ? tp_inst._defaults.secondMin :\n\t\t\t\t\t\t\ttp_inst._defaults.second > tp_inst._defaults.secondMax ? tp_inst._defaults.secondMax : tp_inst._defaults.second;\n\t\t\ttp_inst.millisec = tp_inst._defaults.millisec < tp_inst._defaults.millisecMin ? tp_inst._defaults.millisecMin :\n\t\t\t\t\t\t\ttp_inst._defaults.millisec > tp_inst._defaults.millisecMax ? tp_inst._defaults.millisecMax : tp_inst._defaults.millisec;\n\t\t\ttp_inst.microsec = tp_inst._defaults.microsec < tp_inst._defaults.microsecMin ? tp_inst._defaults.microsecMin :\n\t\t\t\t\t\t\ttp_inst._defaults.microsec > tp_inst._defaults.microsecMax ? tp_inst._defaults.microsecMax : tp_inst._defaults.microsec;\n\t\t\ttp_inst.ampm = '';\n\t\t\ttp_inst.$input = $input;\n\n\t\t\tif (tp_inst._defaults.altField) {\n\t\t\t\ttp_inst.$altInput = $(tp_inst._defaults.altField).css({\n\t\t\t\t\tcursor: 'pointer'\n\t\t\t\t}).focus(function () {\n\t\t\t\t\t$input.trigger(\"focus\");\n\t\t\t\t});\n\t\t\t}\n\n\t\t\tif (tp_inst._defaults.minDate === 0 || tp_inst._defaults.minDateTime === 0) {\n\t\t\t\ttp_inst._defaults.minDate = new Date();\n\t\t\t}\n\t\t\tif (tp_inst._defaults.maxDate === 0 || tp_inst._defaults.maxDateTime === 0) {\n\t\t\t\ttp_inst._defaults.maxDate = new Date();\n\t\t\t}\n\n\t\t\t// datepicker needs minDate/maxDate, timepicker needs minDateTime/maxDateTime..\n\t\t\tif (tp_inst._defaults.minDate !== undefined && tp_inst._defaults.minDate instanceof Date) {\n\t\t\t\ttp_inst._defaults.minDateTime = new Date(tp_inst._defaults.minDate.getTime());\n\t\t\t}\n\t\t\tif (tp_inst._defaults.minDateTime !== undefined && tp_inst._defaults.minDateTime instanceof Date) {\n\t\t\t\ttp_inst._defaults.minDate = new Date(tp_inst._defaults.minDateTime.getTime());\n\t\t\t}\n\t\t\tif (tp_inst._defaults.maxDate !== undefined && tp_inst._defaults.maxDate instanceof Date) {\n\t\t\t\ttp_inst._defaults.maxDateTime = new Date(tp_inst._defaults.maxDate.getTime());\n\t\t\t}\n\t\t\tif (tp_inst._defaults.maxDateTime !== undefined && tp_inst._defaults.maxDateTime instanceof Date) {\n\t\t\t\ttp_inst._defaults.maxDate = new Date(tp_inst._defaults.maxDateTime.getTime());\n\t\t\t}\n\t\t\ttp_inst.$input.bind('focus', function () {\n\t\t\t\ttp_inst._onFocus();\n\t\t\t});\n\n\t\t\treturn tp_inst;\n\t\t},\n\n\t\t/*\n\t\t* add our sliders to the calendar\n\t\t*/\n\t\t_addTimePicker: function (dp_inst) {\n\t\t\tvar currDT = (this.$altInput && this._defaults.altFieldTimeOnly) ? this.$input.val() + ' ' + this.$altInput.val() : this.$input.val();\n\n\t\t\tthis.timeDefined = this._parseTime(currDT);\n\t\t\tthis._limitMinMaxDateTime(dp_inst, false);\n\t\t\tthis._injectTimePicker();\n\t\t},\n\n\t\t/*\n\t\t* parse the time string from input value or _setTime\n\t\t*/\n\t\t_parseTime: function (timeString, withDate) {\n\t\t\tif (!this.inst) {\n\t\t\t\tthis.inst = $.datepicker._getInst(this.$input[0]);\n\t\t\t}\n\n\t\t\tif (withDate || !this._defaults.timeOnly) {\n\t\t\t\tvar dp_dateFormat = $.datepicker._get(this.inst, 'dateFormat');\n\t\t\t\ttry {\n\t\t\t\t\tvar parseRes = parseDateTimeInternal(dp_dateFormat, this._defaults.timeFormat, timeString, $.datepicker._getFormatConfig(this.inst), this._defaults);\n\t\t\t\t\tif (!parseRes.timeObj) {\n\t\t\t\t\t\treturn false;\n\t\t\t\t\t}\n\t\t\t\t\t$.extend(this, parseRes.timeObj);\n\t\t\t\t} catch (err) {\n\t\t\t\t\t$.timepicker.log(\"Error parsing the date/time string: \" + err +\n\t\t\t\t\t\t\t\t\t\"\\ndate/time string = \" + timeString +\n\t\t\t\t\t\t\t\t\t\"\\ntimeFormat = \" + this._defaults.timeFormat +\n\t\t\t\t\t\t\t\t\t\"\\ndateFormat = \" + dp_dateFormat);\n\t\t\t\t\treturn false;\n\t\t\t\t}\n\t\t\t\treturn true;\n\t\t\t} else {\n\t\t\t\tvar timeObj = $.datepicker.parseTime(this._defaults.timeFormat, timeString, this._defaults);\n\t\t\t\tif (!timeObj) {\n\t\t\t\t\treturn false;\n\t\t\t\t}\n\t\t\t\t$.extend(this, timeObj);\n\t\t\t\treturn true;\n\t\t\t}\n\t\t},\n\n\t\t/*\n\t\t* generate and inject html for timepicker into ui datepicker\n\t\t*/\n\t\t_injectTimePicker: function () {\n\t\t\tvar $dp = this.inst.dpDiv,\n\t\t\t\to = this.inst.settings,\n\t\t\t\ttp_inst = this,\n\t\t\t\tlitem = '',\n\t\t\t\tuitem = '',\n\t\t\t\tshow = null,\n\t\t\t\tmax = {},\n\t\t\t\tgridSize = {},\n\t\t\t\tsize = null,\n\t\t\t\ti = 0,\n\t\t\t\tl = 0;\n\n\t\t\t// Prevent displaying twice\n\t\t\tif ($dp.find(\"div.ui-timepicker-div\").length === 0 && o.showTimepicker) {\n\t\t\t\tvar noDisplay = ' style=\"display:none;\"',\n\t\t\t\t\thtml = '<div class=\"ui-timepicker-div' + (o.isRTL ? ' ui-timepicker-rtl' : '') + '\"><dl>' + '<dt class=\"ui_tpicker_time_label\"' + ((o.showTime) ? '' : noDisplay) + '>' + o.timeText + '</dt>' +\n\t\t\t\t\t\t\t\t'<dd class=\"ui_tpicker_time\"' + ((o.showTime) ? '' : noDisplay) + '></dd>';\n\n\t\t\t\t// Create the markup\n\t\t\t\tfor (i = 0, l = this.units.length; i < l; i++) {\n\t\t\t\t\tlitem = this.units[i];\n\t\t\t\t\tuitem = litem.substr(0, 1).toUpperCase() + litem.substr(1);\n\t\t\t\t\tshow = o['show' + uitem] !== null ? o['show' + uitem] : this.support[litem];\n\n\t\t\t\t\t// Added by Peter Medeiros:\n\t\t\t\t\t// - Figure out what the hour/minute/second max should be based on the step values.\n\t\t\t\t\t// - Example: if stepMinute is 15, then minMax is 45.\n\t\t\t\t\tmax[litem] = parseInt((o[litem + 'Max'] - ((o[litem + 'Max'] - o[litem + 'Min']) % o['step' + uitem])), 10);\n\t\t\t\t\tgridSize[litem] = 0;\n\n\t\t\t\t\thtml += '<dt class=\"ui_tpicker_' + litem + '_label\"' + (show ? '' : noDisplay) + '>' + o[litem + 'Text'] + '</dt>' +\n\t\t\t\t\t\t\t\t'<dd class=\"ui_tpicker_' + litem + '\"><div class=\"ui_tpicker_' + litem + '_slider\"' + (show ? '' : noDisplay) + '></div>';\n\n\t\t\t\t\tif (show && o[litem + 'Grid'] > 0) {\n\t\t\t\t\t\thtml += '<div style=\"padding-left: 1px\"><table class=\"ui-tpicker-grid-label\"><tr>';\n\n\t\t\t\t\t\tif (litem === 'hour') {\n\t\t\t\t\t\t\tfor (var h = o[litem + 'Min']; h <= max[litem]; h += parseInt(o[litem + 'Grid'], 10)) {\n\t\t\t\t\t\t\t\tgridSize[litem]++;\n\t\t\t\t\t\t\t\tvar tmph = $.datepicker.formatTime(this.support.ampm ? 'hht' : 'HH', {hour: h}, o);\n\t\t\t\t\t\t\t\thtml += '<td data-for=\"' + litem + '\">' + tmph + '</td>';\n\t\t\t\t\t\t\t}\n\t\t\t\t\t\t}\n\t\t\t\t\t\telse {\n\t\t\t\t\t\t\tfor (var m = o[litem + 'Min']; m <= max[litem]; m += parseInt(o[litem + 'Grid'], 10)) {\n\t\t\t\t\t\t\t\tgridSize[litem]++;\n\t\t\t\t\t\t\t\thtml += '<td data-for=\"' + litem + '\">' + ((m < 10) ? '0' : '') + m + '</td>';\n\t\t\t\t\t\t\t}\n\t\t\t\t\t\t}\n\n\t\t\t\t\t\thtml += '</tr></table></div>';\n\t\t\t\t\t}\n\t\t\t\t\thtml += '</dd>';\n\t\t\t\t}\n\t\t\t\t\n\t\t\t\t// Timezone\n\t\t\t\tvar showTz = o.showTimezone !== null ? o.showTimezone : this.support.timezone;\n\t\t\t\thtml += '<dt class=\"ui_tpicker_timezone_label\"' + (showTz ? '' : noDisplay) + '>' + o.timezoneText + '</dt>';\n\t\t\t\thtml += '<dd class=\"ui_tpicker_timezone\" ' + (showTz ? '' : noDisplay) + '></dd>';\n\n\t\t\t\t// Create the elements from string\n\t\t\t\thtml += '</dl></div>';\n\t\t\t\tvar $tp = $(html);\n\n\t\t\t\t// if we only want time picker...\n\t\t\t\tif (o.timeOnly === true) {\n\t\t\t\t\t$tp.prepend('<div class=\"ui-widget-header ui-helper-clearfix ui-corner-all\">' + '<div class=\"ui-datepicker-title\">' + o.timeOnlyTitle + '</div>' + '</div>');\n\t\t\t\t\t$dp.find('.ui-datepicker-header, .ui-datepicker-calendar').hide();\n\t\t\t\t}\n\t\t\t\t\n\t\t\t\t// add sliders, adjust grids, add events\n\t\t\t\tfor (i = 0, l = tp_inst.units.length; i < l; i++) {\n\t\t\t\t\tlitem = tp_inst.units[i];\n\t\t\t\t\tuitem = litem.substr(0, 1).toUpperCase() + litem.substr(1);\n\t\t\t\t\tshow = o['show' + uitem] !== null ? o['show' + uitem] : this.support[litem];\n\n\t\t\t\t\t// add the slider\n\t\t\t\t\ttp_inst[litem + '_slider'] = tp_inst.control.create(tp_inst, $tp.find('.ui_tpicker_' + litem + '_slider'), litem, tp_inst[litem], o[litem + 'Min'], max[litem], o['step' + uitem]);\n\n\t\t\t\t\t// adjust the grid and add click event\n\t\t\t\t\tif (show && o[litem + 'Grid'] > 0) {\n\t\t\t\t\t\tsize = 100 * gridSize[litem] * o[litem + 'Grid'] / (max[litem] - o[litem + 'Min']);\n\t\t\t\t\t\t$tp.find('.ui_tpicker_' + litem + ' table').css({\n\t\t\t\t\t\t\twidth: size + \"%\",\n\t\t\t\t\t\t\tmarginLeft: o.isRTL ? '0' : ((size / (-2 * gridSize[litem])) + \"%\"),\n\t\t\t\t\t\t\tmarginRight: o.isRTL ? ((size / (-2 * gridSize[litem])) + \"%\") : '0',\n\t\t\t\t\t\t\tborderCollapse: 'collapse'\n\t\t\t\t\t\t}).find(\"td\").click(function (e) {\n\t\t\t\t\t\t\t\tvar $t = $(this),\n\t\t\t\t\t\t\t\t\th = $t.html(),\n\t\t\t\t\t\t\t\t\tn = parseInt(h.replace(/[^0-9]/g), 10),\n\t\t\t\t\t\t\t\t\tap = h.replace(/[^apm]/ig),\n\t\t\t\t\t\t\t\t\tf = $t.data('for'); // loses scope, so we use data-for\n\n\t\t\t\t\t\t\t\tif (f === 'hour') {\n\t\t\t\t\t\t\t\t\tif (ap.indexOf('p') !== -1 && n < 12) {\n\t\t\t\t\t\t\t\t\t\tn += 12;\n\t\t\t\t\t\t\t\t\t}\n\t\t\t\t\t\t\t\t\telse {\n\t\t\t\t\t\t\t\t\t\tif (ap.indexOf('a') !== -1 && n === 12) {\n\t\t\t\t\t\t\t\t\t\t\tn = 0;\n\t\t\t\t\t\t\t\t\t\t}\n\t\t\t\t\t\t\t\t\t}\n\t\t\t\t\t\t\t\t}\n\t\t\t\t\t\t\t\t\n\t\t\t\t\t\t\t\ttp_inst.control.value(tp_inst, tp_inst[f + '_slider'], litem, n);\n\n\t\t\t\t\t\t\t\ttp_inst._onTimeChange();\n\t\t\t\t\t\t\t\ttp_inst._onSelectHandler();\n\t\t\t\t\t\t\t}).css({\n\t\t\t\t\t\t\t\tcursor: 'pointer',\n\t\t\t\t\t\t\t\twidth: (100 / gridSize[litem]) + '%',\n\t\t\t\t\t\t\t\ttextAlign: 'center',\n\t\t\t\t\t\t\t\toverflow: 'hidden'\n\t\t\t\t\t\t\t});\n\t\t\t\t\t} // end if grid > 0\n\t\t\t\t} // end for loop\n\n\t\t\t\t// Add timezone options\n\t\t\t\tthis.timezone_select = $tp.find('.ui_tpicker_timezone').append('<select></select>').find(\"select\");\n\t\t\t\t$.fn.append.apply(this.timezone_select,\n\t\t\t\t$.map(o.timezoneList, function (val, idx) {\n\t\t\t\t\treturn $(\"<option />\").val(typeof val === \"object\" ? val.value : val).text(typeof val === \"object\" ? val.label : val);\n\t\t\t\t}));\n\t\t\t\tif (typeof(this.timezone) !== \"undefined\" && this.timezone !== null && this.timezone !== \"\") {\n\t\t\t\t\tvar local_timezone = (new Date(this.inst.selectedYear, this.inst.selectedMonth, this.inst.selectedDay, 12)).getTimezoneOffset() * -1;\n\t\t\t\t\tif (local_timezone === this.timezone) {\n\t\t\t\t\t\tselectLocalTimezone(tp_inst);\n\t\t\t\t\t} else {\n\t\t\t\t\t\tthis.timezone_select.val(this.timezone);\n\t\t\t\t\t}\n\t\t\t\t} else {\n\t\t\t\t\tif (typeof(this.hour) !== \"undefined\" && this.hour !== null && this.hour !== \"\") {\n\t\t\t\t\t\tthis.timezone_select.val(o.timezone);\n\t\t\t\t\t} else {\n\t\t\t\t\t\tselectLocalTimezone(tp_inst);\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t\tthis.timezone_select.change(function () {\n\t\t\t\t\ttp_inst._onTimeChange();\n\t\t\t\t\ttp_inst._onSelectHandler();\n\t\t\t\t});\n\t\t\t\t// End timezone options\n\t\t\t\t\n\t\t\t\t// inject timepicker into datepicker\n\t\t\t\tvar $buttonPanel = $dp.find('.ui-datepicker-buttonpane');\n\t\t\t\tif ($buttonPanel.length) {\n\t\t\t\t\t$buttonPanel.before($tp);\n\t\t\t\t} else {\n\t\t\t\t\t$dp.append($tp);\n\t\t\t\t}\n\n\t\t\t\tthis.$timeObj = $tp.find('.ui_tpicker_time');\n\n\t\t\t\tif (this.inst !== null) {\n\t\t\t\t\tvar timeDefined = this.timeDefined;\n\t\t\t\t\tthis._onTimeChange();\n\t\t\t\t\tthis.timeDefined = timeDefined;\n\t\t\t\t}\n\n\t\t\t\t// slideAccess integration: http://trentrichardson.com/2011/11/11/jquery-ui-sliders-and-touch-accessibility/\n\t\t\t\tif (this._defaults.addSliderAccess) {\n\t\t\t\t\tvar sliderAccessArgs = this._defaults.sliderAccessArgs,\n\t\t\t\t\t\trtl = this._defaults.isRTL;\n\t\t\t\t\tsliderAccessArgs.isRTL = rtl;\n\t\t\t\t\t\t\n\t\t\t\t\tsetTimeout(function () { // fix for inline mode\n\t\t\t\t\t\tif ($tp.find('.ui-slider-access').length === 0) {\n\t\t\t\t\t\t\t$tp.find('.ui-slider:visible').sliderAccess(sliderAccessArgs);\n\n\t\t\t\t\t\t\t// fix any grids since sliders are shorter\n\t\t\t\t\t\t\tvar sliderAccessWidth = $tp.find('.ui-slider-access:eq(0)').outerWidth(true);\n\t\t\t\t\t\t\tif (sliderAccessWidth) {\n\t\t\t\t\t\t\t\t$tp.find('table:visible').each(function () {\n\t\t\t\t\t\t\t\t\tvar $g = $(this),\n\t\t\t\t\t\t\t\t\t\toldWidth = $g.outerWidth(),\n\t\t\t\t\t\t\t\t\t\toldMarginLeft = $g.css(rtl ? 'marginRight' : 'marginLeft').toString().replace('%', ''),\n\t\t\t\t\t\t\t\t\t\tnewWidth = oldWidth - sliderAccessWidth,\n\t\t\t\t\t\t\t\t\t\tnewMarginLeft = ((oldMarginLeft * newWidth) / oldWidth) + '%',\n\t\t\t\t\t\t\t\t\t\tcss = { width: newWidth, marginRight: 0, marginLeft: 0 };\n\t\t\t\t\t\t\t\t\tcss[rtl ? 'marginRight' : 'marginLeft'] = newMarginLeft;\n\t\t\t\t\t\t\t\t\t$g.css(css);\n\t\t\t\t\t\t\t\t});\n\t\t\t\t\t\t\t}\n\t\t\t\t\t\t}\n\t\t\t\t\t}, 10);\n\t\t\t\t}\n\t\t\t\t// end slideAccess integration\n\n\t\t\t\ttp_inst._limitMinMaxDateTime(this.inst, true);\n\t\t\t}\n\t\t},\n\n\t\t/*\n\t\t* This function tries to limit the ability to go outside the\n\t\t* min/max date range\n\t\t*/\n\t\t_limitMinMaxDateTime: function (dp_inst, adjustSliders) {\n\t\t\tvar o = this._defaults,\n\t\t\t\tdp_date = new Date(dp_inst.selectedYear, dp_inst.selectedMonth, dp_inst.selectedDay);\n\n\t\t\tif (!this._defaults.showTimepicker) {\n\t\t\t\treturn;\n\t\t\t} // No time so nothing to check here\n\n\t\t\tif ($.datepicker._get(dp_inst, 'minDateTime') !== null && $.datepicker._get(dp_inst, 'minDateTime') !== undefined && dp_date) {\n\t\t\t\tvar minDateTime = $.datepicker._get(dp_inst, 'minDateTime'),\n\t\t\t\t\tminDateTimeDate = new Date(minDateTime.getFullYear(), minDateTime.getMonth(), minDateTime.getDate(), 0, 0, 0, 0);\n\n\t\t\t\tif (this.hourMinOriginal === null || this.minuteMinOriginal === null || this.secondMinOriginal === null || this.millisecMinOriginal === null || this.microsecMinOriginal === null) {\n\t\t\t\t\tthis.hourMinOriginal = o.hourMin;\n\t\t\t\t\tthis.minuteMinOriginal = o.minuteMin;\n\t\t\t\t\tthis.secondMinOriginal = o.secondMin;\n\t\t\t\t\tthis.millisecMinOriginal = o.millisecMin;\n\t\t\t\t\tthis.microsecMinOriginal = o.microsecMin;\n\t\t\t\t}\n\n\t\t\t\tif (dp_inst.settings.timeOnly || minDateTimeDate.getTime() === dp_date.getTime()) {\n\t\t\t\t\tthis._defaults.hourMin = minDateTime.getHours();\n\t\t\t\t\tif (this.hour <= this._defaults.hourMin) {\n\t\t\t\t\t\tthis.hour = this._defaults.hourMin;\n\t\t\t\t\t\tthis._defaults.minuteMin = minDateTime.getMinutes();\n\t\t\t\t\t\tif (this.minute <= this._defaults.minuteMin) {\n\t\t\t\t\t\t\tthis.minute = this._defaults.minuteMin;\n\t\t\t\t\t\t\tthis._defaults.secondMin = minDateTime.getSeconds();\n\t\t\t\t\t\t\tif (this.second <= this._defaults.secondMin) {\n\t\t\t\t\t\t\t\tthis.second = this._defaults.secondMin;\n\t\t\t\t\t\t\t\tthis._defaults.millisecMin = minDateTime.getMilliseconds();\n\t\t\t\t\t\t\t\tif (this.millisec <= this._defaults.millisecMin) {\n\t\t\t\t\t\t\t\t\tthis.millisec = this._defaults.millisecMin;\n\t\t\t\t\t\t\t\t\tthis._defaults.microsecMin = minDateTime.getMicroseconds();\n\t\t\t\t\t\t\t\t} else {\n\t\t\t\t\t\t\t\t\tif (this.microsec < this._defaults.microsecMin) {\n\t\t\t\t\t\t\t\t\t\tthis.microsec = this._defaults.microsecMin;\n\t\t\t\t\t\t\t\t\t}\n\t\t\t\t\t\t\t\t\tthis._defaults.microsecMin = this.microsecMinOriginal;\n\t\t\t\t\t\t\t\t}\n\t\t\t\t\t\t\t} else {\n\t\t\t\t\t\t\t\tthis._defaults.millisecMin = this.millisecMinOriginal;\n\t\t\t\t\t\t\t\tthis._defaults.microsecMin = this.microsecMinOriginal;\n\t\t\t\t\t\t\t}\n\t\t\t\t\t\t} else {\n\t\t\t\t\t\t\tthis._defaults.secondMin = this.secondMinOriginal;\n\t\t\t\t\t\t\tthis._defaults.millisecMin = this.millisecMinOriginal;\n\t\t\t\t\t\t\tthis._defaults.microsecMin = this.microsecMinOriginal;\n\t\t\t\t\t\t}\n\t\t\t\t\t} else {\n\t\t\t\t\t\tthis._defaults.minuteMin = this.minuteMinOriginal;\n\t\t\t\t\t\tthis._defaults.secondMin = this.secondMinOriginal;\n\t\t\t\t\t\tthis._defaults.millisecMin = this.millisecMinOriginal;\n\t\t\t\t\t\tthis._defaults.microsecMin = this.microsecMinOriginal;\n\t\t\t\t\t}\n\t\t\t\t} else {\n\t\t\t\t\tthis._defaults.hourMin = this.hourMinOriginal;\n\t\t\t\t\tthis._defaults.minuteMin = this.minuteMinOriginal;\n\t\t\t\t\tthis._defaults.secondMin = this.secondMinOriginal;\n\t\t\t\t\tthis._defaults.millisecMin = this.millisecMinOriginal;\n\t\t\t\t\tthis._defaults.microsecMin = this.microsecMinOriginal;\n\t\t\t\t}\n\t\t\t}\n\n\t\t\tif ($.datepicker._get(dp_inst, 'maxDateTime') !== null && $.datepicker._get(dp_inst, 'maxDateTime') !== undefined && dp_date) {\n\t\t\t\tvar maxDateTime = $.datepicker._get(dp_inst, 'maxDateTime'),\n\t\t\t\t\tmaxDateTimeDate = new Date(maxDateTime.getFullYear(), maxDateTime.getMonth(), maxDateTime.getDate(), 0, 0, 0, 0);\n\n\t\t\t\tif (this.hourMaxOriginal === null || this.minuteMaxOriginal === null || this.secondMaxOriginal === null || this.millisecMaxOriginal === null) {\n\t\t\t\t\tthis.hourMaxOriginal = o.hourMax;\n\t\t\t\t\tthis.minuteMaxOriginal = o.minuteMax;\n\t\t\t\t\tthis.secondMaxOriginal = o.secondMax;\n\t\t\t\t\tthis.millisecMaxOriginal = o.millisecMax;\n\t\t\t\t\tthis.microsecMaxOriginal = o.microsecMax;\n\t\t\t\t}\n\n\t\t\t\tif (dp_inst.settings.timeOnly || maxDateTimeDate.getTime() === dp_date.getTime()) {\n\t\t\t\t\tthis._defaults.hourMax = maxDateTime.getHours();\n\t\t\t\t\tif (this.hour >= this._defaults.hourMax) {\n\t\t\t\t\t\tthis.hour = this._defaults.hourMax;\n\t\t\t\t\t\tthis._defaults.minuteMax = maxDateTime.getMinutes();\n\t\t\t\t\t\tif (this.minute >= this._defaults.minuteMax) {\n\t\t\t\t\t\t\tthis.minute = this._defaults.minuteMax;\n\t\t\t\t\t\t\tthis._defaults.secondMax = maxDateTime.getSeconds();\n\t\t\t\t\t\t\tif (this.second >= this._defaults.secondMax) {\n\t\t\t\t\t\t\t\tthis.second = this._defaults.secondMax;\n\t\t\t\t\t\t\t\tthis._defaults.millisecMax = maxDateTime.getMilliseconds();\n\t\t\t\t\t\t\t\tif (this.millisec >= this._defaults.millisecMax) {\n\t\t\t\t\t\t\t\t\tthis.millisec = this._defaults.millisecMax;\n\t\t\t\t\t\t\t\t\tthis._defaults.microsecMax = maxDateTime.getMicroseconds();\n\t\t\t\t\t\t\t\t} else {\n\t\t\t\t\t\t\t\t\tif (this.microsec > this._defaults.microsecMax) {\n\t\t\t\t\t\t\t\t\t\tthis.microsec = this._defaults.microsecMax;\n\t\t\t\t\t\t\t\t\t}\n\t\t\t\t\t\t\t\t\tthis._defaults.microsecMax = this.microsecMaxOriginal;\n\t\t\t\t\t\t\t\t}\n\t\t\t\t\t\t\t} else {\n\t\t\t\t\t\t\t\tthis._defaults.millisecMax = this.millisecMaxOriginal;\n\t\t\t\t\t\t\t\tthis._defaults.microsecMax = this.microsecMaxOriginal;\n\t\t\t\t\t\t\t}\n\t\t\t\t\t\t} else {\n\t\t\t\t\t\t\tthis._defaults.secondMax = this.secondMaxOriginal;\n\t\t\t\t\t\t\tthis._defaults.millisecMax = this.millisecMaxOriginal;\n\t\t\t\t\t\t\tthis._defaults.microsecMax = this.microsecMaxOriginal;\n\t\t\t\t\t\t}\n\t\t\t\t\t} else {\n\t\t\t\t\t\tthis._defaults.minuteMax = this.minuteMaxOriginal;\n\t\t\t\t\t\tthis._defaults.secondMax = this.secondMaxOriginal;\n\t\t\t\t\t\tthis._defaults.millisecMax = this.millisecMaxOriginal;\n\t\t\t\t\t\tthis._defaults.microsecMax = this.microsecMaxOriginal;\n\t\t\t\t\t}\n\t\t\t\t} else {\n\t\t\t\t\tthis._defaults.hourMax = this.hourMaxOriginal;\n\t\t\t\t\tthis._defaults.minuteMax = this.minuteMaxOriginal;\n\t\t\t\t\tthis._defaults.secondMax = this.secondMaxOriginal;\n\t\t\t\t\tthis._defaults.millisecMax = this.millisecMaxOriginal;\n\t\t\t\t\tthis._defaults.microsecMax = this.microsecMaxOriginal;\n\t\t\t\t}\n\t\t\t}\n\n\t\t\tif (adjustSliders !== undefined && adjustSliders === true) {\n\t\t\t\tvar hourMax = parseInt((this._defaults.hourMax - ((this._defaults.hourMax - this._defaults.hourMin) % this._defaults.stepHour)), 10),\n\t\t\t\t\tminMax = parseInt((this._defaults.minuteMax - ((this._defaults.minuteMax - this._defaults.minuteMin) % this._defaults.stepMinute)), 10),\n\t\t\t\t\tsecMax = parseInt((this._defaults.secondMax - ((this._defaults.secondMax - this._defaults.secondMin) % this._defaults.stepSecond)), 10),\n\t\t\t\t\tmillisecMax = parseInt((this._defaults.millisecMax - ((this._defaults.millisecMax - this._defaults.millisecMin) % this._defaults.stepMillisec)), 10),\n\t\t\t\t\tmicrosecMax = parseInt((this._defaults.microsecMax - ((this._defaults.microsecMax - this._defaults.microsecMin) % this._defaults.stepMicrosec)), 10);\n\n\t\t\t\tif (this.hour_slider) {\n\t\t\t\t\tthis.control.options(this, this.hour_slider, 'hour', { min: this._defaults.hourMin, max: hourMax });\n\t\t\t\t\tthis.control.value(this, this.hour_slider, 'hour', this.hour - (this.hour % this._defaults.stepHour));\n\t\t\t\t}\n\t\t\t\tif (this.minute_slider) {\n\t\t\t\t\tthis.control.options(this, this.minute_slider, 'minute', { min: this._defaults.minuteMin, max: minMax });\n\t\t\t\t\tthis.control.value(this, this.minute_slider, 'minute', this.minute - (this.minute % this._defaults.stepMinute));\n\t\t\t\t}\n\t\t\t\tif (this.second_slider) {\n\t\t\t\t\tthis.control.options(this, this.second_slider, 'second', { min: this._defaults.secondMin, max: secMax });\n\t\t\t\t\tthis.control.value(this, this.second_slider, 'second', this.second - (this.second % this._defaults.stepSecond));\n\t\t\t\t}\n\t\t\t\tif (this.millisec_slider) {\n\t\t\t\t\tthis.control.options(this, this.millisec_slider, 'millisec', { min: this._defaults.millisecMin, max: millisecMax });\n\t\t\t\t\tthis.control.value(this, this.millisec_slider, 'millisec', this.millisec - (this.millisec % this._defaults.stepMillisec));\n\t\t\t\t}\n\t\t\t\tif (this.microsec_slider) {\n\t\t\t\t\tthis.control.options(this, this.microsec_slider, 'microsec', { min: this._defaults.microsecMin, max: microsecMax });\n\t\t\t\t\tthis.control.value(this, this.microsec_slider, 'microsec', this.microsec - (this.microsec % this._defaults.stepMicrosec));\n\t\t\t\t}\n\t\t\t}\n\n\t\t},\n\n\t\t/*\n\t\t* when a slider moves, set the internal time...\n\t\t* on time change is also called when the time is updated in the text field\n\t\t*/\n\t\t_onTimeChange: function () {\n\t\t\tif (!this._defaults.showTimepicker) {\n                                return;\n\t\t\t}\n\t\t\tvar hour = (this.hour_slider) ? this.control.value(this, this.hour_slider, 'hour') : false,\n\t\t\t\tminute = (this.minute_slider) ? this.control.value(this, this.minute_slider, 'minute') : false,\n\t\t\t\tsecond = (this.second_slider) ? this.control.value(this, this.second_slider, 'second') : false,\n\t\t\t\tmillisec = (this.millisec_slider) ? this.control.value(this, this.millisec_slider, 'millisec') : false,\n\t\t\t\tmicrosec = (this.microsec_slider) ? this.control.value(this, this.microsec_slider, 'microsec') : false,\n\t\t\t\ttimezone = (this.timezone_select) ? this.timezone_select.val() : false,\n\t\t\t\to = this._defaults,\n\t\t\t\tpickerTimeFormat = o.pickerTimeFormat || o.timeFormat,\n\t\t\t\tpickerTimeSuffix = o.pickerTimeSuffix || o.timeSuffix;\n\n\t\t\tif (typeof(hour) === 'object') {\n\t\t\t\thour = false;\n\t\t\t}\n\t\t\tif (typeof(minute) === 'object') {\n\t\t\t\tminute = false;\n\t\t\t}\n\t\t\tif (typeof(second) === 'object') {\n\t\t\t\tsecond = false;\n\t\t\t}\n\t\t\tif (typeof(millisec) === 'object') {\n\t\t\t\tmillisec = false;\n\t\t\t}\n\t\t\tif (typeof(microsec) === 'object') {\n\t\t\t\tmicrosec = false;\n\t\t\t}\n\t\t\tif (typeof(timezone) === 'object') {\n\t\t\t\ttimezone = false;\n\t\t\t}\n\n\t\t\tif (hour !== false) {\n\t\t\t\thour = parseInt(hour, 10);\n\t\t\t}\n\t\t\tif (minute !== false) {\n\t\t\t\tminute = parseInt(minute, 10);\n\t\t\t}\n\t\t\tif (second !== false) {\n\t\t\t\tsecond = parseInt(second, 10);\n\t\t\t}\n\t\t\tif (millisec !== false) {\n\t\t\t\tmillisec = parseInt(millisec, 10);\n\t\t\t}\n\t\t\tif (microsec !== false) {\n\t\t\t\tmicrosec = parseInt(microsec, 10);\n\t\t\t}\n\t\t\tif (timezone !== false) {\n\t\t\t\ttimezone = timezone.toString();\n\t\t\t}\n\n\t\t\tvar ampm = o[hour < 12 ? 'amNames' : 'pmNames'][0];\n\n\t\t\t// If the update was done in the input field, the input field should not be updated.\n\t\t\t// If the update was done using the sliders, update the input field.\n\t\t\tvar hasChanged = (\n\t\t\t\t\t\thour !== parseInt(this.hour,10) || // sliders should all be numeric\n\t\t\t\t\t\tminute !== parseInt(this.minute,10) || \n\t\t\t\t\t\tsecond !== parseInt(this.second,10) || \n\t\t\t\t\t\tmillisec !== parseInt(this.millisec,10) || \n\t\t\t\t\t\tmicrosec !== parseInt(this.microsec,10) || \n\t\t\t\t\t\t(this.ampm.length > 0 && (hour < 12) !== ($.inArray(this.ampm.toUpperCase(), this.amNames) !== -1)) || \n\t\t\t\t\t\t(this.timezone !== null && timezone !== this.timezone.toString()) // could be numeric or \"EST\" format, so use toString()\n\t\t\t\t\t);\n\n\t\t\tif (hasChanged) {\n\n\t\t\t\tif (hour !== false) {\n\t\t\t\t\tthis.hour = hour;\n\t\t\t\t}\n\t\t\t\tif (minute !== false) {\n\t\t\t\t\tthis.minute = minute;\n\t\t\t\t}\n\t\t\t\tif (second !== false) {\n\t\t\t\t\tthis.second = second;\n\t\t\t\t}\n\t\t\t\tif (millisec !== false) {\n\t\t\t\t\tthis.millisec = millisec;\n\t\t\t\t}\n\t\t\t\tif (microsec !== false) {\n\t\t\t\t\tthis.microsec = microsec;\n\t\t\t\t}\n\t\t\t\tif (timezone !== false) {\n\t\t\t\t\tthis.timezone = timezone;\n\t\t\t\t}\n\n\t\t\t\tif (!this.inst) {\n\t\t\t\t\tthis.inst = $.datepicker._getInst(this.$input[0]);\n\t\t\t\t}\n\n\t\t\t\tthis._limitMinMaxDateTime(this.inst, true);\n\t\t\t}\n\t\t\tif (this.support.ampm) {\n\t\t\t\tthis.ampm = ampm;\n\t\t\t}\n\n\t\t\t// Updates the time within the timepicker\n\t\t\tthis.formattedTime = $.datepicker.formatTime(o.timeFormat, this, o);\n\t\t\tif (this.$timeObj) {\n\t\t\t\tif (pickerTimeFormat === o.timeFormat) {\n\t\t\t\t\tthis.$timeObj.text(this.formattedTime + pickerTimeSuffix);\n\t\t\t\t}\n\t\t\t\telse {\n\t\t\t\t\tthis.$timeObj.text($.datepicker.formatTime(pickerTimeFormat, this, o) + pickerTimeSuffix);\n\t\t\t\t}\n\t\t\t}\n\n\t\t\tthis.timeDefined = true;\n\t\t\tif (hasChanged) {\n\t\t\t\tthis._updateDateTime();\n\t\t\t\tthis.$input.focus();\n\t\t\t}\n\t\t},\n\n\t\t/*\n\t\t* call custom onSelect.\n\t\t* bind to sliders slidestop, and grid click.\n\t\t*/\n\t\t_onSelectHandler: function () {\n\t\t\tvar onSelect = this._defaults.onSelect || this.inst.settings.onSelect;\n\t\t\tvar inputEl = this.$input ? this.$input[0] : null;\n\t\t\tif (onSelect && inputEl) {\n\t\t\t\tonSelect.apply(inputEl, [this.formattedDateTime, this]);\n\t\t\t}\n\t\t},\n\n\t\t/*\n\t\t* update our input with the new date time..\n\t\t*/\n\t\t_updateDateTime: function (dp_inst) {\n\t\t\tdp_inst = this.inst || dp_inst;\n\t\t\tvar dtTmp = (dp_inst.currentYear > 0? \n\t\t\t\t\t\t\tnew Date(dp_inst.currentYear, dp_inst.currentMonth, dp_inst.currentDay) : \n\t\t\t\t\t\t\tnew Date(dp_inst.selectedYear, dp_inst.selectedMonth, dp_inst.selectedDay)),\n\t\t\t\tdt = $.datepicker._daylightSavingAdjust(dtTmp),\n\t\t\t\t//dt = $.datepicker._daylightSavingAdjust(new Date(dp_inst.selectedYear, dp_inst.selectedMonth, dp_inst.selectedDay)),\n\t\t\t\t//dt = $.datepicker._daylightSavingAdjust(new Date(dp_inst.currentYear, dp_inst.currentMonth, dp_inst.currentDay)),\n\t\t\t\tdateFmt = $.datepicker._get(dp_inst, 'dateFormat'),\n\t\t\t\tformatCfg = $.datepicker._getFormatConfig(dp_inst),\n\t\t\t\ttimeAvailable = dt !== null && this.timeDefined;\n\t\t\tthis.formattedDate = $.datepicker.formatDate(dateFmt, (dt === null ? new Date() : dt), formatCfg);\n\t\t\tvar formattedDateTime = this.formattedDate;\n\t\t\t\n\t\t\t// if a slider was changed but datepicker doesn't have a value yet, set it\n\t\t\tif (dp_inst.lastVal === \"\") {\n                dp_inst.currentYear = dp_inst.selectedYear;\n                dp_inst.currentMonth = dp_inst.selectedMonth;\n                dp_inst.currentDay = dp_inst.selectedDay;\n            }\n\n\t\t\t/*\n\t\t\t* remove following lines to force every changes in date picker to change the input value\n\t\t\t* Bug descriptions: when an input field has a default value, and click on the field to pop up the date picker. \n\t\t\t* If the user manually empty the value in the input field, the date picker will never change selected value.\n\t\t\t*/\n\t\t\t//if (dp_inst.lastVal !== undefined && (dp_inst.lastVal.length > 0 && this.$input.val().length === 0)) {\n\t\t\t//\treturn;\n\t\t\t//}\n\n\t\t\tif (this._defaults.timeOnly === true) {\n\t\t\t\tformattedDateTime = this.formattedTime;\n\t\t\t} else if (this._defaults.timeOnly !== true && (this._defaults.alwaysSetTime || timeAvailable)) {\n\t\t\t\tformattedDateTime += this._defaults.separator + this.formattedTime + this._defaults.timeSuffix;\n\t\t\t}\n\n\t\t\tthis.formattedDateTime = formattedDateTime;\n\n\t\t\tif (!this._defaults.showTimepicker) {\n\t\t\t\tthis.$input.val(this.formattedDate);\n\t\t\t} else if (this.$altInput && this._defaults.timeOnly === false && this._defaults.altFieldTimeOnly === true) {\n\t\t\t\tthis.$altInput.val(this.formattedTime);\n\t\t\t\tthis.$input.val(this.formattedDate);\n\t\t\t} else if (this.$altInput) {\n\t\t\t\tthis.$input.val(formattedDateTime);\n\t\t\t\tvar altFormattedDateTime = '',\n\t\t\t\t\taltSeparator = this._defaults.altSeparator ? this._defaults.altSeparator : this._defaults.separator,\n\t\t\t\t\taltTimeSuffix = this._defaults.altTimeSuffix ? this._defaults.altTimeSuffix : this._defaults.timeSuffix;\n\t\t\t\t\n\t\t\t\tif (!this._defaults.timeOnly) {\n\t\t\t\t\tif (this._defaults.altFormat) {\n\t\t\t\t\t\taltFormattedDateTime = $.datepicker.formatDate(this._defaults.altFormat, (dt === null ? new Date() : dt), formatCfg);\n\t\t\t\t\t}\n\t\t\t\t\telse {\n\t\t\t\t\t\taltFormattedDateTime = this.formattedDate;\n\t\t\t\t\t}\n\n\t\t\t\t\tif (altFormattedDateTime) {\n\t\t\t\t\t\taltFormattedDateTime += altSeparator;\n\t\t\t\t\t}\n\t\t\t\t}\n\n\t\t\t\tif (this._defaults.altTimeFormat) {\n\t\t\t\t\taltFormattedDateTime += $.datepicker.formatTime(this._defaults.altTimeFormat, this, this._defaults) + altTimeSuffix;\n\t\t\t\t}\n\t\t\t\telse {\n\t\t\t\t\taltFormattedDateTime += this.formattedTime + altTimeSuffix;\n\t\t\t\t}\n\t\t\t\tthis.$altInput.val(altFormattedDateTime);\n\t\t\t} else {\n\t\t\t\tthis.$input.val(formattedDateTime);\n\t\t\t}\n\n\t\t\tthis.$input.trigger(\"change\");\n\t\t},\n\n\t\t_onFocus: function () {\n\t\t\tif (!this.$input.val() && this._defaults.defaultValue) {\n\t\t\t\tthis.$input.val(this._defaults.defaultValue);\n\t\t\t\tvar inst = $.datepicker._getInst(this.$input.get(0)),\n\t\t\t\t\ttp_inst = $.datepicker._get(inst, 'timepicker');\n\t\t\t\tif (tp_inst) {\n\t\t\t\t\tif (tp_inst._defaults.timeOnly && (inst.input.val() !== inst.lastVal)) {\n\t\t\t\t\t\ttry {\n\t\t\t\t\t\t\t$.datepicker._updateDatepicker(inst);\n\t\t\t\t\t\t} catch (err) {\n\t\t\t\t\t\t\t$.timepicker.log(err);\n\t\t\t\t\t\t}\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t}\n\t\t},\n\n\t\t/*\n\t\t* Small abstraction to control types\n\t\t* We can add more, just be sure to follow the pattern: create, options, value\n\t\t*/\n\t\t_controls: {\n\t\t\t// slider methods\n\t\t\tslider: {\n\t\t\t\tcreate: function (tp_inst, obj, unit, val, min, max, step) {\n\t\t\t\t\tvar rtl = tp_inst._defaults.isRTL; // if rtl go -60->0 instead of 0->60\n\t\t\t\t\treturn obj.prop('slide', null).slider({\n\t\t\t\t\t\torientation: \"horizontal\",\n\t\t\t\t\t\tvalue: rtl ? val * -1 : val,\n\t\t\t\t\t\tmin: rtl ? max * -1 : min,\n\t\t\t\t\t\tmax: rtl ? min * -1 : max,\n\t\t\t\t\t\tstep: step,\n\t\t\t\t\t\tslide: function (event, ui) {\n\t\t\t\t\t\t\ttp_inst.control.value(tp_inst, $(this), unit, rtl ? ui.value * -1 : ui.value);\n\t\t\t\t\t\t\ttp_inst._onTimeChange();\n\t\t\t\t\t\t},\n\t\t\t\t\t\tstop: function (event, ui) {\n\t\t\t\t\t\t\ttp_inst._onSelectHandler();\n\t\t\t\t\t\t}\n\t\t\t\t\t});\t\n\t\t\t\t},\n\t\t\t\toptions: function (tp_inst, obj, unit, opts, val) {\n\t\t\t\t\tif (tp_inst._defaults.isRTL) {\n\t\t\t\t\t\tif (typeof(opts) === 'string') {\n\t\t\t\t\t\t\tif (opts === 'min' || opts === 'max') {\n\t\t\t\t\t\t\t\tif (val !== undefined) {\n\t\t\t\t\t\t\t\t\treturn obj.slider(opts, val * -1);\n\t\t\t\t\t\t\t\t}\n\t\t\t\t\t\t\t\treturn Math.abs(obj.slider(opts));\n\t\t\t\t\t\t\t}\n\t\t\t\t\t\t\treturn obj.slider(opts);\n\t\t\t\t\t\t}\n\t\t\t\t\t\tvar min = opts.min, \n\t\t\t\t\t\t\tmax = opts.max;\n\t\t\t\t\t\topts.min = opts.max = null;\n\t\t\t\t\t\tif (min !== undefined) {\n\t\t\t\t\t\t\topts.max = min * -1;\n\t\t\t\t\t\t}\n\t\t\t\t\t\tif (max !== undefined) {\n\t\t\t\t\t\t\topts.min = max * -1;\n\t\t\t\t\t\t}\n\t\t\t\t\t\treturn obj.slider(opts);\n\t\t\t\t\t}\n\t\t\t\t\tif (typeof(opts) === 'string' && val !== undefined) {\n\t\t\t\t\t\treturn obj.slider(opts, val);\n\t\t\t\t\t}\n\t\t\t\t\treturn obj.slider(opts);\n\t\t\t\t},\n\t\t\t\tvalue: function (tp_inst, obj, unit, val) {\n\t\t\t\t\tif (tp_inst._defaults.isRTL) {\n\t\t\t\t\t\tif (val !== undefined) {\n\t\t\t\t\t\t\treturn obj.slider('value', val * -1);\n\t\t\t\t\t\t}\n\t\t\t\t\t\treturn Math.abs(obj.slider('value'));\n\t\t\t\t\t}\n\t\t\t\t\tif (val !== undefined) {\n\t\t\t\t\t\treturn obj.slider('value', val);\n\t\t\t\t\t}\n\t\t\t\t\treturn obj.slider('value');\n\t\t\t\t}\n\t\t\t},\n\t\t\t// select methods\n\t\t\tselect: {\n\t\t\t\tcreate: function (tp_inst, obj, unit, val, min, max, step) {\n\t\t\t\t\tvar sel = '<select class=\"ui-timepicker-select\" data-unit=\"' + unit + '\" data-min=\"' + min + '\" data-max=\"' + max + '\" data-step=\"' + step + '\">',\n\t\t\t\t\t\tformat = tp_inst._defaults.pickerTimeFormat || tp_inst._defaults.timeFormat;\n\n\t\t\t\t\tfor (var i = min; i <= max; i += step) {\n\t\t\t\t\t\tsel += '<option value=\"' + i + '\"' + (i === val ? ' selected' : '') + '>';\n\t\t\t\t\t\tif (unit === 'hour') {\n\t\t\t\t\t\t\tsel += $.datepicker.formatTime($.trim(format.replace(/[^ht ]/ig, '')), {hour: i}, tp_inst._defaults);\n\t\t\t\t\t\t}\n\t\t\t\t\t\telse if (unit === 'millisec' || unit === 'microsec' || i >= 10) { sel += i; }\n\t\t\t\t\t\telse {sel += '0' + i.toString(); }\n\t\t\t\t\t\tsel += '</option>';\n\t\t\t\t\t}\n\t\t\t\t\tsel += '</select>';\n\n\t\t\t\t\tobj.children('select').remove();\n\n\t\t\t\t\t$(sel).appendTo(obj).change(function (e) {\n\t\t\t\t\t\ttp_inst._onTimeChange();\n\t\t\t\t\t\ttp_inst._onSelectHandler();\n\t\t\t\t\t});\n\n\t\t\t\t\treturn obj;\n\t\t\t\t},\n\t\t\t\toptions: function (tp_inst, obj, unit, opts, val) {\n\t\t\t\t\tvar o = {},\n\t\t\t\t\t\t$t = obj.children('select');\n\t\t\t\t\tif (typeof(opts) === 'string') {\n\t\t\t\t\t\tif (val === undefined) {\n\t\t\t\t\t\t\treturn $t.data(opts);\n\t\t\t\t\t\t}\n\t\t\t\t\t\to[opts] = val;\t\n\t\t\t\t\t}\n\t\t\t\t\telse { o = opts; }\n\t\t\t\t\treturn tp_inst.control.create(tp_inst, obj, $t.data('unit'), $t.val(), o.min || $t.data('min'), o.max || $t.data('max'), o.step || $t.data('step'));\n\t\t\t\t},\n\t\t\t\tvalue: function (tp_inst, obj, unit, val) {\n\t\t\t\t\tvar $t = obj.children('select');\n\t\t\t\t\tif (val !== undefined) {\n\t\t\t\t\t\treturn $t.val(val);\n\t\t\t\t\t}\n\t\t\t\t\treturn $t.val();\n\t\t\t\t}\n\t\t\t}\n\t\t} // end _controls\n\n\t});\n\n\t$.fn.extend({\n\t\t/*\n\t\t* shorthand just to use timepicker.\n\t\t*/\n\t\ttimepicker: function (o) {\n\t\t\to = o || {};\n\t\t\tvar tmp_args = Array.prototype.slice.call(arguments);\n\n\t\t\tif (typeof o === 'object') {\n\t\t\t\ttmp_args[0] = $.extend(o, {\n\t\t\t\t\ttimeOnly: true\n\t\t\t\t});\n\t\t\t}\n\n\t\t\treturn $(this).each(function () {\n\t\t\t\t$.fn.datetimepicker.apply($(this), tmp_args);\n\t\t\t});\n\t\t},\n\n\t\t/*\n\t\t* extend timepicker to datepicker\n\t\t*/\n\t\tdatetimepicker: function (o) {\n\t\t\to = o || {};\n\t\t\tvar tmp_args = arguments;\n\n\t\t\tif (typeof(o) === 'string') {\n\t\t\t\tif (o === 'getDate') {\n\t\t\t\t\treturn $.fn.datepicker.apply($(this[0]), tmp_args);\n\t\t\t\t} else {\n\t\t\t\t\treturn this.each(function () {\n\t\t\t\t\t\tvar $t = $(this);\n\t\t\t\t\t\t$t.datepicker.apply($t, tmp_args);\n\t\t\t\t\t});\n\t\t\t\t}\n\t\t\t} else {\n\t\t\t\treturn this.each(function () {\n\t\t\t\t\tvar $t = $(this);\n\t\t\t\t\t$t.datepicker($.timepicker._newInst($t, o)._defaults);\n\t\t\t\t});\n\t\t\t}\n\t\t}\n\t});\n\n\t/*\n\t* Public Utility to parse date and time\n\t*/\n\t$.datepicker.parseDateTime = function (dateFormat, timeFormat, dateTimeString, dateSettings, timeSettings) {\n\t\tvar parseRes = parseDateTimeInternal(dateFormat, timeFormat, dateTimeString, dateSettings, timeSettings);\n\t\tif (parseRes.timeObj) {\n\t\t\tvar t = parseRes.timeObj;\n\t\t\tparseRes.date.setHours(t.hour, t.minute, t.second, t.millisec);\n\t\t\tparseRes.date.setMicroseconds(t.microsec);\n\t\t}\n\n\t\treturn parseRes.date;\n\t};\n\n\t/*\n\t* Public utility to parse time\n\t*/\n\t$.datepicker.parseTime = function (timeFormat, timeString, options) {\n\t\tvar o = extendRemove(extendRemove({}, $.timepicker._defaults), options || {}),\n\t\t\tiso8601 = (timeFormat.replace(/\\'.*?\\'/g, '').indexOf('Z') !== -1);\n\n\t\t// Strict parse requires the timeString to match the timeFormat exactly\n\t\tvar strictParse = function (f, s, o) {\n\n\t\t\t// pattern for standard and localized AM/PM markers\n\t\t\tvar getPatternAmpm = function (amNames, pmNames) {\n\t\t\t\tvar markers = [];\n\t\t\t\tif (amNames) {\n\t\t\t\t\t$.merge(markers, amNames);\n\t\t\t\t}\n\t\t\t\tif (pmNames) {\n\t\t\t\t\t$.merge(markers, pmNames);\n\t\t\t\t}\n\t\t\t\tmarkers = $.map(markers, function (val) {\n\t\t\t\t\treturn val.replace(/[.*+?|()\\[\\]{}\\\\]/g, '\\\\$&');\n\t\t\t\t});\n\t\t\t\treturn '(' + markers.join('|') + ')?';\n\t\t\t};\n\n\t\t\t// figure out position of time elements.. cause js cant do named captures\n\t\t\tvar getFormatPositions = function (timeFormat) {\n\t\t\t\tvar finds = timeFormat.toLowerCase().match(/(h{1,2}|m{1,2}|s{1,2}|l{1}|c{1}|t{1,2}|z|'.*?')/g),\n\t\t\t\t\torders = {\n\t\t\t\t\t\th: -1,\n\t\t\t\t\t\tm: -1,\n\t\t\t\t\t\ts: -1,\n\t\t\t\t\t\tl: -1,\n\t\t\t\t\t\tc: -1,\n\t\t\t\t\t\tt: -1,\n\t\t\t\t\t\tz: -1\n\t\t\t\t\t};\n\n\t\t\t\tif (finds) {\n\t\t\t\t\tfor (var i = 0; i < finds.length; i++) {\n\t\t\t\t\t\tif (orders[finds[i].toString().charAt(0)] === -1) {\n\t\t\t\t\t\t\torders[finds[i].toString().charAt(0)] = i + 1;\n\t\t\t\t\t\t}\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t\treturn orders;\n\t\t\t};\n\n\t\t\tvar regstr = '^' + f.toString()\n\t\t\t\t\t.replace(/([hH]{1,2}|mm?|ss?|[tT]{1,2}|[zZ]|[lc]|'.*?')/g, function (match) {\n\t\t\t\t\t\t\tvar ml = match.length;\n\t\t\t\t\t\t\tswitch (match.charAt(0).toLowerCase()) {\n\t\t\t\t\t\t\tcase 'h':\n\t\t\t\t\t\t\t\treturn ml === 1 ? '(\\\\d?\\\\d)' : '(\\\\d{' + ml + '})';\n\t\t\t\t\t\t\tcase 'm':\n\t\t\t\t\t\t\t\treturn ml === 1 ? '(\\\\d?\\\\d)' : '(\\\\d{' + ml + '})';\n\t\t\t\t\t\t\tcase 's':\n\t\t\t\t\t\t\t\treturn ml === 1 ? '(\\\\d?\\\\d)' : '(\\\\d{' + ml + '})';\n\t\t\t\t\t\t\tcase 'l':\n\t\t\t\t\t\t\t\treturn '(\\\\d?\\\\d?\\\\d)';\n\t\t\t\t\t\t\tcase 'c':\n\t\t\t\t\t\t\t\treturn '(\\\\d?\\\\d?\\\\d)';\n\t\t\t\t\t\t\tcase 'z':\n\t\t\t\t\t\t\t\treturn '(z|[-+]\\\\d\\\\d:?\\\\d\\\\d|\\\\S+)?';\n\t\t\t\t\t\t\tcase 't':\n\t\t\t\t\t\t\t\treturn getPatternAmpm(o.amNames, o.pmNames);\n\t\t\t\t\t\t\tdefault:    // literal escaped in quotes\n\t\t\t\t\t\t\t\treturn '(' + match.replace(/\\'/g, \"\").replace(/(\\.|\\$|\\^|\\\\|\\/|\\(|\\)|\\[|\\]|\\?|\\+|\\*)/g, function (m) { return \"\\\\\" + m; }) + ')?';\n\t\t\t\t\t\t\t}\n\t\t\t\t\t\t})\n\t\t\t\t\t.replace(/\\s/g, '\\\\s?') +\n\t\t\t\t\to.timeSuffix + '$',\n\t\t\t\torder = getFormatPositions(f),\n\t\t\t\tampm = '',\n\t\t\t\ttreg;\n\n\t\t\ttreg = s.match(new RegExp(regstr, 'i'));\n\n\t\t\tvar resTime = {\n\t\t\t\thour: 0,\n\t\t\t\tminute: 0,\n\t\t\t\tsecond: 0,\n\t\t\t\tmillisec: 0,\n\t\t\t\tmicrosec: 0\n\t\t\t};\n\n\t\t\tif (treg) {\n\t\t\t\tif (order.t !== -1) {\n\t\t\t\t\tif (treg[order.t] === undefined || treg[order.t].length === 0) {\n\t\t\t\t\t\tampm = '';\n\t\t\t\t\t\tresTime.ampm = '';\n\t\t\t\t\t} else {\n\t\t\t\t\t\tampm = $.inArray(treg[order.t].toUpperCase(), o.amNames) !== -1 ? 'AM' : 'PM';\n\t\t\t\t\t\tresTime.ampm = o[ampm === 'AM' ? 'amNames' : 'pmNames'][0];\n\t\t\t\t\t}\n\t\t\t\t}\n\n\t\t\t\tif (order.h !== -1) {\n\t\t\t\t\tif (ampm === 'AM' && treg[order.h] === '12') {\n\t\t\t\t\t\tresTime.hour = 0; // 12am = 0 hour\n\t\t\t\t\t} else {\n\t\t\t\t\t\tif (ampm === 'PM' && treg[order.h] !== '12') {\n\t\t\t\t\t\t\tresTime.hour = parseInt(treg[order.h], 10) + 12; // 12pm = 12 hour, any other pm = hour + 12\n\t\t\t\t\t\t} else {\n\t\t\t\t\t\t\tresTime.hour = Number(treg[order.h]);\n\t\t\t\t\t\t}\n\t\t\t\t\t}\n\t\t\t\t}\n\n\t\t\t\tif (order.m !== -1) {\n\t\t\t\t\tresTime.minute = Number(treg[order.m]);\n\t\t\t\t}\n\t\t\t\tif (order.s !== -1) {\n\t\t\t\t\tresTime.second = Number(treg[order.s]);\n\t\t\t\t}\n\t\t\t\tif (order.l !== -1) {\n\t\t\t\t\tresTime.millisec = Number(treg[order.l]);\n\t\t\t\t}\n\t\t\t\tif (order.c !== -1) {\n\t\t\t\t\tresTime.microsec = Number(treg[order.c]);\n\t\t\t\t}\n\t\t\t\tif (order.z !== -1 && treg[order.z] !== undefined) {\n\t\t\t\t\tresTime.timezone = $.timepicker.timezoneOffsetNumber(treg[order.z]);\n\t\t\t\t}\n\n\n\t\t\t\treturn resTime;\n\t\t\t}\n\t\t\treturn false;\n\t\t};// end strictParse\n\n\t\t// First try JS Date, if that fails, use strictParse\n\t\tvar looseParse = function (f, s, o) {\n\t\t\ttry {\n\t\t\t\tvar d = new Date('2012-01-01 ' + s);\n\t\t\t\tif (isNaN(d.getTime())) {\n\t\t\t\t\td = new Date('2012-01-01T' + s);\n\t\t\t\t\tif (isNaN(d.getTime())) {\n\t\t\t\t\t\td = new Date('01/01/2012 ' + s);\n\t\t\t\t\t\tif (isNaN(d.getTime())) {\n\t\t\t\t\t\t\tthrow \"Unable to parse time with native Date: \" + s;\n\t\t\t\t\t\t}\n\t\t\t\t\t}\n\t\t\t\t}\n\n\t\t\t\treturn {\n\t\t\t\t\thour: d.getHours(),\n\t\t\t\t\tminute: d.getMinutes(),\n\t\t\t\t\tsecond: d.getSeconds(),\n\t\t\t\t\tmillisec: d.getMilliseconds(),\n\t\t\t\t\tmicrosec: d.getMicroseconds(),\n\t\t\t\t\ttimezone: d.getTimezoneOffset() * -1\n\t\t\t\t};\n\t\t\t}\n\t\t\tcatch (err) {\n\t\t\t\ttry {\n\t\t\t\t\treturn strictParse(f, s, o);\n\t\t\t\t}\n\t\t\t\tcatch (err2) {\n\t\t\t\t\t$.timepicker.log(\"Unable to parse \\ntimeString: \" + s + \"\\ntimeFormat: \" + f);\n\t\t\t\t}\t\t\t\t\n\t\t\t}\n\t\t\treturn false;\n\t\t}; // end looseParse\n\t\t\n\t\tif (typeof o.parse === \"function\") {\n\t\t\treturn o.parse(timeFormat, timeString, o);\n\t\t}\n\t\tif (o.parse === 'loose') {\n\t\t\treturn looseParse(timeFormat, timeString, o);\n\t\t}\n\t\treturn strictParse(timeFormat, timeString, o);\n\t};\n\n\t/**\n\t * Public utility to format the time\n\t * @param {string} format format of the time\n\t * @param {Object} time Object not a Date for timezones\n\t * @param {Object} [options] essentially the regional[].. amNames, pmNames, ampm\n\t * @returns {string} the formatted time\n\t */\n\t$.datepicker.formatTime = function (format, time, options) {\n\t\toptions = options || {};\n\t\toptions = $.extend({}, $.timepicker._defaults, options);\n\t\ttime = $.extend({\n\t\t\thour: 0,\n\t\t\tminute: 0,\n\t\t\tsecond: 0,\n\t\t\tmillisec: 0,\n\t\t\tmicrosec: 0,\n\t\t\ttimezone: null\n\t\t}, time);\n\n\t\tvar tmptime = format,\n\t\t\tampmName = options.amNames[0],\n\t\t\thour = parseInt(time.hour, 10);\n\n\t\tif (hour > 11) {\n\t\t\tampmName = options.pmNames[0];\n\t\t}\n\n\t\ttmptime = tmptime.replace(/(?:HH?|hh?|mm?|ss?|[tT]{1,2}|[zZ]|[lc]|'.*?')/g, function (match) {\n\t\t\tswitch (match) {\n\t\t\tcase 'HH':\n\t\t\t\treturn ('0' + hour).slice(-2);\n\t\t\tcase 'H':\n\t\t\t\treturn hour;\n\t\t\tcase 'hh':\n\t\t\t\treturn ('0' + convert24to12(hour)).slice(-2);\n\t\t\tcase 'h':\n\t\t\t\treturn convert24to12(hour);\n\t\t\tcase 'mm':\n\t\t\t\treturn ('0' + time.minute).slice(-2);\n\t\t\tcase 'm':\n\t\t\t\treturn time.minute;\n\t\t\tcase 'ss':\n\t\t\t\treturn ('0' + time.second).slice(-2);\n\t\t\tcase 's':\n\t\t\t\treturn time.second;\n\t\t\tcase 'l':\n\t\t\t\treturn ('00' + time.millisec).slice(-3);\n\t\t\tcase 'c':\n\t\t\t\treturn ('00' + time.microsec).slice(-3);\n\t\t\tcase 'z':\n\t\t\t\treturn $.timepicker.timezoneOffsetString(time.timezone === null ? options.timezone : time.timezone, false);\n\t\t\tcase 'Z':\n\t\t\t\treturn $.timepicker.timezoneOffsetString(time.timezone === null ? options.timezone : time.timezone, true);\n\t\t\tcase 'T':\n\t\t\t\treturn ampmName.charAt(0).toUpperCase();\n\t\t\tcase 'TT':\n\t\t\t\treturn ampmName.toUpperCase();\n\t\t\tcase 't':\n\t\t\t\treturn ampmName.charAt(0).toLowerCase();\n\t\t\tcase 'tt':\n\t\t\t\treturn ampmName.toLowerCase();\n\t\t\tdefault:\n\t\t\t\treturn match.replace(/'/g, \"\");\n\t\t\t}\n\t\t});\n\n\t\treturn tmptime;\n\t};\n\n\t/*\n\t* the bad hack :/ override datepicker so it doesn't close on select\n\t// inspired: http://stackoverflow.com/questions/1252512/jquery-datepicker-prevent-closing-picker-when-clicking-a-date/1762378#1762378\n\t*/\n\t$.datepicker._base_selectDate = $.datepicker._selectDate;\n\t$.datepicker._selectDate = function (id, dateStr) {\n\t\tvar inst = this._getInst($(id)[0]),\n\t\t\ttp_inst = this._get(inst, 'timepicker');\n\n\t\tif (tp_inst) {\n\t\t\ttp_inst._limitMinMaxDateTime(inst, true);\n\t\t\tinst.inline = inst.stay_open = true;\n\t\t\t//This way the onSelect handler called from calendarpicker get the full dateTime\n\t\t\tthis._base_selectDate(id, dateStr);\n\t\t\tinst.inline = inst.stay_open = false;\n\t\t\tthis._notifyChange(inst);\n\t\t\tthis._updateDatepicker(inst);\n\t\t} else {\n\t\t\tthis._base_selectDate(id, dateStr);\n\t\t}\n\t};\n\n\t/*\n\t* second bad hack :/ override datepicker so it triggers an event when changing the input field\n\t* and does not redraw the datepicker on every selectDate event\n\t*/\n\t$.datepicker._base_updateDatepicker = $.datepicker._updateDatepicker;\n\t$.datepicker._updateDatepicker = function (inst) {\n\n\t\t// don't popup the datepicker if there is another instance already opened\n\t\tvar input = inst.input[0];\n\t\tif ($.datepicker._curInst && $.datepicker._curInst !== inst && $.datepicker._datepickerShowing && $.datepicker._lastInput !== input) {\n\t\t\treturn;\n\t\t}\n\n\t\tif (typeof(inst.stay_open) !== 'boolean' || inst.stay_open === false) {\n\n\t\t\tthis._base_updateDatepicker(inst);\n\n\t\t\t// Reload the time control when changing something in the input text field.\n\t\t\tvar tp_inst = this._get(inst, 'timepicker');\n\t\t\tif (tp_inst) {\n\t\t\t\ttp_inst._addTimePicker(inst);\n\t\t\t}\n\t\t}\n\t};\n\n\t/*\n\t* third bad hack :/ override datepicker so it allows spaces and colon in the input field\n\t*/\n\t$.datepicker._base_doKeyPress = $.datepicker._doKeyPress;\n\t$.datepicker._doKeyPress = function (event) {\n\t\tvar inst = $.datepicker._getInst(event.target),\n\t\t\ttp_inst = $.datepicker._get(inst, 'timepicker');\n\n\t\tif (tp_inst) {\n\t\t\tif ($.datepicker._get(inst, 'constrainInput')) {\n\t\t\t\tvar ampm = tp_inst.support.ampm,\n\t\t\t\t\ttz = tp_inst._defaults.showTimezone !== null ? tp_inst._defaults.showTimezone : tp_inst.support.timezone,\n\t\t\t\t\tdateChars = $.datepicker._possibleChars($.datepicker._get(inst, 'dateFormat')),\n\t\t\t\t\tdatetimeChars = tp_inst._defaults.timeFormat.toString()\n\t\t\t\t\t\t\t\t\t\t\t.replace(/[hms]/g, '')\n\t\t\t\t\t\t\t\t\t\t\t.replace(/TT/g, ampm ? 'APM' : '')\n\t\t\t\t\t\t\t\t\t\t\t.replace(/Tt/g, ampm ? 'AaPpMm' : '')\n\t\t\t\t\t\t\t\t\t\t\t.replace(/tT/g, ampm ? 'AaPpMm' : '')\n\t\t\t\t\t\t\t\t\t\t\t.replace(/T/g, ampm ? 'AP' : '')\n\t\t\t\t\t\t\t\t\t\t\t.replace(/tt/g, ampm ? 'apm' : '')\n\t\t\t\t\t\t\t\t\t\t\t.replace(/t/g, ampm ? 'ap' : '') + \n\t\t\t\t\t\t\t\t\t\t\t\" \" + tp_inst._defaults.separator + \n\t\t\t\t\t\t\t\t\t\t\ttp_inst._defaults.timeSuffix + \n\t\t\t\t\t\t\t\t\t\t\t(tz ? tp_inst._defaults.timezoneList.join('') : '') + \n\t\t\t\t\t\t\t\t\t\t\t(tp_inst._defaults.amNames.join('')) + (tp_inst._defaults.pmNames.join('')) + \n\t\t\t\t\t\t\t\t\t\t\tdateChars,\n\t\t\t\t\tchr = String.fromCharCode(event.charCode === undefined ? event.keyCode : event.charCode);\n\t\t\t\treturn event.ctrlKey || (chr < ' ' || !dateChars || datetimeChars.indexOf(chr) > -1);\n\t\t\t}\n\t\t}\n\n\t\treturn $.datepicker._base_doKeyPress(event);\n\t};\n\n\t/*\n\t* Fourth bad hack :/ override _updateAlternate function used in inline mode to init altField\n\t* Update any alternate field to synchronise with the main field.\n\t*/\n\t$.datepicker._base_updateAlternate = $.datepicker._updateAlternate;\n\t$.datepicker._updateAlternate = function (inst) {\n\t\tvar tp_inst = this._get(inst, 'timepicker');\n\t\tif (tp_inst) {\n\t\t\tvar altField = tp_inst._defaults.altField;\n\t\t\tif (altField) { // update alternate field too\n\t\t\t\tvar altFormat = tp_inst._defaults.altFormat || tp_inst._defaults.dateFormat,\n\t\t\t\t\tdate = this._getDate(inst),\n\t\t\t\t\tformatCfg = $.datepicker._getFormatConfig(inst),\n\t\t\t\t\taltFormattedDateTime = '', \n\t\t\t\t\taltSeparator = tp_inst._defaults.altSeparator ? tp_inst._defaults.altSeparator : tp_inst._defaults.separator, \n\t\t\t\t\taltTimeSuffix = tp_inst._defaults.altTimeSuffix ? tp_inst._defaults.altTimeSuffix : tp_inst._defaults.timeSuffix,\n\t\t\t\t\taltTimeFormat = tp_inst._defaults.altTimeFormat !== null ? tp_inst._defaults.altTimeFormat : tp_inst._defaults.timeFormat;\n\t\t\t\t\n\t\t\t\taltFormattedDateTime += $.datepicker.formatTime(altTimeFormat, tp_inst, tp_inst._defaults) + altTimeSuffix;\n\t\t\t\tif (!tp_inst._defaults.timeOnly && !tp_inst._defaults.altFieldTimeOnly && date !== null) {\n\t\t\t\t\tif (tp_inst._defaults.altFormat) {\n\t\t\t\t\t\taltFormattedDateTime = $.datepicker.formatDate(tp_inst._defaults.altFormat, date, formatCfg) + altSeparator + altFormattedDateTime;\n\t\t\t\t\t}\n\t\t\t\t\telse {\n\t\t\t\t\t\taltFormattedDateTime = tp_inst.formattedDate + altSeparator + altFormattedDateTime;\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t\t$(altField).val(altFormattedDateTime);\n\t\t\t}\n\t\t}\n\t\telse {\n\t\t\t$.datepicker._base_updateAlternate(inst);\n\t\t}\n\t};\n\n\t/*\n\t* Override key up event to sync manual input changes.\n\t*/\n\t$.datepicker._base_doKeyUp = $.datepicker._doKeyUp;\n\t$.datepicker._doKeyUp = function (event) {\n\t\tvar inst = $.datepicker._getInst(event.target),\n\t\t\ttp_inst = $.datepicker._get(inst, 'timepicker');\n\n\t\tif (tp_inst) {\n\t\t\tif (tp_inst._defaults.timeOnly && (inst.input.val() !== inst.lastVal)) {\n\t\t\t\ttry {\n\t\t\t\t\t$.datepicker._updateDatepicker(inst);\n\t\t\t\t} catch (err) {\n\t\t\t\t\t$.timepicker.log(err);\n\t\t\t\t}\n\t\t\t}\n\t\t}\n\n\t\treturn $.datepicker._base_doKeyUp(event);\n\t};\n\n\t/*\n\t* override \"Today\" button to also grab the time.\n\t*/\n\t$.datepicker._base_gotoToday = $.datepicker._gotoToday;\n\t$.datepicker._gotoToday = function (id) {\n\t\tvar inst = this._getInst($(id)[0]),\n\t\t\t$dp = inst.dpDiv;\n\t\tthis._base_gotoToday(id);\n\t\tvar tp_inst = this._get(inst, 'timepicker');\n\t\tselectLocalTimezone(tp_inst);\n\t\tvar now = new Date();\n\t\tthis._setTime(inst, now);\n\t\t$('.ui-datepicker-today', $dp).click();\n\t};\n\n\t/*\n\t* Disable & enable the Time in the datetimepicker\n\t*/\n\t$.datepicker._disableTimepickerDatepicker = function (target) {\n\t\tvar inst = this._getInst(target);\n\t\tif (!inst) {\n\t\t\treturn;\n\t\t}\n\n\t\tvar tp_inst = this._get(inst, 'timepicker');\n\t\t$(target).datepicker('getDate'); // Init selected[Year|Month|Day]\n\t\tif (tp_inst) {\n\t\t\tinst.settings.showTimepicker = false;\n\t\t\ttp_inst._defaults.showTimepicker = false;\n\t\t\ttp_inst._updateDateTime(inst);\n\t\t}\n\t};\n\n\t$.datepicker._enableTimepickerDatepicker = function (target) {\n\t\tvar inst = this._getInst(target);\n\t\tif (!inst) {\n\t\t\treturn;\n\t\t}\n\n\t\tvar tp_inst = this._get(inst, 'timepicker');\n\t\t$(target).datepicker('getDate'); // Init selected[Year|Month|Day]\n\t\tif (tp_inst) {\n\t\t\tinst.settings.showTimepicker = true;\n\t\t\ttp_inst._defaults.showTimepicker = true;\n\t\t\ttp_inst._addTimePicker(inst); // Could be disabled on page load\n\t\t\ttp_inst._updateDateTime(inst);\n\t\t}\n\t};\n\n\t/*\n\t* Create our own set time function\n\t*/\n\t$.datepicker._setTime = function (inst, date) {\n\t\tvar tp_inst = this._get(inst, 'timepicker');\n\t\tif (tp_inst) {\n\t\t\tvar defaults = tp_inst._defaults;\n\n\t\t\t// calling _setTime with no date sets time to defaults\n\t\t\ttp_inst.hour = date ? date.getHours() : defaults.hour;\n\t\t\ttp_inst.minute = date ? date.getMinutes() : defaults.minute;\n\t\t\ttp_inst.second = date ? date.getSeconds() : defaults.second;\n\t\t\ttp_inst.millisec = date ? date.getMilliseconds() : defaults.millisec;\n\t\t\ttp_inst.microsec = date ? date.getMicroseconds() : defaults.microsec;\n\n\t\t\t//check if within min/max times.. \n\t\t\ttp_inst._limitMinMaxDateTime(inst, true);\n\n\t\t\ttp_inst._onTimeChange();\n\t\t\ttp_inst._updateDateTime(inst);\n\t\t}\n\t};\n\n\t/*\n\t* Create new public method to set only time, callable as $().datepicker('setTime', date)\n\t*/\n\t$.datepicker._setTimeDatepicker = function (target, date, withDate) {\n\t\tvar inst = this._getInst(target);\n\t\tif (!inst) {\n\t\t\treturn;\n\t\t}\n\n\t\tvar tp_inst = this._get(inst, 'timepicker');\n\n\t\tif (tp_inst) {\n\t\t\tthis._setDateFromField(inst);\n\t\t\tvar tp_date;\n\t\t\tif (date) {\n\t\t\t\tif (typeof date === \"string\") {\n\t\t\t\t\ttp_inst._parseTime(date, withDate);\n\t\t\t\t\ttp_date = new Date();\n\t\t\t\t\ttp_date.setHours(tp_inst.hour, tp_inst.minute, tp_inst.second, tp_inst.millisec);\n\t\t\t\t\ttp_date.setMicroseconds(tp_inst.microsec);\n\t\t\t\t} else {\n\t\t\t\t\ttp_date = new Date(date.getTime());\n\t\t\t\t\ttp_date.setMicroseconds(date.getMicroseconds());\n\t\t\t\t}\n\t\t\t\tif (tp_date.toString() === 'Invalid Date') {\n\t\t\t\t\ttp_date = undefined;\n\t\t\t\t}\n\t\t\t\tthis._setTime(inst, tp_date);\n\t\t\t}\n\t\t}\n\n\t};\n\n\t/*\n\t* override setDate() to allow setting time too within Date object\n\t*/\n\t$.datepicker._base_setDateDatepicker = $.datepicker._setDateDatepicker;\n\t$.datepicker._setDateDatepicker = function (target, date) {\n\t\tvar inst = this._getInst(target);\n\t\tif (!inst) {\n\t\t\treturn;\n\t\t}\n\n\t\tif (typeof(date) === 'string') {\n\t\t\tdate = new Date(date);\n\t\t\tif (!date.getTime()) {\n\t\t\t\t$.timepicker.log(\"Error creating Date object from string.\");\n\t\t\t}\n\t\t}\n\n\t\tvar tp_inst = this._get(inst, 'timepicker');\n\t\tvar tp_date;\n\t\tif (date instanceof Date) {\n\t\t\ttp_date = new Date(date.getTime());\n\t\t\ttp_date.setMicroseconds(date.getMicroseconds());\n\t\t} else {\n\t\t\ttp_date = date;\n\t\t}\n\t\t\n\t\t// This is important if you are using the timezone option, javascript's Date \n\t\t// object will only return the timezone offset for the current locale, so we \n\t\t// adjust it accordingly.  If not using timezone option this won't matter..\n\t\t// If a timezone is different in tp, keep the timezone as is\n\t\tif (tp_inst && tp_date) {\n\t\t\t// look out for DST if tz wasn't specified\n\t\t\tif (!tp_inst.support.timezone && tp_inst._defaults.timezone === null) {\n\t\t\t\ttp_inst.timezone = tp_date.getTimezoneOffset() * -1;\n\t\t\t}\n\t\t\tdate = $.timepicker.timezoneAdjust(date, tp_inst.timezone);\n\t\t\ttp_date = $.timepicker.timezoneAdjust(tp_date, tp_inst.timezone);\n\t\t}\n\n\t\tthis._updateDatepicker(inst);\n\t\tthis._base_setDateDatepicker.apply(this, arguments);\n\t\tthis._setTimeDatepicker(target, tp_date, true);\n\t};\n\n\t/*\n\t* override getDate() to allow getting time too within Date object\n\t*/\n\t$.datepicker._base_getDateDatepicker = $.datepicker._getDateDatepicker;\n\t$.datepicker._getDateDatepicker = function (target, noDefault) {\n\t\tvar inst = this._getInst(target);\n\t\tif (!inst) {\n\t\t\treturn;\n\t\t}\n\n\t\tvar tp_inst = this._get(inst, 'timepicker');\n\n\t\tif (tp_inst) {\n\t\t\t// if it hasn't yet been defined, grab from field\n\t\t\tif (inst.lastVal === undefined) {\n\t\t\t\tthis._setDateFromField(inst, noDefault);\n\t\t\t}\n\n\t\t\tvar date = this._getDate(inst);\n\t\t\tif (date && tp_inst._parseTime($(target).val(), tp_inst.timeOnly)) {\n\t\t\t\tdate.setHours(tp_inst.hour, tp_inst.minute, tp_inst.second, tp_inst.millisec);\n\t\t\t\tdate.setMicroseconds(tp_inst.microsec);\n\n\t\t\t\t// This is important if you are using the timezone option, javascript's Date \n\t\t\t\t// object will only return the timezone offset for the current locale, so we \n\t\t\t\t// adjust it accordingly.  If not using timezone option this won't matter..\n\t\t\t\tif (tp_inst.timezone != null) {\n\t\t\t\t\t// look out for DST if tz wasn't specified\n\t\t\t\t\tif (!tp_inst.support.timezone && tp_inst._defaults.timezone === null) {\n\t\t\t\t\t\ttp_inst.timezone = date.getTimezoneOffset() * -1;\n\t\t\t\t\t}\n\t\t\t\t\tdate = $.timepicker.timezoneAdjust(date, tp_inst.timezone);\n\t\t\t\t}\n\t\t\t}\n\t\t\treturn date;\n\t\t}\n\t\treturn this._base_getDateDatepicker(target, noDefault);\n\t};\n\n\t/*\n\t* override parseDate() because UI 1.8.14 throws an error about \"Extra characters\"\n\t* An option in datapicker to ignore extra format characters would be nicer.\n\t*/\n\t$.datepicker._base_parseDate = $.datepicker.parseDate;\n\t$.datepicker.parseDate = function (format, value, settings) {\n\t\tvar date;\n\t\ttry {\n\t\t\tdate = this._base_parseDate(format, value, settings);\n\t\t} catch (err) {\n\t\t\t// Hack!  The error message ends with a colon, a space, and\n\t\t\t// the \"extra\" characters.  We rely on that instead of\n\t\t\t// attempting to perfectly reproduce the parsing algorithm.\n\t\t\tif (err.indexOf(\":\") >= 0) {\n\t\t\t\tdate = this._base_parseDate(format, value.substring(0, value.length - (err.length - err.indexOf(':') - 2)), settings);\n\t\t\t\t$.timepicker.log(\"Error parsing the date string: \" + err + \"\\ndate string = \" + value + \"\\ndate format = \" + format);\n\t\t\t} else {\n\t\t\t\tthrow err;\n\t\t\t}\n\t\t}\n\t\treturn date;\n\t};\n\n\t/*\n\t* override formatDate to set date with time to the input\n\t*/\n\t$.datepicker._base_formatDate = $.datepicker._formatDate;\n\t$.datepicker._formatDate = function (inst, day, month, year) {\n\t\tvar tp_inst = this._get(inst, 'timepicker');\n\t\tif (tp_inst) {\n\t\t\ttp_inst._updateDateTime(inst);\n\t\t\treturn tp_inst.$input.val();\n\t\t}\n\t\treturn this._base_formatDate(inst);\n\t};\n\n\t/*\n\t* override options setter to add time to maxDate(Time) and minDate(Time). MaxDate\n\t*/\n\t$.datepicker._base_optionDatepicker = $.datepicker._optionDatepicker;\n\t$.datepicker._optionDatepicker = function (target, name, value) {\n\t\tvar inst = this._getInst(target),\n\t\t\tname_clone;\n\t\tif (!inst) {\n\t\t\treturn null;\n\t\t}\n\n\t\tvar tp_inst = this._get(inst, 'timepicker');\n\t\tif (tp_inst) {\n\t\t\tvar min = null,\n\t\t\t\tmax = null,\n\t\t\t\tonselect = null,\n\t\t\t\toverrides = tp_inst._defaults.evnts,\n\t\t\t\tfns = {},\n\t\t\t\tprop;\n\t\t\tif (typeof name === 'string') { // if min/max was set with the string\n\t\t\t\tif (name === 'minDate' || name === 'minDateTime') {\n\t\t\t\t\tmin = value;\n\t\t\t\t} else if (name === 'maxDate' || name === 'maxDateTime') {\n\t\t\t\t\tmax = value;\n\t\t\t\t} else if (name === 'onSelect') {\n\t\t\t\t\tonselect = value;\n\t\t\t\t} else if (overrides.hasOwnProperty(name)) {\n\t\t\t\t\tif (typeof (value) === 'undefined') {\n\t\t\t\t\t\treturn overrides[name];\n\t\t\t\t\t}\n\t\t\t\t\tfns[name] = value;\n\t\t\t\t\tname_clone = {}; //empty results in exiting function after overrides updated\n\t\t\t\t}\n\t\t\t} else if (typeof name === 'object') { //if min/max was set with the JSON\n\t\t\t\tif (name.minDate) {\n\t\t\t\t\tmin = name.minDate;\n\t\t\t\t} else if (name.minDateTime) {\n\t\t\t\t\tmin = name.minDateTime;\n\t\t\t\t} else if (name.maxDate) {\n\t\t\t\t\tmax = name.maxDate;\n\t\t\t\t} else if (name.maxDateTime) {\n\t\t\t\t\tmax = name.maxDateTime;\n\t\t\t\t}\n\t\t\t\tfor (prop in overrides) {\n\t\t\t\t\tif (overrides.hasOwnProperty(prop) && name[prop]) {\n\t\t\t\t\t\tfns[prop] = name[prop];\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t}\n\t\t\tfor (prop in fns) {\n\t\t\t\tif (fns.hasOwnProperty(prop)) {\n\t\t\t\t\toverrides[prop] = fns[prop];\n\t\t\t\t\tif (!name_clone) { name_clone = $.extend({}, name); }\n\t\t\t\t\tdelete name_clone[prop];\n\t\t\t\t}\n\t\t\t}\n\t\t\tif (name_clone && isEmptyObject(name_clone)) { return; }\n\t\t\tif (min) { //if min was set\n\t\t\t\tif (min === 0) {\n\t\t\t\t\tmin = new Date();\n\t\t\t\t} else {\n\t\t\t\t\tmin = new Date(min);\n\t\t\t\t}\n\t\t\t\ttp_inst._defaults.minDate = min;\n\t\t\t\ttp_inst._defaults.minDateTime = min;\n\t\t\t} else if (max) { //if max was set\n\t\t\t\tif (max === 0) {\n\t\t\t\t\tmax = new Date();\n\t\t\t\t} else {\n\t\t\t\t\tmax = new Date(max);\n\t\t\t\t}\n\t\t\t\ttp_inst._defaults.maxDate = max;\n\t\t\t\ttp_inst._defaults.maxDateTime = max;\n\t\t\t} else if (onselect) {\n\t\t\t\ttp_inst._defaults.onSelect = onselect;\n\t\t\t}\n\t\t}\n\t\tif (value === undefined) {\n\t\t\treturn this._base_optionDatepicker.call($.datepicker, target, name);\n\t\t}\n\t\treturn this._base_optionDatepicker.call($.datepicker, target, name_clone || name, value);\n\t};\n\t\n\t/*\n\t* jQuery isEmptyObject does not check hasOwnProperty - if someone has added to the object prototype,\n\t* it will return false for all objects\n\t*/\n\tvar isEmptyObject = function (obj) {\n\t\tvar prop;\n\t\tfor (prop in obj) {\n\t\t\tif (obj.hasOwnProperty(prop)) {\n\t\t\t\treturn false;\n\t\t\t}\n\t\t}\n\t\treturn true;\n\t};\n\n\t/*\n\t* jQuery extend now ignores nulls!\n\t*/\n\tvar extendRemove = function (target, props) {\n\t\t$.extend(target, props);\n\t\tfor (var name in props) {\n\t\t\tif (props[name] === null || props[name] === undefined) {\n\t\t\t\ttarget[name] = props[name];\n\t\t\t}\n\t\t}\n\t\treturn target;\n\t};\n\n\t/*\n\t* Determine by the time format which units are supported\n\t* Returns an object of booleans for each unit\n\t*/\n\tvar detectSupport = function (timeFormat) {\n\t\tvar tf = timeFormat.replace(/'.*?'/g, '').toLowerCase(), // removes literals\n\t\t\tisIn = function (f, t) { // does the format contain the token?\n\t\t\t\t\treturn f.indexOf(t) !== -1 ? true : false;\n\t\t\t\t};\n\t\treturn {\n\t\t\t\thour: isIn(tf, 'h'),\n\t\t\t\tminute: isIn(tf, 'm'),\n\t\t\t\tsecond: isIn(tf, 's'),\n\t\t\t\tmillisec: isIn(tf, 'l'),\n\t\t\t\tmicrosec: isIn(tf, 'c'),\n\t\t\t\ttimezone: isIn(tf, 'z'),\n\t\t\t\tampm: isIn(tf, 't') && isIn(timeFormat, 'h'),\n\t\t\t\tiso8601: isIn(timeFormat, 'Z')\n\t\t\t};\n\t};\n\n\t/*\n\t* Converts 24 hour format into 12 hour\n\t* Returns 12 hour without leading 0\n\t*/\n\tvar convert24to12 = function (hour) {\n\t\thour %= 12;\n\n\t\tif (hour === 0) {\n\t\t\thour = 12;\n\t\t}\n\n\t\treturn String(hour);\n\t};\n\n\tvar computeEffectiveSetting = function (settings, property) {\n\t\treturn settings && settings[property] ? settings[property] : $.timepicker._defaults[property];\n\t};\n\n\t/*\n\t* Splits datetime string into date and time substrings.\n\t* Throws exception when date can't be parsed\n\t* Returns {dateString: dateString, timeString: timeString}\n\t*/\n\tvar splitDateTime = function (dateTimeString, timeSettings) {\n\t\t// The idea is to get the number separator occurrences in datetime and the time format requested (since time has\n\t\t// fewer unknowns, mostly numbers and am/pm). We will use the time pattern to split.\n\t\tvar separator = computeEffectiveSetting(timeSettings, 'separator'),\n\t\t\tformat = computeEffectiveSetting(timeSettings, 'timeFormat'),\n\t\t\ttimeParts = format.split(separator), // how many occurrences of separator may be in our format?\n\t\t\ttimePartsLen = timeParts.length,\n\t\t\tallParts = dateTimeString.split(separator),\n\t\t\tallPartsLen = allParts.length;\n\n\t\tif (allPartsLen > 1) {\n\t\t\treturn {\n\t\t\t\tdateString: allParts.splice(0, allPartsLen - timePartsLen).join(separator),\n\t\t\t\ttimeString: allParts.splice(0, timePartsLen).join(separator)\n\t\t\t};\n\t\t}\n\n\t\treturn {\n\t\t\tdateString: dateTimeString,\n\t\t\ttimeString: ''\n\t\t};\n\t};\n\n\t/*\n\t* Internal function to parse datetime interval\n\t* Returns: {date: Date, timeObj: Object}, where\n\t*   date - parsed date without time (type Date)\n\t*   timeObj = {hour: , minute: , second: , millisec: , microsec: } - parsed time. Optional\n\t*/\n\tvar parseDateTimeInternal = function (dateFormat, timeFormat, dateTimeString, dateSettings, timeSettings) {\n\t\tvar date,\n\t\t\tparts,\n\t\t\tparsedTime;\n\n\t\tparts = splitDateTime(dateTimeString, timeSettings);\n\t\tdate = $.datepicker._base_parseDate(dateFormat, parts.dateString, dateSettings);\n\n\t\tif (parts.timeString === '') {\n\t\t\treturn {\n\t\t\t\tdate: date\n\t\t\t};\n\t\t}\n\n\t\tparsedTime = $.datepicker.parseTime(timeFormat, parts.timeString, timeSettings);\n\n\t\tif (!parsedTime) {\n\t\t\tthrow 'Wrong time format';\n\t\t}\n\n\t\treturn {\n\t\t\tdate: date,\n\t\t\ttimeObj: parsedTime\n\t\t};\n\t};\n\n\t/*\n\t* Internal function to set timezone_select to the local timezone\n\t*/\n\tvar selectLocalTimezone = function (tp_inst, date) {\n\t\tif (tp_inst && tp_inst.timezone_select) {\n\t\t\tvar now = date || new Date();\n\t\t\ttp_inst.timezone_select.val(-now.getTimezoneOffset());\n\t\t}\n\t};\n\n\t/*\n\t* Create a Singleton Instance\n\t*/\n\t$.timepicker = new Timepicker();\n\n\t/**\n\t * Get the timezone offset as string from a date object (eg '+0530' for UTC+5.5)\n\t * @param {number} tzMinutes if not a number, less than -720 (-1200), or greater than 840 (+1400) this value is returned\n\t * @param {boolean} iso8601 if true formats in accordance to iso8601 \"+12:45\"\n\t * @return {string}\n\t */\n\t$.timepicker.timezoneOffsetString = function (tzMinutes, iso8601) {\n\t\tif (isNaN(tzMinutes) || tzMinutes > 840 || tzMinutes < -720) {\n\t\t\treturn tzMinutes;\n\t\t}\n\n\t\tvar off = tzMinutes,\n\t\t\tminutes = off % 60,\n\t\t\thours = (off - minutes) / 60,\n\t\t\tiso = iso8601 ? ':' : '',\n\t\t\ttz = (off >= 0 ? '+' : '-') + ('0' + Math.abs(hours)).slice(-2) + iso + ('0' + Math.abs(minutes)).slice(-2);\n\t\t\n\t\tif (tz === '+00:00') {\n\t\t\treturn 'Z';\n\t\t}\n\t\treturn tz;\n\t};\n\n\t/**\n\t * Get the number in minutes that represents a timezone string\n\t * @param  {string} tzString formatted like \"+0500\", \"-1245\", \"Z\"\n\t * @return {number} the offset minutes or the original string if it doesn't match expectations\n\t */\n\t$.timepicker.timezoneOffsetNumber = function (tzString) {\n\t\tvar normalized = tzString.toString().replace(':', ''); // excuse any iso8601, end up with \"+1245\"\n\n\t\tif (normalized.toUpperCase() === 'Z') { // if iso8601 with Z, its 0 minute offset\n\t\t\treturn 0;\n\t\t}\n\n\t\tif (!/^(\\-|\\+)\\d{4}$/.test(normalized)) { // possibly a user defined tz, so just give it back\n\t\t\treturn tzString;\n\t\t}\n\n\t\treturn ((normalized.substr(0, 1) === '-' ? -1 : 1) * // plus or minus\n\t\t\t\t\t((parseInt(normalized.substr(1, 2), 10) * 60) + // hours (converted to minutes)\n\t\t\t\t\tparseInt(normalized.substr(3, 2), 10))); // minutes\n\t};\n\n\t/**\n\t * No way to set timezone in js Date, so we must adjust the minutes to compensate. (think setDate, getDate)\n\t * @param  {Date} date\n\t * @param  {string} toTimezone formatted like \"+0500\", \"-1245\"\n\t * @return {Date}\n\t */\n\t$.timepicker.timezoneAdjust = function (date, toTimezone) {\n\t\tvar toTz = $.timepicker.timezoneOffsetNumber(toTimezone);\n\t\tif (!isNaN(toTz)) {\n\t\t\tdate.setMinutes(date.getMinutes() + -date.getTimezoneOffset() - toTz);\n\t\t}\n\t\treturn date;\n\t};\n\n\t/**\n\t * Calls `timepicker()` on the `startTime` and `endTime` elements, and configures them to\n\t * enforce date range limits.\n\t * n.b. The input value must be correctly formatted (reformatting is not supported)\n\t * @param  {Element} startTime\n\t * @param  {Element} endTime\n\t * @param  {Object} options Options for the timepicker() call\n\t * @return {jQuery}\n\t */\n\t$.timepicker.timeRange = function (startTime, endTime, options) {\n\t\treturn $.timepicker.handleRange('timepicker', startTime, endTime, options);\n\t};\n\n\t/**\n\t * Calls `datetimepicker` on the `startTime` and `endTime` elements, and configures them to\n\t * enforce date range limits.\n\t * @param  {Element} startTime\n\t * @param  {Element} endTime\n\t * @param  {Object} options Options for the `timepicker()` call. Also supports `reformat`,\n\t *   a boolean value that can be used to reformat the input values to the `dateFormat`.\n\t * @param  {string} method Can be used to specify the type of picker to be added\n\t * @return {jQuery}\n\t */\n\t$.timepicker.datetimeRange = function (startTime, endTime, options) {\n\t\t$.timepicker.handleRange('datetimepicker', startTime, endTime, options);\n\t};\n\n\t/**\n\t * Calls `datepicker` on the `startTime` and `endTime` elements, and configures them to\n\t * enforce date range limits.\n\t * @param  {Element} startTime\n\t * @param  {Element} endTime\n\t * @param  {Object} options Options for the `timepicker()` call. Also supports `reformat`,\n\t *   a boolean value that can be used to reformat the input values to the `dateFormat`.\n\t * @return {jQuery}\n\t */\n\t$.timepicker.dateRange = function (startTime, endTime, options) {\n\t\t$.timepicker.handleRange('datepicker', startTime, endTime, options);\n\t};\n\n\t/**\n\t * Calls `method` on the `startTime` and `endTime` elements, and configures them to\n\t * enforce date range limits.\n\t * @param  {string} method Can be used to specify the type of picker to be added\n\t * @param  {Element} startTime\n\t * @param  {Element} endTime\n\t * @param  {Object} options Options for the `timepicker()` call. Also supports `reformat`,\n\t *   a boolean value that can be used to reformat the input values to the `dateFormat`.\n\t * @return {jQuery}\n\t */\n\t$.timepicker.handleRange = function (method, startTime, endTime, options) {\n\t\toptions = $.extend({}, {\n\t\t\tminInterval: 0, // min allowed interval in milliseconds\n\t\t\tmaxInterval: 0, // max allowed interval in milliseconds\n\t\t\tstart: {},      // options for start picker\n\t\t\tend: {}         // options for end picker\n\t\t}, options);\n\n\t\tfunction checkDates(changed, other) {\n\t\t\tvar startdt = startTime[method]('getDate'),\n\t\t\t\tenddt = endTime[method]('getDate'),\n\t\t\t\tchangeddt = changed[method]('getDate');\n\n\t\t\tif (startdt !== null) {\n\t\t\t\tvar minDate = new Date(startdt.getTime()),\n\t\t\t\t\tmaxDate = new Date(startdt.getTime());\n\n\t\t\t\tminDate.setMilliseconds(minDate.getMilliseconds() + options.minInterval);\n\t\t\t\tmaxDate.setMilliseconds(maxDate.getMilliseconds() + options.maxInterval);\n\n\t\t\t\tif (options.minInterval > 0 && minDate > enddt) { // minInterval check\n\t\t\t\t\tendTime[method]('setDate', minDate);\n\t\t\t\t}\n\t\t\t\telse if (options.maxInterval > 0 && maxDate < enddt) { // max interval check\n\t\t\t\t\tendTime[method]('setDate', maxDate);\n\t\t\t\t}\n\t\t\t\telse if (startdt > enddt) {\n\t\t\t\t\tother[method]('setDate', changeddt);\n\t\t\t\t}\n\t\t\t}\n\t\t}\n\n\t\tfunction selected(changed, other, option) {\n\t\t\tif (!changed.val()) {\n\t\t\t\treturn;\n\t\t\t}\n\t\t\tvar date = changed[method].call(changed, 'getDate');\n\t\t\tif (date !== null && options.minInterval > 0) {\n\t\t\t\tif (option === 'minDate') {\n\t\t\t\t\tdate.setMilliseconds(date.getMilliseconds() + options.minInterval);\n\t\t\t\t}\n\t\t\t\tif (option === 'maxDate') {\n\t\t\t\t\tdate.setMilliseconds(date.getMilliseconds() - options.minInterval);\n\t\t\t\t}\n\t\t\t}\n\t\t\tif (date.getTime) {\n\t\t\t\tother[method].call(other, 'option', option, date);\n\t\t\t}\n\t\t}\n\n\t\t$.fn[method].call(startTime, $.extend({\n\t\t\tonClose: function (dateText, inst) {\n\t\t\t\tcheckDates($(this), endTime);\n\t\t\t},\n\t\t\tonSelect: function (selectedDateTime) {\n\t\t\t\tselected($(this), endTime, 'minDate');\n\t\t\t}\n\t\t}, options, options.start));\n\t\t$.fn[method].call(endTime, $.extend({\n\t\t\tonClose: function (dateText, inst) {\n\t\t\t\tcheckDates($(this), startTime);\n\t\t\t},\n\t\t\tonSelect: function (selectedDateTime) {\n\t\t\t\tselected($(this), startTime, 'maxDate');\n\t\t\t}\n\t\t}, options, options.end));\n\n\t\tcheckDates(startTime, endTime);\n\t\tselected(startTime, endTime, 'minDate');\n\t\tselected(endTime, startTime, 'maxDate');\n\t\treturn $([startTime.get(0), endTime.get(0)]);\n\t};\n\n\t/**\n\t * Log error or data to the console during error or debugging\n\t * @param  {Object} err pass any type object to log to the console during error or debugging\n\t * @return {void}\n\t */\n\t$.timepicker.log = function (err) {\n\t\tif (window.console) {\n\t\t\twindow.console.log(err);\n\t\t}\n\t};\n\n\t/*\n\t * Add util object to allow access to private methods for testability.\n\t */\n\t$.timepicker._util = {\n\t\t_extendRemove: extendRemove,\n\t\t_isEmptyObject: isEmptyObject,\n\t\t_convert24to12: convert24to12,\n\t\t_detectSupport: detectSupport,\n\t\t_selectLocalTimezone: selectLocalTimezone,\n\t\t_computeEffectiveSetting: computeEffectiveSetting,\n\t\t_splitDateTime: splitDateTime,\n\t\t_parseDateTimeInternal: parseDateTimeInternal\n\t};\n\n\t/*\n\t* Microsecond support\n\t*/\n\tif (!Date.prototype.getMicroseconds) {\n\t\tDate.prototype.microseconds = 0;\n\t\tDate.prototype.getMicroseconds = function () { return this.microseconds; };\n\t\tDate.prototype.setMicroseconds = function (m) {\n\t\t\tthis.setMilliseconds(this.getMilliseconds() + Math.floor(m / 1000));\n\t\t\tthis.microseconds = m % 1000;\n\t\t\treturn this;\n\t\t};\n\t}\n\n\t/*\n\t* Keep up with the version\n\t*/\n\t$.timepicker.version = \"1.4.3\";\n\n}));"}
}});
